import contextlib

from qgis.core import Qgis, QgsMessageLog, QgsPluginLayer, QgsProject
from qgis.PyQt.QtCore import QSignalBlocker


def _repr_plugin_layer(self: QgsPluginLayer):
    return f"{self.__class__.__name__}: '{self.name()}'"


def set_unique_name_for_layer(layer):
    name = layer.name()
    names = [
        lyr.name()
        for lyr in QgsProject.instance().mapLayers().values()
        if isinstance(lyr, QgsPluginLayer) and lyr.name() != name
    ]
    if name in names:
        i, unique = 1, name
        while unique in names:
            unique = f"{name} {i}"
            i += 1
        name = unique
    with QSignalBlocker(layer):
        layer.setName(name)
    return name


def connect(data_layer, layer):
    if data_layer is None:
        return
    disconnect(data_layer, layer)
    data_layer.editingStopped.connect(layer.update_dataset)
    # See also layersWillBeRemoved(layers)
    QgsProject.instance().layerStore().layersRemoved.connect(
        layer.on_data_layers_deletion
    )
    data_layer.crsChanged.connect(layer.update_dataset)
    data_layer.crs3DChanged.connect(layer.update_dataset)
    QgsMessageLog.logMessage(
        f"Connected {layer.name()} and {data_layer.name()}", level=Qgis.Info
    )


def disconnect(data_layer, layer):
    if data_layer is None:
        return
    with contextlib.suppress(Exception):
        data_layer.editingStopped.disconnect(layer.update_dataset)
    with contextlib.suppress(Exception):
        QgsProject.instance().layerStore().layersRemoved.disconnect(
            layer.on_data_layers_deletion
        )
    with contextlib.suppress(Exception):
        data_layer.crsChanged.disconnect(layer.update_dataset)
    with contextlib.suppress(Exception):
        data_layer.crs3DChanged.disconnect(layer.update_dataset)
    QgsMessageLog.logMessage(
        f"Disconnected {layer.name()} and {data_layer.name()}", level=Qgis.Info
    )
