import xml.etree.ElementTree as ET

from forgeo.io.xml import Serializer

from ...settings import PluginSettings
from ...utils import QgisVectorDataFilter
from .converters import converter
from .legacy.v0 import FiltersSerializer as v0Serializer


class UnknownXmlTagException(Exception):
    pass


class QgisVectorDataFilterSerializer(
    Serializer, target=QgisVectorDataFilter, tag="QgisVectorDataFilter"
):
    @classmethod
    def dump_element(cls, data_filter, e):
        if data_filter.layer_id:
            node = ET.Element("LayerID")
            node.text = data_filter.layer_id
            e.append(node)
        if data_filter.value:
            node = ET.Element("Value")
            node.text = data_filter.value
            e.append(node)
        if data_filter.expression:
            node = ET.Element("Expression")
            node.text = data_filter.expression
            e.append(node)
        if data_filter.dip_fields:
            dip_node = ET.Element("DipMeasurement")
            for key, value in data_filter.dip_fields.items():
                if value:  # Avoids dumping "ReversePolarity" = ""
                    node = ET.Element(key)
                    node.text = value
                    dip_node.append(node)
            e.append(dip_node)

    @classmethod
    def load_element(cls, e):
        layer_id = value = expression = dip_fields = None
        for elem in e:
            tag = elem.tag
            if tag == "LayerID":
                layer_id = elem.text
            elif tag == "Value":
                value = elem.text
            elif tag == "Expression":
                expression = elem.text
            elif tag == "DipMeasurement":
                dip_fields = {child.tag: child.text for child in elem}
            else:
                msg = f"In {cls.__name__}: {tag = }"
                raise UnknownXmlTagException(msg)
        return QgisVectorDataFilter(layer_id, value, expression, dip_fields)


class ModelFilters(dict):
    pass


class FiltersSerializer(Serializer, target=ModelFilters, tag="ModelFilters"):
    @classmethod
    def dump_element(cls, modelfilters, e):
        node = ET.Element("Version")
        node.text = PluginSettings().get("version")
        e.append(node)
        for name, dataselection in modelfilters.items():
            node = ET.Element("DataSelection")
            e.append(node)
            node.attrib["name"] = name
            for datafilter in dataselection:
                node.append(QgisVectorDataFilterSerializer.dump(datafilter))

    @classmethod
    def load_element(cls, e):
        modelfilters = {}
        version = None
        for elem in e:
            if elem.tag == "Version":
                version = elem.text
        if version is None:
            old_modelfilters = v0Serializer.load(e)
            modelfilters = converter(
                version, PluginSettings().get("version"), old_modelfilters
            )
        else:
            old_filters = []
            is_old_filters = False
            for elem in e:
                if elem.tag == "DataSelection":
                    name = elem.attrib.get("name")
                    if name is None:
                        is_old_filters = True
                    dataselection = []
                    for datafilter in elem:
                        dataselection.append(
                            QgisVectorDataFilterSerializer.load(datafilter)
                        )
                    modelfilters[name] = dataselection
                    old_filters.append(dataselection)
            # FIXME Remove very soon old_filters
            if is_old_filters:
                modelfilters = old_filters
        return modelfilters
