# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Flutwelle
                                 A QGIS plugin
 Flutwelle
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2026-02-10
        git sha              : $Format:%H$
        copyright            : (C) 2026 by retep relleum
        email                : retep.relleum@bluewin.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE 
USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

from qgis.PyQt.QtCore import Qt,QVariant
from qgis.PyQt.QtGui import QColor,QGuiApplication
from qgis.PyQt.QtWidgets import QDialogButtonBox

import math

# Initialize Qt resources from file resources.py
from ..resources import *
# Import the code for the dialog

from qgis.gui import QgsMapTool
from qgis.utils import iface
from qgis.gui import QgsRubberBand
from qgis.core import *



import glob
import numpy as np

class Raster():
    """
    A class for managing raster layers in QGIS projects.
    This class handles loading, visibility control, and value sampling from raster layers
    organized within a "Raster" group in the QGIS layer tree. It supports dynamic loading
    of raster files based on geographic coordinates and provides functionality for visualizing
    specific elevation ranges.
    Attributes:
        rg (QgsLayerTreeGroup): The "Raster" group in the QGIS layer tree.
        tiffPath (str): The base path where raster TIFF files are stored.
        r_layer (QgsRasterLayer): The currently loaded raster layer.
    """
    def __init__(self,path):
        root = QgsProject.instance().layerTreeRoot()
        self.rg = root.findGroup("Raster") 
        if not self.rg :
            self.rg=root.insertGroup(0,"Raster")
        self.tiffPath=path
        self.r_layer=QgsRasterLayer()
        self.rg.setItemVisibilityChecked(True)
    
    def setVisibility(self,vis:bool):
        self.rg.setItemVisibilityChecked(vis)

    def getValue(self,p:QgsPoint):
        r_layer=self.getR_Layer(p)
        if  r_layer==0:
            return 0
        val, res = r_layer.dataProvider().sample(QgsPointXY(p.x(),p.y()), 1)
        if res:
            return val
        else:
            return 0


    def getR_Layer(self,p:QgsPoint):
        def getR_Layer(self, p: QgsPoint):
            """
            Retrieve or load a raster layer based on the geographic coordinates of a point.
            
            This method attempts to find a raster layer corresponding to the 1km x 1km grid cell
            containing the given point. It first checks if the layer is already loaded in memory,
            then searches the QGIS project, and finally attempts to load it from the file system
            if not found.
            
            Args:
                p (QgsPoint): A point object containing x and y coordinates (typically in meters)
                             from which the layer name is derived by integer division by 1000.
            
            Returns:
                QgsRasterLayer: The raster layer corresponding to the grid cell, or 0 if the
                               layer cannot be found or loaded from the file system.
            
            Raises:
                None: Returns 0 instead of raising exceptions if layer is not found.
            
            Notes:
                - Layer names are generated from coordinates: "{int(x//1000)}-{int(y//1000)}"
                - Searches for .tif files matching the pattern "*{lname}*.tif" in self.tiffPath
                - Loaded layers are cached in self.r_layer and added to the layer group self.rg
                - The method uses glob for recursive file searching
            """
        lname=f"{int(p.x()//1000)}-{int(p.y()//1000)}"
        if lname==self.r_layer.name():
            return self.r_layer
        r_layer = QgsProject.instance().mapLayersByName(lname)
        if not r_layer  :
            p=f'{self.tiffPath}/*{lname}*.tif'
            pfad=glob.glob(p, recursive=True)
            if len(pfad)>0:
                self.r_layer = QgsRasterLayer(pfad[0], lname)
                QgsProject.instance().addMapLayer(self.r_layer,False)
                self.rg.addLayer(self.r_layer)
                return self.r_layer 
            else:
              #  iface.messageBar().pushMessage("Error", "Bitte Kopieren sie die nötigen Dateien herunter https://www.swisstopo.admin.ch/de/hoehenmodell-swissalti3d ", level=Qgis.Critical)
                return 0
        else:
            self.r_layer=r_layer[0]
            return self.r_layer 
        
    def mark_line(self,point:QgsPoint):
        def mark_line(self, point: QgsPoint):
            """
            Highlights a specific value range in raster layers by applying a color ramp shader.
            
            Retrieves the pixel value at the given point and creates a color ramp that 
            highlights values within ±1.5 of that value in cyan, with transparent areas 
            outside this range. The color ramp is applied to all raster layers whose names 
            start with "2" and contain the target value within their min-max range.
            
            Args:
                point (QgsPoint): The point location to sample the raster value from.
            
            Returns:
                None
            
            Side Effects:
                - Modifies the renderer of matching raster layers
                - Triggers repaints and style change signals on updated layers
                - Reloads all layers in the current QGIS project
                - Prints exception details to console if renderer application fails
            """
        val=self.getValue(point)
        fcn = QgsColorRampShader()
        fcn.setColorRampType(QgsColorRampShader.Interpolated)
        fcn.setClassificationMode(QgsColorRampShader.Continuous)
        lst = [ QgsColorRampShader.ColorRampItem(val-1.5, QColor(0,0,0,0)),
                QgsColorRampShader.ColorRampItem(val, QColor(0,255,255)),
                QgsColorRampShader.ColorRampItem(val+1.5, QColor(0,0,0,0)) ]
        fcn.setColorRampItemList(lst)


        layers = QgsProject.instance().mapLayers().values()

        for layerx in layers:
            layerType = layerx.type()
            if layerType == QgsMapLayer.RasterLayer:
                if layerx.name().startswith("2"):
                    provider=layerx.dataProvider()
                    stats = provider.bandStatistics(1, QgsRasterBandStats.All)
                    min = stats.minimumValue
                    max = stats.maximumValue
                    if min <= val <= max:
                        shader = QgsRasterShader()
                        shader.setRasterShaderFunction(fcn)
                        renderer = QgsSingleBandPseudoColorRenderer(layerx.dataProvider(), 1, shader)
                        try:
                            layerx.setRenderer(renderer)
                            layerx.triggerRepaint()
                            layerx.emitStyleChanged()
                        except Exception as inst:
                            print(type(inst))    # the exception type
                            print(inst.args)     # arguments stored in .args
                            print(inst)
        QgsProject.instance().reloadAllLayers()

class Mupe:
    """
    A utility class for performing geometric and raster-based operations on QGIS point objects.

    This class provides methods for calculating distances, vector operations, and retrieving
    point values from a raster dataset. It serves as a helper for 3D raster manipulations.

    Attributes:
        raster (Raster): The raster dataset to query for point values.

    Methods:
        qgsDisXy(p1, p2) -> float:
            Calculate the Euclidean distance between two points in the XY plane.
        
        qgsVecDirNorm(p1, p2) -> QgsPoint:
            Compute the normalized direction vector from p2 to p1 in 3D space.
        
        qgsVecAdd(p1, p2) -> QgsPoint:
            Add two points' XY coordinates and set Z from raster value.
        
        qgsVecAddM(p1, p2) -> QgsPoint:
            Subtract p2's XY coordinates from p1's and set Z from raster value.
        
        qgsVec90add(p1, p2) -> QgsPoint:
            Add p1 to p2 rotated 90 degrees counterclockwise and set Z from raster.
        
        qgsVec90addM(p1, p2) -> QgsPoint:
            Subtract p2 rotated 90 degrees counterclockwise from p1 and set Z from raster.
        
        getPoint(x, y) -> QgsPoint:
            Create a QgsPoint from various input formats with Z value from raster.
            Supports float coordinates, QgsPointXY, or QgsPoint as input.
        
        getPointXY(p) -> QgsPointXY:
            Convert a QgsPoint to a QgsPointXY (2D projection).
    """
    def __init__(self,raster:Raster):
        self.raster=raster
    def qgsDisXy(self,p1:QgsPoint,p2:QgsPoint)->float:
        return math.sqrt((p1.x()-p2.x())**2+(p1.y()-p2.y())**2)
    def qgsVecDirNorm(self,p1:QgsPoint,p2:QgsPoint)->QgsPoint:
        dirP=QgsPoint(p1.x()-p2.x(),p1.y()-p2.y(),p1.z()-p2.z())
        dirP=QgsPoint(dirP.y()/math.sqrt(dirP.x()**2+dirP.y()**2),-dirP.x()/math.sqrt(dirP.x()**2+dirP.y()**2),dirP.z())
        return dirP
    def qgsVecAdd(self,p1:QgsPoint,p2:QgsPoint)->QgsPoint:
        p=QgsPoint(p1.x()+p2.x(),p1.y()+p2.y(),0)
        p.setZ(self.raster.getValue(p))
        return p
    def qgsVecAddM(self,p1:QgsPoint,p2:QgsPoint)->QgsPoint:
        p=QgsPoint(p1.x()-p2.x(),p1.y()-p2.y(),0)
        p.setZ(self.raster.getValue(p))
        return p 
    def qgsVec90add(self,p1:QgsPoint,p2:QgsPoint)->QgsPoint:
        p=QgsPoint(p1.x()+p2.y(),p1.y()-p2.x(),0)
        p.setZ(self.raster.getValue(p))
        return p         
    def qgsVec90addM(self,p1:QgsPoint,p2:QgsPoint)->QgsPoint:
        p=QgsPoint(p1.x()-p2.y(),p1.y()+p2.x(),0)
        p.setZ(self.raster.getValue(p))
        return p       
    def getPoint(self,x:float|QgsPoint|QgsPointXY,y:float=0)->QgsPoint:
        if isinstance(x,float):
            p=QgsPoint(x,y,0)
        elif isinstance(x,QgsPointXY):
            p=QgsPoint(x.x(),x.y(),0)
        else:
            p=x
        p.setZ(self.raster.getValue(p))
        return p   
    def getPointXY(self,p:QgsPoint)->QgsPointXY:
        return QgsPointXY(p.x(),p.y())          
  
