# -*- coding: utf-8 -*-
import sys, os
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingUtils,
                       QgsFeatureSink,
                       QgsFeature,
                       QgsGeometry,
                       QgsPoint,
                       QgsPointXY,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsUnitTypes,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFeatureSink)
from qgis import processing


class RussianDollsProcessingAlgorithm(QgsProcessingAlgorithm):

    INPUT = 'INPUT'
    FIELD = 'FIELD'
    FIELD_ORDER = 'FIELD_ORDER'
    CUT = 'CUT'
    RING = 'RING'
    DISTANCE = 'DISTANCE'
    OUTLINE = 'OUTLINE'
    OUTPUT = 'OUTPUT'
    OUTPUT2 = 'OUTPUT2'
    liste_order = ['Croissant', 'Décroissant']
    liste_cut = ['Rogner','Aggrandir']
    
    # dict_order = {'up':"Croissant", 'down':"Décroissant"}
    # dict_cut = {'cut':"Rogner", 'merge':"Aggrandir"}
    
    

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RussianDollsProcessingAlgorithm()

    def name(self):
        return 'russiandolls'

    def displayName(self):
        return '4_' + self.tr('Emboîter')
        

    def shortHelpString(self):
        return self.tr("Tronque les géométries pour s'emboiter par ordre croissant d'un champ de la couche")

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Polygones à emboîter"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.FIELD,
                self.tr("Selon le champs"),
                parentLayerParameterName = self.INPUT,
                allowMultiple=False,
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.FIELD_ORDER,
                self.tr("Ordre d'emboîtement"),
                options=self.liste_order, 
                allowMultiple=False, 
                usesStaticStrings=False, 
                defaultValue=0
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.CUT,
                self.tr("Option d'emboîtement"),
                options=self.liste_cut, 
                allowMultiple=False, 
                usesStaticStrings=False, 
                defaultValue=0
            )
        )
        
        distance = QgsProcessingParameterDistance(
            self.DISTANCE,
            self.tr("Distance d'emboitement (accrochage si négatif)"),
            defaultValue = 0
        )
        distance.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        self.addParameter(distance)
        

        # distance = QgsProcessingParameterDistance(
            # self.SNAP,
            # self.tr("Tolérance d'accrochage"),
            # defaultValue = 0
        # )
        # distance.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        # self.addParameter(distance)

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.RING,
                self.tr('Découper en anneaux distincts')
            )
        )
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.OUTLINE,
                self.tr("Extraire l'emprise totale")
            )
        )
        
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Polygones emboîtés')
            )
        )
        
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT2,
                self.tr('Emprise totale')
            )
        )
        

    def processAlgorithm(self, parameters, context, feedback):
        source = self.parameterAsSource(
            parameters,
            self.INPUT,
            context
        )
        field = self.parameterAsFields(
            parameters,
            self.FIELD,
            context
        )

        ring = self.parameterAsBoolean(
            parameters,
            self.RING, 
            context
        )
        outline = self.parameterAsBoolean(
            parameters,
            self.OUTLINE, 
            context
        )
        
        if source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
     

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            source.fields(),
            source.wkbType(),
            source.sourceCrs()
        )
        
        if outline:
            (sink2, dest_id2) = self.parameterAsSink(
                parameters,
                self.OUTPUT2,
                context,
                source.fields(),
                source.wkbType(),
                source.sourceCrs()
            )
        

        feedback.pushInfo(f"Starting russian dolls on {parameters[self.FIELD]}")
        feedback.pushInfo(f"Entity regroupment on identifying list values")
        
        if feedback.isCanceled():
            return {}
        
        alg = processing.run('native:removenullgeometries', 
            {
            'INPUT': parameters[self.INPUT],
            'REMOVE_EMPTY': True,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback, is_child_algorithm=True)
        
        
        alg = processing.run('native:dissolve', {
            'FIELD': [parameters[self.FIELD]],
            'INPUT': alg['OUTPUT'],
            'SEPARATE_DISJOINT': False,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback, is_child_algorithm=True)
    
    
        alg = processing.run("native:splitvectorlayer", {
            'INPUT': alg['OUTPUT'],
            'FIELD': parameters[self.FIELD],
            'FILE_TYPE': 0,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.pushInfo(f"")    
        files = {}
        for f in os.listdir(alg['OUTPUT']):
            n = os.path.join(alg['OUTPUT'], f)
            s = QgsProcessingUtils.mapLayerFromString(n, context)
            for f in s.getFeatures():
                v = f.attribute(parameters[self.FIELD])
                break
            try: files[v] = s
            except: return {}

        vals = list(files.keys())
        
        if feedback.isCanceled():
            return {}
        
        if parameters[self.CUT]==1:
            if parameters[self.FIELD_ORDER]==1:
                vals.sort()
            else:
                vals.sort(reverse=True)
            feedback.pushInfo(f"Making unions")
            for i,v in enumerate(vals):
                if feedback.isCanceled():
                    return {}
                if i>0:
                    overlay = files[vals[i-1]]
                    feedback.pushInfo(f"Intersection for :{v}")
                    
                    if parameters[self.DISTANCE]>0:
                        alg2 = processing.run('native:buffer', 
                            {
                                'INPUT': overlay,
                                'DISSOLVE': False,
                                'DISTANCE': parameters[self.DISTANCE],
                                'END_CAP_STYLE': 0,
                                'JOIN_STYLE': 0,
                                'MITER_LIMIT': 2,
                                'SEGMENTS': 5,
                                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                            },
                            context=context, feedback=feedback, is_child_algorithm=True)
                        overlay = alg2['OUTPUT']
                    
                    alg = processing.run("native:union", {
                        'INPUT': files[v],
                        'OVERLAY': overlay,
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    alg = processing.run("native:dissolve", {
                        'INPUT': alg['OUTPUT'],
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    alg = processing.run('native:fixgeometries', 
                    {
                        'INPUT': alg['OUTPUT'],
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, 
                    context=context, feedback=feedback, is_child_algorithm=True)
                    files[v] = alg['OUTPUT']
            if parameters[self.FIELD_ORDER]==1:
                vals.sort(reverse=True)
            else:
                vals.sort()
        else:
            if parameters[self.FIELD_ORDER]==1:
                vals.sort(reverse=True)
            else:
                vals.sort()
            feedback.pushInfo(f"Making intersections")
            for i,v in enumerate(vals):
                if i>0:
                    overlay = files[vals[i-1]]
                    feedback.pushInfo(f"Intersection for :{v}")
                    
                    if parameters[self.DISTANCE]>0:
                        alg2 = processing.run('native:buffer', 
                            {
                                'INPUT': overlay,
                                'DISSOLVE': False,
                                'DISTANCE': -1*parameters[self.DISTANCE],
                                'END_CAP_STYLE': 0,
                                'JOIN_STYLE': 0,
                                'MITER_LIMIT': 2,
                                'SEGMENTS': 5,
                                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                            },
                            context=context, feedback=feedback, is_child_algorithm=True)
                        overlay = alg2['OUTPUT']
                    
                    alg = processing.run("native:intersection", {
                        'INPUT': files[v],
                        'OVERLAY': overlay,
                        'OVERLAY_FIELDS': [],
                        'OVERLAY_FIELDS_PREFIX': 'toto',
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    alg = processing.run('native:fixgeometries', 
                    {
                        'INPUT': alg['OUTPUT'],
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, 
                    context=context, feedback=feedback, is_child_algorithm=True)
                    files[v] = alg['OUTPUT']
        
        
        if feedback.isCanceled():
            return {}
        
        if parameters[self.DISTANCE]<0:
            feedback.pushInfo(f"Snaping polygon")
            for i,v in enumerate(vals):
                if i>0:
                    feedback.pushInfo(f"Snapping for :{v}")
                    alg = processing.run("native:snapgeometries", {
                        'INPUT': files[v],
                        'REFERENCE_LAYER': files[vals[i-1]],
                        'TOLERANCE': -1*parameters[self.DISTANCE],
                        'BEHAVIOR': 0,
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    alg = processing.run('native:fixgeometries', 
                    {
                        'INPUT': alg['OUTPUT'],
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, 
                    context=context, feedback=feedback, is_child_algorithm=True)
                    files[v] = alg['OUTPUT']
        
        zitotale = files[vals[0]]
        
        if feedback.isCanceled():
            return {}
        
        if ring:
            feedback.pushInfo(f"Cutting disk into rings")
            for i,v in enumerate(vals):
                if i<len(vals)-1:
                    feedback.pushInfo(f"Cutting disk for :{v}")
                    alg = processing.run("native:difference", {
                        'INPUT': files[v],
                        'OVERLAY': files[vals[i+1]],
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, context=context, feedback=feedback, is_child_algorithm=True)
                    files[v] = alg['OUTPUT']
        
        liste = []
        for v in vals:
            liste.append(files[v])
        
        feedback.pushInfo(f"Merging layers")
        alg = processing.run("native:mergevectorlayers",
            {
                'LAYERS': liste,
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }, context=context, feedback=feedback, is_child_algorithm=True)
        
        
        alg = processing.run('native:fixgeometries', 
            {
                'INPUT': alg['OUTPUT'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }, 
            context=context, feedback=feedback, is_child_algorithm=True)

        if feedback.isCanceled():
            return {}

        s = QgsProcessingUtils.mapLayerFromString(alg['OUTPUT'], context)
        if source.fields().lookupField('fid')<0 and s.fields().lookupField('fid')>=0:
            feedback.pushInfo(f"deleting fid column")
            alg = processing.run('native:deletecolumn', 
                {
                    'COLUMN': ['fid'],
                    'INPUT': alg['OUTPUT'],
                    'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                }, 
                context=context, feedback=feedback, is_child_algorithm=True)
            s = QgsProcessingUtils.mapLayerFromString(alg['OUTPUT'], context)
        sink.addFeatures(s.getFeatures(), QgsFeatureSink.FastInsert)
        
        
        out = {self.OUTPUT: dest_id}
        
        if outline:
            if isinstance(zitotale, str):
                zitotale = QgsProcessingUtils.mapLayerFromString(zitotale, context)
            if source.fields().lookupField('fid')<0 and zitotale.fields().lookupField('fid')>=0:
                alg = processing.run('native:deletecolumn', 
                    {
                        'COLUMN': ['fid'],
                        'INPUT': zitotale,
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }, 
                    context=context, feedback=feedback, is_child_algorithm=True)
                zitotale = QgsProcessingUtils.mapLayerFromString(alg['OUTPUT'], context)
            sink2.addFeatures(zitotale.getFeatures(), QgsFeatureSink.FastInsert)
            out[self.OUTPUT2] = dest_id2
        
        return out
        
        
        # list_values = list(feats.keys())
        # list_values.sort()
        # feats_disk = {}
        # output = feats_disk
        # tot = len(list_values)
        
        
        # if tot>0:
            # feedback.pushInfo(f"Working intersections on classes : {list_values}")
            # v = list_values.pop(0)
            # memoFeat = feats[v]
            # feats_disk[v] = memoFeat
            # for c,v in enumerate(list_values):
                # feat = feats[v]
                # geom = feat.geometry().intersection(memoFeat.geometry())
                # feat.setGeometry(geom)
                # feats_disk[v] = feat
                # memoFeat = feat
                # feedback.setProgress(100*c/tot)
            # output = feats_disk
        # else:
            # feedback.pushInfo(f"No value available")
        
        # if tot>0 and ring:
            # feats_ring = {}
            # list_values = list(feats_disk.keys())
            # list_values.sort(reverse=True)
            # tot = len(list_values)
            # if tot>0:
                # feedback.pushInfo(f"Working difference on classes : {list_values}")
                # v = list_values.pop(0)
                # memoFeat = feats_disk[v]
                # feats_ring[v] = memoFeat
                # for c,v in enumerate(list_values):
                    # feat = QgsFeature(feats_disk[v])
                    # geom = feat.geometry().difference(memoFeat.geometry())
                    # feat.setGeometry(geom)
                    # feats_ring[v] = feat
                    # memoFeat = feats_disk[v]
                    # feedback.setProgress(100*c/tot)
            # output = feats_ring
            
        # feedback.pushInfo(f"Writing final layer")
        # l = output.values()
        # tot = len(l)
        # for c,f in enumerate(l):
            # sink.addFeature(f, QgsFeatureSink.FastInsert)
            # feedback.setProgress(100*c/tot)

        
        # return {self.OUTPUT: dest_id}

    












