# -*- coding: utf-8 -*-
import sys, os
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingUtils,
                       QgsFeatureSink,
                       QgsFeature,
                       QgsGeometry,
                       QgsPoint,
                       QgsPointXY,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsUnitTypes,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFeatureSink)
from qgis import processing


class DiskToRingProcessingAlgorithm(QgsProcessingAlgorithm):

    INPUT = 'INPUT'
    FIELD = 'FIELD'
    FIELD_ORDER = 'FIELD_ORDER'
    OUTPUT = 'OUTPUT'
    liste_order = ['Croissant', 'Décroissant']

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return DiskToRingProcessingAlgorithm()

    def name(self):
        return 'disktoring'

    def displayName(self):
        return self.tr('Découper en anneaux')

    def group(self):
        return self.tr('Scripts')

    def groupId(self):
        return 'scripts'

    def shortHelpString(self):
        return self.tr("Découpe les géométries en disques pour former des anneaux")

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Couche source"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.FIELD,
                self.tr("Selon le champs"),
                parentLayerParameterName = self.INPUT,
                allowMultiple=False,
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.FIELD_ORDER,
                self.tr("Ordre d'emboîtement"),
                options=self.liste_order, 
                allowMultiple=False, 
                usesStaticStrings=False, 
                defaultValue=0
            )
        )
 
        
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Anneaux')
            )
        )
        

    def processAlgorithm(self, parameters, context, feedback):
        source = self.parameterAsSource(
            parameters,
            self.INPUT,
            context
        )
        field = self.parameterAsFields(
            parameters,
            self.FIELD,
            context
        )

        if source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
     

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            source.fields(),
            source.wkbType(),
            source.sourceCrs()
        )
        
        feedback.pushInfo(f"Cutting disk into rings")

        alg = processing.run('native:dissolve', {
            'FIELD': [parameters[self.FIELD]],
            'INPUT': parameters[self.INPUT],
            'SEPARATE_DISJOINT': False,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback, is_child_algorithm=True)
    

        alg = processing.run("native:splitvectorlayer", {
            'INPUT': alg['OUTPUT'],
            'FIELD': parameters[self.FIELD],
            'FILE_TYPE': 0,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback, is_child_algorithm=True)
  
        files = {}
        for f in os.listdir(alg['OUTPUT']):
            n = os.path.join(alg['OUTPUT'], f)
            s = QgsProcessingUtils.mapLayerFromString(n, context)
            for f in s.getFeatures():
                v = f.attribute(parameters[self.FIELD])
                break
            files[v] = s

        vals = list(files.keys())

        if parameters[self.FIELD_ORDER]==1:
            vals.sort(reverse=True)
        else:
            vals.sort()



        for i,v in enumerate(vals):
            if i<len(vals)-1:
                feedback.pushInfo(f"Cutting disk for :{v}")
                alg = processing.run("native:difference", {
                    'INPUT': files[v],
                    'OVERLAY': files[vals[i+1]],
                    'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                }, context=context, feedback=feedback, is_child_algorithm=True)
                files[v] = alg['OUTPUT']

                

        liste = []
        for v in vals:
            liste.append(files[v])
        
        feedback.pushInfo(f"Merging layers")
        alg = processing.run("native:mergevectorlayers",
            {
                'LAYERS': liste,
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }, context=context, feedback=feedback, is_child_algorithm=True)

        s = QgsProcessingUtils.mapLayerFromString(alg['OUTPUT'], context)
        if source.fields().lookupField('fid')<0 and s.fields().lookupField('fid')>=0:
            feedback.pushInfo(f"deleting fid column")
            alg = processing.run('native:deletecolumn', 
                {
                    'COLUMN': ['fid'],
                    'INPUT': alg['OUTPUT'],
                    'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                }, 
                context=context, feedback=feedback, is_child_algorithm=True)
            s = QgsProcessingUtils.mapLayerFromString(alg['OUTPUT'], context)
        sink.addFeatures(s.getFeatures(), QgsFeatureSink.FastInsert)

        return {self.OUTPUT: dest_id}






