# -*- coding: utf-8 -*-
import sys
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsMapLayer,
                       QgsProject,
                       QgsFeatureSink,
                       QgsFeature,
                       QgsGeometry,
                       QgsPoint,
                       QgsPointXY,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsUnitTypes,
                       QgsProcessingFeatureSourceDefinition,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFeatureSink)
from qgis import processing

class LinearInterpolationProcessingAlgorithm(QgsProcessingAlgorithm):

    INPUT = 'INPUT'
    DISTANCE = 'DISTANCE'
    FIELD = 'FIELD'
    INPUT_LINES = 'INPUT_LINES'
    OUTPUT = 'OUTPUT'
    TOLERANCE = 'TOLERANCE'
    CLEAN = 'CLEAN'
    OUTPUT_LINES = 'OUTPUT_LINES'
    JUNCTION = 'JUNCTION'
    ERROR = 'ERROR'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return LinearInterpolationProcessingAlgorithm()

    def name(self):
        return 'multilineinterpolation'

    def displayName(self):
        return self.tr('Multiligne Interpolation')

    def group(self):
        return self.tr('Scripts')

    def groupId(self):
        return 'scripts'

    def shortHelpString(self):
        return self.tr("Interpole des points le long d'une multiligne en calculant la valeur des champs cibles linéarisée selon l'abscisse curviligne")

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Points à interpoler"),
                [QgsProcessing.TypeVectorPoint]
            )
        )
        distance = QgsProcessingParameterDistance(
                self.DISTANCE,
                self.tr("Distance entre deux points"),
                defaultValue = 1
            )
        distance.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        self.addParameter(distance)
        self.addParameter(
            QgsProcessingParameterField(
                self.FIELD,
                self.tr("Champ contenant les valeurs"),
                parentLayerParameterName = self.INPUT,
                allowMultiple=False,
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_LINES,
                self.tr("Lignes d'interpolation"),
                [QgsProcessing.TypeVectorLine]
            )
        )
        distance = QgsProcessingParameterDistance(
                self.TOLERANCE,
                self.tr("Tolérance de connexion"),
                defaultValue = 10
            )
        distance.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        self.addParameter(distance)
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CLEAN,
                self.tr('Supprimer les branches de longueur inférieure à l''interpolation')
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.JUNCTION,
                self.tr('Récupérer les perpendiculaires de projection')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Points interpolés')
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LINES,
                self.tr('Lignes reconstituées')
            )
        )
        error = QgsProcessingParameterFeatureSink(
                self.ERROR,
                self.tr('Erreurs')
            )
        error.setCreateByDefault(False)
        self.addParameter(error)
        
    def getMapLayer(self, v, context):
        if not isinstance(v, (QgsMapLayer,QgsProcessingFeatureSourceDefinition)):
            try: v = context.getMapLayer(v)
            except: pass
        return v

    def processAlgorithm(self, parameters, context, feedback):
        sourceL = self.parameterAsSource(
            parameters,
            self.INPUT_LINES,
            context
        )
        source = self.parameterAsSource(
            parameters,
            self.INPUT,
            context
        )
        field = self.parameterAsFields(
            parameters,
            self.FIELD,
            context
        )
        junction = self.parameterAsBoolean(
            parameters,
            self.JUNCTION, 
            context
        )
        clean = self.parameterAsBoolean(
            parameters,
            self.CLEAN, 
            context
        )

        if source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
        if sourceL is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT_Lines))


        # crs = QgsProject.instance().crs()
        
        for v in (self.INPUT_LINES,self.INPUT):
            parameters[v] = self.getMapLayer(parameters[v], context)

        if sourceL.sourceCrs()!=source.sourceCrs():
            alg = processing.run("native:reprojectlayer", {
                'INPUT': parameters[self.INPUT_LINES],
                'TARGET_CRS': source.sourceCrs(),
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }, context=context, feedback=feedback)
            parameters[self.INPUT_LINES] = self.getMapLayer(alg['OUTPUT'], context)
        
        (sinkL, destL_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_LINES,
            context,
            sourceL.fields(),
            sourceL.wkbType(),
            sourceL.sourceCrs()
        )        
        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            source.fields(),
            source.wkbType(),
            source.sourceCrs()
        )
        (sinkE, destE_id) = self.parameterAsSink(
            parameters,
            self.ERROR,
            context,
            sourceL.fields(),
            sourceL.wkbType(),
            sourceL.sourceCrs()
        )
        

        feedback.pushInfo(f"")
        feedback.pushInfo(f"Step 1/3 : Making branches")
        alg = processing.run("fmt:branchconnect", {
            'INPUT': parameters[self.INPUT_LINES],
            'TOLERANCE': parameters[self.TOLERANCE],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }, context=context, feedback=feedback)
        
        
        outLayer = self.getMapLayer(alg['OUTPUT'], context)
        
        if clean: 
            delIds = []
            for l in outLayer.getFeatures():
                if l.geometry().length()<parameters[self.DISTANCE]:
                    delIds.append(l.id())
                    sinkE.addFeature(l, QgsFeatureSink.FastInsert)
       
            for id in delIds:
                outLayer.deleteFeature(id)
            
            alg = processing.run("fmt:branchconnect", {
                'INPUT': outLayer,
                'TOLERANCE': parameters[self.TOLERANCE],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }, context=context, feedback=feedback)
        
            outLayer = self.getMapLayer(alg['OUTPUT'], context)
        
        feedback.pushInfo(f"")
        feedback.pushInfo(f"Step 2/3 : Snapping point to closest and cutting lines")
        exclude = 0
        error = 0
        points = [p for p in source.getFeatures()]
        lines = [l for l in outLayer.getFeatures()]
        tot = len(points)
        
        if tot==0:
            return
        
        square = parameters[self.TOLERANCE]*parameters[self.TOLERANCE]
        for c,p in enumerate(points):
            if feedback.isCanceled():
                break
            feedback.setProgress(100*c/tot)
            pvalue = p.attribute(parameters[self.FIELD])
            try:
                mini = min((l.geometry().closestSegmentWithContext( p.geometry().asPoint() ), l, i) for i,l in enumerate(lines))
            except:
                error += 1
                continue
            dist = mini[0][0]    
            if dist>square:
                exclude += 1
                continue
            minDistPoint = mini[0][1]
            vertex = mini[0][2]
            line = mini[1]
            ind = mini[2]
            
            if junction:
                pd = QgsPointXY(minDistPoint)
                pi = p.geometry().asPoint()
                if pi.distance(pd)>0:
                    geom = QgsGeometry.fromPolyline([QgsPoint(pd), QgsPoint(pi)])
                    feat = QgsFeature()
                    feat.setGeometry(geom)
                    sinkL.addFeature(feat, QgsFeatureSink.FastInsert)
            
            pts = line.geometry().asPolyline()
            ptsg = pts[:vertex]
            ptsg.append(QgsPointXY(minDistPoint))
            lg = QgsFeature()
            lg.setGeometry(QgsGeometry.fromPolyline(map(QgsPoint, ptsg)))
            try:
                lg.first = line.first
            except:
                pass
            lg.last = pvalue
            ptsd = pts[vertex:]
            ptsd.insert(0, QgsPointXY(minDistPoint))
            ld = QgsFeature()
            ld.setGeometry(QgsGeometry.fromPolyline(map(QgsPoint, ptsd)))
            try:
                ld.last = line.last
            except:
                pass
            ld.first = pvalue
            lines.pop(ind)
            lines.insert(ind, ld)
            lines.insert(ind, lg)
            
            geom = QgsGeometry.fromPointXY(QgsPointXY(minDistPoint))
            feat = QgsFeature(source.fields())
            feat.setGeometry(geom)
            feat.setAttribute(parameters[self.FIELD], pvalue)
            sink.addFeature(feat)
        
        
        s = 0
        if exclude>1:
            s = "s"
        if exclude>0:
            feedback.pushInfo(f"{exclude} point{s} didn't match the tolerance")
            
        s = 0
        if error>1:
            s = "s"
        if error>0:
            feedback.pushInfo(f"{error} point{s} failed calculating")
        
        missingLines = []
        pointsMaked = []
        feedback.pushInfo(f"")
        feedback.pushInfo(f"Step 3/3 : Interpolating points on {len(lines)} lines")
        tot = len(lines)
        for c,l in enumerate(lines):
            if feedback.isCanceled():
                break
            
            try:
                first = l.first
                last = l.last
                sinkL.addFeature(l, QgsFeatureSink.FastInsert)
                maker = pointMaker(l, source.fields(), parameters[self.FIELD], parameters[self.DISTANCE], first, last, feedback)
                maker.makePoints(sink)
                pointsMaked.append(maker)
                feedback.setProgress(100*c/tot)
            except:
                missingLines.append(l)
                pass
        
        
        
        
        
        
        
        n = len(missingLines)
        s = 0
        if n>1:
            s = "s"
        if n>0:
            feedback.pushInfo(f"   need extent interpolation for {n} line{s}")
        d = 0
        for c,l in enumerate(missingLines):
            if feedback.isCanceled():
                break

            for e in ['first', 'last']:    
                try:
                    if e=='first':
                        v = l.first
                    if e=='last':
                        v = l.last
                except:
                    list = l.geometry().asPolyline()
                    ind = len(list)-1 if e=='last' else 0
                    pt = list[ind]
                    try:
                        mini = min((m.line.geometry().closestSegmentWithContext( pt ), m, i) for i,m in enumerate(pointsMaked) if len(m.listPoints)>0)
                    except:
                        continue
                    dist = mini[0][0]    
                    if dist>square:
                        continue
                    m = mini[1]
                    mini2 = min((pt.distance(f.geometry().asPoint()), f) for i,f in enumerate(m.listPoints))
                    dist = mini2[0]  
                    if dist>square:
                        continue
                    f = mini2[1]
                    if e=='first':
                        l.first = f.attribute(parameters[self.FIELD])
                    if e=='last':
                        l.last = f.attribute(parameters[self.FIELD])
               
            try:
                first = l.first
                last = l.last
                sinkL.addFeature(l, QgsFeatureSink.FastInsert)
                maker = pointMaker(l, source.fields(), parameters[self.FIELD], parameters[self.DISTANCE], first, last, feedback)
                maker.makePoints(sink)
                feedback.setProgress(100*c/tot)
                d += 1
            except:
                # print(sys.exc_info())
                sinkE.addFeature(l, QgsFeatureSink.FastInsert)
                pass
        
        feedback.pushInfo(f"recuperation successful : {d}/{n}")

        return {self.OUTPUT: dest_id, self.OUTPUT_LINES: destL_id}

    
    

    

class pointMaker():
    def  __init__(self, line, fields, field, distance, first, last, feedback):
        self.line = line
        self.fields = fields
        self.distance = distance
        self.feedback = feedback
        self.field = field
        self.first = first
        self.last = last
        self.setList()
    
    def setList(self):
        self.list = self.line.geometry().asPolyline()
        self.cumul = 0
        self.use = 0
        self.id = 0
        self.acc = (self.last-self.first)/(self.line.geometry().length())
        self.listPoints = []
        
    def makePoints(self, sink):
        self.sink = sink
        while len(self.list)>1:
            if self.feedback.isCanceled():
                break
            d = self.list.pop(0)
            f = self.list[0]
            self.makePointLine(d, f)
            
    def makePointLine(self, d, f):
        dist = d.distance(f)
        need = self.distance - self.use
        self.use += dist
        while dist>=need:
            if self.feedback.isCanceled():
                break
            self.use -= self.distance
            self.cumul += self.distance
            x = d.x() + need*(f.x()-d.x())/dist
            y = d.y() + need*(f.y()-d.y())/dist
            self.addPoint(x, y, self.first + self.cumul*self.acc)
            need += self.distance

    def addPoint(self, x, y, val):
        self.id += 1
        pt = QgsPointXY(x,y)
        geom = QgsGeometry.fromPointXY(pt)
        feat = QgsFeature(self.fields)
        feat.setGeometry(geom)
        feat.setAttribute(self.field, val)
        self.sink.addFeature(feat)
        self.listPoints.append(feat)










