from qgis.core import QgsProcessing
from qgis.core import QgsProcessingAlgorithm
from qgis.core import QgsProcessingMultiStepFeedback
from qgis.core import QgsProcessingParameterField
from qgis.core import QgsProcessingParameterFeatureSource
from qgis.core import QgsProcessingParameterFeatureSink
from qgis.core import QgsProcessingParameterNumber
from qgis.core import QgsProcessingParameterRasterLayer
from qgis.core import QgsProcessingParameterRasterDestination
from qgis.core import QgsProcessingParameterDistance
from qgis.core import QgsProcessingParameterEnum
from qgis.core import QgsProcessingParameterDefinition
from qgis.core import QgsProcessingParameterBoolean
from qgis.core import QgsProcessingUtils
from qgis.core import QgsUnitTypes
from qgis.core import QgsVectorLayer
from qgis.core import QgsProcessingUtils
from qgis.core import QgsFeatureSink
from qgis.core import QgsWkbTypes


from qgis.PyQt.QtCore import QCoreApplication
from qgis import processing

import time



class CpheSpatialisationProcessingAlgorithm(QgsProcessingAlgorithm):

    def name(self):
        return 'cphespatialisation'

    def displayName(self):
        return '1_' + self.tr('Spatialiser')

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def shortHelpString(self):
        return """<html><body><p>Cet algorithme interpole une couche vectorielle contenant des données spatiales et les projète sur un Modèle Numérique de Terrain (MNT) si celui-ci est donnée en paramètre.</p>
<br></body></html>"""

    def createInstance(self):
        return CpheSpatialisationProcessingAlgorithm()

    def initAlgorithm(self, config=None):
        choixSpatialisationListe = ['Interpolation rst','Interpolation Bilineaire', 'Interpolation Bicubique', 'En Escalier']
        
        self.addParameter(QgsProcessingParameterFeatureSource('cotes_des_plus_hautes_eau', 'Cotes des Plus Hautes Eau', types=[QgsProcessing.TypeVectorAnyGeometry], defaultValue=None))
        self.addParameter(QgsProcessingParameterBoolean('force_centroid','Cocher pour forcer le recalcul des cenntroïdes (mailles)', defaultValue=True))
        self.addParameter(QgsProcessingParameterField('champCPHE', 'Champ contenant la valeur de la CPHE', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='cotes_des_plus_hautes_eau', allowMultiple=False, defaultValue=None))
        distance = QgsProcessingParameterDistance('rasterSize', f"Taille d'une cellule Raster", defaultValue=0.5)
        distance.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        self.addParameter(distance)
        self.addParameter(QgsProcessingParameterEnum('interpolation', 'Choix de la spatialisation', options=choixSpatialisationListe, allowMultiple=False, usesStaticStrings=False, defaultValue=3))
        self.addParameter(QgsProcessingParameterRasterDestination('res_CPHE', 'CPHE', createByDefault=False, defaultValue=None))
        param = QgsProcessingParameterDistance('buffer', 'Taille du tampon', parentParameterName='cotes_des_plus_hautes_eau', defaultValue=100)
        
        param.setDefaultUnit(QgsUnitTypes.DistanceMeters)
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)
        
        
        param = QgsProcessingParameterFeatureSource('masque', 'Traitement par zones', optional=True, types=[QgsProcessing.TypeVectorAnyGeometry], defaultValue=None)
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)
             
       
    def processAlgorithm(self, parameters, context, model_feedback):

        feedback = QgsProcessingMultiStepFeedback(7, model_feedback)
        
        nicenames = {
            'res_CPHE': "CPHE",
        }  
        results = {}
        

        if parameters['masque'] is None:
            parameters['cotes_des_plus_hautes_eau_unitaire'] = parameters['cotes_des_plus_hautes_eau']
            status, results = self.traitement_unitaire(parameters, context, feedback)
            if status=='cancel':
                return {}
        else:  
            results_assemb = {}

            alg = processing.run('native:fixgeometries', 
                {
                    'INPUT': parameters['masque'],
                    'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                }, 
                context=context, feedback=feedback, is_child_algorithm=True)
            source = context.getMapLayer(alg['OUTPUT'])
            
            for f in source.getFeatures():
                (sink, dest_id) = QgsProcessingUtils.createFeatureSink(
                    'memory:',
                    context,
                    source.fields(),
                    source.wkbType(),
                    source.sourceCrs()
                )
                
                sink.addFeature(f, QgsFeatureSink.FastInsert)
                alg_params = {
                    'INPUT': parameters['cotes_des_plus_hautes_eau'],
                    'INTERSECT': dest_id,
                    'PREDICATE': [0], 
                    'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                }
                alg = processing.run('native:extractbylocation', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
                parameters['cotes_des_plus_hautes_eau_unitaire'] = alg['OUTPUT']

                if context.getMapLayer(alg['OUTPUT']).featureCount()<1:
                    continue
  
                try:
                    status, results = self.traitement_unitaire(parameters, context, feedback)
                except Exception as e:
                    continue 
                       
                if status=='cancel':
                    return {}
                    
                for key,val in results.items():
                    if key not in results_assemb: results_assemb[key] = []
                    
                    alg_params = {
                        'COPY_SUBDATASETS': False,
                        'DATA_TYPE': 0,  # Utiliser le type de donnée de la couche en entrée
                        'EXTRA': '',
                        'INPUT': val,
                        'NODATA': 0,
                        'OPTIONS': '',
                        'TARGET_CRS': source.sourceCrs(),
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }
                    alg = processing.run('gdal:translate', alg_params, context=context, feedback=feedback, is_child_algorithm=True)     
                    
                    alg_params = {
                        'ALPHA_BAND': False,
                        'CROP_TO_CUTLINE': True,
                        'DATA_TYPE': 0,  # Utiliser le type de donnée de la couche en entrée
                        'EXTRA': '',
                        'INPUT': alg['OUTPUT'],
                        'KEEP_RESOLUTION': False,
                        'MASK': dest_id,
                        'MULTITHREADING': False,
                        'NODATA': None,
                        'OPTIONS': '',
                        'SET_RESOLUTION': False,
                        'SOURCE_CRS': None,
                        'TARGET_CRS': None,
                        'TARGET_EXTENT': None,
                        'X_RESOLUTION': None,
                        'Y_RESOLUTION': None,
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }
                    alg = processing.run('gdal:cliprasterbymasklayer', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    results_assemb[key].append(alg['OUTPUT'])  
            
            for key,liste in results_assemb.items():
                
                if len(liste)>1:
                    alg_params = {
                        '-z': False,
                        'GRASS_RASTER_FORMAT_META': '',
                        'GRASS_RASTER_FORMAT_OPT': '',
                        'GRASS_REGION_CELLSIZE_PARAMETER': 0,
                        'GRASS_REGION_PARAMETER': None,
                        'input': liste,
                        'output': parameters[key]
                    }
                    alg = processing.run('grass7:r.patch', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
                    results[key] = alg['output']
                
                elif len(liste)>0:
                    alg_params = {
                        'COPY_SUBDATASETS': False,
                        'DATA_TYPE': 0,  # Utiliser le type de donnée de la couche en entrée
                        'EXTRA': '',
                        'INPUT': liste[0],
                        'NODATA': 0,
                        'OPTIONS': '',
                        'TARGET_CRS': None,
                        'OUTPUT': parameters[key]
                    }
                    alg = processing.run('gdal:translate', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
                    
                    results[key] = alg['OUTPUT']
                    
                else:
                    results[key] = None
            
        
        for key in results.keys():  
            details = context.layerToLoadOnCompletionDetails(results[key])
            details.name = nicenames.get(key,details.name)
            details.forceName = True    

      
        # results['parameters'] = parameters
        
        return results
           
    def traitement_unitaire(self, parameters, context, feedback):    
        results = {}
        outputs = {}    
        m = 1 if parameters['masque'] is None else 0        
        outputs['res_CPHE'] = [QgsProcessing.TEMPORARY_OUTPUT,parameters['res_CPHE']][m]
        

    # Pré-traitement_unitaire

        outputs['pre_traitement']=parameters['cotes_des_plus_hautes_eau_unitaire']

        # De morceaux multiples à morceaux uniques
        alg_params = {
            'INPUT': outputs['pre_traitement'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['centroides'] = processing.run('native:multiparttosingleparts', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

       
        if parameters['force_centroid']:
            # Centroïdes
            alg_params = {
                'ALL_PARTS': False,
                'INPUT': outputs['pre_traitement'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['centroides'] = processing.run('native:centroids', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
         
        if feedback.isCanceled():
            return 'cancel', None  
            
        else : 
            # Géométrie d'emprise minimale
            alg_params = {
                'FIELD': '',
                'INPUT': outputs['centroides']['OUTPUT'],
                'TYPE': 3,  # Enveloppe convexe
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['GomtrieDempriseMinimale'] = processing.run('qgis:minimumboundinggeometry', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
            outputs['pre_traitement']=outputs['GomtrieDempriseMinimale']['OUTPUT']

      
        if feedback.isCanceled():
            return 'cancel', None   

    #Interpolation Escalier        
        if parameters['interpolation'] == 3:
            status,outputs['res_CPHE'] = self.interpolEscalier(parameters, context, feedback, outputs)
            
            if status=='cancel':
                return 'cancel', None
                
    #Interpolaiton avec Rfillnull
        else :
            status,outputs['res_CPHE'] = self.rfillnull(parameters, context, feedback, outputs)
            
            if status=='cancel':
                return 'cancel', None
        
        results['res_CPHE'] = outputs['res_CPHE']

        feedback.setCurrentStep(7)        
        if feedback.isCanceled():
            return 'cancel', None
             
        return 'ok', results      

    def rfillnull(self, parameters, context, feedback, outputs):
        status = 'ok'

        # Tampon
        alg_params = {
            'DISSOLVE': True,
            'DISTANCE': parameters['buffer'],
            'END_CAP_STYLE': 0,  # Rond
            'INPUT': outputs['centroides']['OUTPUT'],
            'JOIN_STYLE': 0,  # Rond
            'MITER_LIMIT': 2,
            'SEGMENTS': 5,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['Tampon'] = processing.run('native:buffer', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        
        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return 'cancel', None
    
        # Limite
        alg_params = {
            'INPUT': outputs['Tampon']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('native:boundary', alg_params, context=context, feedback=feedback, is_child_algorithm=True)


        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return 'cancel', None

        # Multiligne Interpolation
        alg_params = {
            'CLEAN': False,
            'DISTANCE': 10,
            'FIELD': parameters['champCPHE'],
            'INPUT': outputs['centroides']['OUTPUT'],
            'INPUT_LINES': alg['OUTPUT'],
            'JUNCTION': False,
            'TOLERANCE': parameters['buffer'],
            'ERROR': QgsProcessing.TEMPORARY_OUTPUT,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT,
            'OUTPUT_LINES': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('fmt:multilineinterpolation', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        

        feedback.setCurrentStep(3)       
        if feedback.isCanceled():
            return 'cancel', None
        
        couches=[outputs['centroides']['OUTPUT'],alg['OUTPUT']]
        
        
        # Fusionner des couches vecteur
        alg_params = {
            'CRS': None,
            'LAYERS': couches,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('native:mergevectorlayers', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        

        feedback.setCurrentStep(4)
        if feedback.isCanceled():
            return 'cancel', None

        if parameters['interpolation'] == 0 :
            # v.surf.rst
            alg_params = {
                '-d': False,
                '-t': False,
                'GRASS_MIN_AREA_PARAMETER': 0.0001,
                'GRASS_OUTPUT_TYPE_PARAMETER': 0,  # auto
                'GRASS_RASTER_FORMAT_META': '',
                'GRASS_RASTER_FORMAT_OPT': '',
                'GRASS_REGION_CELLSIZE_PARAMETER': parameters['rasterSize'],
                'GRASS_REGION_PARAMETER': None,
                'GRASS_SNAP_TOLERANCE_PARAMETER': -1,
                'GRASS_VECTOR_DSCO': '',
                'GRASS_VECTOR_EXPORT_NOCAT': False,
                'GRASS_VECTOR_LCO': '',
                'dmax': None,
                'dmin': None,
                'input':  alg['OUTPUT'],
                'mask': None,
                'npmin': 300,
                'scalex': None,
                'segmax': 40,
                'smooth': None,
                'smooth_column': '',
                'tension': 40,
                'theta': None,
                'where': '',
                'zcolumn': parameters['champCPHE'],
                'zscale': 1,
                'elevation' : QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['Vsurfrst'] = processing.run('grass7:v.surf.rst', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
            # alg['OUTPUT'] = context.getMapLayer(outputs['Vsurfrst']['elevation'])
            alg['OUTPUT'] = outputs['Vsurfrst']['elevation']
        else : 
            if parameters['interpolation'] == 1:
                i = 0
            elif parameters['interpolation'] == 2: 
                i = 1
            # v.surf.bspline
            alg_params = {
                'GRASS_MIN_AREA_PARAMETER': 0.0001,
                'GRASS_OUTPUT_TYPE_PARAMETER': 0,  # auto
                'GRASS_RASTER_FORMAT_META': '',
                'GRASS_RASTER_FORMAT_OPT': '',
                'GRASS_REGION_CELLSIZE_PARAMETER': parameters['rasterSize'],
                'GRASS_REGION_PARAMETER': None,
                'GRASS_SNAP_TOLERANCE_PARAMETER': -1,
                'GRASS_VECTOR_DSCO': '',
                'GRASS_VECTOR_EXPORT_NOCAT': False,
                'GRASS_VECTOR_LCO': '',
                'column': parameters['champCPHE'],
                'error': 1e-06,
                'ew_step': 4,
                'input': alg['OUTPUT'],
                'lambda_i': 0.01,
                'maxit': 10000,
                'memory': 300,
                'method': i,  
                'ns_step': 2,
                'solver': 0,  # cholesky
                'sparse_input': '',
                'raster_output': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['Vsurfbspline'] = processing.run('grass7:v.surf.bspline', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
            alg['OUTPUT'] =  outputs['Vsurfbspline']['raster_output']

        feedback.setCurrentStep(5)        
        if feedback.isCanceled():
            return 'cancel', None
          
         # Convertir
        alg_params = {
            'COPY_SUBDATASETS': False,
            'DATA_TYPE': 0,  # Utiliser le type de donnée de la couche en entrée
            'EXTRA': '',
            'INPUT':alg['OUTPUT'],
            'NODATA': 0,
            'OPTIONS': '',
            'TARGET_CRS': None,
            'OUTPUT':outputs['res_CPHE']
        }
        alg = processing.run('gdal:translate', alg_params, context=context, feedback=feedback, is_child_algorithm=True)


        feedback.setCurrentStep(6)
        if feedback.isCanceled():
            return 'cancel', None
            
        return  status,alg['OUTPUT']
        

    def interpolEscalier(self, parameters, context, feedback, outputs):
        status = 'ok'                
        # Calcul de la taille de la zone pour recalculer le buffer minimal du voronoi
        maxi = 0
        alg_params = {
            'INPUT': outputs['pre_traitement'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg2 = processing.run('native:polygonfromlayerextent', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        for f in context.getMapLayer(alg2['OUTPUT']).getFeatures():
            maxi = max(maxi, f.attribute('WIDTH'))
            maxi = max(maxi, f.attribute('HEIGHT'))
        percent = max(100*parameters['buffer']/maxi, 10)
        
        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return 'cancel', None  

        # Polygones de Voronoï
        alg_params = {
            'BUFFER': percent,
            'INPUT': outputs['centroides']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('qgis:voronoipolygons', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        
        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return 'cancel', None
            
        # Tampon
        alg_params = {
            'DISSOLVE': True,
            'DISTANCE': parameters['buffer'],
            'END_CAP_STYLE': 0,  # Rond
            'INPUT': outputs['pre_traitement'],
            'JOIN_STYLE': 0,  # Rond
            'MITER_LIMIT': 2,
            'SEGMENTS': 5,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['Tampon'] = processing.run('native:buffer', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(3)
        if feedback.isCanceled():
            return 'cancel', None

        # Intersection
        alg_params = {
            'GRID_SIZE': None,
            'INPUT': alg['OUTPUT'],
            'INPUT_FIELDS': None,
            'OVERLAY':  outputs['Tampon']['OUTPUT'],
            'OVERLAY_FIELDS': None,
            'OVERLAY_FIELDS_PREFIX': None,
            'OUTPUT':  QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('native:intersection', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        

        feedback.setCurrentStep(4)        
        if feedback.isCanceled():
            return 'cancel', None

        # Rasteriser (vecteur vers raster)
        alg_params = {
            'BURN': 0,
            'DATA_TYPE': 5,  # Float32
            'EXTENT': alg['OUTPUT'],
            'EXTRA': '',
            'FIELD': parameters['champCPHE'],
            'HEIGHT': parameters['rasterSize'],
            'INIT': None,
            'INPUT': alg['OUTPUT'],
            'INVERT': False,
            # 'NODATA': -9999,
            'NODATA': '',
            'OPTIONS': '',
            'UNITS': 1,  # Unités géoréférencées
            'USE_Z': False,
            'WIDTH': parameters['rasterSize'],
            'OUTPUT':outputs['res_CPHE']
        }
        alg = processing.run('gdal:rasterize', alg_params, context=context, feedback=feedback, is_child_algorithm=True)


        feedback.setCurrentStep(5)        
        if feedback.isCanceled():
            return 'cancel', None

        return status,alg['OUTPUT']