from qgis.core import QgsProcessing
from qgis.core import QgsProcessingAlgorithm
from qgis.core import QgsProcessingMultiStepFeedback
from qgis.core import QgsProcessingParameterRasterLayer
from qgis.core import QgsProcessingParameterNumber
from qgis.core import QgsProcessingParameterFeatureSink
from qgis.core import QgsProcessingParameterString
from qgis.core import QgsProcessingParameterDefinition
try:  from qgis.core import QgsProcessingParameterArea
except: pass
from qgis.core import QgsProcessingParameterRasterDestination
from qgis.core import QgsProcessingException
from qgis.core import QgsUnitTypes
from qgis.PyQt.QtCore import QCoreApplication
from qgis import processing

import os, tempfile

class ClassRasterisationProcessingAlgorithm(QgsProcessingAlgorithm):
    def name(self):
        return 'classrasterisation'

    def displayName(self):
        return self.tr('Rasteriser par classe')

    def tr(self, string):
        return QCoreApplication.translate('processing', string)

    def shortHelpString(self):
        return """<html><body><p>Cet algorithme crée une couche vecteur contenant toutes les classes de hauteurs d'eau renseignées dans "classes" à partir d'un MNT</p>
<br></body></html>"""

    def group(self):
        return self.tr('Scripts')

    def groupId(self):
        return 'scripts'

    def createInstance(self):
        return ClassRasterisationProcessingAlgorithm()

    NUMBERS = 'NUMBERS'
    file_null = os.path.join(tempfile.gettempdir(), 'null.tif')

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterRasterLayer('hauteurs', 'Raster des hauteurs d\'eau', defaultValue=None))
        self.addParameter(QgsProcessingParameterString(self.NUMBERS,'Classes (séparer les valeurs par un ";")',defaultValue='0;0.5;1;1.5;2'))
        self.addParameter(QgsProcessingParameterFeatureSink('output', 'Raster de classes', createByDefault=True, defaultValue=None))
        try:
            param = QgsProcessingParameterArea('tamisage', 'Tamisage :', minValue=0, defaultValue=400)
            param.setDefaultUnit(QgsUnitTypes.AreaSquareMeters )
        except:
            param = QgsProcessingParameterNumber('tamisage', 'Tamisage (m²) :', minValue=0, defaultValue=400)
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)
       

    def chemin(self):
        chem= tempfile.gettempdir()
        file_null = os.path.join(chem, 'null.tif')
        return file_null

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        results = {}
        rasters = []
        v=0
        # L=sorted(parameters[self.NUMBERS].split(';'))

        liste = parameters[self.NUMBERS].split(';')
        # L = list(map(float, liste))
        try: L = list(map(float, liste))
        except Exception as e: raise QgsProcessingException(f"Donnée invalide pour Classes={parameters[self.NUMBERS]}: {e}")
            
        L.sort(reverse=True)
        
        feedback = QgsProcessingMultiStepFeedback(12, model_feedback)
        
        expL = map(lambda x: f"(A>{x})", L)
        
        
        # print(" + ".join(expL))
        
        alg = processing.run('native:rasterlayerproperties', 
            {
                'INPUT': parameters['hauteurs']
            }, context=context, feedback=feedback, is_child_algorithm=True)
        parameters['rasterSize'] = min(alg['PIXEL_WIDTH'],alg['PIXEL_HEIGHT'])
        parameters['threshold'] = int(parameters['tamisage']/(alg['PIXEL_WIDTH']*alg['PIXEL_HEIGHT']))
        
        
        # r.mapcalc.simple1
        alg_params = {
            'GRASS_RASTER_FORMAT_META': '',
            'GRASS_RASTER_FORMAT_OPT': '',
            'GRASS_REGION_CELLSIZE_PARAMETER': 0,
            'GRASS_REGION_PARAMETER': '',
            'a': parameters['hauteurs'],
            'b': None,
            'c': None,
            'd': None,
            'e': None,
            'expression': " + ".join(expL),
            'f': None,
            'output': QgsProcessing.TEMPORARY_OUTPUT
        }
        alg = processing.run('grass7:r.mapcalc.simple', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        

        
        alg_params = {
            'EIGHT_CONNECTEDNESS': False,
            'EXTRA': '',
            'INPUT': alg['output'],
            'MASK_LAYER': None,
            'NO_MASK': False,
            'THRESHOLD': parameters['threshold'],
            'OUTPUT':  parameters['output']
        }
        alg = processing.run('gdal:sieve', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        
        return {'output': alg['OUTPUT']}
        
        
