from qgis.PyQt import QtWidgets, QtCore, QtGui
from qgis.core import (QgsProject,
                    QgsApplication,
                    QgsMapLayerProxyModel,
                    QgsProcessing,
                    QgsVectorLayer,
                    QgsRasterLayer,
                    QgsField,
                    QgsGeometry,
                    QgsMultiPoint,
                    QgsPointXY,
                    QgsFeature,
                    QgsProcessingParameterRasterDestination,
                    QgsProcessingFeatureSourceDefinition,
                    QgsEditorWidgetSetup,
                    QgsMapLayer,
                    QgsVectorLayerCache,
                    QgsFeatureRequest,
                    QgsProcessingContext,
                    QgsProcessingFeedback,
                    QgsWkbTypes,
                    QgsProcessingAlgRunnerTask,
                    QgsTask,
                    QgsProcessingAlgorithm,
                    QgsCategorizedSymbolRenderer,
                    QgsRendererCategory,
                    QgsSymbol,
                    QgsSettings,
                    QgsCategorizedSymbolRenderer,
                    QgsSymbol,
                    QgsRendererCategory,
                    QgsColorRamp,
                    QgsStyle,
                    QgsVectorLayerSimpleLabeling,
                    QgsFeatureRequest,
                    QgsFieldProxyModel,
                    QgsLayerTreeLayer,
                    QgsProperty
                    )

from qgis.gui import (QgsMapLayerComboBox, 
                    QgsFieldComboBox, 
                    QgsAttributeTableView, 
                    QgsAttributeTableModel, 
                    QgsMapTool, 
                    QgsRubberBand,
                    QgsMapToolExtent
                    )
from qgis.utils import iface
from qgis import processing
import os, configparser, time, traceback, random, math
from functools import partial

from .fmt_processing import launchMultiprocessing

def clearLayout(layout):
        try:
            for i in reversed(range(layout.count())): 
                l = layout.itemAt(i).layout()
                w = layout.itemAt(i).widget()
                if isinstance(w, QtWidgets.QComboBox):
                    w.addItem("", "")
                if isinstance(w, QgsMapLayerComboBox):
                    w.setAllowEmptyLayer(True)    
                if w:
                    layout.removeWidget(w)
                    w.setParent(None)
                    del w
                if l:
                    clearLayout(l)
                    layout.removeItem(l)
                    del l
        except:
            # print(layout, isinstance(layout, QtWidgets.QLayout))
            # print(sys.exc_info())
            pass


def makeGroup(name):
    group = QgsProject.instance().layerTreeRoot().findGroup(name)
    if not group:
        group = QgsProject.instance().layerTreeRoot().addGroup(name)
    return group


class fmtGroupBox(QtWidgets.QWidget):
    def __init__(self, txt="", title = True, parent=None):
        super().__init__( parent)
        self.title = title
        self.mainlayout = QtWidgets.QVBoxLayout(self)
        self.mainlayout.setContentsMargins(0,5,0,5)
        self.mainlayout.setSpacing(0)
              
        w = QtWidgets.QWidget()
        self.mainlayout.addWidget(w)
        l = QtWidgets.QHBoxLayout(w)   
        l.setContentsMargins(0,0,0,0)
        

        self.labelimg = fmtClickLabel(txt)
        self.labelimg.setMaximumWidth(15)
        self.state = 'minus'
        self.aff()
        
        l.addWidget(self.labelimg)    
        label = fmtClickLabel(txt)

        font = QtGui.QFont()
        font.setBold(True)
        label.setFont(font)
        l.addWidget(label)
        
        if not title :
            w.hide()
        
        self.box = QtWidgets.QGroupBox()        
        self.lay = QtWidgets.QVBoxLayout(self.box)     
        self.mainlayout.addWidget(self.box)

        self.labelimg.clicked.connect(self.change)
        label.clicked.connect(self.change)
                
    def layout(self):
        return self.lay
        
    def aff(self):
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"{self.state}.png"))
        self.labelimg.setPixmap(ico.pixmap(12,12))
        
    def change(self):
        if self.state=='minus': self.state = 'plus'
        else: self.state = 'minus'
        self.aff()
        self.box.setVisible(self.state=='minus')
    
    def clear(self):
        self.box.hide()
        self.box = QtWidgets.QGroupBox()        
        self.lay = QtWidgets.QVBoxLayout(self.box)     
        self.mainlayout.addWidget(self.box)
       
class fmtAddFeaturesLayer(QtWidgets.QWidget):
    # rightClicked = QtCore.pyqtSignal()
    layerChanged = QtCore.pyqtSignal()
    fieldChanged = QtCore.pyqtSignal()
    addClicked = QtCore.pyqtSignal()
    execSelected =QtCore.pyqtSignal(bool)
    selectionRequired = QtCore.pyqtSignal(QgsVectorLayer)
    def __init__(self , id = None, params = None, **kw):      
        super().__init__()
        self.kw = kw      
        self.vectParam = f"vect{id}"
        self.field = f"champ{id}"
        self.dict = f"dict{id}"
        self.params = params
        layout = QtWidgets.QVBoxLayout(self)
        layout.setContentsMargins(0,0,0,0)
        layout.setSpacing(0) 
        self.applyLegend = None
        
        if self.kw.get('title') :
            title = self.kw.get('title')
            layout.addWidget(QtWidgets.QLabel(title))
            
        self.layer = QgsMapLayerComboBox()        
        self.layer.setAllowEmptyLayer(True)
        self.layer.setLayer(None)

        self.layer.setToolTip("Clic droit pour plus d'options")
        if self.kw.get('polygon'):
            self.layer.setFilters(QgsMapLayerProxyModel.PolygonLayer) 
        else : 
            self.layer.setFilters(QgsMapLayerProxyModel.VectorLayer)         
       
        self.selection = QtWidgets.QCheckBox("Sélection")
        self.selection.setToolTip("Le calcul s'éxécutera uniquement sur les entitées séléctionnées")
        self.selection.clicked.connect(partial(self.enableSelection))


        h = QtWidgets.QHBoxLayout()
        layout.addLayout(h)
        h.setContentsMargins(0,0,0,0)
        h.setSpacing(0)
        h.addWidget(self.layer)
        
        getfile = fmtGetFile(self.vectParam, params = self.params)
        getfile.pathLoaded.connect(self.getFromFile)
        h.addWidget(getfile)            

        if self.kw.get('add'):
            l = fmtClickLabel()
            l.clicked.connect(self.addClicked.emit)
            l.setMaximumWidth(20)
            ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"add.png"))
            l.setPixmap(ico.pixmap(20,20))
            h.addWidget(l)
            self.addButton = l     
        
        self.ligne = QtWidgets.QWidget()
        self.ligne.hide()
        l = QtWidgets.QHBoxLayout(self.ligne)
        l.setContentsMargins(5,0,0,0)        
        l.setSpacing(20)
        l.setAlignment(QtCore.Qt.AlignmentFlag.AlignLeft)        
        layout.addWidget(self.ligne)
        
        self.champ = QgsFieldComboBox()
        # self.champ.setMinimumWidth(100)
        self.champ.view().setMinimumWidth(150)
        self.champ.setToolTip("Nom du champ")
        if kw.get('numeric'):
            self.champ.setFilters(QgsFieldProxyModel.Numeric)
        if kw.get('text'):
            self.champ.setFilters(QgsFieldProxyModel.String)
        
        l.addWidget(QtWidgets.QLabel("Champ :"))
        l.addWidget(self.champ)
        
        if self.kw.get('selection'):
            l.addWidget(self.selection)


        self.layer.layerChanged.connect(self.changeLayer)  
        self.layer.activated.connect(self.save)       
        self.champ.currentTextChanged.connect(self.changeField)
        self.champ.activated.connect(self.save)
        # self.initValue()
        
        self.lastLayer = None
        
    def fieldExist(self):
        field = self.params.get(self.field)
        if  field!= '':
            self.setField(field)
            return
        l = self.params.get(self.dict).split(",")
        for i in self.champ.fields().names():
            if i.casefold() in l :
                self.setField(i)
                break
                
    def save(self):
        if self.layer.currentLayer():
            self.params.save(self.vectParam, self.layer.currentLayer().dataProvider().dataSourceUri())
            self.params.save(self.field, self.champ.currentText())
        else : 
            self.params.save(self.vectParam, '')
            self.params.save(self.field, '')
    
    def setCurrentLayer(self, layer, save=True):
        try: self.layer.setLayer(layer) 
        except: 
            print(traceback.format_exc())
            pass
        if save:
            self.save()
    
    def currentIfaceLayer(self):
        return self.layer.currentLayer()
    
    def currentLayer(self, select = False):        
        if self.layer.currentLayer():
            if select: 
                return self.fmtselect()
            if self.selection.isEnabled() and  self.selection.isChecked():
                out = self.fmtselect()               
            else : 
                out = self.layer.currentLayer()
            # if isinstance(out, QgsMapLayer):
                # return out
            # try:
                # return QgsVectorLayer(out, self.title, 'ogr')
            # except:
                # print("impossible de charger", out)
            return out


    def chemin(self): 
        if self.layer.currentLayer():
            return self.layer.currentLayer().dataProvider().dataSourceUri()        

    def fieldText(self):
        return self.champ.currentText()
        
    def setField(self, txt):
        self.champ.setField(txt)
    
    def fmtselect(self):  
        alg = processing.run("native:savefeatures",
             {'INPUT': QgsProcessingFeatureSourceDefinition(self.layer.currentLayer().source(),  selectedFeaturesOnly=True),
             'OUTPUT':QgsProcessing.TEMPORARY_OUTPUT,
             'LAYER_NAME':'',
             'DATASOURCE_OPTIONS':'',
             'LAYER_OPTIONS':''
                }
            )       
        return QgsVectorLayer(alg['OUTPUT'], 'select', 'ogr')
        
    def changeLayer(self):
        try:
            self.lastLayer.selectionChanged.disconnect(self.enableSelection)
            self.lastLayer.configChanged.disconnect(self.fieldExist)
        except: pass
        
        
        self.champ.setLayer(self.layer.currentLayer())        
        # print(self.field, self.params.get(self.field))
        
        # try : self.setField(self.params.get(self.field))
        try : self.fieldExist()
        except: pass
        self.ligne.setVisible(self.layer.currentLayer() is not None)
        try: self.addButton.setVisible(self.layer.currentLayer() is None)
        except: pass
        self.champ.setLayer(self.layer.currentLayer())
        if self.layer.currentLayer():
            self.selection.setEnabled(self.layer.currentLayer().selectedFeatureCount()>0)
            self.layer.currentLayer().selectionChanged.connect(self.enableSelection)
            self.layer.currentLayer().configChanged.connect(self.fieldExist)
            self.lastLayer = self.layer.currentLayer()
            # self.memoLayer = self.layer.currentLayer() 
        # if not self.selection.isEnabled() and self.selection.isChecked():
            # self.selection.setChecked(False)            
        self.layerChanged.emit()

    def changeField(self):
        # self.params.save(self.field, self.champ.currentText())
        self.fieldChanged.emit()

    def enableSelection(self):
        self.selection.setEnabled(self.layer.currentLayer().selectedFeatureCount()>0)   
        # if self.selection.isEnabled() is False:
            # self.selection.setChecked(False)            
        if self.selection.isEnabled():
            self.execSelected.emit(self.selection.isChecked())
        else : self.execSelected.emit(True)
            
    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.MouseButton.RightButton:
            position = event.globalPos()
            self.openMenu(position)
            # self.rightClicked.emit()
          
    def setApplyLegend(self, function):
        self.applyLegend = function
    
    def openMenu(self, position):        
        if self.currentLayer() is None:
            return
        w = QtWidgets.QMenu(self)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"mActionSelectRectangle.png"))
        w.addAction(ico,"Sélectionner", lambda: self.selectionRequired.emit(self.layer.currentLayer()))  
        if self.applyLegend:
            ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"symbology.png"))
            w.addAction(ico,"Colorier", lambda: self.applyLegend(self.currentIfaceLayer(), champ=self.fieldText()))
        w.exec(position)

    def getFromFile(self, path):
        self.initValue()
    
    def initValue(self, group=None):  
        try: path,name = self.params.get(self.vectParam).split("|")
        except : path = self.params.get(self.vectParam).split("|")[0]
        try: name = name.split("=")[1]
        except: name = None
        
        try:
            if not os.path.exists(path) : 
                return
            found = False
            if name is None:
                name = os.path.basename(path)
                name = os.path.splitext(name)[0]
            for n,l in QgsProject.instance().mapLayers().items():
                p = l.dataProvider().dataSourceUri().split("|")[0]
                if p==path:
                    found = True
                    self.setCurrentLayer(l, save=False)   
                    # if group:
                        # group = makeGroup(group)
                        # group.addLayer(l)
                    break
            if not found:
                layer = QgsVectorLayer(path, name, "ogr")
                if group:
                    group = makeGroup(group)
                    QgsProject.instance().addMapLayer(layer, False)
                    group.addLayer(layer)
                else:
                    QgsProject.instance().addMapLayer(layer)
                self.setCurrentLayer(layer, save=False) 
        except:
            print(traceback.format_exc())
            pass
            
    def name(self):
        if self.layer.currentLayer():
            return self.layer.currentText()
        else : 
            return ""
            
class fmtSelectionRequired(QtWidgets.QWidget):
    selectionRequired = QtCore.pyqtSignal()
    def __init__(self):
        super().__init__()

              
class fmtRaster(QtWidgets.QWidget):
    layerChanged = QtCore.pyqtSignal(QgsMapLayer)
    def __init__(self, name=None, title=None, params=None, **kw):
        super().__init__()
        self.title = title
        self.name = name
        self.params = params
        self.notFirst = False    
        self.insertLast = kw.get('insertLast')
        l = QtWidgets.QVBoxLayout(self)
        l.setContentsMargins(0,0,0,5)
        l.setSpacing(0) 
        if title : 
            l.addWidget(QtWidgets.QLabel(self.title))
        w = QtWidgets.QWidget()
        m = QtWidgets.QHBoxLayout(w)
        m.setContentsMargins(0,0,0,0)
        m.setSpacing(0) 
        self.raster = QgsMapLayerComboBox()
        self.raster.activated.connect(self.save)
        self.raster.setFilters(QgsMapLayerProxyModel().RasterLayer)
        self.raster.setAllowEmptyLayer(True)
        self.raster.setLayer(None)
        m.addWidget(self.raster) 
        getfile = fmtGetFile(self.name, params = self.params)
        getfile.pathLoaded.connect(self.getFromFile)
        m.addWidget(getfile)
        l.addWidget(w)
        self.raster.layerChanged.connect(partial(self.changedLayer))
        # self.initValue()

    def save(self):
        if self.raster.currentLayer():
            self.params.save(self.name, self.raster.currentLayer().dataProvider().dataSourceUri())
        else : 
            self.params.save(self.name, '')
    
    def updateCombo(self, otherLayer):
        if self.notFirst :
            return
        self.notFirst = True
        if otherLayer is None :
            self.raster.blockSignals(True)
            self.raster.setCurrentIndex(0)
            self.raster.blockSignals(False)            
        else : 
            if self.raster.currentLayer() != otherLayer :
                self.raster.blockSignals(True)
                self.raster.setLayer(otherLayer)  
                self.raster.blockSignals(False)
        self.notFirst = False

    def changedLayer(self):
        self.layerChanged.emit(self.raster.currentLayer())           
     
    def currentLayer(self):
        return self.raster.currentLayer()
    
    def getRaster(self) : 
        return self.raster
    
    def setLayer(self, layer, save=True):
        self.raster.setLayer(layer)
        if save:
            self.save()
            
    def getFromFile(self, path):
        self.initValue()
        
    def initValue(self, group=None):  
        tt = self.params.get(self.name).split("|")
        path = tt[0]
        try: name = tt[1].split("=")[1]
        except: name = None
        try:
            if not os.path.exists(path): return
            found = False
            if name is None:
                t = os.path.basename(path)
                t = os.path.splitext(t)[0]
            else:
                t = name
            if t is None:
                t = self.name
            for n,l in QgsProject.instance().mapLayers().items():
                p = l.dataProvider().dataSourceUri().split("|")[0]
                if p==path:
                    found = True
                    self.raster.setLayer(l)   
                    # if group:
                        # group = makeGroup(group)
                        # if self.insertLast:
                            # group.insertChildNode(-1, QgsLayerTreeLayer(l))
                        # else:
                            # group.addLayer(l)
                    break
            if not found:
                layer = QgsRasterLayer(path, t)
                if group:
                    group = makeGroup(group)
                    QgsProject.instance().addMapLayer(layer, False)
                    if self.insertLast:
                        group.insertChildNode(-1, QgsLayerTreeLayer(layer))
                    else:
                        group.addLayer(layer)
                else:
                    QgsProject.instance().addMapLayer(layer, not self.insertLast)
                    if self.insertLast:
                        QgsProject.instance().layerTreeRoot().insertChildNode(-1, QgsLayerTreeLayer(layer)) 

                self.raster.setLayer(layer) 
        except:
            print(traceback.format_exc())
            pass

class fmtClickLabel(QtWidgets.QLabel):
    clicked = QtCore.pyqtSignal()
    enterOverflown = QtCore.pyqtSignal()
    leaveOverflown = QtCore.pyqtSignal()
    rightClicked = QtCore.pyqtSignal(QtCore.QPoint)
    def __init__(self, txt="", hovered = None, parent=None):
        super().__init__(txt, parent)
        self.hovered = hovered
        self.setCursor(QtCore.Qt.CursorShape.PointingHandCursor)
        if self.hovered : 
            self.setMouseTracking(True)
            self.base = self.font()
            self.hover = QtGui.QFont(self.base)
            self.hover.setBold(True)
            self.hover.setPointSizeF(self.base.pointSizeF() * 1.05)

    def enterEvent(self, event):
        if not self.hovered : return
        self.setFont(self.hover)
        self.enterOverflown.emit()
        super().enterEvent(event)

    def leaveEvent(self, event):
        if not self.hovered : return 
        self.setFont(self.base)
        self.leaveOverflown.emit()
        super().leaveEvent(event)

    def mousePressEvent(self, e):
        # super().mousePressEvent(e)
        self.clicked.emit()
        
    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.MouseButton.RightButton:
            position = event.globalPos()
            self.rightClicked.emit(position)
        else : 
            self.clicked.emit()
   
class fmtComboBox(QtWidgets.QComboBox):
    def __init__(self, name, params, parent=None):
        super().__init__(parent)
        self.params = params 
        self.name = name
        self.currentIndexChanged.connect(self.save)

    def save(self):
        self.params.save(self.name, self.currentData())

    def initValue(self):
        self.setCurrentIndex(self.findData(self.params.get(self.name)))

    def addItem(self, *v):
        ico, txt, data = None, "", None
        try: ico, txt, data = v
        except:
            try: txt, data = v
            except: txt = v
        self.blockSignals(True)
        if ico:
            super().addItem(ico, txt, data)
        else:
            super().addItem(txt, data)
        self.blockSignals(False)  

    def addItems(self, lst):    
        for e in lst:
            self.addItem(e,e)

class fmtSpatialisationComboBox(fmtComboBox):   
    def __init__(self, spaDict, *v, **kw):
        empty = kw.pop('empty',None)
        self.spaDict = spaDict
        super().__init__(*v, **kw)
        if empty:
            self.addItem("", "")
        for k,v in spaDict.items() :
            self.addItem(v.get('name',k), k)
        self.initValue()
        
    def getIndice(self):
        return self.spaDict.get(self.currentData(),{}).get('indice')
            
            
class fmtSlider(QtWidgets.QSlider):
    def __init__(self, orientation=QtCore.Qt.Orientation.Horizontal, decimals=2):
        super().__init__(orientation)
        self.setDecimals(decimals)
        
    def coeff(self):
        return 10**self.decimals
    
    def setDecimals(self, decimals):
        self.decimals = decimals
    
    def setMaximum(self, v):
        super().setMaximum(int(v*self.coeff()))
    def maximum(self):
        return super().maximum()/self.coeff()
        
    def setMinimum(self, v):
        super().setMinimum(int(v*self.coeff()))
    def minimum(self):
        return super().minimum()/self.coeff()  
    
    def setRange(self, v1, v2):
        self.setMinimum(v1)
        self.setMaximum(v2)
        
    def value(self):
        return super().value()/self.coeff()  
        
    def setValue(self, v):
        super().setValue(int(v*self.coeff()))
        
    def setPageStep(self, v) :
        super().setPageStep(int(v*self.coeff()))
        
    def setSingleStep(self, v) :
        super().setSingleStep(int(v*self.coeff()))

UNITE = {
    "mm": 0.001,
    "cm": 0.01,
    "dm": 0.1,
    "m": 1,
    "dam": 10,
    "hm": 100,
    "km": 1000,
}        
class fmtSetValue(QtWidgets.QWidget):
    rightClicked = QtCore.pyqtSignal()
    valueChanged = QtCore.pyqtSignal(float)
    def __init__(self, name, title="", tooltip=False, params=None, **kw):
        super().__init__()
        self.title = title
        self.params = params 
        self.name = name
        self.tooltip = tooltip
        self.noUnity = kw.get('noUnity')
        self.area = kw.get('area')
        self.strictPositive = kw.get('strictPositive')
        self.absoluteMin = kw.get('absoluteMin')
        self.absoluteMax = kw.get('absoluteMax') 
        l =  QtWidgets.QHBoxLayout(self) 
        l.setContentsMargins(0,0,0,0)
        l.setSpacing(0)
        self.paint()
    
    def paint(self):
        clearLayout(self.layout())
        self.unite = self.params.get(f"{self.name}Unity")
        self.decimals = self.params.get(f"{self.name}Decimals") or 0
        if self.strictPositive:
            self.absoluteMin = 10**(-self.decimals)
        self.defaultValue = self.params.get(f"{self.name}Default")
        self.max = self.params.get(f"{self.name}Max")
        self.min = self.params.get(f"{self.name}Min")       
        if self.absoluteMin is not None: self.min = max(self.min, self.absoluteMin)
        if self.absoluteMax is not None: self.max = min(self.max, self.absoluteMax)
        if self.noUnity:
            self.unite = None
            self.decimals = 0
        label = QtWidgets.QLabel(self.title)
        self.layout().addWidget(label)
        self.slider = fmtSlider(decimals=self.decimals)   
        self.slider.setRange(self.min, self.max)
        self.slider.setValue(self.defaultValue)      
        self.spinbox = QtWidgets.QDoubleSpinBox()
        self.spinbox.setDecimals(self.decimals)
        self.spinbox.setStepType(QtWidgets.QAbstractSpinBox.StepType.AdaptiveDecimalStepType)
        self.spinbox.setRange(self.min, self.max)
        self.spinbox.setValue(self.defaultValue)
        self.layout().addWidget(self.slider)     
        self.layout().addWidget(self.spinbox)
        self.slider.valueChanged.connect(partial(self.changedValue, self.slider, self.spinbox))
        self.spinbox.valueChanged.connect(partial(self.changedValue, self.spinbox, self.slider))
        self.setToolTip("Clique droit pour modifier les bornes")  
        if self.tooltip:
            label.setToolTip(self.tooltip)
        if self.unite :
            txt = f" {self.unite}"
            if self.area: txt += '²'
            self.spinbox.setSuffix(txt)

    def value(self):
        coeff = 1
        if self.unite: coeff = UNITE.get(self.unite, 1)
        if self.area: coeff = coeff*coeff
        return self.spinbox.value() *coeff

        
    def changedValue(self, ori, dest):
        dest.blockSignals(True)
        dest.setValue(ori.value())
        dest.blockSignals(False)
        self.valueChanged.emit(ori.value()) 
        self.params.save(f"{self.name}Default", ori.value())
   
    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.MouseButton.RightButton:
            position = event.globalPos()
            self.openMenu(position)
            self.rightClicked.emit()
   
    def openMenu(self, position):        
        w = QtWidgets.QMenu(self)   
        w.addAction("Modifier les bornes", partial(fmtTakeNewValue, self))
        if self.unite:
            w.addAction("Modifier l'affichage", partial(fmtTakeNewAff, self))
        w.exec(position)
        
class fmtTakeNewValue(QtWidgets.QWidget):
    def __init__(self, parent):
        super().__init__()
        self.parent = parent
        if isinstance(parent, fmtSetValue): self.fmtValue = parent
        else: self.fmtValue = parent.fmtValue
        self.params = self.fmtValue.params 
        title = self.fmtValue.title
        if title=="": title = self.fmtValue.name
        dlg = QtWidgets.QDialog()
        dlg.setWindowTitle(title)
        dlg.setWindowIcon(QtGui.QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        minl = QtWidgets.QLabel("Borne minimale : ")
        minValue = QtWidgets.QDoubleSpinBox()
        maxl = QtWidgets.QLabel("Borne maximale : ")
        maxValue =QtWidgets.QDoubleSpinBox()
        for w in (minValue,maxValue):
            w.setRange(-9999999,9999999)
            w.setDecimals(self.fmtValue.decimals) 
            if self.fmtValue.unite: w.setSuffix(self.fmtValue.spinbox.suffix()) 
            if self.fmtValue.absoluteMin is not None:
                w.setMinimum(self.fmtValue.absoluteMin)
            if self.fmtValue.absoluteMax is not None:
                w.setMinimum(self.fmtValue.absoluteMax)
        minValue.setValue(self.fmtValue.min)
        maxValue.setValue(self.fmtValue.max)
        ok = QtWidgets.QPushButton("OK")
        ok.clicked.connect(lambda :self.save( minValue.value(), maxValue.value()))
        ok.clicked.connect(lambda : dlg.accept())
        cancel = QtWidgets.QPushButton("Annuler")
        cancel.clicked.connect(lambda : dlg.reject()) 
        layout = QtWidgets.QVBoxLayout(dlg)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(minl)
        l.addWidget(minValue)
        layout.addLayout(l)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(maxl)
        l.addWidget(maxValue)
        layout.addLayout(l)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(ok)
        l.addWidget(cancel)
        layout.addLayout(l)      
        dlg.exec()
   
    def save(self,min,max):
        self.params.save(f"{self.fmtValue.name}Min", min)
        self.params.save(f"{self.fmtValue.name}Max", max)
        self.parent.paint()

class fmtTakeNewAff(QtWidgets.QWidget):
    def __init__(self, parent):
        super().__init__()
        self.parent = parent
        if isinstance(parent, fmtSetValue): self.fmtValue = parent
        else: self.fmtValue = parent.fmtValue
        self.params = self.fmtValue.params  
        title = self.fmtValue.title
        if title=="": title = self.fmtValue.name
        dlg = QtWidgets.QDialog()
        dlg.setWindowTitle(title)
        dlg.setWindowIcon(QtGui.QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        self.uniteValue = QtWidgets.QComboBox()
        for k,v in UNITE.items():
            aff = k
            if self.fmtValue.area: aff+='²'
            self.uniteValue.addItem(aff,k)
        self.uniteValue.setCurrentIndex(self.uniteValue.findData(self.fmtValue.unite))
        self.decimalsValue =QtWidgets.QSpinBox()
        self.decimalsValue.setRange(0,3)
        self.decimalsValue.setValue(self.fmtValue.decimals)
        self.uniteValue.currentIndexChanged.connect(self.adaptDecimals)
        ok = QtWidgets.QPushButton("OK")
        ok.clicked.connect(lambda :self.save( self.uniteValue.currentData(), self.decimalsValue.value()))
        ok.clicked.connect(lambda : dlg.accept())
        cancel = QtWidgets.QPushButton("Annuler")
        cancel.clicked.connect(lambda : dlg.reject()) 
        layout = QtWidgets.QVBoxLayout(dlg)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(QtWidgets.QLabel("Unité : "))
        l.addWidget(self.uniteValue)
        layout.addLayout(l)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(QtWidgets.QLabel("Décimales : "))
        l.addWidget(self.decimalsValue)
        layout.addLayout(l)
        l = QtWidgets.QHBoxLayout()
        l.addWidget(ok)
        l.addWidget(cancel)
        layout.addLayout(l)      
        dlg.exec()
        
    def adaptDecimals(self):
        d = round(math.log(UNITE.get(self.fmtValue.unite,1)/UNITE.get(self.uniteValue.currentData(),1), 10))
        self.decimalsValue.setValue(self.fmtValue.decimals-d)
   
    def save(self, unite, decimals):
        coeff = UNITE.get(self.fmtValue.unite,1)/UNITE.get(unite,1)
        self.params.save(f"{self.fmtValue.name}Unity", unite)
        self.params.save(f"{self.fmtValue.name}Decimals", decimals)
        self.params.save(f"{self.fmtValue.name}Min", self.fmtValue.min*coeff)
        self.params.save(f"{self.fmtValue.name}Max", self.fmtValue.max*coeff)
        self.params.save(f"{self.fmtValue.name}Default", self.fmtValue.defaultValue*coeff)
        if isinstance(self.parent, fmtThresholdInputWidget):
            self.parent.recalc(coeff)
        self.parent.paint()
        

class fmtLoadLayer(QtWidgets.QWidget):
    textChanged = QtCore.pyqtSignal(str)
    def __init__(self, tuple=(), params = None):
        super().__init__()
        self.params = params
        tupleKey = tuple[0]
        tupleValue = tuple[1]

        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(0,0,0,5)
        l.setSpacing(0)  
        l.addWidget(QtWidgets.QLabel(f"{tupleKey} : "))
        if os.path.exists(self.params.get(tupleValue)):
            txt = self.params.get(tupleValue)
        else :
            txt = ""
        self.link = QtWidgets.QLineEdit(txt)
        self.link.setToolTip(txt)
        l.addWidget(self.link)  
        get = fmtGetFile(parameter = tupleValue, params = self.params)
        l.addWidget(get)    
        
        self.link.setText(get.getPath())
        self.link.setToolTip(get.getPath())    

    def changedText(self, k, v, txt) :
        self.params.save(v, txt)
        if self.params.get('activeSession')!= '':
            self.params.save(v, txt, self.params.get('activeSession'))
        self.link.setText(txt)
        self.textChanged.emit(self.link.text())
    
    def setText(self, txt):
        if self.link.text() != txt :
            self.link.setText(txt) 
    
    def text(self):
        return self.link.text()
    
class fmtClickDoubleSpinbox(QtWidgets.QDoubleSpinBox): 
    rightClicked = QtCore.pyqtSignal(QtWidgets.QDoubleSpinBox,QtCore.QPoint)
    def __init__(self, parent = None):
        super().__init__(parent)

    def contextMenuEvent(self, event):
        self.rightClicked.emit(self, event.globalPos())
        event.accept()

class MultiSliderWidget(QtWidgets.QWidget):
    thresholdsChanged = QtCore.pyqtSignal(list)
    lineRightClicked = QtCore.pyqtSignal(float)
    circleRightClicked = QtCore.pyqtSignal(int)
    def __init__(self, vmin=0.0, vmax=1.0, values=None, parent=None):
        super().__init__(parent)
        self.setMinimumHeight(25)
        self.vmin = vmin
        self.vmax = vmax
        self.thresholds = values if values else []
        self.handleRadius = 6
        self.draggingIndex = None
        self.setMouseTracking(True)

    def setThresholds(self, values, vmin=None, vmax=None, mini=None, maxi=None):
        self.thresholds = list(values)
        self.vmin = vmin if vmin is not None else min(values) if values else 0
        self.vmax = vmax if vmax is not None else max(values) if values else 1
        
        self.margeMin = (1 + 2*min(abs(min(values)-mini),1)) * (self.handleRadius+1)
        self.margeMax = (1 + 2*min(abs(max(values)-maxi),1)) * (self.handleRadius+1)
        
        self.update()

    def paintEvent(self, event):
        painter = QtGui.QPainter(self)
        lineY = self.rect().center().y()
        painter.setPen(QtGui.QPen(QtGui.QColor("gray"), 2))
        painter.drawLine(self.rect().left(), lineY, self.rect().right(), lineY)

        painter.setBrush(QtGui.QBrush(QtGui.QColor("blue")))
        for val in self.thresholds:
            if self.vmax == self.vmin:
                continue
            x = self._valueToPos(val)
            painter.drawEllipse(QtCore.QPointF(x, lineY), self.handleRadius, self.handleRadius)

    def _valueToPos(self, value):
        return self.margeMin + (value - self.vmin) / (self.vmax - self.vmin) * (self.width()-self.margeMin-self.margeMax)

    def _posToValue(self, x):
        return self.vmin + (x-self.margeMin) / (self.width()-self.margeMin-self.margeMax) * (self.vmax - self.vmin)

    def mousePressEvent(self, event):
        x = event.pos().x()
        found = False
        for i, val in enumerate(self.thresholds):
            if abs(x - self._valueToPos(val)) <= self.handleRadius + 2:
                if event.button() == QtCore.Qt.MouseButton.RightButton:
                    self.circleRightClicked.emit(i)
                    return
                found = True
                self.draggingIndex = i
                break
        if not found:
            if event.button() == QtCore.Qt.MouseButton.RightButton:
                self.lineRightClicked.emit(self._posToValue(x))

    def mouseMoveEvent(self, event):
        if self.draggingIndex is not None:
            x = event.pos().x()
            value = self._posToValue(x)
            self.thresholds[self.draggingIndex] = value
            self.update()
            self.thresholdsChanged.emit(list(self.thresholds))

    def mouseReleaseEvent(self, event):
        self.draggingIndex = None


class fmtThresholdInputWidget(QtWidgets.QWidget):
    miniChanged = QtCore.pyqtSignal()
    def __init__(self, name, title="", params=None, **kw):
        super().__init__()
        self.name = name
        self.params = params
        layout = QtWidgets.QVBoxLayout(self)
        layout.setContentsMargins(0,0,0,20)
        self.paint()
    
    def paint(self):
        clearLayout(self.layout())
        self.mini = self.params.get(f"{self.name}Min")
        self.maxi = self.params.get(f"{self.name}Max")
        self.unite = self.params.get(f"{self.name}Unity")
        self.decimals = self.params.get(f"{self.name}Decimals") or 0
        self.pas = 1
        self.fmtValue = fmtSetValue(self.name, params=self.params, default=0)
        
        w = QtWidgets.QWidget()
        self.layout().addWidget(w)
        l = QtWidgets.QHBoxLayout(w)
        l.setContentsMargins(0,0,0,0)
        label = fmtClickLabel(f"Classes de hauteurs ({self.unite}): ")
        label.setToolTip('clique droit pour Sauvegarder/Restaurer les classes') 
        label.rightClicked.connect(self.openMenuTitle)
        l.addWidget(label)
        sBtn = fmtClickLabel()
        sBtn.setMaximumWidth(19)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"save.png"))
        sBtn.setPixmap(ico.pixmap(19,19))
        sBtn.setToolTip("Enregistrer ces valeurs comme classes par default")    
        l.addWidget(sBtn)
        rBtn = fmtClickLabel()
        rBtn.setMaximumWidth(20)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"restart.png"))
        rBtn.setPixmap(ico.pixmap(20,20))
        rBtn.setToolTip("Valeurs par default")    
        l.addWidget(rBtn)
        sBtn.clicked.connect(self.saveDefaultParameters)
        rBtn.clicked.connect(self.defaultParameters)
        
        self.thresholdWidgets = []
        self.slider = MultiSliderWidget()
        self.layout().addWidget(self.slider)
        self.slider.thresholdsChanged.connect(self.updateFromSlider)
        self.thresholdLayout = QtWidgets.QHBoxLayout()
        self.layout().addLayout(self.thresholdLayout)
        self.slider.circleRightClicked.connect(self.openFromSliderCircle)
        self.slider.lineRightClicked.connect(self.openFromSliderLine)
        self.thresholdLayout.setContentsMargins(0,0,0,0)
        self.thresholdLayout.setSpacing(0)     
        self.initValues()
    
    def recalc(self, coeff):
        for s in ("","Memo"):
            l = [float(val)*coeff for val in self.params.get(self.name+s).split(";")]
            self.params.save(self.name+s, ";".join(f"{v:.2f}" for v in l))
        
    def openMenuTitle(self, position):
        menu = QtWidgets.QMenu()
        menu.addAction("Enregistrer ces valeurs comme classes par default", partial(self.saveDefaultParameters))
        menu.addAction("Restaurer les valeurs par default", partial(self.defaultParameters))
        menu.addSeparator()
        menu.addAction("Modifier les bornes", partial(fmtTakeNewValue, self))
        menu.addAction("Modifier l'affichage", partial(fmtTakeNewAff, self))
        menu.exec_(position)
    
    def defaultParameters(self):
        clearLayout(self.thresholdLayout)
        self.thresholdWidgets = []
        self.params.save(self.name, self.params.get(self.name+"Memo"))
        self.initValues()    

    def saveDefaultParameters(self):
        self.params.save(self.name+"Memo", self.params.get(self.name))     
        
    def initValues(self):
        l = [float(val) for val in self.params.get(self.name).split(";")]
        for val in l:
            self.addThreshold(val, init=True)
        # self.sortThresholds()
        self.updateVisuals(save=False)

    def addThreshold(self, val=0.0, init=False):
        spinbox = fmtClickDoubleSpinbox()
        spinbox.setDecimals(self.decimals)
        self.adapSpinboxlength(spinbox)
        spinbox.setRange(self.mini, self.maxi)
        spinbox.setStepType(QtWidgets.QAbstractSpinBox.StepType.AdaptiveDecimalStepType)
        spinbox.valueChanged.connect(partial(self.adapSpinboxlength, spinbox))
        spinbox.setValue(val)
        spinbox.rightClicked.connect(self.openMenu)
        spinbox.editingFinished.connect(self.onSpinboxChanged)
        self.thresholdWidgets.append(spinbox)
        self.thresholdLayout.addWidget(spinbox)
        if not init:
            self.sortThresholds()
            self.updateVisuals()

    def removeThreshold(self, spinbox):
        self.thresholdLayout.removeWidget(spinbox)
        spinbox.setParent(None)
        self.thresholdWidgets.remove(spinbox)
        self.sortThresholds()
        self.updateVisuals()

    def sortThresholds(self):
        self.thresholdWidgets.sort(key=lambda sb: sb.value())
        for i in reversed(range(self.thresholdLayout.count())):
            self.thresholdLayout.itemAt(i).widget().setParent(None)
        for sb in self.thresholdWidgets:
            self.thresholdLayout.addWidget(sb)

    def onSpinboxChanged(self):
        self.sortThresholds()
        self.updateVisuals()

    def adapSpinboxlength(self, w):
        d = int(math.log10(abs(w.value()))) if abs(w.value())>1 else 0
        s = 35 + d*5 + self.decimals*5 + (w.value()<0)*5
        w.setMaximumWidth(s)
    
    def updateFromSlider(self, values):
        values.sort()
        if len(values) != len(self.thresholdWidgets):
            self.rebuildFromValues(values)
        else:
            for val, sb in zip(values, self.thresholdWidgets):
                sb.blockSignals(True)
                sb.setValue(val)
                sb.blockSignals(False)
        self.sortThresholds()
        self.updateVisuals()


    def rebuildFromValues(self, values):
        for sb in self.thresholdWidgets:
            self.thresholdLayout.removeWidget(sb)
            sb.setParent(None)
        self.thresholdWidgets.clear()
        for val in sorted(values):
            self.addThreshold(val)

    def updateVisuals(self, save=True):
        values = [sb.value() for sb in self.thresholdWidgets]
        if values:
            self.slider.setThresholds(values, vmin=min(values), vmax=max(values), mini=self.mini, maxi=self.maxi)
        if save:
            self.params.save(self.name, self.getThresholds())
        self.miniChanged.emit()

    def openMenu(self, widget, position):
        menu = QtWidgets.QMenu()
        action = menu.addAction("Supprimer cette classe")
        action.triggered.connect(partial(self.removeThreshold, widget))
        menu.addSeparator()
        mini = max([sb.value() for sb in self.thresholdWidgets if sb.value()<widget.value()-0.02], default=self.mini)
        if widget.value()>mini+0.01:
            if mini>self.mini: val = (widget.value()+mini)/2
            else: val = max(widget.value()-self.pas, mini)
            menu.addAction("Ajouter avant", partial(self.addThreshold, val))
        maxi = min([sb.value() for sb in self.thresholdWidgets if sb.value()>widget.value()+0.02], default=self.maxi)
        if widget.value()<maxi-0.01:
            if maxi<self.maxi: val = (widget.value()+maxi)/2
            else: val = min(widget.value()+self.pas, maxi)
            menu.addAction("Ajouter après", partial(self.addThreshold, val))
        menu.exec_(position)
       
    def openFromSliderCircle(self, index):
        widget = self.thresholdWidgets[index]
        self.openMenu(widget, QtGui.QCursor.pos())
        
    def openFromSliderLine(self, val):
        menu = QtWidgets.QMenu()
        menu.addAction("Ajouter ici", partial(self.addThreshold, val))    
        menu.exec_(QtGui.QCursor.pos())
    
    def getThresholds(self, as_float=False):
        values = sorted([sb.value() for sb in self.thresholdWidgets])
        if as_float:
            return values
        return ";".join(f"{v:.2f}" for v in values)
        
    def getValues(self):
        coeff = UNITE.get(self.unite, 1)
        values = self.getThresholds(as_float=True)
        return ";".join(f"{v*coeff:.2f}" for v in values)
    
    def getMinim(self):
        coeff = UNITE.get(self.unite, 1)
        values = self.getThresholds(as_float=True)
        return values[0]*coeff
       
        
        
class fmtGetFile(QtWidgets.QWidget):
    pathLoaded = QtCore.pyqtSignal(str)
    def __init__(self, parameter = '', params = None, parent=None):
        super().__init__(parent)
        self.parameter = parameter
        self.params = params
        self.path = ''
        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(3, 0, 3, 0)
        l.setSpacing(0)
        but = QtWidgets.QPushButton('...')
        but.clicked.connect(self.getFile)
        but.setToolTip("Sélectionner l'entrée")
        self.setFixedWidth(25)
        l.addWidget(but)
        
        
    def getFile(self):    
        lastPath = self.params.get('datasFile') or os.path.expanduser("~")
        if not os.path.isdir(lastPath):
            lastPath = os.path.expanduser("~") 
        self.path, _ = QtWidgets.QFileDialog.getOpenFileName(self, "Sélectionner un fichier", lastPath, "Tous les fichiers (*)") 
        if self.path:
            file = os.path.dirname(self.path)
            self.params.save('datasFile', file)
            try :
                self.params.save(self.parameter, self.path)
            except :
                print(traceback.format_exc())
                pass
            self.pathLoaded.emit(self.path)
   
    def getPath(self):
        if self.path != '':
            return self.path

class fmtExecuteButton(QtWidgets.QWidget):    
    def __init__(self, dock, alg, params, **kw):
        self.dock = dock
        super().__init__()
        QtWidgets.QVBoxLayout(self)
        self.alg = alg
        self.params = params
       
        self.fmtProcessing = launchMultiprocessing(self.dock, noAffichage=True)
        self.fmtProcessing.finished.connect(self.recupLayers)
        
        h = QtWidgets.QHBoxLayout()
        self.layout().addLayout(h)
            
        w = QtWidgets.QPushButton("Exécuter")
        w.clicked.connect(self.launch) 
        h.addWidget(w)
        self.execute = w
        w = fmtClickLabel()
        w.setMaximumWidth(25)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"sensitive.png"))
        w.setPixmap(ico.pixmap(20,20))
        w.clicked.connect(self.switchMode) 
        h.addWidget(w)
        self.test = w
        
        w = QtWidgets.QLabel()
        font = QtGui.QFont()
        font.setItalic(True)
        w.setFont(font)
        w.hide()
        self.layout().addWidget(w)
        self.tryLabel = w
        
        self.result = None
        self.mode = False
        self.setWidgets(kw.get('widgets',[]))
        self.setCuts(kw.get('cuts',[]))
        self.setSignals(kw.get('signals',[]))
        
        self.inputs = {}
        
        if not kw.get('sensitive',True):
            self.test.hide()
        
    def setWidgets(self, lst=[]):
        lst.append(self.execute)
        self.widgets = lst
        
    def setCuts(self, lst):
        self.cuts = lst
        
    def setSignals(self, lst):
        for s in lst:
            s.connect(self.tryZone)
        
    def launch(self):
        params = {}
        for p,v in self.params.items():
            if callable(v):
                params[p] = v() 
            else:
                params[p] = v
        self.dock.fmtProcessing.run(self.alg, params)     
         
    def switchMode(self):
        self.mode = not self.mode
        for w in self.widgets:
            w.setEnabled(not self.mode)
        if self.mode:
            self.tryLabel.setText("Sélectionnez une zone sur le canvas")
            self.tryLabel.setStyleSheet("")
            self.tryLabel.show()
            ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"ko.png"))
            self.test.setPixmap(ico.pixmap(20,20))
            self.execute.setText("Test sensibilité")
            self.dock.setTryTool()
            self.dock.tryTool.drawFinished.connect(self.cutZone)
        else:
            self.tryLabel.hide()
            ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"sensitive.png"))
            self.test.setPixmap(ico.pixmap(20,20))
            self.execute.setText("Exécuter")
            self.dock.setTryTool(False)
            try: self.dock.tryTool.drawFinished.disconnect(self.cutZone)
            except: pass
            try: self.result.commitChanges()
            except: pass
            try: QgsProject.instance().removeMapLayer(self.result)
            except: pass
            self.dock.canvas.refresh()
            self.result = None
            
    def cutZone(self, lst):
        self.inputs = {}
        crs_id = QgsProject.instance().crs().authid()
        if crs_id in (None, ""): crs_id = "EPSG:2154"
        zone = QgsVectorLayer(f"Polygon?crs={crs_id}", "zone", "memory")
        zone.startEditing()
        geom = QgsGeometry.fromPolygonXY([lst])
        feat = QgsFeature()
        feat.setGeometry(geom)
        zone.addFeature(feat)
        zone.commitChanges()
        bb = geom.boundingBox()
        projwin = f"{bb.xMinimum()},{bb.xMaximum()},{bb.yMinimum()},{bb.yMaximum()} [{crs_id}]"
        self.area = geom.area()
        for k in self.cuts:
            l = self.params[k]
            if callable(l): l = l()  
            if isinstance(l, QgsVectorLayer):
                try: 
                    alg = processing.run('native:clip', 
                    { 
                        'INPUT' : l,
                        'OVERLAY' : zone, 
                        'OUTPUT' :QgsProcessing.TEMPORARY_OUTPUT,   
                    })
                    self.inputs[k] = alg['OUTPUT']
                except:
                    pass
            if isinstance(l, QgsRasterLayer):
                try: 
                    alg = processing.run('gdal:cliprasterbyextent', 
                    { 
                        'INPUT' : l,
                        'PROJWIN' :projwin, 
                        'OUTPUT' :QgsProcessing.TEMPORARY_OUTPUT,   
                    })
                    self.inputs[k] = alg['OUTPUT']
                except:
                    print(k, l, traceback.format_exc())
                    pass
        self.tryZone()
        
    def makeUniqueIdent(self):
        t = hex(int(time.time()*1000))[2:]
        r = hex(random.randint(1000000,10000000))[2:]
        return str(t) + str(r)
    
    def tryZone(self):
        if self.inputs=={}: 
            if self.mode: self.switchMode()
            return
        self.tryLabel.hide()
        if not self.mode: return
        params = {}
        for p,v in self.params.items():
            if p in self.cuts:
                params[p] = self.inputs.get(p)
            elif callable(v):
                params[p] = v() 
            else:
                params[p] = v        
        alg = None
        output = None
        self.processingId = self.makeUniqueIdent()
        params['processingId'] = self.processingId
        
        self.fmtProcessing.clearTask()
        
        try:
            if self.alg == 'spatialisation':
                pass
            elif self.alg == 'projection' :
                if not self.inputs.get('rasterClassesHauteur'):
                    self.fmtProcessing.run('projectionSensitiveInitiale', params)
                else:
                    params['input'] = self.inputs['rasterClassesHauteur']
                    self.fmtProcessing.run('projectionSensitiveEnchaine', params)
                if params['tamisage']>self.area/10:
                    self.tryLabel.setText("Attention, emprise trop faible pour tester ce tamisage")
                    self.tryLabel.setStyleSheet("color:red;")
                    self.tryLabel.show()
                else:
                    self.tryLabel.hide()
            elif self.alg == 'affinage' : 
                self.fmtProcessing.run(self.alg, params) 
            
            elif self.alg == 'emboiter':
                self.fmtProcessing.run(self.alg, params) 
        
        except:
            print(traceback.format_exc())
            self.switchMode()
            return        
  

    def recupLayers(self, alg_id, d):     
        if self.processingId!=d['processingId']: return
        output = None
        oriCombo = None
        if alg_id == 'fmt:classrasterisation':
            self.inputs['rasterClassesHauteur'] = d['output']
            self.tryZone()
        if alg_id == 'gdal:sieve':
            output = d['OUTPUT']     
        if alg_id == 'fmt:refinepolygon':
            output = d['output']
            oriCombo = self.dock.vectProjeter
        if alg_id == 'fmt:russiandolls':
            output = d['OUTPUT']
            oriCombo = self.dock.vectAffiner
        
        if output is None: return
        
        if isinstance(output, QgsVectorLayer):
            if self.result is None:
                output.setName("sensibility_test")
                self.result = QgsProject.instance().addMapLayer(output)
                self.dock.applyClassSymbology(output, oriCombo)
            else:
                if not self.result.isEditable():
                    self.result.startEditing()
                for f in self.result.getFeatures():
                    self.result.deleteFeature(f.id())
                for f in output.getFeatures():
                    self.result.addFeature(f)    
        if isinstance(output, QgsRasterLayer):
            if self.result:
                QgsProject.instance().removeMapLayer(self.result)
            output.setName("sensibility_test")
            self.result = QgsProject.instance().addMapLayer(output)   


              

class fmtTryTool(QgsMapTool):
    drawFinished = QtCore.pyqtSignal(list)
    def __init__(self, iface):
        self.canvas = iface.mapCanvas()
        super().__init__(self.canvas)
        self.memo = {}
        self.rubberBand = QgsRubberBand(self.canvas, QgsWkbTypes.GeometryType.PolygonGeometry)
        self.rubberBand.setStrokeColor(QtCore.Qt.GlobalColor.darkGray)
        self.rubberBand.setLineStyle(QtCore.Qt.PenStyle.DashLine)
        self.rubberBand.setWidth(2)
        self.started = False
        
    def activate(self):
        super().activate()
        self.canvas.setCursor(QtGui.QCursor(QtCore.Qt.CursorShape.CrossCursor))

    def deactivate(self):
        super().deactivate()
        self.rubberBand.reset(QgsWkbTypes.GeometryType.PolygonGeometry)
        self.canvas.setCursor(QtGui.QCursor(QtCore.Qt.CursorShape.ArrowCursor))
        self.deactivated.emit()

    def canvasMoveEvent(self, e):
        if not self.started: return
        self.memo['pts'] = []
        pt1 = self.memo['pt1']
        pt2 = e.mapPoint()
        self.memo['pts'].append(QgsPointXY(pt1.x(), pt1.y()))
        self.memo['pts'].append(QgsPointXY(pt1.x(), pt2.y()))
        self.memo['pts'].append(QgsPointXY(pt2.x(), pt2.y()))
        self.memo['pts'].append(QgsPointXY(pt2.x(), pt1.y()))
        self.drawPoly(self.memo['pts'])
    
    def canvasPressEvent(self, e):
        self.memo = {'pts':[], 'crs':self.canvas.mapSettings().destinationCrs().authid(), 'type':'polygon'}
        self.started = True
        self.drawPoly()
        self.memo['pt1'] = e.mapPoint()

    def canvasReleaseEvent(self, e):
        self.started = False
        self.drawPoly(self.memo['pts'])
        self.drawFinished.emit(self.memo['pts'])

    def drawPoly(self, lst=[]):
        self.rubberBand.reset(QgsWkbTypes.GeometryType.PolygonGeometry)
        for i,p in enumerate(lst):
            b = i==len(lst)-1
            self.rubberBand.addPoint(p, b)
        self.rubberBand.show()
 
 
# class fmtSymbolButton(QtWidgets.QWidget):
    # toggled = QtCore.pyqtSignal(bool)
    # def __init__(self, params):  
        # super().__init__()     
        # self.params = params
        
        # b = QtWidgets.QHBoxLayout(self)
        # self.w = QtWidgets.QPushButton()
        # self.w.setCheckable(True) 
        # self.w.setMaximumWidth(30)
        # ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"symbology.png"))
        # self.w.setIcon(ico)
        # self.w.setIconSize(QtCore.QSize(30, 25))
        # self.w.setChecked(self.params.get('symbol'))
        # self.w.setToolTip("Appliquer la Symbologie")
        # b.addStretch() 
        # b.setContentsMargins(0,0,10,0)
        # b.addWidget(self.w)
        # self.w.toggled.connect(self.buttonToggle)
        # self.w.toggled.connect(partial(self.params.save,'symbol'))
        
    # def buttonToggle(self) :   
        # self.toggled.emit(self.w.isChecked())
        # style = QgsStyle().defaultStyle()        

    # def updateTwin(self):
        # self.w.blockSignals(True)
        # self.w.setChecked(self.w.isChecked() == False)
        # self.w.blockSignals(False)      

class ToogleToolBar(QtWidgets.QToolBar):
    selectionRequired = QtCore.pyqtSignal()
    changed = QtCore.pyqtSignal()
    beforeChanged = QtCore.pyqtSignal()
    def __init__(self):
        super().__init__()
        self.lst = []
        self.tool = None
        self.actionTriggered.connect(self.trigg)
        
    def addButton(self, code, e, tooltip=None):
        a = None
        if isinstance(e, QtGui.QIcon):
            a = self.addAction(e, "")
        elif isinstance(e, str):
            a = self.addAction(e)
        if a:
            a.code = code
            a.setCheckable(True)
            if tooltip:
                a.setToolTip(tooltip)
            self.lst.append(a)
        return a

    def trigg(self, a):
        self.beforeChanged.emit()
        if a.isChecked():
            self.tool = a.code
            self.uncheck(a)
            self.changed.emit()
        else:
            a.setChecked(True)
        

    def uncheck(self, ai=None):
        if ai is None: self.tool = None
        for a in self.lst:
            if ai!=a:
                a.setChecked(False)
    
    def currentTool(self):
        return self.tool

    def setCurrentTool(self, k=None):
        if k is None:
            self.uncheck()
            self.tool = None
            self.changed.emit()
            return
        for a in self.lst:
            if a.code==k and not a.isChecked():
                a.trigger()
                
    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.MouseButton.RightButton:
            position = event.globalPos()
            self.openMenu(position)
            # self.rightClicked.emit()
          
    def openMenu(self, position):        
        w = QtWidgets.QMenu(self)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__), "images", f"mActionSelectRectangle.png"))
        w.addAction(ico,"Sélection à partir des CPHE", lambda: self.selectionRequired.emit())  
        w.exec(position)

    def setEnabled(self, code, b):
        for a in self.lst:
            if a.code==code:
                a.setEnabled(b)
