# -*- coding: utf-8 -*-
"""
/***************************************************************************
 HeatNetTool
                                 A QGIS plugin
 This plugin provides tools for district heating planning
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-04
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Lars Goray
        email                : lars.goray@fh-muenster.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread, pyqtSignal
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
from qgis.core import QgsProject, QgsMapLayer, QgsVectorLayer, QgsMessageLog, QgsLayerTreeLayer

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .heat_net_tool_dialog import HeatNetToolDialog

import os.path
import subprocess
import sys
import re
from pathlib import Path
import traceback

try:
    import pandas as pd
    import geopandas as gpd
    from shapely import Point
    from .src.download_files import file_list_from_URL, search_filename, read_file_from_zip, filter_df, get_shape_from_wfs, clean_data, add_point, create_square, get_area_for_zensus
    from .src.adjust_files import Streets_adj, Buildings_adj, Parcels_adj, spatial_join
    from .src.status_analysis import WLD, Polygons
    from .src.net_analysis import Streets, Source, Buildings, Graph, Net, Result, get_closest_point, calculate_GLF, calculate_volumeflow, calculate_diameter_velocity_loss
    from .src.load_curve import Temperature, LoadProfile
    from workalendar.europe import Germany
    from matplotlib.figure import Figure
    import matplotlib.pyplot as plt
except:
    pass

class Worker(QThread):
    '''
    Worker class that runs long-running tasks in a separate thread and emits signals for GUI updates.

    This class inherits from QThread and is designed to handle the execution of background tasks. 
    It provides signals to update the GUI (progress bar, label, etc.) and notify when the task is completed. 
    The task is passed as a callable and is executed in the `run()` method.

    Attributes
    ----------
    progress_update : pyqtSignal(int)
        Signal to update the progress bar with an integer value (0-100).
    label_update : pyqtSignal(str)
        Signal to update the text of a QLabel or similar widget.
    color_update : pyqtSignal(str)
        Signal to update the stylesheet (e.g., color) of a widget.
    add_layer_signal : pyqtSignal(str, str, str)
        Signal to trigger the addition of a layer to the QGIS project, passing necessary parameters like file path, style, and group name.
    finished : pyqtSignal()
        Signal emitted when the task is finished.

    Parameters
    ----------
    task_function : callable
        The function that will be executed in the background thread. It should accept `progress_update`, `label_update`, and `color_update` as the first arguments.
    gui_elements : dict
        Dictionary of GUI elements (like progress bars and labels) that will be updated by the task.
    *args, **kwargs :
        Additional arguments and keyword arguments to be passed to the task function.

    Methods
    -------
    run()
        Executes the `task_function` with the provided arguments and emits the `finished` signal when done.
    '''
    progress_update = pyqtSignal(int) 
    label_update = pyqtSignal(str, str)
    error_occurred = pyqtSignal(str)
    finished = pyqtSignal()

    def __init__(self, task_function, gui_elements, *args, **kwargs):
        super().__init__()
        self.task_function = task_function
        self.args = args
        self.kwargs = kwargs
        self.gui_elements = gui_elements  # GUI elements to update

    def run(self):
        try:
            # Run the task function with the provided arguments
            self.task_function(self.progress_update, self.label_update, *self.args, **self.kwargs)
        except Exception as e:
            print(e)
            # stacktrace error
            stacktrace = traceback.format_exc()
            self.error_occurred.emit(f'Error: {e}\n{stacktrace}')
        finally:
            # always send finished signal
            self.finished.emit()

class HeatNetTool:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&F|Heat ')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Gemarkung (Name and info of municipalities and cities in NRW)
        self.gemarkungen_df = pd.DataFrame()

        # set worker status
        self.worker_running = False

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('HeatNetTool', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/heat_net_tool/icon.png' # dev environment. This path is saved in resources.qrc. If icon needs to be changed resources.qrc has to be recompiled to .py
        self.add_action(
            icon_path,
            text=self.tr(u'F|Heat'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&F|Heat '),
                action)
            self.iface.removeToolBarIcon(action)

    def check_python_version(self):
        '''Checks QGIS python version'''

        python_version = sys.version
        self.dlg.intro_label.setText(self.tr('Your QGIS python version is: ')+ python_version)
        self.dlg.intro_label.setStyleSheet('color: white')
        self.dlg.intro_label.repaint()

    def install_package(self):
        '''
        Installs Python packages using pip.

        Parameters
        ----------
        None

        Returns
        -------
        None
        '''
        # feedback
        self.dlg.intro_label.setText(self.tr('Starting Installation. Check cmd for progress.'))
        self.dlg.intro_label.setStyleSheet('color: orange')
        self.dlg.intro_label.repaint()

        # get requirements
        requirements_path =  self.plugin_dir+'/requirements.txt'
        if not os.path.exists(requirements_path):
            self.dlg.intro_label.setText(f'Error: {requirements_path} not found.')
            self.dlg.intro_label.setStyleSheet('color: #ff5555')
            self.dlg.intro_label.repaint()
            
        with open(requirements_path, 'r') as file:
            package_list = [line.strip() for line in file if line.strip()]

        # check if pip is installed
        try:
            import pip
        except ImportError:
            try:
                import ensurepip
                ensurepip.bootstrap()
            except Exception as e:
                print(f'Error while installing pip: {e}')

        current_executable = sys.executable
        python_executable = os.path.join(os.path.dirname(current_executable), 'python.exe')

        try:
            # Execute the "pip install" command to install all packages
            cmd = [python_executable, "-m", "pip", "install", "--upgrade"] + package_list
            subprocess.check_call(["cmd", "/K"] + cmd)  # /K ceeps window open
        except subprocess.CalledProcessError as e:
            # feedback
            self.dlg.intro_label.setText(self.tr(f"Process finished or aborted."))
            self.dlg.intro_label.setStyleSheet("color: white")
            self.dlg.intro_label.repaint()

        # Import all packages
        import pandas as pd
        import geopandas as gpd
        from shapely import Point
        from .src.download_files import file_list_from_URL, search_filename, read_file_from_zip, filter_df, get_shape_from_wfs, clean_data, add_point, create_square, get_area_for_zensus
        from .src.adjust_files import Streets_adj, Buildings_adj, Parcels_adj, spatial_join
        from .src.status_analysis import WLD, Polygons
        from .src.net_analysis import Streets, Source, Buildings, Graph, Net, Result, get_closest_point, calculate_GLF, calculate_volumeflow, calculate_diameter_velocity_loss
        from .src.load_curve import Temperature, LoadProfile
        from workalendar.europe import Germany
        from matplotlib.figure import Figure
        import matplotlib.pyplot as plt
        
    def select_output_file(self, dir, lineEdit, filetype):
        '''
        Opens a file dialog to select an output file and sets the selected path to a QLineEdit.

        Parameters
        ----------
        dir : str
            The directory to start the file dialog in.
        lineEdit : QLineEdit
            The QLineEdit widget to display the selected file path.
        filetype : str
            The file type filter for the dialog (e.g., "*.txt").

        Returns
        -------
        None
        '''
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, self.tr("Select output file "),dir, filetype)
        lineEdit.setText(filename)
    
    def select_input_file(self, dir, lineEdit, filetype):
        '''
        Opens a file dialog to select an input file and sets the selected path to a QLineEdit.

        Parameters
        ----------
        dir : str
            The directory to start the file dialog in.
        lineEdit : QLineEdit
            The QLineEdit widget to display the selected file path.
        filetype : str
            The file type filter for the dialog (e.g., "*.txt").

        Returns
        -------
        None
        '''
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, self.tr("Select input file"), dir, filetype)
        if filename:  # Check if the user selected a file
            lineEdit.setText(filename)

    def get_layer_path_from_combobox(self, combobox):
        '''
        Gets the path of the selected layer from the given ComboBox.

        Parameters
        ----------
        combobox : QComboBox
            The QComboBox object representing the layer selection.

        Returns
        -------
        tuple
            A tuple containing the path of the selected layer (str), the layer name (str), and the selected layer object (QgsVectorLayer). Returns (None, None, None) if no layer is found.
        '''
        # Get the name of the selected layer from the ComboBox
        selected_layer_name = combobox.currentText()
        selected_layer_name = re.sub(r'\s*\[.*?\]', '', selected_layer_name).strip()

        # Find the layer in the project
        layers = QgsProject.instance().mapLayersByName(selected_layer_name)

        if layers:
            # Assume we take the first found layer if multiple layers have the same name
            selected_layer = layers[0]
            # Extract the path from the layer information
            path = selected_layer.source()

            # Check if '|' character exists in the path
            if '|' in path:
                # Split the path and layer name
                path_parts = path.split('|')
                path = path_parts[0]  # Path is the first part
                # Layer name is the second part, remove 'layername='
                selected_layer_name = path_parts[1].replace('layername=', '')

            return path, selected_layer_name, selected_layer
        else:
            return None, None, None

    def create_layer_tree_structure(self):
        '''
        Creates the desired layer tree structure if it does not already exist.

        This function checks if the group 'FHeat' exists in the layer tree. If it does not, it creates the group
        and adds the subgroups 'Basics', 'adjusted', 'status', and 'net' under it. 

        Returns
        -------
        None
        '''
        # Get the root group in the Layer tree
        root = QgsProject.instance().layerTreeRoot()

        # Check if the 'FHeat' group already exists
        fheat_group = root.findGroup('F|Heat')

        if fheat_group is None:
            # Create the 'FHeat' group
            fheat_group = root.addGroup('F|Heat')

            # Define the subgroups
            subgroups = [self.tr('Net'), self.tr('Heat Density'), self.tr('Adjusted Files'), self.tr('Basic Data')]
            
            # Add each subgroup to 'FHeat'
            for subgroup in subgroups:
                fheat_group.addGroup(subgroup)

    def add_shapefile_to_project(self, shapefile_path, style=None, group_name=None):
        '''
        Adds a shapefile to the QGIS project.

        Parameters
        ----------
        shapefile_path : str
            The file path of the shapefile to add.
        style : str, optional
            The name of the style to apply to the layer. Options include 'wld', 'polygons', 'net', 'streets', 'buildings', 'parcels'.
        group_name : str
            Layer group where the layer should be added

        Returns
        -------
        None
        '''
        layer_name = os.path.splitext(os.path.basename(shapefile_path))[0]
        layer = QgsVectorLayer(path=shapefile_path, baseName=layer_name, providerLib='ogr')
        if not layer.isValid():
            print("Layer failed to load!")
            return

        # Get the root group in the Layer tree
        root = QgsProject.instance().layerTreeRoot()

        if group_name:
            # Check if the group exists
            group = root.findGroup(group_name)
            
            # If the group doesn't exist, create it
            if group is None:
                group = root.addGroup(group_name)
            
            # Add the layer to the specified group
            QgsProject.instance().addMapLayer(layer, False)  # False prevents adding to the top-level group
            group.insertChildNode(0, QgsLayerTreeLayer(layer))  # Add layer to group

        else:
            # If no group is specified, add the layer to the root
            QgsProject.instance().addMapLayer(layer)

        # Apply style
        if style == 'wld':
            style_path = self.plugin_dir + '/layerstyles/wld.qml'
            layer.loadNamedStyle(style_path)

        if style == 'polygons':
            style_path = self.plugin_dir + '/layerstyles/heat density polygons.qml'
            layer.loadNamedStyle(style_path)

        if style == 'net':
            style_path = self.plugin_dir + '/layerstyles/net.qml'
            layer.loadNamedStyle(style_path)

        if style == 'streets':
            style_path = self.plugin_dir + '/layerstyles/streets.qml'
            layer.loadNamedStyle(style_path)

        if style == 'buildings':
            style_path = self.plugin_dir + '/layerstyles/buildings.qml'
            layer.loadNamedStyle(style_path)
        
        if style == 'parcels':
            style_path = self.plugin_dir + '/layerstyles/parcels.qml'
            layer.loadNamedStyle(style_path)

        if style == 'buildings_adj':
            style_path = self.plugin_dir + '/layerstyles/buildings_adj.qml'
            layer.loadNamedStyle(style_path)

        if style == 'streets_adj':
            style_path = self.plugin_dir + '/layerstyles/streets_adj.qml'
            layer.loadNamedStyle(style_path)

        if style == 'zensus':
            style_path = self.plugin_dir + '/layerstyles/zensus.qml'
            layer.loadNamedStyle(style_path)

    def load_download_options(self):
        '''
        Loads municipality and city names of NRW into comboBoxes.

        This method reads an Excel file containing information about municipalities and cities, and populates comboBoxes with this information.

        Returns
        -------
        None
        '''
        path = Path(self.plugin_dir) / 'data/cities.xlsx'
        df = pd.read_excel(path, dtype={'schluessel': str, 'gmdschl': str})

        # convert string to list with floats
        df['bbox'] = df['bbox'].apply(lambda x: [float(coord) for coord in x.replace('(', '').replace(')', '').split(',')])

        municipalities = sorted(df['gemeinde'].unique().tolist())
        cities = sorted(df['name'].tolist())

        # add options to comboBoxes
        self.dlg.load_comboBox_municipality.addItems(municipalities)
        self.dlg.load_comboBox_city.addItems(cities)

        # save df for later operations
        self.gemarkungen_df = df
    
    def adapt_download_options(self):
        '''
        Adjusts the city ComboBox based on the selected municipality.
        '''
        value = self.dlg.load_comboBox_municipality.currentText()

        # Filter the dataframe based on the selected municipality
        filtered_cities = self.gemarkungen_df[self.gemarkungen_df['gemeinde'] == value]['name'].tolist()

        # Sort the list of cities for the selected municipality (value)
        sorted_cities = sorted(filtered_cities)

        # Clear the city ComboBox
        self.dlg.load_comboBox_city.clear()

        # Add the filtered cities to the city ComboBox
        self.dlg.load_comboBox_city.addItems(sorted_cities)

    def update_label(self, label, text, color):
        label.setText(text)
        label.setStyleSheet(f"color: {color}")
        label.repaint()  # Repaint to apply both text and color changes immediately

    def show_error_message(self, message, gui_elements):

        if 'label' in gui_elements:
            label = gui_elements['label']
            self.update_label(label, self.tr('aborted due to error'), '#ff5555')
    
        # Show error in QMessageBox
        QMessageBox.critical(self.dlg, "Error", message)
        # update runner status
        self.worker_running = False

    def run_long_task(self, task, gui_elements, on_task_finished):
        '''
        Executes a long-running task in a separate thread, preventing the GUI from freezing.

        This function moves a worker to a new thread to run a specified long task asynchronously. The worker is responsible for emitting signals that update GUI elements such as a progress bar and status label during the task. Upon task completion, the GUI elements are updated and cleanup is performed.

        Parameters
        ----------
        task : callable
            The long-running task to be executed. This should be a function that contains the main logic for the background task.
        gui_elements : dict
            A dictionary containing the GUI elements (like progress bars and labels) to be updated during the task. Expected keys are:
            - 'progressBar': The progress bar widget to display task progress.
            - 'label': The label widget to display status updates.
        on_task_finished : callable
            A callback function to be executed when the task finishes. This function is connected to the worker's `finished` signal.

        Returns
        -------
        None
        '''
        self.worker = Worker(task, gui_elements)

        # Connect signals to GUI elements
        self.worker.progress_update.connect(gui_elements['progressBar'].setValue)
        self.worker.label_update.connect(lambda text, color: self.update_label(gui_elements['label'], text, color))
        
        # Connect error massage signal
        self.worker.error_occurred.connect(lambda message: self.show_error_message(message, gui_elements))

        self.thread = QThread()
        self.worker.moveToThread(self.thread)

        # Start worker
        self.thread.started.connect(self.worker.run)
        self.worker.finished.connect(self.thread.quit)
        self.worker.finished.connect(self.worker.deleteLater)
        self.thread.finished.connect(self.thread.deleteLater)

        # Start on_task_finished in main thread when worker finished
        self.worker.finished.connect(on_task_finished)

        self.thread.start()

    # Methods for loading layers and attributes to comboboxes

    def get_all_loaded_layers(self):
        '''
        Get a list of all loaded layers in the project, including layers within groups.

        Returns
        -------
        list of QgsMapLayer
            A list containing all the loaded layers in the current QGIS project.
        '''
        root = QgsProject.instance().layerTreeRoot()
        all_layers = root.layerOrder()
        loaded_layers = []

        for layer in all_layers:
            if isinstance(layer, QgsMapLayer):
                loaded_layers.append(layer)

        return loaded_layers

    #obsolete
    def load_layers_to_combobox(self, combobox):
        '''
        Load names of all the loaded layers into a QComboBox.

        Parameters
        ----------
        combobox : QComboBox
            The ComboBox widget to populate with layer names.

        Returns
        -------
        None
        '''
        # Fetch the currently loaded layers
        layers = self.get_all_loaded_layers()
        # Clear the contents of the comboBox from previous runs
        combobox.clear()
        # Add a default translated item as the first item in the ComboBox
        combobox.addItem(self.tr("Select Layer"))
        # Populate the comboBox with names of all the loaded layers
        combobox.addItems([layer.name() for layer in layers])
        # Set the default item as the current index
        combobox.setCurrentIndex(0)
    
    def load_attributes_to_combobox(self, layer_name, combobox):
        '''
        Load attributes of the selected layer into a QComboBox.

        Parameters
        ----------
        layer_name : str
            The name of the layer whose attributes are to be loaded.
        combobox : QComboBox
            The ComboBox widget to populate with attribute names.

        Returns
        -------
        None
        '''
        # Find the layer by its name
        layer = QgsProject.instance().mapLayersByName(layer_name)[0]  # Assuming unique names
        # Clear the contents of the comboBox from previous runs
        combobox.clear()
        # Add a default item as the first item in the ComboBox
        combobox.addItem(self.tr("Select Attribute"))
        # Populate the comboBox with names of all the attributes of the layer
        for field in layer.fields():
            combobox.addItem(field.name())

        # Set the default item as the current index
        combobox.setCurrentIndex(0)

    def load_attributes(self, combobox_in, combobox_out):
        '''
        Load attributes of the layer selected in one combobox into another combobox.

        Parameters
        ----------
        combobox_in : str
            The name of the ComboBox widget containing the layer selection.
        combobox_out : str
            The name of the ComboBox widget to populate with attribute names.

        Returns
        -------
        None
        '''
        # Get the current layer name selected in the specified combobox
        layer_name_with_epsg = getattr(self.dlg, combobox_in).currentText()

        # Use a regular expression to strip out the EPSG code (or anything in brackets)
        layer_name = re.sub(r'\s*\[.*?\]', '', layer_name_with_epsg).strip()

        layers = self.get_all_loaded_layers()

        # Load attributes to the specified combobox
        if layer_name in [layer.name() for layer in layers]:
            self.load_attributes_to_combobox(layer_name, getattr(self.dlg, combobox_out))
    
    # obsolete
    def tab_change(self):
        '''
        Update ComboBoxes with loaded layers when the tab is changed.

        Returns
        -------
        None
        '''
        # Load layers into comboBoxes
        self.load_layers_to_combobox(self.dlg.adjust_comboBox_buildings)
        self.load_layers_to_combobox(self.dlg.adjust_comboBox_parcels)
        self.load_layers_to_combobox(self.dlg.adjust_comboBox_streets)
        self.load_layers_to_combobox(self.dlg.status_comboBox_streets)
        self.load_layers_to_combobox(self.dlg.status_comboBox_parcels)
        self.load_layers_to_combobox(self.dlg.status_comboBox_buildings)
        self.load_layers_to_combobox(self.dlg.net_comboBox_buildings)
        self.load_layers_to_combobox(self.dlg.net_comboBox_streets)
        self.load_layers_to_combobox(self.dlg.net_comboBox_source)
        self.load_layers_to_combobox(self.dlg.net_comboBox_polygon)

    # Main Methods for background thread
    def download_files(self, progress_update, label_update):
        '''
        Downloads and processes shapefiles for buildings, streets, and parcels for a selected city or municipality.

        This function performs the following steps:

        1. **Progress Bar Initialization**:
        - Sets the progress bar to 0, indicating the start of the download and processing operation.

        2. **Determine Selection**:
        - Retrieves the selected city or municipality from the graphical user interface (GUI).
        - Checks whether a city or municipality is selected and sets the appropriate parameter.

        3. **Filter DataFrame**:
        - Filters a DataFrame (`gemarkungen_df`) for the selected city or municipality based on the user’s selection.

        4. **Download Buildings Shapefiles**:
        - Accesses the URL for building data and downloads the relevant shapefiles based on the municipality key.

        5. **Download Streets Shapefiles**:
        - Accesses the URL for street data and downloads the relevant shapefiles based on the municipality key.

        6. **Download Parcel Data**:
        - Accesses a Web Feature Service (WFS) to download parcel data, ensuring that if a city is selected, only the relevant parcels are retrieved.

        7. **Filter Geometries**:
        - If only a city is selected, filters the building and street data to include only geometries that intersect with the selected parcels.

        8. **Save Shapefile gdf as variable**:
        - Saves the processed building, street, and parcel shapefiles to specified paths.

        Parameters
        ----------
        None

        Returns
        -------
        None
        '''
        # Note: In GUI city = district, municipality = city
        
        progress_update.emit(0) # update progressBar
        self.download_status = '0'
        

        if self.dlg.load_lineEdit_buildings.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the buildings'),'orange')

        elif self.dlg.load_lineEdit_streets.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the streets'),'orange')

        elif self.dlg.load_lineEdit_parcels.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the parcels'),'orange')
        
        else:
            label_update.emit(self.tr('Starting...'),'white')

            # URLs
            # buildings
            url_buildings = 'https://www.opengeodata.nrw.de/produkte/umwelt_klima/energie/kwp/'

            # parcels
            url_parcels = 'https://www.wfs.nrw.de/geobasis/wfs_nw_inspire-flurstuecke_alkis'
            layer_parcels = 'cp:CadastralParcel'

            # get name from combo box
            if self.dlg.load_radioButton_municipality.isChecked():
                name = self.dlg.load_comboBox_municipality.currentText()
                parameter = 'municipality'
            elif self.dlg.load_radioButton_city.isChecked():
                name = self.dlg.load_comboBox_city.currentText()
                parameter = 'city'
            else:
                label_update.emit(self.tr('please choose city or district'),'orange')
                return
            
            progress_update.emit(1) # update progressBar
            
            # filter df for city/municipality name
            filtered_df = filter_df(name, self.gemarkungen_df, parameter)
            
            # city/municipality keys
            municipality_key = filtered_df['gmdschl'][0]

            progress_update.emit(5) # update progressBar
            label_update.emit(self.tr('Downloading...'), 'white')

            # buildings shapes
            all_buildings_files = file_list_from_URL(url_buildings+'index.json')
            progress_update.emit(10) # update progressBar
            buildings_zip = search_filename(all_buildings_files, municipality_key)
            
            # Check if Data is found
            if buildings_zip == 'No data found':
                label_update.emit(self.tr('Error, no data found. Data source was possibly renamed.'), '#ff5555')
                return
            
            progress_update.emit(15) # update progressBar
            buildings_file_pattern = f'WBM-NRW_{municipality_key}' # file pattern maybe has to be renamed, when changes on the website occur
            buildings_gdf = read_file_from_zip(url_buildings, buildings_zip, buildings_file_pattern)

            progress_update.emit(35) # update progressBar

            # streets shapes
            streets_file_pattern = f'WBM-NRW-Waermelinien_{municipality_key}' # file pattern maybe has to be renamed, when changes on the website occur
            streets_gdf = read_file_from_zip(url_buildings, buildings_zip, streets_file_pattern)

            progress_update.emit(55) # update progressBar

            # parcels
            gdf_list_parcels=[] # if a whole municipality is selected filtered df consists of multiple cities which parcels will be saved in this list and later merged
            for row  in filtered_df.itertuples():
                bbox = row.bbox
                key = row.schluessel
                parcel_gdf_i, e = get_shape_from_wfs(url_parcels, key, bbox, layer_parcels)
                if e == 1:
                    label_update.emit(self.tr('Too many parcels for key: {}!\nMax. 100,000 parcels can be downloaded at once\nparcels incomplete').format(key), '#ff5555')
                gdf_list_parcels.append(parcel_gdf_i)
            parcels_gdf = pd.concat(gdf_list_parcels, ignore_index=True)

            progress_update.emit(90) # update progressBar

            # buffer(0) can sometimes repair invalid geometries
            buildings_gdf['geometry'] = buildings_gdf['geometry'].buffer(0)
            parcels_gdf['geometry'] = parcels_gdf['geometry'].buffer(0)

            # The buildings and streets can only be downloaded at municipality level, if you only want one city, the 
            # additional buildings are superfluous and only extend the calculation time of the following programs.
            # Therefore, only buildings that are located on the parcels that are available at municipality level are retained
            if parameter == 'city':
                union = gpd.GeoDataFrame(geometry=[parcels_gdf.unary_union])
                buildings_gdf = gpd.sjoin(buildings_gdf, union, predicate='intersects')
                streets_gdf = gpd.sjoin(streets_gdf, union, predicate='intersects')

            progress_update.emit(98) # update progressBar

            # save gdfs as instance attributes
            self.buildings_gdf = buildings_gdf
            self.streets_gdf = streets_gdf
            self.parcels_gdf = parcels_gdf

            self.download_status = 'complete'

    def download_zensus(self, progress_update, label_update):
    
        # set status for on_task_finished
        self.download_zensus_status = 0

        if self.dlg.load_lineEdit_zensus.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the Zensus data'), 'orange')
        else:

            # get name from combo box
            if self.dlg.load_radioButton_municipality.isChecked():
                name = self.dlg.load_comboBox_municipality.currentText()
                parameter = 'municipality'
            elif self.dlg.load_radioButton_city.isChecked():
                name = self.dlg.load_comboBox_city.currentText()
                parameter = 'city'
            else:
                label_update.emit(self.tr('please choose city or district'), 'orange')
                return

            label_update.emit(self.tr('Loading...'), 'white')
            
            progress_update.emit(1)

            ## get area from self.gemarkungen in epsg3035

            # filter df for city/municipality name
            filtered_df = filter_df(name, self.gemarkungen_df, parameter)

            zensus_bbox, zensus_area = get_area_for_zensus(filtered_df)

            xmin, ymin, xmax, ymax = zensus_bbox

            progress_update.emit(5)
            
            ## load zensus data
            url_zensus = 'https://www.zensus2022.de/static/Zensus_Veroeffentlichung/'

            heizungsart_zip = 'Zensus2022_Heizungsart.zip'
            energietraeger_zip = 'Zensus2022_Energietraeger.zip'

            df_Heizungsart = read_file_from_zip(url_zensus, heizungsart_zip, 'Zensus2022_Heizungsart_100m-Gitter', '.csv', encoding='latin1')
            progress_update.emit(20)
            df_Energietraeger = read_file_from_zip(url_zensus, energietraeger_zip, 'Zensus2022_Energietraeger_100m-Gitter', '.csv', encoding='latin1')

            # only data within bbox
            df_Heizungsart = df_Heizungsart[(df_Heizungsart['x_mp_100m'] > xmin) & (df_Heizungsart['x_mp_100m'] < xmax) & (df_Heizungsart['y_mp_100m'] > ymin) & (df_Heizungsart['y_mp_100m'] < ymax)]
            df_Energietraeger = df_Energietraeger[(df_Energietraeger['x_mp_100m'] > xmin) & (df_Energietraeger['x_mp_100m'] < xmax) & (df_Energietraeger['y_mp_100m'] > ymin) & (df_Energietraeger['y_mp_100m'] < ymax)]
            progress_update.emit(40)

            ### clean data
            df_Heizungsart = clean_data(df_Heizungsart)
            progress_update.emit(50)
            df_Energietraeger = clean_data(df_Energietraeger)
            progress_update.emit(60)

            ### add points
            df_Heizungsart = add_point(df_Heizungsart)
            progress_update.emit(70)
            df_Energietraeger = add_point(df_Energietraeger)
            progress_update.emit(80)

            # create GeoDataFrames
            gdf_Heizungsart = gpd.GeoDataFrame(df_Heizungsart, geometry='point')
            gdf_Energietraeger = gpd.GeoDataFrame(df_Energietraeger, geometry='point')

            # set crs
            gdf_Heizungsart.set_crs(epsg=3035, inplace=True)
            gdf_Energietraeger.set_crs(epsg=3035, inplace=True)

            # Spatial Join
            gdf_combined = gpd.sjoin(gdf_Heizungsart, gdf_Energietraeger, how='inner', predicate='intersects')

            # drop unwanted columns
            gdf_combined = gdf_combined.drop(columns=['index_right'])

            # add square geometry
            gdf_combined['geometry'] = gdf_combined.apply(lambda row: create_square(row['point'], 100), axis=1)

            # drop point column
            zensus_gdf = gdf_combined.drop(columns=['point'])

            # Set the 'geometry' column as the active geometry column for the 'shape' object
            zensus_gdf = zensus_gdf.set_geometry('geometry')

            # Set the CRS for 'shape'
            zensus_gdf.set_crs(epsg=3035, inplace=True)
            
            # save zensus_gdf as instance attribute
            self.zensus_gdf = zensus_gdf

            # set status for on_task_finished
            self.download_zensus_status = 'complete'

    def adjust_files(self, progress_update, label_update):
        '''
        Adjust and process building, street, and parcel data layers in a GIS project.

        This method performs several operations on geospatial data, including:
        
        1. **Initialization**:
        - Sets the initial value of the progress bar and provides feedback to the user interface.

        2. **Parameter Definitions**:
        - Defines the heat demand attribute (`heat_att`) and building age class bins (`bak_bins`) and labels (`bak_labels`).

        3. **Data Loading**:
        - Reads additional building information from an Excel file.

        4. **Layer Path and Object Retrieval**:
        - Retrieves file paths and layer objects for streets, buildings, and parcels from the selected options in the user interface.

        5. **Adjustment Class Initialization**:
        - Initializes instances of `Parcels_adj`, `Buildings_adj`, and `Streets_adj` classes for data processing.

        6. **Building Data Adjustments**:
        - **Heat Demand Filtering**: Filters out buildings without heat demand.
        - **Spatial Join**: Joins buildings with parcels to add building age information.
        - **Age Class Addition**: Adds building age classes and LANUV age and type information.
        - **Load Profile Addition**: Integrates load profiles from the Excel data.
        - **Data Cleanup**: Drops unwanted attributes and adds thermal power information.
        - **ID Assignment**: Assigns new IDs to buildings and merges building data as needed.

        7. **Street Data Adjustments**:
        - Rounds street coordinates and adds a boolean column to indicate possible routes.

        8. **Saving and Layer Update**:
        - Determines whether to create new files or overwrite existing ones based on user input.
        - Saves the modified shapefiles and updates the QGIS project layers accordingly.

        9. **Completion**:
        - Finalizes the progress bar and provides completion feedback.

        Returns
        -------
        None
        '''
        # set status for on_task_finished
        self.adjust_files_status = 0
        
        progress_update.emit(0) # update progressBar
        label_update.emit(self.tr('Calculating...'), 'white') # update label
        
        # heat demand attribute
        heat_att = self.dlg.adjust_comboBox_heat.currentText()

        # bool to check if buildings are already adjusted
        self.bool_files_already_adjusted = None

        if heat_att == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as heat demand.'),'orange')
            return
        
        if self.dlg.adjust_radioButton_new.isChecked():
            if self.dlg.adjust_lineEdit_buildings.text().strip() == "":
                label_update.emit(self.tr('Specify a file path for the buildings'),'orange')
                return
            if self.dlg.adjust_lineEdit_streets.text().strip() == "":
                label_update.emit(self.tr('Specify a file path for the streets'),'orange')
                return

        # building age classes
        bak_bins = [0, 1918, 1948, 1957, 1968, 1978, 1983, 1994, 2001, 9999]
        bak_labels = ['B','C','D','E','F','G','H','I','J']
        
        streets_path, streets_layer_name, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.adjust_comboBox_streets)
        buildings_path, buildings_layer_name, buildings_layer_obj = self.get_layer_path_from_combobox(self.dlg.adjust_comboBox_buildings)
        parcels_path, parcels_layer_name, parcels_layer_obj  = self.get_layer_path_from_combobox(self.dlg.adjust_comboBox_parcels)

        progress_update.emit(5) # update progressBar

        parcels = Parcels_adj(parcels_path)
        buildings = Buildings_adj(buildings_path, heat_att)
        streets = Streets_adj(streets_path)

        # test if buildings already have been adjusted
        if 'Leistung_th [kW]' in buildings.gdf.columns:
            self.bool_files_already_adjusted = True
            label_update.emit(self.tr('Buildings already adjusted!'), 'rgb(0, 255, 0)') # update label
            progress_update.emit(100) # update progressBar
        else:
            self.bool_files_already_adjusted = False
            buildings.gdf = buildings.gdf[buildings.gdf[heat_att]>0].reset_index(drop=True) # only buildings with heat demand
            progress_update.emit(10) # update progressBar
            buildings.add_LANUV_age_and_type() # add building age and type by LANUV
            progress_update.emit(20) # update progressBar
            buildings.merge_buildings()
            buildings.gdf['new_ID'] = buildings.gdf.index.astype('int32')
            progress_update.emit(30) # update progressBar
            buildings.gdf = spatial_join(buildings.gdf.copy(), parcels.gdf, ['validFrom']) # building age from parcels
            progress_update.emit(40) # update progressBar
            buildings.add_BAK(bak_bins,bak_labels) # add building age class
            progress_update.emit(50) # update progressBar
            buildings.add_Vlh_Loadprofile(self.excel_building_info)
            progress_update.emit(60) # update progressBar
            buildings.drop_unwanted()
            progress_update.emit(70) # update progressBar
            buildings.add_power()
            buildings.add_custom_heat_demand(self.excel_building_demand_wg, self.excel_building_info)
            buildings.add_connect_option()
            buildings.rename_and_order_columns()

            progress_update.emit(80) # update progressBar

            streets.round_streets()
            progress_update.emit(90) # update progressBar
            streets.add_bool_column() # possible routes

            # save gdfs as instance attributes
            self.buildings_gdf = buildings.gdf
            self.streets_gdf = streets.gdf
            self.parcels_gdf = parcels.gdf
            progress_update.emit(95) # update progressBar

    def status_analysis(self, progress_update, label_update):
        '''
         Perform a status analysis of building and street data, updating the GIS project with new attributes and geometries.

        This method conducts an analysis that involves the following steps:

        1. **Progress Bar Initialization**:
        - Sets the initial value of the progress bar to indicate the start of the process.

        2. **Layer Path Retrieval**:
        - Retrieves paths and layer objects for streets, parcels, and buildings from the respective combo boxes in the user interface.

        3. **Attribute Selection**:
        - Retrieves the selected heat and thermal power attributes from the combo boxes.

        4. **Output Path Specification**:
        - Specifies the file path for saving the polygon data from a line edit field.

        5. **Geospatial Data Loading**:
        - Reads the shapefiles for streets, parcels, and buildings into GeoDataFrames.

        6. **Warmth Load Density (WLD) Calculation**:
        - Initializes a `WLD` object with the loaded buildings and streets.
        - Calculates the centroid for buildings.
        - Identifies the closest street to each building.
        - Adds the length of connections between buildings and streets.
        - Adds the specified heat attribute and calculates the Warmth Load Density (WLD).
        - Updates the streets GeoDataFrame with the new attributes and saves it back to a shapefile.
        - Adds the updated street layer to the GIS project with a specific style.

        7. **Polygon Processing**:
        - Initializes a `Polygons` object with the parcels, updated streets, and buildings.
        - Selects parcels based on their connection to buildings within a specified distance.
        - Applies a buffer, dissolve, and explode operation to refine the polygon geometries.
        - Adds the specified heat and thermal power attributes to the polygons.
        - Saves the processed polygons to a shapefile.
        - Adds the updated polygon layer to the GIS project with a specific style.

        8. **Completion**:
        - Updates the progress bar to indicate the completion of the analysis.

        Returns
        -------
        None
        '''
        
        progress_update.emit(0) # update progressBar
        self.status_analysis_status = 0

        # feedback
        label_update.emit(self.tr('Calculating...'), 'white')

        # layer from combo box
        streets_path, streets_layer_name, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.status_comboBox_streets)
        parcels_path, parcels_layer_name, parcels_layer_obj = self.get_layer_path_from_combobox(self.dlg.status_comboBox_parcels)
        buildings_path, buildings_layer_name, buildings_layer_obj = self.get_layer_path_from_combobox(self.dlg.status_comboBox_buildings)
        
        # attributes from layer
        heat_attribute = self.dlg.status_comboBox_heat.currentText()
        power_attribute = self.dlg.status_comboBox_power.currentText()

        # check if attributes and paths are selected correctly
        if heat_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as heat demand.'),'orange')
            return
        if power_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as thermal power.'),'orange')
            return
        
        if self.dlg.status_lineEdit_polygons.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the heat density output'),'orange')
            return

        # path from lineEdit
        polygon_path = self.dlg.status_lineEdit_polygons.text()

        progress_update.emit(2) # update progressBar

        # shapes to gdf
        streets = gpd.read_file(streets_path)
        parcels = gpd.read_file(parcels_path)
        buildings = gpd.read_file(buildings_path)

        # HLD/WLD
        wld = WLD(buildings,streets)
        progress_update.emit(5) # update progressBar
        wld.get_centroid()
        progress_update.emit(20) # update progressBar
        wld.closest_street_buildings()
        progress_update.emit(30) # update progressBar
        wld.add_lenght()
        progress_update.emit(40) # update progressBar
        wld.add_heat_att(heat_att=heat_attribute)
        progress_update.emit(50) # update progressBar
        wld.add_WLD(heat_att=heat_attribute)
        progress_update.emit(60) # update progressBar
        
        # polygons
        polygons = Polygons(parcels, wld.streets, buildings)
        polygons.select_parcels_by_building_connection(0.1)
        progress_update.emit(70) # update progressBar
        polygons.buffer_dissolve_and_explode(0.5)
        progress_update.emit(80) # update progressBar
        polygons.add_attributes(heat_attribute, power_attribute)
        progress_update.emit(90) # update progressBar
        
        # translate columns
        wld.rename_columns()
        polygons.rename_columns()

        # save gdfs as instance attributes
        self.wld = wld.streets
        self.polygons = polygons.polygons

        # Set status as complete
        self.status_analysis_status = 'complete'
    
    def network_analysis(self, progress_update, label_update):
        '''
        Conduct a network analysis for a district heating system, including setup, data loading,
        and computation of the optimal network path based on building heat demand and street layout.

        This method performs the following steps:

        1. **Progress Bar Initialization**:
        - Initializes the progress bar to indicate the start of the analysis.

        2. **User Feedback**:
        - Displays an initial status message indicating that calculations are in progress.

        3. **Load Pipe Data**:
        - Loads pipe data from an Excel file for use in the network analysis.

        4. **Retrieve Temperatures**:
        - Retrieves supply and return temperatures from SpinBoxes in the user interface.

        5. **Retrieve Layer Paths**:
        - Retrieves the file paths and objects for source, streets, and buildings layers from combo boxes.

        6. **Select Attributes**:
        - Retrieves the selected heat and power attributes from combo boxes.

        7. **Specify Output Path**:
        - Retrieves the path for saving the resulting network shapefile from a line edit field.

        8. **Progress Bar Update**:
        - Updates the progress bar to reflect the completion of data retrieval steps.

        9. **Instantiate Classes**:
        - Creates instances of classes for buildings, source, and streets, loading the respective data.

        10. **Polygon Filtering**:
            - If a polygon is selected, filters buildings to those within the polygon boundaries.

        11. **Drop Unwanted Routes**:
            - Removes street segments marked as not possible routes, if the attribute exists.

        12. **Create Connection Points**:
            - Adds centroids to buildings and finds closest points to streets.
            - Establishes connection points for sources to the street network.

        13. **Graph Construction**:
            - Constructs a network graph based on street geometry.
            - Connects building centroids and source points to the graph.
            - Adds edge attributes, such as length, to the graph.

        14. **Connectivity Check**:
            - Verifies that all points in the network are connected.
            - If not, provides feedback and visualization to assist in fixing disconnections.

        15. **Progress Bar Update**:
            - Updates the progress bar after constructing the graph.

        16. **Network Analysis**:
            - Creates a `Net` object and performs a detailed network analysis, computing the optimal network for heat distribution based on the supply and return temperatures, pipe data, and building attributes.

        17. **GeoDataFrame Creation and Saving**:
            - Converts the graph to a GeoDataFrame with the computed network.
            - Saves the GeoDataFrame as a shapefile at the specified output path.

        18. **Add Layer to Project**:
            - Adds the resulting network shapefile as a layer in the GIS project.

        19. **Completion**:
            - Finalizes the progress bar and provides user feedback indicating the successful completion of the network analysis.

        20. **Error Handling**:
            - Logs any exceptions encountered during the process and exits gracefully.

        Returns
        -------
        None
        '''
        
        progress_update.emit(0) # update progressBar
        # set status
        self.network_analysis_status = 0

        # feedback
        label_update.emit(self.tr('Calculating...'), 'white')
    
        # Temperatures from SpinBox
        t_supply = self.dlg.net_doubleSpinBox_supply.value()
        t_return = self.dlg.net_doubleSpinBox_return.value()

        if t_supply <= t_return:
            # feedback
            self.dlg.net_label_response.setText(self.tr('The return temperature has to be smaller than the supply temperature!'))
            self.dlg.net_label_response.setStyleSheet("color: #ff5555")
            self.dlg.net_label_response.repaint()
            return

        # Layer paths
        source_path, source_layer, source_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_source)
        streets_path, streets_layer, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_streets)
        buildings_path, buildings_layer, buildings_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_buildings)
        
        heat_attribute = self.dlg.net_comboBox_heat.currentText()
        power_attribute = self.dlg.net_comboBox_power.currentText()

        # check if attributes and paths are selected correctly
        if heat_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as heat demand.'),'orange')
            return
        if power_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as thermal power.'),'orange')
            return

        if self.dlg.net_lineEdit_net.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the net shapefile output'),'orange')
            return
        if self.dlg.net_lineEdit_net.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the result file'),'orange')
            return
       
        progress_update.emit(2) # update progressBar

        # Instantiate classes
        buildings = Buildings(buildings_path, heat_attribute, buildings_layer)
        source = Source(source_path, source_layer)
        streets = Streets(streets_path, streets_layer)
        
        # check if polygon checkbox is checked
        if self.dlg.net_checkBox_polygon.isChecked():
            polygon_path, polygon_layer, polygon_layer_obj  = self.get_layer_path_from_combobox(self.dlg.net_comboBox_polygon)
            # load polygon as gdf
            if polygon_layer == None:
                polygon = gpd.read_file(polygon_path)
            else: 
                polygon = gpd.read_file(polygon_path, layer=polygon_layer)

            # only buildings within polygon
            buildings.gdf = gpd.sjoin(buildings.gdf, polygon, how="inner", predicate="within")

        # Drop unwanted routes if existing
        try:
            streets.gdf = streets.gdf[streets.gdf['Moegliche_Route']==1]
        except:
            pass

        # Drop unconnected buildings if existing
        try:
            buildings.gdf = buildings.gdf[buildings.gdf['Anschluss']==1]
        except:
            pass

        
        progress_update.emit(5) # update progressBar

        # create connection points
        buildings.add_centroid()
        buildings.closest_points_buildings(streets.gdf)
        source.closest_points_sources(streets.gdf)
        streets.add_connection_to_streets(buildings.gdf, source.gdf)
        
        progress_update.emit(15) # update progressBar

        # Create graph
        graph = Graph(crs=buildings.gdf.crs)
        graph.create_street_network(streets.gdf)
        graph.connect_centroids(buildings.gdf)
        graph.connect_source(source.gdf)
        graph.add_attribute_length()

        progress_update.emit(25) # update progressBar

        # Test connection
        start_point = (source.gdf['geometry'][0].x, source.gdf['geometry'][0].y)
        connected_points = graph.get_connected_points(start_point)
        all_points = list(graph.graph.nodes)
        if start_point not in connected_points:
            connected_points.append(start_point)
        if len(all_points) > len(connected_points):
            # get disconnected points
            disconnected_points = [point for point in all_points if point not in connected_points and point != start_point]
            # building centroids
            building_centroids = [(centroid.x, centroid.y) for centroid in buildings.gdf['centroid']]
            # check if building centroids are disconnected
            disconnected_buildings = [centroid for centroid in building_centroids if centroid in disconnected_points]
            if disconnected_buildings:
                # feedback
                label_update.emit(self.tr('Some Buildings are not connected to the street network! Please connect the nearest street to the street network by using the snapping tool or set the "Moegliche_Route/possoble_route"-attribute of their corresponding street to zero to connect them to another street.'), '#ff5555')
                # save parameters as self attributes to plot in main thread
                graph.start_point = start_point
                graph.connected_points = connected_points
                graph.disconnected_buildings = disconnected_buildings
                self.graph = graph
                self.network_analysis_status = 'plot'
                return
            
            # # check if polygon is activated
            # if self.dlg.net_checkBox_polygon.isChecked():
            #     # check if disconnected points are inside the polygon
            #     disconnected_points = [point for point in all_points if point not in connected_points and point != start_point]
            #     #print( f'disconnected nodes: {len(disconnected_points)}')
            #     #print(disconnected_points)
            #     if any(polygon['geometry'][0].contains(Point(point)) for point in disconnected_points):
            #         # feedback
            #         label_update.emit('Some points of the street network in your area are not connected! Please set their "possible_route"-attribute to zero or connect them to the street network by using the snapping tool.','#ff5555')
            #         # save parameters as self attributes to plot in main thread
            #         graph.start_point = start_point
            #         graph.connected_points = connected_points
            #         self.graph = graph
            #         self.network_analysis_status = 'plot'
            #         return
            # else:
                
            #     #print( f'{len(disconnected_points)} disconnected nodes')
            #     #print(disconnected_points)
            #     # feedback
            #     label_update.emit('Some points of the street network are not connected! Please set their "possible_route"-attribute to zero or connect them to the street network by using the snapping tool.', '#ff5555')
            #     # save parameters as self attributes to plot in main thread
            #     graph.start_point = start_point
            #     graph.connected_points = connected_points
            #     self.graph = graph
            #     self.network_analysis_status = 'plot'
            #     return

        progress_update.emit(30) # update progressBar

        ### Net Analysis ###
        net = Net(t_supply,t_return,crs=buildings.gdf.crs)
        net.network_analysis(graph.graph, buildings.gdf, source.gdf, self.pipe_info, power_th_att=power_attribute, progressBar=self.dlg.net_progressBar)

        progress_update.emit(70) # update progressBar

        # GeoDataFrame from net
        net.ensure_power_th_attribute()
        net.graph_to_gdf()

        # translate
        net.rename_columns()
        
        # save net as instance attribute
        self.net_gdf = net.gdf

        # set net status as completed
        self.network_analysis_status = 'complete'

    def create_result(self, progress_update, label_update):
        '''
        Generate and save the results of the network analysis, including a detailed
        energy demand profile and associated data for a district heating system.

        This method performs the following steps:

        1. **Progress Bar Initialization**:
        - Sets the progress bar to 0, indicating the start of the result creation process.

        2. **User Feedback**:
        - Displays an initial status message indicating that calculations are in progress.

        3. **Load Pipe Data**:
        - Loads pipe information from an Excel file, including pipe diameters (DN).

        4. **Define Load Profiles**:
        - Sets the load profiles for different building types (e.g., EFH, MFH).

        5. **Retrieve Temperature Values**:
        - Retrieves the supply and return temperatures from the user interface.

        6. **Retrieve Layer Paths**:
        - Gets file paths and objects for source, streets, and buildings layers from combo boxes.

        7. **Select Attributes**:
        - Retrieves selected heat and power attributes from the user interface.

        8. **Load Network Shapefile**:
        - Loads the network data from the specified shapefile path.

        9. **Instantiate Classes**:
        - Creates instances of relevant classes for buildings, source, and streets.

        10. **Polygon Filtering**:
            - If a polygon is selected, filters buildings to include only those within the polygon boundaries.

        11. **Progress Bar Update**:
            - Updates the progress bar after loading and preparing data.

        12. **Create Result Data Structure**:
            - Initializes the `Result` class and prepares a data dictionary from the buildings and network data.

        13. **Generate DataFrame**:
            - Converts the data dictionary into a DataFrame and saves it as an Excel file.

        14. **Load Curve Generation**:
            - If a temperature file is provided, loads the temperature data; otherwise, retrieves historical temperature data from an external source.

        15. **Energy Demand Profile Creation**:
            - Creates a time series DataFrame for energy demand based on load profiles, temperature data, and building heat demand.

        16. **Aggregate Demand and Losses**:
            - Adds a column for the sum of all buildings' demands and calculates losses. 
            - Adds a total column that includes both the demand and losses.

        17. **Visualization**:
            - Plots and saves bar charts for the energy demand profile and sorted demand profile.

        18. **Save and Embed Results**:
            - Saves the demand profile in an Excel file and embeds the generated plots as images.

        19. **Open Result File**:
            - Opens the resulting Excel file for user review.

        20. **Completion**:
            - Updates the progress bar to 100% and displays a completion message to the user.

        Returns
        -------
        None
        '''
        self.result_status = 0
        progress_update.emit(0) # update progressBar

        # Check for a valid project_path. Give feedback if project is not saved
        project_file_path = QgsProject.instance().fileName()
        if project_file_path:
            self.project_dir = os.path.dirname(project_file_path)
        else:
            # feedback
            label_update(self.tr('This QGIS project has no valid directory!\nPlease save the project.'), '#ff5555')
            return

        # Load Profiles
        load_profiles = ['EFH', 'MFH', 'GHA', 'GMK', 'GKO']

        # Temperatures from SpinBox
        t_supply = self.dlg.net_doubleSpinBox_supply.value()
        t_return = self.dlg.net_doubleSpinBox_return.value()

        # Layer paths
        source_path, source_layer, source_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_source)
        streets_path, streets_layer, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_streets)
        buildings_path, buildings_layer, buildings_layer_obj = self.get_layer_path_from_combobox(self.dlg.net_comboBox_buildings)
        
        heat_attribute = self.dlg.net_comboBox_heat.currentText()
        power_attribute = self.dlg.net_comboBox_power.currentText()

        # check if attributes and paths are selected correctly
        if heat_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as heat demand.'),'orange')
            return
        if power_attribute == self.tr('Select Attribute'):
            label_update.emit(self.tr('Please select an Attribute as thermal power.'),'orange')
            return

        if self.dlg.net_lineEdit_net.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the net shapefile output'),'orange')
            return
        if self.dlg.net_lineEdit_result.text().strip() == "":
            label_update.emit(self.tr('Specify a file path for the result file'),'orange')
            return
        
        # net path
        net_path = self.dlg.net_lineEdit_net.text()
        net_gdf = gpd.read_file(net_path)
        
        # feedback
        label_update.emit(self.tr('Calculating...'), 'white')

        # Instantiate classes
        buildings = Buildings(buildings_path, heat_attribute, buildings_layer)
        source = Source(source_path, source_layer)
        streets = Streets(streets_path, streets_layer)

        # filter buildings
        buildings.gdf = buildings.gdf[buildings.gdf['Anschluss']==1]
        
        # check if polygon checkbox is checked
        if self.dlg.net_checkBox_polygon.isChecked():
            polygon_path, polygon_layer, polygon_layer_obj  = self.get_layer_path_from_combobox(self.dlg.net_comboBox_polygon)
            # load polygon as gdf
            if polygon_layer == None:
                polygon = gpd.read_file(polygon_path)
            else:
                polygon = gpd.read_file(polygon_path, layer=polygon_layer)

            # only buildings within polygon
            buildings.gdf = gpd.sjoin(buildings.gdf, polygon, how="inner", predicate="within")
       
        progress_update.emit(10) # update progressBar

        ### result ###
        result_path = self.dlg.net_lineEdit_result.text()
        result = Result(result_path)

        # Check if excel file is already open
        if result.is_excel_file_open():
            label_update.emit(self.tr('Excel result file is already open. Please close the result file to save new result.'), 'orange')
            self.worker_running = False # Reset worker_running
            return

        result.create_data_dict(buildings.gdf, net_gdf, load_profiles, self.dn_list, heat_attribute, t_supply, t_return)
        result.create_df_from_dataDict(net_name = os.path.splitext(os.path.basename(net_path))[0])
        
        progress_update.emit(15) # update progressBar

        ### building statistic ###
        statistic = result.building_statistic(buildings.gdf)
        # save statistic as result attribute
        result.statistic = statistic
        
        # save result as instance attribute
        self.result = result
        
        progress_update.emit(20) # update progressBar

        ### Load Curve ###

        # set up time data
        year = 2022
        resolution = 8760
        freq = 'H'

        # Holidays
        cal = Germany()
        holidays = dict(cal.holidays(year))

        # temperature
        temperature_data = self.temp_profile['TT_TU']

        # load_profile class
        load_profile = LoadProfile(result.gdf, result_path, year, temperature_data, holidays)
        
        # dataframe for collecting generated profiles
        demand = load_profile.set_up_df(year, resolution, freq)

        progress_update.emit(30) # update progressBar

        for row in load_profile.net_result.itertuples(index=False):
            building_type = row.Lastprofil
            building_class = 3 # Baualtersklasse NRW:3 Quelle:Praxisinformation P 2006 / 8 Gastransport / Betriebswirtschaft, BGW, 2006, Seite 43 Tabelle 2 und 3
            if pd.isna(building_type):
                break
            elif building_type.lower() not in ('efh', 'mfh'):
                building_class = 0
            hd = row[2]
            demand[building_type] = load_profile.create_heat_demand_profile(building_type, building_class, 0, 1, hd)

        progress_update.emit(35) # update progressBar

        # add sum column for all buildings
        demand_with_sum_buildings = load_profile.add_sum_buildings(demand)

        # add loss
        demand_with_loss = load_profile.add_loss(demand_with_sum_buildings, load_profile.net_result, resolution)
        
        # add sum buildings+loss
        demand_with_sum = load_profile.add_sum(demand_with_loss)

        progress_update.emit(40) # update progressBar

        # plot and save fig
        load_profile.plot_bar_chart(demand_with_sum, column_names=['Gesamtsumme', 'Verlust'], filename = self.project_dir+'/Lastprofil.png')
        progress_update.emit(50) # update progressBar
        load_profile.plot_bar_chart(demand_with_sum, column_names=['Gesamtsumme (extra Dämmung)', 'Verlust'], colors=['green','orange'], filename = self.project_dir+'/Lastprofil_extra_Daemmung.png', ylabel='Wärmebedarf und Verlust bei extra Dämmung [MW]', title='Wärmebedarf und Verlust bei extra Dämmung pro Stunde im Jahr')

        progress_update.emit(60) # update progressBar

        # order
        sorted_demand = demand_with_sum.sort_values(by='Gesamtsumme', ascending=False)

        # plot and save fig
        load_profile.plot_bar_chart(sorted_demand, column_names=['Gesamtsumme', 'Verlust'], filename = self.project_dir+'/Lastprofil_geordnet.png', title='Geordnetes Lastprofil')

        progress_update.emit(70) # update progressBar

        # order extra insulation
        sorted_demand = demand_with_sum.sort_values(by='Gesamtsumme (extra Dämmung)', ascending=False)

        # plot and save fig extra insulation
        load_profile.plot_bar_chart(sorted_demand, column_names=['Gesamtsumme (extra Dämmung)', 'Verlust'], colors=['green','orange'], filename = self.project_dir+'/Lastprofil_extra_Daemmung_geordnet.png', ylabel='Wärmebedarf und Verlust bei extra Dämmung [MW]', title='Geordnetes Lastprofil (extra Dämmung)')

        progress_update.emit(80) # update progressBar

        # round columns of demand dataframe
        demand_with_sum = demand_with_sum.round(decimals=3)

        # Save load_profile as instance attribute
        load_profile.demand_with_sum = demand_with_sum
        self.load_profile = load_profile

        self.result_status = 'complete'


    # Methods to start main methods in background
    def start_download_files(self):
        '''
        Starts the process of downloading shapefiles for buildings, streets, and parcels and updates the GUI elements accordingly.

        This method initializes the GUI elements (progress bar and label), starts the download process in a background thread using the 
        `run_long_task` method, and defines a callback function `on_task_finished` to handle the post-download operations. 
        Upon successful download, the shapefiles are saved to the user-specified paths and added to the QGIS project.

        Steps:
        1. Initializes GUI elements such as the progress bar and status label.
        2. Starts the download task in the background.
        3. When the download is complete, the shapefiles for buildings, streets, and parcels are saved to specified locations.
        4. The shapefiles are added to the QGIS project.
        5. Updates the GUI with a completion message and sets the progress bar to 100%.

        Notes:
        - If the `download_status` is not 'complete', no further action is taken.

        Parameters
        ----------
        None

        Returns
        -------
        None
        '''
        # Chek if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        
        gui_elements = {
            'progressBar': self.dlg.load_progressBar, # progress bar 
            'label': self.dlg.load_label_feedback     # feedback label
        }
        
        def on_task_finished():
            '''function that is executed, once the background task is complete'''

            # Check if background tasl is complete
            if self.download_status == 'complete':
                # Get paths from line edits
                buildings_path = self.dlg.load_lineEdit_buildings.text()
                streets_path = self.dlg.load_lineEdit_streets.text()
                parcels_path = self.dlg.load_lineEdit_parcels.text()

                # Save shapefiles 
                self.buildings_gdf.to_file(buildings_path)
                self.streets_gdf.to_file(streets_path)
                self.parcels_gdf.to_file(parcels_path)

                # Füge die Shapefiles zum QGIS Projekt hinzu
                self.add_shapefile_to_project(parcels_path, style = 'parcels', group_name = self.tr('Basic Data'))
                self.add_shapefile_to_project(buildings_path, style = 'buildings', group_name = self.tr('Basic Data'))
                self.add_shapefile_to_project(streets_path, style = 'streets', group_name = self.tr('Basic Data'))

                # GUI-Feedback aktualisieren
                self.dlg.load_progressBar.setValue(100)
                self.dlg.load_label_feedback.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.load_label_feedback.setText(self.tr('Download complete!'))
            # Reset worker_running
            self.worker_running = False
            
        # Start the background process
        self.worker_running = True
        self.run_long_task(self.download_files, gui_elements, on_task_finished)
    
    def start_download_zensus(self):
        # check if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        # define GUI elements
        gui_elements = {
            'progressBar': self.dlg.load_progressBar_zensus, # progress bar 
            'label': self.dlg.load_label_zensus     # feedback label
        }
        def on_task_finished():
            '''function that is executed, once the background task is complete'''
            # check if the backround task is complete
            if self.download_zensus_status == 'complete':
                # get path from lineEdit
                path = self.dlg.load_lineEdit_zensus.text()
                
                # save gdf to file
                self.zensus_gdf.to_file(path)

                # add shapefile to project
                self.add_shapefile_to_project(path, 'zensus')

                # update progressBar
                self.dlg.load_progressBar_zensus.setValue(100)
                self.dlg.load_label_zensus.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.load_label_zensus.setText(self.tr('Download complete!'))
            # Reset worker_running
            self.worker_running = False
            
        self.worker_running = True
        self.run_long_task(self.download_zensus, gui_elements, on_task_finished)

    def start_adjust_files(self):
        # check if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        
        # import building_info ### Excel files have to be imported in main thread. Imports in background lead to a crasg because of windows acces violation
        excel_path = self.plugin_dir+'/data/building_info.xlsx'
        self.excel_building_info = pd.read_excel(excel_path, sheet_name='database')
        self.excel_building_demand_wg = pd.read_excel(excel_path, sheet_name='Grunddaten_Gebaeude', nrows=13, usecols='A:D')

        # define GUI elements
        gui_elements = {
            'progressBar': self.dlg.adjust_progressBar, # progress bar 
            'label': self.dlg.adjust_label_feedback     # feedback label
        }
        def on_task_finished():
            '''function that is executed, once the background task is complete'''
            # check if the backround task is complete
            if self.bool_files_already_adjusted == False:
                # get paths from layers
                streets_path, streets_layer_name, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.adjust_comboBox_streets)
                buildings_path, buildings_layer_name, buildings_layer_obj = self.get_layer_path_from_combobox(self.dlg.adjust_comboBox_buildings)
                
                # get path from lineEdit if new is checked
                if self.dlg.adjust_radioButton_new.isChecked():
                    buildings_path = self.dlg.adjust_lineEdit_buildings.text()
                    streets_path = self.dlg.adjust_lineEdit_streets.text()

                # save shapes
                self.buildings_gdf.to_file(buildings_path)
                self.streets_gdf.to_file(streets_path)

                # check if files are overwritten or newly created
                if self.dlg.adjust_radioButton_new.isChecked():
                    self.add_shapefile_to_project(streets_path, style = 'streets_adj', group_name = self.tr('Adjusted Files'))
                    self.add_shapefile_to_project(buildings_path, style = 'buildings_adj', group_name = self.tr('Adjusted Files'))
                else:
                    QgsProject.instance().removeMapLayer(buildings_layer_obj)
                    QgsProject.instance().removeMapLayer(streets_layer_obj)
                    self.add_shapefile_to_project(streets_path, style = 'streets', group_name = self.tr('Adjusted Files'))
                    self.add_shapefile_to_project(buildings_path, style = 'buildings', group_name = self.tr('Adjusted Files'))
                
                self.dlg.adjust_progressBar.setValue(100) # update progressBar

                self.dlg.adjust_label_feedback.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.adjust_label_feedback.setText(self.tr('Completed!'))
                self.dlg.adjust_label_feedback.repaint()
            # Reset worker_running
            self.worker_running = False
        self.worker_running = True
        self.run_long_task(self.adjust_files, gui_elements, on_task_finished)

    def start_status_analysis(self):
        # check if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        # define GUI elements
        gui_elements = {
            'progressBar': self.dlg.status_progressBar, # progress bar 
            'label': self.dlg.status_label_response   # feedback label
        }
        def on_task_finished():
            '''function that is executed, once the background task is complete'''
            # check if the backround task is complete
            if self.status_analysis_status == 'complete':
                # layer from combo box
                streets_path, streets_layer_name, streets_layer_obj = self.get_layer_path_from_combobox(self.dlg.status_comboBox_streets)
                
                # path from lineEdit
                polygon_path = self.dlg.status_lineEdit_polygons.text()

                # save shapefiles
                self.wld.to_file(streets_path)
                self.polygons.to_file(polygon_path)

                # add shapefiles to project
                self.add_shapefile_to_project(streets_path, 'wld', self.tr('Heat Density'))
                self.add_shapefile_to_project(polygon_path, 'polygons', self.tr('Heat Density'))

                # update progressBar
                self.dlg.status_progressBar.setValue(100)
                self.dlg.status_label_response.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.status_label_response.setText(self.tr('Completed!'))

            # Reset worker_running
            self.worker_running = False
            
        self.worker_running = True
        self.run_long_task(self.status_analysis, gui_elements, on_task_finished)
    
    def start_network_analysis(self):
        # check if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        
        # define GUI elements
        gui_elements = {
            'progressBar': self.dlg.net_progressBar, # progress bar 
            'label': self.dlg.net_label_response   # feedback label
        }

        # pipe info
        excel_file_path = Path(self.plugin_dir) / 'data/pipe_data.xlsx'
        self.pipe_info = pd.read_excel(excel_file_path, sheet_name='pipe_data')

        def on_task_finished():
            '''function that is executed, once the background task is complete'''
            # check if the backround task is complete
            if self.network_analysis_status == 'complete':
                # path to save net shape file
                net_path = self.dlg.net_lineEdit_net.text()

                # save net shapefile
                self.net_gdf.to_file(net_path)

                # load net as layer
                self.add_shapefile_to_project(net_path, 'net', group_name = self.tr('Net'))

                # update progressBar
                self.dlg.net_progressBar.setValue(100)
                # feedback
                self.dlg.net_label_response.setText(self.tr('Completed!'))
                self.dlg.net_label_response.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.net_label_response.repaint()
            elif self.network_analysis_status == 'plot':
                self.graph.plot_graph(self.graph.start_point, self.graph.connected_points, self.graph.disconnected_buildings)
            # Reset worker_running
            self.worker_running = False
            return
        self.worker_running = True
        self.run_long_task(self.network_analysis, gui_elements, on_task_finished)
    
    def start_create_result(self):
        
        # check if another process is already running
        if self.worker_running == True:
            QMessageBox.warning(self.dlg, self.tr('Warning'), self.tr('A process is already running. Please wait for completion to start a new process.'))
            return
        
        # define GUI elements
        gui_elements = {
            'progressBar': self.dlg.net_progressBar, # progress bar 
            'label': self.dlg.net_label_response   # feedback label
        }
        # pipe info
        excel_file_path = Path(self.plugin_dir) / 'data/pipe_data.xlsx'
        self.pipe_info = pd.read_excel(excel_file_path, sheet_name='pipe_data')
        self.dn_list = self.pipe_info['DN'].to_list()

        # temperature
        if self.dlg.net_checkBox_temperature.isChecked():
            temp_path = self.dlg.net_lineEdit_temperature.text()

            # chek if line edit is empty
            if temp_path.strip() == "":
                self.dlg.net_label_response.setText(self.tr('Specify a file path for your custom temperature or uncheck the box.'))
                self.dlg.net_label_response.setStyleSheet('color: orange')
                self.dlg.net_label_response.repaint()
                return
        else:
            temp_path = Path(self.plugin_dir) / 'data/example_temperature.xlsx'
        self.temp_profile = pd.read_excel(temp_path)

        def on_task_finished():
            '''function that is executed, once the background task is complete'''
            if self.result_status == 'complete':

                result = self.result

                # Copy result file to user path
                costs_path = Path(self.plugin_dir) / 'data/result.xlsx'
                result.copy_excel_file(costs_path)

                # Save result
                result.save_in_excel(result_table = result.gdf)

                # Save Statistic
                result.save_in_excel(result_table = result.statistic, sheet = 'Statistik')
                
                # get load_profile from instance attributes
                load_profile = self.load_profile

                # save demand profile in excel
                load_profile.save_in_excel(load_profile.demand_with_sum, index_bool=True)

                # save load curve plot in excel
                load_profile.embed_image_in_excel(0,load_profile.demand_with_sum.shape[1]+1, image_filename = self.project_dir+'/Lastprofil.png')

                # save sorted load curve plot in excel
                load_profile.embed_image_in_excel(22,load_profile.demand_with_sum.shape[1]+1, image_filename = self.project_dir+'/Lastprofil_geordnet.png')

                # save load curve plot in excel extra insulation
                load_profile.embed_image_in_excel(44,load_profile.demand_with_sum.shape[1]+1, image_filename = self.project_dir+'/Lastprofil_extra_Daemmung.png')

                # save sorted load curve plot in excel extra insulation
                load_profile.embed_image_in_excel(66,load_profile.demand_with_sum.shape[1]+1, image_filename = self.project_dir+'/Lastprofil_extra_Daemmung_geordnet.png')

                # open result
                load_profile.open_excel_file()

                # update progressBar
                self.dlg.net_progressBar.setValue(100)
                # feedback
                self.dlg.net_label_response.setText(self.tr('Completed!'))
                self.dlg.net_label_response.setStyleSheet("color: rgb(0, 255, 0)")
                self.dlg.net_label_response.repaint()
            # Reset worker_running
            self.worker_running = False
        
        self.worker_running = True
        self.run_long_task(self.create_result, gui_elements, on_task_finished)

    # run method that starts, when the icon is clicked in QGIS
    def run(self):
        """Run method that performs all the real work"""

        # Project path
        project_file_path = QgsProject.instance().fileName()
        self.project_dir = os.path.dirname(project_file_path)

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = HeatNetToolDialog()

            # check modules
            try:
                import pandas as pd
                import geopandas as gpd
                from shapely import Point
                from .src.download_files import file_list_from_URL, search_filename, read_file_from_zip, filter_df, get_shape_from_wfs, clean_data, add_point, create_square, get_area_for_zensus
                from .src.adjust_files import Streets_adj, Buildings_adj, Parcels_adj, spatial_join
                from .src.status_analysis import WLD, Polygons
                from .src.net_analysis import Streets, Source, Buildings, Graph, Net, Result, get_closest_point, calculate_GLF, calculate_volumeflow, calculate_diameter_velocity_loss
                from .src.load_curve import Temperature, LoadProfile
                from workalendar.europe import Germany
                from matplotlib.figure import Figure
                import matplotlib.pyplot as plt
            except Exception as e:
                message_box = QMessageBox()
                message_box.setIcon(QMessageBox.Warning)
                message_box.setWindowTitle('Import Error')
                message_box.setText(self.tr('Failed to import a required module: {}').format(str(e)))
                message_box.setInformativeText(self.tr('Please install all required Python packages by pressing "Install Packages" in the Introduction part of the F|Heat plugin.'))
                message_box.exec_()

            # check python version
            self.dlg.intro_pushButton_python_version.clicked.connect(lambda: self.check_python_version())

            # install python packages
            self.dlg.intro_pushButton_load_packages.clicked.connect(lambda: self.install_package())

            ### Load ###

            # download options
            self.load_download_options()
            
            # shape paths
            self.dlg.load_pushButton_buildings.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.load_lineEdit_buildings,'*.gpkg;;*.shp'))
            self.dlg.load_pushButton_parcels.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.load_lineEdit_parcels,'*.gpkg;;*.shp'))
            self.dlg.load_pushButton_streets.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.load_lineEdit_streets,'*.gpkg;;*.shp'))
            
            # Start Download Files 
            #self.dlg.load_pushButton_start.clicked.connect(self.download_files_alt) # main thread
            self.dlg.load_pushButton_start.clicked.connect(self.start_download_files) # background

            # zensus
            self.dlg.load_pushButton_zensus.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.load_lineEdit_zensus,'*.gpkg;;*.shp'))
            
            #self.dlg.load_pushButton_start_zensus.clicked.connect(self.download_zensus_alt) # main thread
            self.dlg.load_pushButton_start_zensus.clicked.connect(self.start_download_zensus) # background

            ### Adjust ###

            # shape paths
            self.dlg.adjust_pushButton_buildings.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.adjust_lineEdit_buildings,'*.gpkg;;*.shp'))
            self.dlg.adjust_pushButton_streets.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.adjust_lineEdit_streets,'*.gpkg;;*.shp'))
            
            # Start Adjust Files 
            # self.dlg.adjust_pushButton_start.clicked.connect(self.adjust_files_alt) # main thread
            self.dlg.adjust_pushButton_start.clicked.connect(self.start_adjust_files) # background thread

            ### status ###

            # shape paths
            self.dlg.status_pushButton_polygons.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.status_lineEdit_polygons,'*.gpkg;;*.shp'))
            
            # start status analysis
            self.dlg.status_pushButton_start.clicked.connect(self.start_status_analysis)

            ### Net ###

            # select output file
            self.dlg.net_pushButton_net_output.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.net_lineEdit_net,'*.gpkg;;*.shp'))
            self.dlg.net_pushButton_result.clicked.connect(
                lambda: self.select_output_file(self.project_dir, self.dlg.net_lineEdit_result,'*.xlsx'))
            self.dlg.net_pushButton_temperature.clicked.connect(
                lambda: self.select_input_file(self.project_dir, self.dlg.net_lineEdit_temperature,'*.xlsx'))

            # start network analysis
            self.dlg.net_pushButton_start.clicked.connect(self.start_network_analysis)

            # create result file
            self.dlg.net_pushButton_create_result.clicked.connect(self.start_create_result)
        
        # Create F|Heat layer structure
        self.create_layer_tree_structure()

        # update the download options
        self.dlg.load_comboBox_municipality.currentIndexChanged.connect(self.adapt_download_options)

        # Connect attribute combobox to layer combobox
        self.load_attributes('adjust_comboBox_buildings', 'adjust_comboBox_heat')
        self.dlg.adjust_comboBox_buildings.currentIndexChanged.connect(
            lambda: self.load_attributes('adjust_comboBox_buildings', 'adjust_comboBox_heat'))

        # Connect attribute combobox to layer combobox
        self.load_attributes('status_comboBox_buildings', 'status_comboBox_heat')
        self.load_attributes('status_comboBox_buildings', 'status_comboBox_power')
        self.dlg.status_comboBox_buildings.currentIndexChanged.connect(
            lambda: self.load_attributes('status_comboBox_buildings', 'status_comboBox_heat'))
        self.dlg.status_comboBox_buildings.currentIndexChanged.connect(
            lambda: self.load_attributes('status_comboBox_buildings', 'status_comboBox_power'))
        
        # Connect attribute combobox to layer combobox
        self.load_attributes('net_comboBox_buildings', 'net_comboBox_heat')
        self.load_attributes('net_comboBox_buildings', 'net_comboBox_power')
        self.dlg.net_comboBox_buildings.currentIndexChanged.connect(
            lambda: self.load_attributes('net_comboBox_buildings', 'net_comboBox_heat'))
        self.dlg.net_comboBox_buildings.currentIndexChanged.connect(
            lambda: self.load_attributes('net_comboBox_buildings', 'net_comboBox_power'))

        # show the dialog
        self.dlg.show()