import os
import re
import sys
import numpy as np
import rasterio
from rasterio.enums import Resampling
from rasterio.warp import reproject
from rasterio.mask import mask
import geopandas as gpd
import math
import logging

# Configuração básica do logging
logging.basicConfig(level=logging.DEBUG, filename='app.log', filemode='w', 
                    format='%(asctime)s - %(name)s - %(levellevelname)s - %(message)s')

def read_mtl(caminho_mtl):
    mtl_data = {}
    try:
        with open(caminho_mtl, 'r') as file:
            for line in file:
                parts = line.strip().split('=')
                if len(parts) == 2:
                    key, value = parts
                    mtl_data[key.strip()] = value.strip().strip('"')
    except IOError:
        print("Erro ao ler o arquivo MTL. Verifique o caminho fornecido.")
        sys.exit(1)
    return mtl_data

def recortar_e_aliar_mdt(caminho_mdt, shapefile_path, output_path, caminho_raster_referencia):
    try:
        with rasterio.open(caminho_raster_referencia) as ref_raster:
            ref_transform = ref_raster.transform
            ref_crs = ref_raster.crs
            ref_width = ref_raster.width
            ref_height = ref_raster.height
        
        with rasterio.open(caminho_mdt) as src:
            shapefile = gpd.read_file(shapefile_path)
            if shapefile.empty:
                print("Erro: Shapefile vazio ou não intersecta o MDT.")
                return None, None
            geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
            out_image, out_transform = mask(src, geometrias, crop=True)
            if out_image.size == 0:
                print("Erro: A máscara resultou em uma imagem vazia.")
                return None, None
            out_meta = src.meta.copy()
            out_meta.update({
                "driver": "GTiff",
                "height": ref_height,
                "width": ref_width,
                "transform": ref_transform,
                "crs": ref_crs,
                "nodata": src.nodata
            })
            reamostrado_image = np.zeros((ref_height, ref_width), dtype=src.dtypes[0])
            reproject(
                source=out_image,
                destination=reamostrado_image,
                src_transform=out_transform,
                src_crs=src.crs,
                dst_transform=ref_transform,
                dst_crs=ref_crs,
                resampling=Resampling.nearest
            )
            with rasterio.open(output_path, 'w', **out_meta) as dst:
                dst.write(reamostrado_image, 1)
            return reamostrado_image, out_meta
    except Exception as e:
        print(f"Erro ao recortar e alinhar MDT: {e}")
        return None, None

def process_images(caminho_bandas, shapefile_path, output_dir, mtl_data):
    bandas = {}
    meta_data = None
    if not os.path.exists(caminho_bandas):
        print("Diretório das bandas não encontrado.")
        return None

    arquivos = [os.path.join(caminho_bandas, arquivo) for arquivo in os.listdir(caminho_bandas) if arquivo.endswith(('.TIF', '.tif'))]
    out_meta = None  # Inicialização de out_meta

    for arquivo in arquivos:
        nome_banda = os.path.basename(arquivo)
        match = re.search(r'_B(\d+)', nome_banda)
        if match:
            band_number = int(match.group(1))
        else:
            print(f"Nenhum número de banda encontrado em {nome_banda}. Pulando este arquivo.")
            continue

        nome_saida = f'band{band_number}.tif'
        try:
            with rasterio.open(arquivo) as src:
                shapefile = gpd.read_file(shapefile_path)
                geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
                out_image, out_transform = mask(src, geometrias, crop=True)
                out_meta = src.meta.copy()
                out_meta.update({
                    "driver": "GTiff",
                    "height": out_image.shape[1],
                    "width": out_image.shape[2],
                    "transform": out_transform
                })
                
                if out_image.ndim == 4:
                    out_image = out_image.reshape((1, *out_image.shape[-2:]))

                if band_number == 10:
                    processed_data = out_image[0]  # Para a banda 10, use a imagem recortada diretamente
                else:
                    toa_reflectance = out_image[0] * float(mtl_data[f'REFLECTANCE_MULT_BAND_{band_number}']) + float(mtl_data[f'REFLECTANCE_ADD_BAND_{band_number}'])
                    processed_data = toa_reflectance
                
                output_path = os.path.join(output_dir, nome_saida)
                with rasterio.open(output_path, 'w', **out_meta) as dst:
                    dst.write(processed_data, 1)
                bandas[nome_saida.replace('.tif', '')] = processed_data

        except Exception as e:
            print(f"Erro ao processar a banda {band_number}: {e}. Pulando.")
            continue

    return bandas, meta_data, out_meta

def run_processing(caminho_mtl, caminho_mdt, caminho_bandas, shapefile_path, output_dir, raster_referencia_path, u_2m, EToi, ETo):
    mtl_data = read_mtl(caminho_mtl)
    
    bandas = {}  # Inicialize bandas como dicionário vazio para evitar NameError

    if mtl_data:
        mdt_output_path = os.path.join(output_dir, 'MDT_Sebal_recorte.tif')
        mdt_recortado, mdt_meta = recortar_e_aliar_mdt(caminho_mdt, shapefile_path, mdt_output_path, raster_referencia_path)
        
        if mdt_recortado is not None:
            print("Processamento do MDT concluído com sucesso.")
            bandas, meta_data, out_meta = process_images(caminho_bandas, shapefile_path, output_dir, mtl_data)
            if not bandas:
                print("Nenhuma banda processada.")
        else:
            print("Falha ao processar MDT.")
    else:
        print("Falha ao carregar dados MTL, terminando o programa.")

    # Acessando bandas de forma segura
    if bandas:
        for banda, dados in bandas.items():
            print(f"Tipo de dados para {banda}: {type(dados)}")
        
        # Seguramente acessando bandas específicas usando get para evitar KeyError
        nir_band = bandas.get('band5')  # Banda Infravermelho próximo
        red_band = bandas.get('band4')  # Banda vermelha
        green_band = bandas.get('band3') # Banda verde
        blue_band = bandas.get('band2')  # Banda azul
        band1 = bandas.get('band1')
        band2 = bandas.get('band2')
        band3 = bandas.get('band3')
        band4 = bandas.get('band4')
        band5 = bandas.get('band5')
        band6 = bandas.get('band6')
        band7 = bandas.get('band7')
        band8 = bandas.get('band8')
        band9 = bandas.get('band9')
        band10 = bandas.get('band10')
                
        # Verificar se as bandas foram carregadas
        if nir_band is None or red_band is None or green_band is None or blue_band is None:
            print("Algumas bandas necessárias estão faltando.")
        else:
            # Processamento adicional se todas as bandas necessárias estiverem presentes
            print("Todas as bandas necessárias foram carregadas com sucesso.")
    else:
        print("Nenhuma banda disponível para processamento.")

    # Verifica se as bandas são arrays do NumPy e cria um composto RGB
    if isinstance(red_band, np.ndarray) and isinstance(green_band, np.ndarray) and isinstance(blue_band, np.ndarray):
        rgb_composite = np.stack((red_band, green_band, blue_band), axis=0)
        rgb_output_path = os.path.join(output_dir, 'CC_432.tif')
        rgb_meta = out_meta.copy()
        rgb_meta.update({
            'driver': 'GTiff',
            'count': 3,
            'dtype': rgb_composite.dtype,
            'width': rgb_composite.shape[2],
            'height': rgb_composite.shape[1]
        })

        try:
            with rasterio.open(rgb_output_path, 'w', **rgb_meta) as dst:
                for i, band in enumerate(rgb_composite, start=1):
                    dst.write(band, i)
            print("Composite RGB Landsat 8, be patient... Done!")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")
    else:
        print("Erro: Uma ou mais bandas não são arrays do NumPy.")

    print("Done!")

    np.seterr(divide='ignore', invalid='ignore')

    if nir_band is not None and red_band is not None:
        NDVI = (nir_band - red_band) / (nir_band + red_band + 1e-10)
        NDVI = np.clip(NDVI, -1, 1)
    else:
        print("Erro: Bandas NIR ou Red estão ausentes ou não foram carregadas corretamente.")
        NDVI = None

    if NDVI is not None:
        ndvi_output_path = os.path.join(output_dir, 'NDVI.tif')
        ndvi_meta = out_meta.copy()
        ndvi_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': NDVI.shape[1],
            'height': NDVI.shape[0]
        })

        try:
            with rasterio.open(ndvi_output_path, 'w', **ndvi_meta) as dst:
                dst.write(NDVI, 1)
            print("NDVI salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

    if nir_band is not None and red_band is not None:
        Lsavi = 0.5
        SAVI = ((nir_band - red_band) / (nir_band + red_band + Lsavi)) * (1 + Lsavi)
        SAVI = np.clip(SAVI, -1, 1)

        savi_output_path = os.path.join(output_dir, 'SAVI.tif')
        savi_meta = out_meta.copy()
        savi_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': SAVI.shape[1],
            'height': SAVI.shape[0]
        })

        try:
            with rasterio.open(savi_output_path, 'w', **savi_meta) as dst:
                dst.write(SAVI, 1)
            print("SAVI salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")
    
    def calculate_lai(savi):
        lai = np.zeros_like(savi, dtype=float)
        lai[savi < 0.1] = 0.00001
        mask = (savi >= 0.1) & (savi < 0.687)
        lai[mask] = -np.log((0.69 - savi[mask]) / 0.59) / 0.91
        lai[savi >= 0.687] = 6
        return lai

    LAI = calculate_lai(SAVI)

    print(LAI)

    lai_output_path = os.path.join(output_dir, 'LAI.tif')

    lai_meta = out_meta.copy()
    lai_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': LAI.shape[1],
        'height': LAI.shape[0]
    })

    try:
        with rasterio.open(lai_output_path, 'w', **lai_meta) as dst:
            dst.write(LAI, 1)
        print("LAI salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    mtl_values = read_mtl(caminho_mtl)
    radiance_mult = mtl_values['RADIANCE_MULT_BAND_10']
    radiance_add = mtl_values['RADIANCE_ADD_BAND_10']

    lai_path = os.path.join(output_dir, 'LAI.tif')
    ndvi_path = os.path.join(output_dir, 'NDVI.tif')
    band10_path = os.path.join(output_dir, 'band10.tif')

    try:
        with rasterio.open(lai_path) as src_lai, rasterio.open(ndvi_path) as src_ndvi, rasterio.open(band10_path) as src_band10:
            lai = src_lai.read(1)
            ndvi = src_ndvi.read(1)
            band10 = src_band10.read(1)
        
        print("Dados carregados com sucesso.")
    except Exception as e:
        print(f"Erro ao carregar arquivos raster: {e}")

    radiance = radiance_mult * band10 + radiance_add

    K1 = 774.8853
    K2 = 1321.0789

    temperature_brightness = K2 / np.log((K1 / radiance) + 1)

    eNB = np.where((lai < 3) & (ndvi > 0), 0.97 + 0.0033 * lai, 
                   np.where((lai >= 3) & (ndvi > 0), 0.98, np.nan))
    eNBf = np.where(np.isnan(eNB), 0.99, eNB)

    eNBf_output_path = os.path.join(output_dir, 'eNBf.tif')

    eNBf_meta = out_meta.copy()
    eNBf_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': eNBf.dtype,
        'width': eNBf.shape[1],
        'height': eNBf.shape[0]
    })

    try:
        with rasterio.open(eNBf_output_path, 'w', **eNBf_meta) as dst:
            dst.write(eNBf, 1)
        print("eNBf salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    e0 = np.where((lai < 3) & (ndvi > 0), 0.95 + 0.01 * lai, 
                  np.where((lai >= 3) & (ndvi > 0), 0.98, np.nan))
    e0f = np.where(np.isnan(e0), 0.985, e0)

    Ts = temperature_brightness / (1 + ((10.8 * temperature_brightness / 14380) * np.log(eNBf)))

    Ts_output_path = os.path.join(output_dir, 'Ts.tif')

    Ts_meta = src_band10.meta.copy()
    Ts_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Ts.shape[1],
        'height': Ts.shape[0]
    })

    try:
        with rasterio.open(Ts_output_path, 'w', **Ts_meta) as dst:
            dst.write(Ts.astype(rasterio.float32), 1)
        print("Ts salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Média da Temperatura de Brilho:", np.mean(temperature_brightness))
    print("Média da Emissividade (Banda Estreita):", np.mean(eNBf))
    print("Média da Emissividade (Banda Larga):", np.mean(e0f))
    print("Média da Temperatura de Superfície:", np.mean(Ts))

    if os.path.isdir(caminho_mtl):
        MTLfile = [f for f in os.listdir(caminho_mtl) if f.endswith('MTL.txt')]
        if MTLfile:
            caminho_completo_mtl = os.path.join(caminho_mtl, MTLfile[0])
        else:
            raise FileNotFoundError("Arquivo MTL não encontrado no diretório especificado.")
    elif os.path.isfile(caminho_mtl) and caminho_mtl.endswith('MTL.txt'):
        caminho_completo_mtl = caminho_mtl
    else:
        raise FileNotFoundError("O caminho fornecido não é um arquivo MTL válido.")

    if caminho_completo_mtl:
        with open(caminho_completo_mtl, 'r') as file:
            radiance_maximum = {}
            reflectance_maximum = {}
            d = SUN_ELEVATION = None

            for line in file:
                if 'EARTH_SUN_DISTANCE' in line:
                    d = float(line.split('=')[-1].strip())
                elif 'SUN_ELEVATION' in line:
                    SUN_ELEVATION = float(line.split('=')[-1].strip())
                elif 'RADIANCE_MAXIMUM_BAND_' in line or 'REFLECTANCE_MAXIMUM_BAND_' in line:
                    key, value = line.split('=')
                    key = key.strip()
                    value = float(value.strip())
                    if 'RADIANCE' in key:
                        radiance_maximum[key] = value
                    else:
                        reflectance_maximum[key] = value
    else:
        print("Não foi possível encontrar um arquivo MTL válido para abrir.")

    ESUN = [(math.pi * d * d) * (radiance_maximum[f'RADIANCE_MAXIMUM_BAND_{i}'] / 
                                 reflectance_maximum[f'REFLECTANCE_MAXIMUM_BAND_{i}']) for i in range(1, 8)]

    print(ESUN)

    W = [value / sum(ESUN) for value in ESUN]
    print(W)

    W1, W2, W3, W4, W5, W6, W7 = W

    try:
        with rasterio.open(caminho_mdt) as src:
            MDT_Sebal = src.read(1)
            print("MDT carregado com sucesso.")
    except Exception as e:
        print(f"Erro ao carregar o MDT: {e}")
        exit()

    aTOA = band1 * W1 + band2 * W2 + band3 * W3 + band4 * W4 + band5 * W5 + band6 * W6 + band7 * W7

    aTOA_output_path = os.path.join(output_dir, 'aTOA.tif')

    aTOA_meta = out_meta.copy()
    aTOA_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': aTOA.dtype,
        'width': aTOA.shape[1],
        'height': aTOA.shape[0]
    })

    try:
        with rasterio.open(aTOA_output_path, 'w', **aTOA_meta) as dst:
            dst.write(aTOA, 1)
        print("aTOA salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    Tsw = 0.75 + 0.00002 * mdt_recortado

    Tsw_output_path = os.path.join(output_dir, 'Tsw.tif')

    Tsw_meta = out_meta.copy()
    Tsw_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': Tsw.dtype,
        'width': Tsw.shape[1],
        'height': Tsw.shape[0]
    })

    try:
        with rasterio.open(Tsw_output_path, 'w', **Tsw_meta) as dst:
            dst.write(Tsw, 1)
        print("Tsw salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    aS = (aTOA - 0.03) / (Tsw ** 2)

    aS_output_path = os.path.join(output_dir, 'aS.tif')

    aS_meta = out_meta.copy()
    aS_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': aS.dtype,
        'width': aS.shape[1],
        'height': aS.shape[0]
    })

    try:
        with rasterio.open(aS_output_path, 'w', **aS_meta) as dst:
            dst.write(aS, 1)
        print("aS salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    SUN_ELEVATION_rad = np.deg2rad(90 - SUN_ELEVATION)
    Rsi = 1367 * np.cos(SUN_ELEVATION_rad) * (1 / (d**2)) * Tsw

    Rsi_output_path = os.path.join(output_dir, 'Rsi.tif')

    Rsi_meta = out_meta.copy()
    Rsi_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Rsi.shape[1],
        'height': Rsi.shape[0]
    })

    try:
        with rasterio.open(Rsi_output_path, 'w', **Rsi_meta) as dst:
            dst.write(Rsi.astype('float32'), 1)
        print("Rsi salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    RLo = e0f * 5.67e-8 * (Ts ** 4)

    RLo_output_path = os.path.join(output_dir, 'RLo.tif')

    RLo_meta = out_meta.copy()
    RLo_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': RLo.shape[1],
        'height': RLo.shape[0]
    })

    try:
        with rasterio.open(RLo_output_path, 'w', **RLo_meta) as dst:
            dst.write(RLo.astype('float32'), 1)
        print("RLo salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    ndvi_path = os.path.join(output_dir, 'NDVI.tif')
    with rasterio.open(ndvi_path) as src:
        NDVI = src.read(1)
        ndvi_nodata = src.nodata

    ts_path = os.path.join(output_dir, 'Ts.tif')
    with rasterio.open(ts_path) as src:
        Ts = src.read(1)
        ts_nodata = src.nodata

    Ts_masked = np.ma.masked_equal(Ts, ts_nodata)
    Ts_median = np.nanmedian(Ts_masked)

    print("Mediana de Ts calculada ignorando o valor de no-data:", Ts_median)

    Pcold = np.where((NDVI > 0.4) & (Ts < Ts_median), Ts, np.nan)

    Pcold_output_path = os.path.join(output_dir, 'Pcold.tif')

    Pcold_meta = out_meta.copy()
    Pcold_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Pcold.shape[1],
        'height': Pcold.shape[0]
    })

    try:
        with rasterio.open(Pcold_output_path, 'w', **Pcold_meta) as dst:
            dst.write(np.float32(Pcold), 1)
        print("Pcold salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    ts_file_path = os.path.join(output_dir, 'Ts.tif')
    tsw_file_path = os.path.join(output_dir, 'Tsw.tif')

    print("Please enter the coordinates of the cold pixel (format: easting,northing):")
    xy_Pcold = input().strip().split(',')
    xy_Pcold = tuple(map(float, xy_Pcold))

    try:
        with rasterio.open(ts_file_path) as ts_src, rasterio.open(tsw_file_path) as tsw_src:
            row, col = ts_src.index(*xy_Pcold)
            z_TsPcold = ts_src.read(1)[row, col]
            Tsw_value = tsw_src.read(1)[row, col]

            print("Cold pixel temperature:", z_TsPcold, "K")
            print("Tsw value at cold pixel:", Tsw_value)

            RLi = 0.85 * ((-np.log(Tsw_value)) ** 0.09) * 5.67e-8 * z_TsPcold ** 4
            print("Calculating incoming longwave radiation (RLi) - W/m2... Done!")

            RLi_output_path = os.path.join(output_dir, 'RLi.tif')

            RLi_array = np.full(ts_src.shape, RLi, dtype='float32')

            RLi_meta = ts_src.meta.copy()
            RLi_meta.update({
                'dtype': 'float32',
                'count': 1
            })

            with rasterio.open(RLi_output_path, 'w', **RLi_meta) as dst:
                dst.write(RLi_array, 1)
            print("RLi salvo com sucesso.")

    except FileNotFoundError as e:
        print(f"Erro: {e}")
    except Exception as e:
        print(f"Ocorreu um erro inesperado: {e}")

    rsi_file_path = os.path.join(output_dir, 'Rsi.tif')
    as_file_path = os.path.join(output_dir, 'aS.tif')

    try:
        with rasterio.open(rsi_file_path) as src_rsi:
            Rsi = src_rsi.read(1)
        with rasterio.open(as_file_path) as src_as:
            aS = src_as.read(1)

        Rn = (1 - aS) * Rsi + RLi - RLo - (1 - e0f) * RLi

        Rn_output_path = os.path.join(output_dir, 'Rn.tif')

        Rn_meta = src_rsi.meta.copy()
        Rn_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': Rn.shape[1],
            'height': Rn.shape[0]
        })

        with rasterio.open(Rn_output_path, 'w', **Rn_meta) as dst:
            dst.write(Rn.astype('float32'), 1)
        print("Rn salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao processar ou salvar os rasters: {e}")

    G_Rn = np.where(NDVI < 0, 0.5, ((Ts - 273.15) / aS) * (0.0038 * aS + 0.0074 * aS ** 2) * (1 - 0.98 * NDVI ** 4))
    G = G_Rn * Rn

    G_output_path = os.path.join(output_dir, 'G.tif')

    G_meta = out_meta.copy()
    G_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': G.shape[1],
        'height': G.shape[0]
    })

    try:
        with rasterio.open(G_output_path, 'w', **G_meta) as dst:
            dst.write(G.astype('float32'), 1)
        print("G salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating soil heat flux (G) - W/m2... Done!")

    Phot = np.where((SAVI > 0.18) & (SAVI < 0.3), Ts, np.nan)

    print("Making the hot pixel mask... Done!")

    Phot_output_path = os.path.join(output_dir, 'Phot.tif')

    Phot_meta = out_meta.copy()
    Phot_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Phot.shape[1],
        'height': Phot.shape[0]
    })

    try:
        with rasterio.open(Phot_output_path, 'w', **Phot_meta) as dst:
            dst.write(np.float32(Phot), 1)
        print("Phot salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    ts_file_path = os.path.join(output_dir, 'Ts.tif')

    print("Choose the hot pixel coordinates in bare soil areas. Use the raster Phot and CC_432 for help...")
    xy_Phot = input('Place coordinates (east,north): ').strip('()')
    print(xy_Phot)

    try:
        with rasterio.open(ts_file_path) as src:
            row, col = src.index(*map(float, xy_Phot.split(',')))
            z_TsPhot = src.read(1)[row, col]
            print(f"Hot pixel temperature: {z_TsPhot} K")

            h = 0.15
            Zom = 0.123 * h
            u_ast = 0.41 * u_2m / (math.log(2 / Zom))
            u_200m = u_ast * (math.log(200 / Zom)) / 0.41
            print("Calculating friction velocity (u*) for weather station - m/s... Done!")
    except Exception as e:
        print(f"Erro ao processar ou salvar os rasters: {e}")    

    savi_file_path = os.path.join(output_dir, 'SAVI.tif')

    try:
        with rasterio.open(savi_file_path) as src_savi:
            SAVI = src_savi.read(1)

        Z0map = np.exp(-5.809 + 5.62 * SAVI)

        Z0map_output_path = os.path.join(output_dir, 'Z0map.tif')

        Z0map_meta = src_savi.meta.copy()
        Z0map_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': Z0map.shape[1],
            'height': Z0map.shape[0]
        })

        with rasterio.open(Z0map_output_path, 'w', **Z0map_meta) as dst:
            dst.write(Z0map.astype('float32'), 1)
        print("Z0map salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao processar ou salvar os rasters: {e}")
    
    u_astmap = 0.41 * u_200m / np.log(200 / Z0map)

    u_astmap_output_path = os.path.join(output_dir, 'u_astmap.tif')

    u_astmap_meta = out_meta.copy()
    u_astmap_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': u_astmap.shape[1],
        'height': u_astmap.shape[0]
    })

    try:
        with rasterio.open(u_astmap_output_path, 'w', **u_astmap_meta) as dst:
            dst.write(np.float32(u_astmap), 1)
        print("u_astmap salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating the friction velocity map (u*map) - m/s... Done!")

    rah = np.log(2 / 0.1) / (u_astmap * 0.41)
    print("Calculating aerodynamic resistance to heat transport map in terms of neutral stability (rah) - s/m... Done!")

    rah_output_path = os.path.join(output_dir, 'rah.tif')

    rah_meta = out_meta.copy()
    rah_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': rah.shape[1],
        'height': rah.shape[0]
    })

    try:
        with rasterio.open(rah_output_path, 'w', **rah_meta) as dst:
            dst.write(np.float32(rah), 1)
        print("rah salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    g_file_path = os.path.join(output_dir, 'G.tif')
    rah_file_path = os.path.join(output_dir, 'rah.tif')
    rn_file_path = os.path.join(output_dir, 'Rn.tif')

    try:
        with rasterio.open(g_file_path) as src:
            row, col = src.index(*map(float, xy_Phot.split(',')))

            if 0 <= row < src.height and 0 <= col < src.width:
                z_GPhot = src.read(1)[row, col]
                print(f"Hot pixel G value: {z_GPhot}")
            else:
                raise IndexError("Coordenadas fornecidas estão fora dos limites do raster G.tif")

        with rasterio.open(rah_file_path) as src:
            z_rahPhot = src.read(1)[row, col] if 0 <= row < src.height and 0 <= col < src.width else None

        with rasterio.open(rn_file_path) as src:
            z_RnPhot = src.read(1)[row, col] if 0 <= row < src.height and 0 <= col < src.width else None

        if z_rahPhot is not None and z_RnPhot is not None:
            print("Aerodynamic resistance correction, be patient...")
        else:
            print("Algumas coordenadas estão fora dos limites para rah.tif ou Rn.tif")
    except rasterio.errors.RasterioIOError as e:
        print(f"Erro ao abrir arquivo: {e}")
    except Exception as e:
        print(f"Erro durante o processamento: {e}")

    z_rahPhot_i = 0
    i = 0
    while abs(z_rahPhot_i - z_rahPhot) > 0.00001:
        print("Iteration number:", i)
        i += 1
        z_rahPhot_i = z_rahPhot
        print("Equation coefficients estimate >> dT = a * Ts + b")
        a = (z_RnPhot - z_GPhot) * z_rahPhot_i / ((z_TsPhot - z_TsPcold) * 1.25 * 1004)
        b = -a * z_TsPcold
        print('a:', a, 'b:', b)
        print("Calculating dT map - K...")

    dT = a * Ts + b

    dT_output_path = os.path.join(output_dir, 'dT.tif')

    dT_meta = out_meta.copy()
    dT_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': dT.shape[1],
        'height': dT.shape[0]
    })

    try:
        with rasterio.open(dT_output_path, 'w', **dT_meta) as dst:
            dst.write(np.float32(dT), 1)
        print("dT salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}") 

    dt_file_path = os.path.join(output_dir, 'dT.tif')
    rah_file_path = os.path.join(output_dir, 'rah.tif')

    try:
        with rasterio.open(dt_file_path) as src_dt:
            dT = src_dt.read(1)
        with rasterio.open(rah_file_path) as src_rah:
            rah = src_rah.read(1)

        H = (dT / rah) * 1.25 * 1004

        H_output_path = os.path.join(output_dir, 'H.tif')

        H_meta = out_meta.copy()
        H_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': H.shape[1],
        'height': H.shape[0]
        })

        with rasterio.open(H_output_path, 'w', **H_meta) as dst:
            dst.write(np.float32(H), 1)
        print("H salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating sensible heat flux (H) - W/m2... Done!")

    g = 9.81
    L = -(1.25 * 1004 * Ts * u_astmap ** 3) / (0.41 * g * H)

    L_output_path = os.path.join(output_dir, 'L.tif')

    L_meta = out_meta.copy()
    L_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': L.shape[1],
        'height': L.shape[0]
    })

    try:
        with rasterio.open(L_output_path, 'w', **L_meta) as dst:
            dst.write(np.float32(L), 1)
        print("L salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}") 
    print("Calculating the Monin-Obukhov length map (L) - m... Done!")

    L200m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (200 / L))) / 2), np.where(L > 0, -5 * (200 / L), 0))
    L2m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (2 / L))) / 2), np.where(L > 0, -5 * (2 / L), 0))
    L01m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (0.1 / L))) / 2), np.where(L > 0, -5 * (0.1 / L), 0))
    L200m_output_path = os.path.join(output_dir, 'L200m.tif')
    L2m_output_path = os.path.join(output_dir, 'L2m.tif')
    L01m_output_path = os.path.join(output_dir, 'L01m.tif')

    L200m_meta = out_meta.copy()
    L200m_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': L200m.shape[1],
        'height': L200m.shape[0]
    })

    try:
        with rasterio.open(L200m_output_path, 'w', **L200m_meta) as dst:
            dst.write(np.float32(L200m), 1)
        print("L200m salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}") 

    L2m_meta = out_meta.copy()
    L2m_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': L2m.shape[1],
        'height': L2m.shape[0]
    })

    try:
        with rasterio.open(L2m_output_path, 'w', **L2m_meta) as dst:
            dst.write(np.float32(L2m), 1)
        print("L2m salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}") 

    L01m_meta = out_meta.copy()
    L01m_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': L01m.shape[1],
        'height': L01m.shape[0]
    })

    try:
        with rasterio.open(L01m_output_path, 'w', **L01m_meta) as dst:
            dst.write(np.float32(L01m), 1)
        print("L01m salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating atmospheric stability correction (L200m, L2m, L01m)... Done!")

    u_astmap = 0.41 * u_200m / (np.log(200 / Z0map) - L200m)

    u_astmap_output_path = os.path.join(output_dir, 'u_astmap.tif')

    u_astmap_meta = out_meta.copy()
    u_astmap_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': u_astmap.shape[1],
        'height': u_astmap.shape[0]
    })

    try:
        with rasterio.open(u_astmap_output_path, 'w', **u_astmap_meta) as dst:
            dst.write(np.float32(u_astmap), 1)
        print("u_astmap salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating corrected friction velocity map (u_astmap) - m/s... Done!")

    rah = (np.log(2 / 0.1) - L2m + L01m) / (u_astmap * 0.41)

    rah_output_path = os.path.join(output_dir, 'rah_corrected.tif')

    rah_meta = out_meta.copy()
    rah_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': rah.shape[1],
        'height': rah.shape[0]
    })

    try:
        with rasterio.open(rah_output_path, 'w', **rah_meta) as dst:
            dst.write(np.float32(rah), 1)
        print("rah salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

    print("Calculating corrected aerodynamic resistance to heat transport (rah) - s/m... Done!")

    rah_file_path = os.path.join(output_dir, 'rah_corrected.tif')
    dt_file_path = os.path.join(output_dir, 'dT.tif')

    try:
        with rasterio.open(rah_file_path) as src_rah, rasterio.open(dt_file_path) as src_dt:
            rah = src_rah.read(1)
            dT = src_dt.read(1)

        H_corrected = (dT / rah) * 1.25 * 1004

        H_corrected_output_path = os.path.join(output_dir, 'H_corrected.tif')

        H_corrected_meta = out_meta.copy()
        H_corrected_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': H_corrected.shape[1],
            'height': H_corrected.shape[0]
        })

        try:
            with rasterio.open(H_corrected_output_path, 'w', **H_corrected_meta) as dst:
                dst.write(np.float32(H_corrected), 1)
            print("H_corrected salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

        print("Calculating corrected sensible heat flux (H_corrected) - W/m2... Done!")

        LET_corrected = Rn - G - H_corrected

        LET_corrected_output_path = os.path.join(output_dir, 'LET_corrected.tif')

        LET_corrected_meta = out_meta.copy()
        LET_corrected_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': LET_corrected.shape[1],
            'height': LET_corrected.shape[0]
        })

        try:
            with rasterio.open(LET_corrected_output_path, 'w', **LET_corrected_meta) as dst:
                dst.write(np.float32(LET_corrected), 1)
            print("LET_corrected salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

        print("Calculating corrected latent heat flux (LET_corrected) - W/m2... Done!")

        ETi_corrected = np.where(3600 * (LET_corrected / (2.45 * 10**6)) < 0, 0, 3600 * (LET_corrected / (2.45 * 10**6)))

        ETi_corrected_output_path = os.path.join(output_dir, 'ETi_corrected.tif')

        ETi_corrected_meta = out_meta.copy()
        ETi_corrected_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': ETi_corrected.shape[1],
            'height': ETi_corrected.shape[0]
        })

        try:
            with rasterio.open(ETi_corrected_output_path, 'w', **ETi_corrected_meta) as dst:
                dst.write(np.float32(ETi_corrected), 1)
            print("ETi_corrected salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

        print("Calculating corrected instantaneous evapotranspiration (ETi_corrected) - mm/h... Done!")

        ETof_corrected = ETi_corrected / EToi

        ETof_corrected_output_path = os.path.join(output_dir, 'ETof_corrected.tif')

        ETof_corrected_meta = out_meta.copy()
        ETof_corrected_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': ETof_corrected.shape[1],
            'height': ETof_corrected.shape[0]
        })

        try:
            with rasterio.open(ETof_corrected_output_path, 'w', **ETof_corrected_meta) as dst:
                dst.write(np.float32(ETof_corrected), 1)
            print("ETof_corrected salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

        print("Calculating corrected reference evapotranspiration fraction (ETof_corrected)... Done!")

        ETday_corrected = ETof_corrected * ETo

        ETday_corrected_output_path = os.path.join(output_dir, 'ETday_corrected.tif')

        ETday_corrected_meta = out_meta.copy()
        ETday_corrected_meta.update({
            'driver': 'GTiff',
            'count': 1,
            'dtype': 'float32',
            'width': ETday_corrected.shape[1],
            'height': ETday_corrected.shape[0]
        })

        try:
            with rasterio.open(ETday_corrected_output_path, 'w', **ETday_corrected_meta) as dst:
                dst.write(np.float32(ETday_corrected), 1)
            print("ETday_corrected salvo com sucesso.")
        except Exception as e:
            print(f"Erro ao escrever o arquivo TIFF: {e}")

        print("Calculating corrected daily evapotranspiration (ETday_corrected) - mm/day... Done!")

    except Exception as e:
        print(f"Erro ao processar ou salvar os arquivos: {e}")

if __name__ == "__main__":
    main()

