import os
import sys
import re
import numpy as np
import rasterio
from rasterio.mask import mask
from rasterio.warp import reproject, Resampling
import geopandas as gpd
from PyQt5 import uic, QtWidgets

# Carrega a interface do arquivo .ui
FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'EvapoGIS.ui'))

class EvapoGISDialog(QtWidgets.QWidget, FORM_CLASS):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setupUi(self)
        self.runButton.clicked.connect(self.run_algorithm)


    def run_algorithm(self):
        caminho_mtl = self.caminhoMTL.text()
        caminho_mdt = self.caminhoMDT.text()
        caminho_bandas = self.caminhoBandas.text()
        shapefile_path = self.shapefilePath.text()
        output_dir = self.outputDir.text()
        raster_referencia_path = self.rasterReferencia.text()
        epsg_code = self.epsgCode.text()
        u_2m = float(self.u2m.text())
        EToi = float(self.EToi.text())
        ETo = float(self.ETo.text())

        # Aqui você pode chamar suas funções com os parâmetros coletados
        mtl_data = read_mtl(caminho_mtl)
        if mtl_data:
            mdt_output_path = os.path.join(output_dir, 'MDT_Sebal_recorte.tif')
            mdt_recortado, mdt_meta = recortar_e_aliar_mdt(caminho_mdt, shapefile_path, mdt_output_path, raster_referencia_path)
            if mdt_recortado is not None:
                print("Processamento do MDT concluído com sucesso.")
                bandas, meta_data, out_meta = process_images(caminho_bandas, shapefile_path, output_dir, mtl_data)
                if not bandas:
                    print("Nenhuma banda processada.")
            else:
                print("Falha ao processar MDT.")
        else:
            print("Falha ao carregar dados MTL, terminando o programa.")

def read_mtl(caminho_mtl):
    mtl_data = {}
    try:
        with open(caminho_mtl, 'r') as file:
            for line in file:
                parts = line.strip().split('=')
                if len(parts) == 2:
                    key, value = parts
                    mtl_data[key.strip()] = value.strip().strip('"')
    except IOError:
        print("Erro ao ler o arquivo MTL. Verifique o caminho fornecido.")
        sys.exit(1)
    return mtl_data

def recortar_e_aliar_mdt(caminho_mdt, shapefile_path, output_path, caminho_raster_referencia):
    try:
        with rasterio.open(caminho_raster_referencia) as ref_raster:
            ref_transform = ref_raster.transform
            ref_crs = ref_raster.crs
            ref_width = ref_raster.width
            ref_height = ref_raster.height
        
        with rasterio.open(caminho_mdt) as src:
            shapefile = gpd.read_file(shapefile_path)
            if shapefile.empty:
                print("Erro: Shapefile vazio ou não intersecta o MDT.")
                return None, None
            geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
            out_image, out_transform = mask(src, geometrias, crop=True)
            if out_image.size == 0:
                print("Erro: A máscara resultou em uma imagem vazia.")
                return None, None
            out_meta = src.meta.copy()
            out_meta.update({
                "driver": "GTiff",
                "height": ref_height,
                "width": ref_width,
                "transform": ref_transform,
                "crs": ref_crs,
                "nodata": src.nodata
            })
            reamostrado_image = np.zeros((ref_height, ref_width), dtype=src.dtypes[0])
            reproject(
                source=out_image,
                destination=reamostrado_image,
                src_transform=out_transform,
                src_crs=src.crs,
                dst_transform=ref_transform,
                dst_crs=ref_crs,
                resampling=Resampling.nearest
            )
            with rasterio.open(output_path, 'w', **out_meta) as dst:
                dst.write(reamostrado_image, 1)
            return reamostrado_image, out_meta
    except Exception as e:
        print(f"Erro ao recortar e alinhar MDT: {e}")
        return None, None

def process_images(caminho_bandas, shapefile_path, output_dir, mtl_data):
    bandas = {}
    meta_data = None
    if not os.path.exists(caminho_bandas):
        print("Diretório das bandas não encontrado.")
        return None

    arquivos = [os.path.join(caminho_bandas, arquivo) for arquivo in os.listdir(caminho_bandas) if arquivo.endswith(('.TIF', '.tif'))]
    out_meta = None  # Inicialização de out_meta

    for arquivo in arquivos:
        nome_banda = os.path.basename(arquivo)
        match = re.search(r'_B(\d+)', nome_banda)
        if match:
            band_number = int(match.group(1))
        else:
            print(f"Nenhum número de banda encontrado em {nome_banda}. Pulando este arquivo.")
            continue

        nome_saida = f'band{band_number}.tif'
        try:
            with rasterio.open(arquivo) as src:
                shapefile = gpd.read_file(shapefile_path)
                geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
                out_image, out_transform = mask(src, geometrias, crop=True)
                out_meta = src.meta.copy()
                out_meta.update({
                    "driver": "GTiff",
                    "height": out_image.shape[1],
                    "width": out_image.shape[2],
                    "transform": out_transform
                })
                
                if out_image.ndim == 4:
                    out_image = out_image.reshape((1, *out_image.shape[-2:]))

                if band_number == 10:
                    processed_data = out_image[0]  # Para a banda 10, use a imagem recortada diretamente
                else:
                    toa_reflectance = out_image[0] * float(mtl_data[f'REFLECTANCE_MULT_BAND_{band_number}']) + float(mtl_data[f'REFLECTANCE_ADD_BAND_{band_number}'])
                    processed_data = toa_reflectance
                
                output_path = os.path.join(output_dir, nome_saida)
                with rasterio.open(output_path, 'w', **out_meta) as dst:
                    dst.write(processed_data, 1)
                bandas[nome_saida.replace('.tif', '')] = processed_data

        except Exception as e:
            print(f"Erro ao processar a banda {band_number}: {e}. Pulando.")
            continue

    return bandas, meta_data, out_meta
