# -*- coding: utf-8 -*-
import re
import numpy as np
import math  
import os  
import rasterio
from rasterio.enums import Resampling
from rasterio.warp import calculate_default_transform, reproject
from rasterio.mask import mask
import geopandas as gpd
import tkinter as tk
from tkinter import filedialog, simpledialog
import numpy as np
import os
import rasterio
import geopandas as gpd
import sys
import logging

# Configuração básica do logging
logging.basicConfig(level=logging.DEBUG, filename='app.log', filemode='w', 
                    format='%(asctime)s - %(name)s - %(levellevel)s - %(message)s')


def select_file(entry_field, filetypes):
    filepath = filedialog.askopenfilename(filetypes=filetypes)
    entry_field.delete(0, tk.END)
    entry_field.insert(0, filepath)

def select_directory(entry_field):
    directory_path = filedialog.askdirectory()
    entry_field.delete(0, tk.END)
    entry_field.insert(0, directory_path)

def get_user_input_gui():
    root = tk.Tk()
    root.title("Configuração de Entrada")

    # Criação de labels e campos de entrada
    labels = ["Caminho do arquivo MTL:", 
              "Caminho do arquivo MDT:", 
              "Diretório das bandas raster (imagens TIF):", 
              "Caminho do shapefile para o recorte:", 
              "Diretório de saída para salvar os arquivos processados:", 
              "Caminho do raster de referência (imagem Landsat):", 
              "Código EPSG:", 
              "Velocidade do vento a 2m (m/s):", 
              "ETo instantâneo (mm):", 
              "ETo diário (mm):"]

    entry_fields = {}

    for idx, label_text in enumerate(labels):
        label = tk.Label(root, text=label_text)
        label.grid(row=idx, column=0, padx=10, pady=5, sticky=tk.W)

        entry_field = tk.Entry(root, width=50)
        entry_field.grid(row=idx, column=1, padx=10, pady=5)

        # Guardar os campos de entrada para uso posterior
        entry_fields[label_text] = entry_field

        # Botões para navegar e selecionar arquivos/diretórios
        if idx in [0, 1, 3, 5]:  # Arquivos
            browse_button = tk.Button(root, text="Navegar", command=lambda e=entry_field: select_file(e, [("Todos os Arquivos", "*.*")]))
            browse_button.grid(row=idx, column=2, padx=5, pady=5)
        elif idx == 2 or idx == 4:  # Diretórios
            browse_button = tk.Button(root, text="Navegar", command=lambda e=entry_field: select_directory(e))
            browse_button.grid(row=idx, column=2, padx=5, pady=5)

    def submit():
        # Recuperar todos os valores inseridos nos campos
        caminho_mtl = entry_fields["Caminho do arquivo MTL:"].get()
        caminho_mdt = entry_fields["Caminho do arquivo MDT:"].get()
        caminho_bandas = entry_fields["Diretório das bandas raster (imagens TIF):"].get()
        shapefile_path = entry_fields["Caminho do shapefile para o recorte:"].get()
        output_dir = entry_fields["Diretório de saída para salvar os arquivos processados:"].get()
        raster_referencia_path = entry_fields["Caminho do raster de referência (imagem Landsat):"].get()
        epsg_code = entry_fields["Código EPSG:"].get()
        u_2m = float(entry_fields["Velocidade do vento a 2m (m/s):"].get())
        EToi = float(entry_fields["ETo instantâneo (mm):"].get())
        ETo = float(entry_fields["ETo diário (mm):"].get())

        # Armazenar os valores em variáveis globais para que possam ser acessadas fora da função
        global user_input_values
        caminho_mtl, caminho_mdt, caminho_bandas, shapefile_path, output_dir, raster_referencia_path, epsg_code, u_2m, EToi, ETo = get_user_input_gui()


        root.quit()

    # Botão de submissão
    submit_button = tk.Button(root, text="Submeter", command=submit)
    submit_button.grid(row=len(labels), columnspan=3, pady=20)

    root.mainloop()

    return user_input_values

def clear_screen():
    # Clear console in a cross-platform way
    command = 'cls' if os.name == 'nt' else 'clear'
    os.system(command)

# Substituímos a função de entrada por uma interface gráfica
if __name__ == "__main__":
    clear_screen()
    caminho_mtl, caminho_mdt, caminho_bandas, shapefile_path, output_dir, raster_referencia_path, epsg_code, u_2m, EToi, ETo = get_user_input_gui()
    # Resto do código permanece o mesmo



def read_mtl(caminho_mtl):
    mtl_data = {}
    try:
        with open(caminho_mtl, 'r') as file:
            for line in file:
                parts = line.strip().split('=')
                if len(parts) == 2:
                    key, value = parts
                    mtl_data[key.strip()] = value.strip().strip('"')
    except IOError:
        print("Erro ao ler o arquivo MTL. Verifique o caminho fornecido.")
        sys.exit(1)
    return mtl_data

def recortar_e_aliar_mdt(caminho_mdt, shapefile_path, output_path, caminho_raster_referencia):
    try:
        with rasterio.open(caminho_raster_referencia) as ref_raster:
            ref_transform = ref_raster.transform
            ref_crs = ref_raster.crs
            ref_width = ref_raster.width
            ref_height = ref_raster.height
        
        with rasterio.open(caminho_mdt) as src:
            shapefile = gpd.read_file(shapefile_path)
            if shapefile.empty:
                print("Erro: Shapefile vazio ou não intersecta o MDT.")
                return None, None
            geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
            out_image, out_transform = mask(src, geometrias, crop=True)
            if out_image.size == 0:
                print("Erro: A máscara resultou em uma imagem vazia.")
                return None, None
            out_meta = src.meta.copy()
            out_meta.update({
                "driver": "GTiff",
                "height": ref_height,
                "width": ref_width,
                "transform": ref_transform,
                "crs": ref_crs,
                "nodata": src.nodata
            })
            reamostrado_image = np.zeros((ref_height, ref_width), dtype=src.dtypes[0])
            reproject(
                source=out_image,
                destination=reamostrado_image,
                src_transform=out_transform,
                src_crs=src.crs,
                dst_transform=ref_transform,
                dst_crs=ref_crs,
                resampling=Resampling.nearest
            )
            with rasterio.open(output_path, 'w', **out_meta) as dst:
                dst.write(reamostrado_image, 1)
            return reamostrado_image, out_meta
    except Exception as e:
        print(f"Erro ao recortar e alinhar MDT: {e}")
        return None, None

def calculate_toa(band_array, band_number, mtl_data):
    try:
        mult_factor = float(mtl_data[f'REFLECTANCE_MULT_BAND_{band_number}'])
        add_factor = float(mtl_data[f'REFLECTANCE_ADD_BAND_{band_number}'])
        toa_reflectance = band_array * mult_factor + add_factor
        return toa_reflectance
    except KeyError:
        print("Erro ao calcular TOA. Verifique os metadados e o número da banda.")
        exit()


def recortar_com_shapefile(banda_path, shapefile_path):
    try:
        with rasterio.open(banda_path) as src:
            shapefile = gpd.read_file(shapefile_path)
            geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
            
            # Cálculo do bounding box em coordenadas de pixel, arredondando para garantir a inclusão completa
            bounds = shapefile.total_bounds
            pixel_min_x, pixel_min_y = ~src.transform * (bounds[0], bounds[3])
            pixel_max_x, pixel_max_y = ~src.transform * (bounds[2], bounds[1])
            pixel_min_x, pixel_min_y = np.floor(pixel_min_x), np.floor(pixel_min_y)
            pixel_max_x, pixel_max_y = np.ceil(pixel_max_x), np.ceil(pixel_max_y)
            expected_height = int(pixel_max_y - pixel_min_y)
            expected_width = int(pixel_max_x - pixel_min_x)
            
            # Recorte sem padding, apenas considerando a máscara
            out_image, out_transform = mask(src, geometrias, crop=True, pad=True)
            out_meta = src.meta.copy()
            
            # Verifica e ajusta a imagem recortada para que corresponda ao bounding box completo
            if out_image.shape[1] < expected_height or out_image.shape[2] < expected_width:
                padded_image = np.zeros((out_image.shape[0], expected_height, expected_width), dtype=out_image.dtype)
                padded_image[:, :out_image.shape[1], :out_image.shape[2]] = out_image
                out_image = padded_image

            # Atualização dos metadados para refletir as novas dimensões
            out_meta.update({
                "driver": "GTiff",
                "height": expected_height,
                "width": expected_width,
                "transform": rasterio.transform.from_bounds(bounds[0], bounds[3], bounds[2], bounds[1], expected_width, expected_height)
            })
            
            return out_image, out_meta

    except Exception as e:
        print(f"Erro ao recortar a imagem: {e}")

def process_images(caminho_bandas, shapefile_path, output_dir, mtl_data):
    bandas = {}
    meta_data = None
    if not os.path.exists(caminho_bandas):
        print("Diretório das bandas não encontrado.")
        return None

    arquivos = [os.path.join(caminho_bandas, arquivo) for arquivo in os.listdir(caminho_bandas) if arquivo.endswith(('.TIF', '.tif'))]
    out_meta = None  # Inicialização de out_meta

    for arquivo in arquivos:
        nome_banda = os.path.basename(arquivo)
        match = re.search(r'_B(\d+)', nome_banda)
        if match:
            band_number = int(match.group(1))
        else:
            print(f"Nenhum número de banda encontrado em {nome_banda}. Pulando este arquivo.")
            continue

        nome_saida = f'band{band_number}.tif'
        try:
            with rasterio.open(arquivo) as src:
                shapefile = gpd.read_file(shapefile_path)
                geometrias = [feature["geometry"] for feature in shapefile.geometry.__geo_interface__['features']]
                out_image, out_transform = mask(src, geometrias, crop=True)
                out_meta = src.meta.copy()
                out_meta.update({
                    "driver": "GTiff",
                    "height": out_image.shape[1],
                    "width": out_image.shape[2],
                    "transform": out_transform
                })
                
                if out_image.ndim == 4:
                    out_image = out_image.reshape((1, *out_image.shape[-2:]))

                if band_number == 10:
                    processed_data = out_image[0]  # Para a banda 10, use a imagem recortada diretamente
                else:
                    toa_reflectance = out_image[0] * float(mtl_data[f'REFLECTANCE_MULT_BAND_{band_number}']) + float(mtl_data[f'REFLECTANCE_ADD_BAND_{band_number}'])
                    processed_data = toa_reflectance
                
                output_path = os.path.join(output_dir, nome_saida)
                with rasterio.open(output_path, 'w', **out_meta) as dst:
                    dst.write(processed_data, 1)
                bandas[nome_saida.replace('.tif', '')] = processed_data

        except Exception as e:
            print(f"Erro ao processar a banda {band_number}: {e}. Pulando.")
            continue

    return bandas, meta_data, out_meta

if __name__ == "__main__":
    clear_screen()
    caminho_mtl, caminho_mdt, caminho_bandas, shapefile_path, output_dir, raster_referencia_path, epsg_code, u_2m, EToi, ETo = get_user_input()
    mtl_data = read_mtl(caminho_mtl)
    
    bandas = {}  # Inicialize bandas como dicionário vazio para evitar NameError

    if mtl_data:
        mdt_output_path = os.path.join(output_dir, 'MDT_Sebal_recorte.tif')
        mdt_recortado, mdt_meta = recortar_e_aliar_mdt(caminho_mdt, shapefile_path, mdt_output_path, raster_referencia_path)
        
        if mdt_recortado is not None:
            print("Processamento do MDT concluído com sucesso.")
            bandas, meta_data, out_meta = process_images(caminho_bandas, shapefile_path, output_dir, mtl_data)
            if not bandas:
                print("Nenhuma banda processada.")
        else:
            print("Falha ao processar MDT.")
    else:
        print("Falha ao carregar dados MTL, terminando o programa.")

    # Acessando bandas de forma segura
    if bandas:
        for banda, dados in bandas.items():
            print(f"Tipo de dados para {banda}: {type(dados)}")
        
        #bandas = {'band4': <dados da banda vermelha>, 'band3': <dados da banda verde>, 'band2': <dados da banda azul>}
        
        # Seguramente acessando bandas específicas usando get para evitar KeyError
        nir_band = bandas.get('band5')  # Banda Infravermelho próximo
        red_band = bandas.get('band4')  # Banda vermelha
        green_band = bandas.get('band3') # Banda verde
        blue_band = bandas.get('band2')  # Banda azul
        band1 = bandas.get ('band1')
        band2 = bandas.get ('band2')
        band3 = bandas.get ('band3')
        band4 = bandas.get ('band4')
        band5 = bandas.get ('band5')
        band6 = bandas.get ('band6')
        band7 = bandas.get ('band7')
        band8 = bandas.get ('band8')
        band9 = bandas.get ('band9')
        band10 = bandas.get ('band10')
                
        # Verificar se as bandas foram carregadas
        if nir_band is None or red_band is None or green_band is None or blue_band is None:
            print("Algumas bandas necessárias estão faltando.")
        else:
            # Processamento adicional se todas as bandas necessárias estiverem presentes
            print("Todas as bandas necessárias foram carregadas com sucesso.")
    else:
        print("Nenhuma banda disponível para processamento.")


# Verifica se as bandas são arrays do NumPy e cria um composto RGB
if isinstance(red_band, np.ndarray) and isinstance(green_band, np.ndarray) and isinstance(blue_band, np.ndarray):
    rgb_composite = np.stack((red_band, green_band, blue_band), axis=0)

    # Construir o caminho completo para o arquivo de saída
    rgb_output_path = os.path.join(output_dir, 'CC_432.tif')

    # Assumindo que out_meta contém as informações de CRS atualizadas do último arquivo recortado
    # Aqui, não precisamos definir o CRS novamente, pois ele será herdado de out_meta
    rgb_meta = out_meta.copy()  # Faz uma cópia do out_meta para preservar os metadados originais
    rgb_meta.update({
        'driver': 'GTiff',
        'count': 3,  # Atualizando a contagem de bandas para 3 (RGB)
        'dtype': rgb_composite.dtype,  # Atualiza o dtype com base no composto RGB
        'width': rgb_composite.shape[2],
        'height': rgb_composite.shape[1]
    })

    # Salvar o composto RGB usando rasterio
    try:
        with rasterio.open(rgb_output_path, 'w', **rgb_meta) as dst:
            for i, band in enumerate(rgb_composite, start=1):
                dst.write(band, i)
        print("Composite RGB Landsat 8, be patient... Done!")
    except Exception as e:  # Captura de exceções de forma mais genérica
        print(f"Erro ao escrever o arquivo TIFF: {e}")
else:
    print("Erro: Uma ou mais bandas não são arrays do NumPy.")

print("Done!")


# Cálculo do NDVI
np.seterr(divide='ignore', invalid='ignore')  # Ignorar avisos de divisão por zero

if nir_band is not None and red_band is not None:
    NDVI = (nir_band - red_band) / (nir_band + red_band + 1e-10)
    NDVI = np.clip(NDVI, -1, 1)  # Garantir que os valores estejam entre -1 e 1
else:
    print("Erro: Bandas NIR ou Red estão ausentes ou não foram carregadas corretamente.")
    NDVI = None  # Define um valor padrão ou uma ação específica aqui

# Salvar o NDVI
if NDVI is not None:
    ndvi_output_path = os.path.join(output_dir, 'NDVI.tif')
    ndvi_meta = out_meta.copy()
    ndvi_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': NDVI.shape[1],
        'height': NDVI.shape[0]
    })

    try:
        with rasterio.open(ndvi_output_path, 'w', **ndvi_meta) as dst:
            dst.write(NDVI, 1)
        print("NDVI salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")

# Cálculo do SAVI
if nir_band is not None and red_band is not None:
    Lsavi = 0.5
    SAVI = ((nir_band - red_band) / (nir_band + red_band + Lsavi)) * (1 + Lsavi)
    SAVI = np.clip(SAVI, -1, 1)

    # Salvar o SAVI
    savi_output_path = os.path.join(output_dir, 'SAVI.tif')
    savi_meta = out_meta.copy()
    savi_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': SAVI.shape[1],
        'height': SAVI.shape[0]
    })

    try:
        with rasterio.open(savi_output_path, 'w', **savi_meta) as dst:
            dst.write(SAVI, 1)
        print("SAVI salvo com sucesso.")
    except Exception as e:
        print(f"Erro ao escrever o arquivo TIFF: {e}")
    
# Cálculo do LAI
    
def calculate_lai(savi):
    lai = np.zeros_like(savi, dtype=float)
    lai[savi < 0.1] = 0.00001
    mask = (savi >= 0.1) & (savi < 0.687)
    lai[mask] = -np.log((0.69 - savi[mask]) / 0.59) / 0.91
    lai[savi >= 0.687] = 6
    return lai

# Cálculo do LAI
LAI = calculate_lai(SAVI)

# Mostrar o resultado
print(LAI)

# Salvar o LAI
lai_output_path = os.path.join(output_dir, 'LAI.tif')

# Preparar os metadados para salvar o LAI, utilizando os metadados da imagem recortada
lai_meta = out_meta.copy()
lai_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': LAI.shape[1],
    'height': LAI.shape[0]
})

# Salvar o LAI usando rasterio
try:
    with rasterio.open(lai_output_path, 'w', **lai_meta) as dst:
        dst.write(LAI, 1)
    print("LAI salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")
    
def read_mtl(caminho_mtl):
    """ Read the MTL file and extract radiance multipliers and addends for band 10. """
    values = {}
    with open(caminho_mtl, 'r') as file:
        for line in file:
            if "RADIANCE_MULT_BAND_10" in line or "RADIANCE_ADD_BAND_10" in line:
                key, value = line.strip().split(' = ')
                values[key.strip()] = float(value)
    return values

# Caminho para o arquivo MTL
mtl_values = read_mtl(caminho_mtl)  # chamando a função com a variável correta

# Ler valores do MTL
mtl_values = read_mtl(caminho_mtl)
radiance_mult = mtl_values['RADIANCE_MULT_BAND_10']
radiance_add = mtl_values['RADIANCE_ADD_BAND_10']

# Definindo os caminhos dos arquivos
lai_path = os.path.join(output_dir, 'LAI.tif')
ndvi_path = os.path.join(output_dir, 'NDVI.tif')
band10_path = os.path.join(output_dir, 'band10.tif')

try:
    with rasterio.open(lai_path) as src_lai, rasterio.open(ndvi_path) as src_ndvi, rasterio.open(band10_path) as src_band10:
        # Ler a primeira banda de cada arquivo raster
        lai = src_lai.read(1)
        ndvi = src_ndvi.read(1)
        band10 = src_band10.read(1)
        
    print("Dados carregados com sucesso.")

except Exception as e:
    print(f"Erro ao carregar arquivos raster: {e}")

# Convertendo DN para Radiância
radiance = radiance_mult * band10 + radiance_add

# Constantes do sensor (Landsat 8, banda 10)
K1 = 774.8853
K2 = 1321.0789

# Calcula a temperatura de brilho usando a equação de Planck inversa
temperature_brightness = K2 / np.log((K1 / radiance) + 1)

# Cálculo da Emissividade de Banda Estreita (eNB)
eNB = np.where((lai < 3) & (ndvi > 0), 0.97 + 0.0033 * lai, 
               np.where((lai >= 3) & (ndvi > 0), 0.98, np.nan))
eNBf = np.where(np.isnan(eNB), 0.99, eNB)

eNBf_output_path = os.path.join(output_dir, 'eNBf.tif')

#Preparar os metadados para salvar o eNBf, utilizando os metadados da imagem recortada
eNBf_meta = out_meta.copy()  # Utilizar out_meta que contém o CRS correto e outros metadados necessários
eNBf_meta.update({
    'driver': 'GTiff',
    'count': 1,  # eNBf é uma única banda
    'dtype': eNBf.dtype,  # Ou 'float32' se você quiser forçar o tipo, conforme seu dado
    'width': eNBf.shape[1],  # Largura do eNBf
    'height': eNBf.shape[0]  # Altura do eNBf
})

# Salvar o eNBf usando rasterio
try:
    with rasterio.open(eNBf_output_path, 'w', **eNBf_meta) as dst:
        dst.write(eNBf, 1)  # Escreve a matriz eNBf no arquivo
    print("eNBf salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Cálculo da Emissividade de Banda Larga (e0f)
e0 = np.where((lai < 3) & (ndvi > 0), 0.95 + 0.01 * lai, 
              np.where((lai >= 3) & (ndvi > 0), 0.98, np.nan))
e0f = np.where(np.isnan(e0), 0.985, e0)

# Cálculo da Temperatura de Superfície (Ts)
Ts = temperature_brightness / (1 + ((10.8 * temperature_brightness / 14380) * np.log(eNBf)))

# Definir o caminho de saída para Ts
Ts_output_path = os.path.join(output_dir, 'Ts.tif')

# Preparar os metadados para salvar o Ts
Ts_meta = src_band10.meta.copy()  # Utilizar metadados do arquivo de banda 10
Ts_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': Ts.shape[1],
    'height': Ts.shape[0]
})

# Salvar o Ts usando rasterio
try:
    with rasterio.open(Ts_output_path, 'w', **Ts_meta) as dst:
        dst.write(Ts.astype(rasterio.float32), 1)
    print("Ts salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Impressão de diagnósticos
print("Média da Temperatura de Brilho:", np.mean(temperature_brightness))
print("Média da Emissividade (Banda Estreita):", np.mean(eNBf))
print("Média da Emissividade (Banda Larga):", np.mean(e0f))
print("Média da Temperatura de Superfície:", np.mean(Ts))

if os.path.isdir(caminho_mtl):
    MTLfile = [f for f in os.listdir(caminho_mtl) if f.endswith('MTL.txt')]
    if MTLfile:
        caminho_completo_mtl = os.path.join(caminho_mtl, MTLfile[0])
    else:
        raise FileNotFoundError("Arquivo MTL não encontrado no diretório especificado.")
elif os.path.isfile(caminho_mtl) and caminho_mtl.endswith('MTL.txt'):
    caminho_completo_mtl = caminho_mtl
else:
    raise FileNotFoundError("O caminho fornecido não é um arquivo MTL válido.")

# Verifica se o caminho completo do arquivo MTL foi obtido antes de tentar abri-lo
if caminho_completo_mtl:
    with open(caminho_completo_mtl, 'r') as file:
        # Inicialização de variáveis
        radiance_maximum = {}
        reflectance_maximum = {}
        d = SUN_ELEVATION = None

        # Leitura e processamento do arquivo MTL
        for line in file:
            if 'EARTH_SUN_DISTANCE' in line:
                d = float(line.split('=')[-1].strip())
            elif 'SUN_ELEVATION' in line:
                SUN_ELEVATION = float(line.split('=')[-1].strip())
            elif 'RADIANCE_MAXIMUM_BAND_' in line or 'REFLECTANCE_MAXIMUM_BAND_' in line:
                key, value = line.split('=')
                key = key.strip()
                value = float(value.strip())
                if 'RADIANCE' in key:
                    radiance_maximum[key] = value
                else:
                    reflectance_maximum[key] = value
else:
    print("Não foi possível encontrar um arquivo MTL válido para abrir.")

# Cálculo de ESUN para cada banda
ESUN = [(math.pi * d * d) * (radiance_maximum[f'RADIANCE_MAXIMUM_BAND_{i}'] / 
                             reflectance_maximum[f'REFLECTANCE_MAXIMUM_BAND_{i}']) for i in range(1, 8)]

print(ESUN)

# Cálculo dos pesos relativos para cada banda
W = [value / sum(ESUN) for value in ESUN]
print(W)

# Descompactando os pesos em variáveis individuais
W1, W2, W3, W4, W5, W6, W7 = W

# Tentar ler o arquivo MDT usando rasterio
try:
    with rasterio.open(caminho_mdt) as src:
        MDT_Sebal = src.read(1)  # Lê a primeira banda do raster
        print("MDT carregado com sucesso.")
except Exception as e:
    print(f"Erro ao carregar o MDT: {e}")
    exit()  # Encerra o script se o arquivo não puder ser carregado

# Suponha que os rasters corrigidos bandX já foram lidos anteriormente
# Exemplo: band1, band2, ..., band7

# Calculando albedo na parte superior da atmosfera (aTOA)
aTOA = band1 * W1 + band2 * W2 + band3 * W3 + band4 * W4 + band5 * W5 + band6 * W6 + band7 * W7

# Definir o caminho de saída para aTOA
aTOA_output_path = os.path.join(output_dir, 'aTOA.tif')

# Preparar os metadados para salvar o aTOA, utilizando os metadados da imagem recortada
aTOA_meta = out_meta.copy()  # Utilizar out_meta que contém o CRS correto e outros metadados necessários
aTOA_meta.update({
    'driver': 'GTiff',
    'count': 1,  # aTOA é uma única banda
    'dtype': aTOA.dtype,  # Ou 'float32' se você quiser forçar o tipo, conforme seu dado
    'width': aTOA.shape[1],  # Largura do aTOA
    'height': aTOA.shape[0]  # Altura do aTOA
})

# Salvar o aTOA usando rasterio
try:
    with rasterio.open(aTOA_output_path, 'w', **aTOA_meta) as dst:
        dst.write(aTOA, 1)  # Escreve a matriz aTOA no arquivo
    print("aTOA salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")


# Cálculo de Tsw com base no MDT recortado
Tsw = 0.75 + 0.00002 * mdt_recortado

Tsw_output_path = os.path.join(output_dir, 'Tsw.tif')
            
# Preparando os metadados para salvar o Tsw
Tsw_meta = out_meta.copy()
Tsw_meta.update({
    'driver': 'GTiff',
    'count': 1,  # Tsw é uma única banda
    'dtype': Tsw.dtype,  # Ou 'float32' se você quiser forçar o tipo, conforme seu dado
    'width': Tsw.shape[1],  # Largura do Tsw
    'height': Tsw.shape[0]  # Altura do Tsw
})

# Salvar o Tsw usando rasterio
try:
    with rasterio.open(Tsw_output_path, 'w', **Tsw_meta) as dst:
        dst.write(Tsw, 1)  # Escreve a matriz Tsw no arquivo
    print("Tsw salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")


#def recortar_imagem_com_padding(src, geometrias, bounds, expected_width, expected_height):
    #out_image, out_transform = mask(src, geometrias, crop=True)
    #print("Original shape:", out_image.shape)  # Imprimir as dimensões originais
    #if out_image.shape[1] < expected_height or out_image.shape[2] < expected_width:
        #padded_image = np.zeros((out_image.shape[0], expected_height, expected_width), dtype=out_image.dtype)
        #padded_image[:, :out_image.shape[1], :out_image.shape[2]] = out_image
        #out_image = padded_image
    #print("Padded shape:", out_image.shape)  # Imprimir as dimensões após padding
    #return out_image

#def recortar_imagem_com_padding(src, geometrias, bounds, expected_width, expected_height):
    #Expandir o bounding box para garantir cobertura completa
    #pixel_min_x, pixel_min_y = ~src.transform * (bounds[0], bounds[3])
    #pixel_max_x, pixel_max_y = ~src.transform * (bounds[2], bounds[1])
    #pixel_min_x = int(pixel_min_x) - 1
    #pixel_min_y = int(pixel_min_y) - 1
    #pixel_max_x = int(pixel_max_x) + 1
    #pixel_max_y = int(pixel_max_y) + 1
    #expected_height = pixel_max_y - pixel_min_y
    #expected_width = pixel_max_x - pixel_min_x

    #out_image, out_transform = mask(src, geometrias, crop=True)
    # Adicionar lógica para verificar e ajustar as dimensões aqui

        
# Calculando o albedo da superfície (aS)
aS = (aTOA - 0.03) / (Tsw ** 2)
  
# Definir o caminho de saída para aS
aS_output_path = os.path.join(output_dir, 'aS.tif')

# Preparar os metadados para salvar o aS, utilizando os metadados da imagem recortada
aS_meta = out_meta.copy()  # Utilizar out_meta que contém o CRS correto e outros metadados necessários
aS_meta.update({
    'driver': 'GTiff',
    'count': 1,  # aS é uma única banda
    'dtype': aS.dtype,  # Ou 'float32' se você quiser forçar o tipo, conforme seu dado
    'width': aS.shape[1],  # Largura do aS
    'height': aS.shape[0]  # Altura do aS
})

# Salvar o aS usando rasterio
try:
    with rasterio.open(aS_output_path, 'w', **aS_meta) as dst:
        dst.write(aS, 1)  # Escreve a matriz aS no arquivo
    print("aS salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Calcular Rsi
SUN_ELEVATION_rad = np.deg2rad(90 - SUN_ELEVATION)
Rsi = 1367 * np.cos(SUN_ELEVATION_rad) * (1 / (d**2)) * Tsw

# Definir o caminho de saída para Rsi
Rsi_output_path = os.path.join(output_dir, 'Rsi.tif')

# Preparar os metadados para salvar o Rsi
Rsi_meta = out_meta.copy()
Rsi_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',  # Garantindo que o tipo seja float32 para cálculo
    'width': Rsi.shape[1],
    'height': Rsi.shape[0]
})

# Salvar o Rsi usando rasterio
try:
    with rasterio.open(Rsi_output_path, 'w', **Rsi_meta) as dst:
        dst.write(Rsi.astype('float32'), 1)
    print("Rsi salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Calcular RLo
RLo = e0f * 5.67e-8 * (Ts ** 4)

# Definir o caminho de saída para RLo
RLo_output_path = os.path.join(output_dir, 'RLo.tif')

# Preparar os metadados para salvar o RLo
RLo_meta = out_meta.copy()
RLo_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': RLo.shape[1],
    'height': RLo.shape[0]
})

# Salvar o RLo usando rasterio
try:
    with rasterio.open(RLo_output_path, 'w', **RLo_meta) as dst:
        dst.write(RLo.astype('float32'), 1)
    print("RLo salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")


# Carregar NDVI raster
ndvi_path = os.path.join(output_dir, 'NDVI.tif')
with rasterio.open(ndvi_path) as src:
    NDVI = src.read(1)  # Assume que NDVI é uma única banda
    ndvi_nodata = src.nodata  # Capturar o valor de no-data do raster

# Carregar Ts raster
ts_path = os.path.join(output_dir, 'Ts.tif')
with rasterio.open(ts_path) as src:
    Ts = src.read(1)
    ts_nodata = src.nodata  # Capturar o valor de no-data de Ts

# Calcular a mediana de Ts ignorando o valor de no-data
Ts_masked = np.ma.masked_equal(Ts, ts_nodata)
Ts_median = np.nanmedian(Ts_masked)

print("Mediana de Ts calculada ignorando o valor de no-data:", Ts_median)

# Criar a máscara de pixel frio
Pcold = np.where((NDVI > 0.4) & (Ts < Ts_median), Ts, np.nan)

# Definir o caminho de saída para Pcold
Pcold_output_path = os.path.join(output_dir, 'Pcold.tif')

# Preparar os metadados para salvar Pcold
Pcold_meta = out_meta.copy()
Pcold_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': Pcold.shape[1],
    'height': Pcold.shape[0]
})

# Salvar Pcold usando rasterio
try:
    with rasterio.open(Pcold_output_path, 'w', **Pcold_meta) as dst:
        dst.write(np.float32(Pcold), 1)
    print("Pcold salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Caminhos completos para os arquivos Ts.tif e Tsw.tif
ts_file_path = os.path.join(output_dir, 'Ts.tif')
tsw_file_path = os.path.join(output_dir, 'Tsw.tif')

# Solicitar ao usuário para inserir as coordenadas do pixel frio
print("Please enter the coordinates of the cold pixel (format: easting,northing):")
xy_Pcold = input().strip().split(',')
xy_Pcold = tuple(map(float, xy_Pcold))  # Converter strings para float

# Verificar e abrir os arquivos Ts e Tsw
try:
    with rasterio.open(ts_file_path) as ts_src, rasterio.open(tsw_file_path) as tsw_src:
        # Converter coordenadas para índices de matriz de pixel
        row, col = ts_src.index(*xy_Pcold)
        z_TsPcold = ts_src.read(1)[row, col]
        Tsw_value = tsw_src.read(1)[row, col]

        print("Cold pixel temperature:", z_TsPcold, "K")
        print("Tsw value at cold pixel:", Tsw_value)

        # Calcular a radiação de onda longa incidente (RLi)
        RLi = 0.85 * ((-np.log(Tsw_value)) ** 0.09) * 5.67e-8 * z_TsPcold ** 4
        print("Calculating incoming longwave radiation (RLi) - W/m2... Done!")

        # Definir o caminho de saída para RLi
        RLi_output_path = os.path.join(output_dir, 'RLi.tif')

        # Criar uma matriz RLi para todo o raster baseado nas dimensões de Ts
        RLi_array = np.full(ts_src.shape, RLi, dtype='float32')

        # Preparar os metadados para salvar o RLi
        RLi_meta = ts_src.meta.copy()
        RLi_meta.update({
            'dtype': 'float32',
            'count': 1
        })

        # Salvar o RLi usando rasterio
        with rasterio.open(RLi_output_path, 'w', **RLi_meta) as dst:
            dst.write(RLi_array, 1)
        print("RLi salvo com sucesso.")

except FileNotFoundError as e:
    print(f"Erro: {e}")
except Exception as e:
    print(f"Ocorreu um erro inesperado: {e}")

# Cálculo do Fluxo de Radiação Líquido (Rn)

# Caminhos completos para os arquivos Rsi.tif e aS.tif
rsi_file_path = os.path.join(output_dir, 'Rsi.tif')
as_file_path = os.path.join(output_dir, 'aS.tif')

try:
    # Abrindo e lendo os rasters
    with rasterio.open(rsi_file_path) as src_rsi:
        Rsi = src_rsi.read(1)  # Lendo a primeira banda
    with rasterio.open(as_file_path) as src_as:
        aS = src_as.read(1)  # Lendo a primeira banda

    # Cálculo do Fluxo de Radiação Líquido (Rn)
    Rn = (1 - aS) * Rsi + RLi - RLo - (1 - e0f) * RLi

    # Definir o caminho de saída para Rn
    Rn_output_path = os.path.join(output_dir, 'Rn.tif')

    # Preparar os metadados para salvar o Rn
    Rn_meta = src_rsi.meta.copy()
    Rn_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Rn.shape[1],
        'height': Rn.shape[0]
    })

    # Salvar o Rn usando rasterio
    with rasterio.open(Rn_output_path, 'w', **Rn_meta) as dst:
        dst.write(Rn.astype('float32'), 1)
    print("Rn salvo com sucesso.")
except Exception as e:
    print(f"Erro ao processar ou salvar os rasters: {e}")

# Cálculo do Fluxo de Calor no Solo (G)
G_Rn = np.where(NDVI < 0, 0.5, ((Ts - 273.15) / aS) * (0.0038 * aS + 0.0074 * aS ** 2) * (1 - 0.98 * NDVI ** 4))
G = G_Rn * Rn

# Definir o caminho de saída para Rn
G_output_path = os.path.join(output_dir, 'G.tif')

# Preparar os metadados para salvar o G
G_meta = out_meta.copy()
G_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': G.shape[1],
    'height': G.shape[0]
})

# Salvar o G usando rasterio
try:
    with rasterio.open(G_output_path, 'w', **G_meta) as dst:
        dst.write(G.astype('float32'), 1)
    print("G salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

print("Calculating soil heat flux (G) - W/m2... Done!")

# Criação da Máscara do Pixel Quente
Phot = np.where((SAVI > 0.18) & (SAVI < 0.3), Ts, np.nan)

print("Making the hot pixel mask... Done!")

# Definir o caminho de saída para Phot
Phot_output_path = os.path.join(output_dir, 'Phot.tif')

# Preparar os metadados para salvar Phot
Phot_meta = out_meta.copy()
Phot_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': Phot.shape[1],
    'height': Phot.shape[0]
})

# Salvar Phot usando rasterio
try:
    with rasterio.open(Phot_output_path, 'w', **Phot_meta) as dst:
        dst.write(np.float32(Phot), 1)
    print("Phot salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

# Caminho completo para o arquivo Ts.tif 
ts_file_path = os.path.join(output_dir, 'Ts.tif')

# Solicitação ao usuário para inserir as coordenadas do pixel quente
print("Choose the hot pixel coordinates in bare soil areas. Use the raster Phot and CC_432 for help...")
xy_Phot = input('Place coordinates (east,north): ').strip('()')
print(xy_Phot)

# Lendo a temperatura no pixel quente usando o caminho completo
try:
    with rasterio.open(ts_file_path) as src:
        # Converter as coordenadas em índices
        row, col = src.index(*map(float, xy_Phot.split(',')))
        z_TsPhot = src.read(1)[row, col]
        print(f"Hot pixel temperature: {z_TsPhot} K")

   # Cálculo da Velocidade de Fricção (u*)
    h = 0.15
    Zom = 0.123 * h
    u_ast = 0.41 * u_2m / (math.log(2 / Zom))
    u_200m = u_ast * (math.log(200 / Zom)) / 0.41
    print("Calculating friction velocity (u*) for weather station - m/s... Done!")
except Exception as e:
    print(f"Erro ao processar ou salvar os rasters: {e}")    

# Cálculo do Comprimento de Rugosidade do Momento (Z0map)  
  
# Definindo o caminho completo para o arquivo SAVI.tif

savi_file_path = os.path.join(output_dir, 'SAVI.tif')

try:
    # Abrindo e lendo o raster SAVI
    with rasterio.open(savi_file_path) as src_savi:
        SAVI = src_savi.read(1)  # Lendo a primeira banda

    # Cálculo do Comprimento de Rugosidade do Momento (Z0map)
    Z0map = np.exp(-5.809 + 5.62 * SAVI)

    # Definir o caminho de saída para Z0map
    Z0map_output_path = os.path.join(output_dir, 'Z0map.tif')

    # Preparar os metadados para salvar o Z0map
    Z0map_meta = src_savi.meta.copy()
    Z0map_meta.update({
        'driver': 'GTiff',
        'count': 1,
        'dtype': 'float32',
        'width': Z0map.shape[1],
        'height': Z0map.shape[0]
    })

    # Salvar o Z0map usando rasterio
    with rasterio.open(Z0map_output_path, 'w', **Z0map_meta) as dst:
        dst.write(Z0map.astype('float32'), 1)
    print("Z0map salvo com sucesso.")
except Exception as e:
    print(f"Erro ao processar ou salvar os rasters: {e}")
    
# Cálculo da Velocidade de Fricção do Mapa (u*map)
u_astmap = 0.41 * u_200m / np.log(200 / Z0map)

# Definir o caminho de saída para u_astmap
u_astmap_output_path = os.path.join(output_dir, 'u_astmap.tif')

# Preparar os metadados para salvar u_astmap
u_astmap_meta = out_meta.copy()
u_astmap_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': u_astmap.shape[1],
    'height': u_astmap.shape[0]
})

# Salvar u_astmap usando rasterio
try:
    with rasterio.open(u_astmap_output_path, 'w', **u_astmap_meta) as dst:
        dst.write(np.float32(u_astmap), 1)
    print("u_astmap salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

print("Calculating the friction velocity map (u*map) - m/s... Done!")

# Calculating aerodynamic resistance to heat transport map in terms of neutral stability (rah)
rah = np.log(2 / 0.1) / (u_astmap * 0.41)
print("Calculating aerodynamic resistance to heat transport map in terms of neutral stability (rah) - s/m... Done!")

# Definir o caminho de saída para rah
rah_output_path = os.path.join(output_dir, 'rah.tif')

# Preparar os metadados para salvar rah
rah_meta = out_meta.copy()
rah_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': rah.shape[1],
    'height': rah.shape[0]
})

# Salvar rah usando rasterio
try:
    with rasterio.open(rah_output_path, 'w', **rah_meta) as dst:
        dst.write(np.float32(rah), 1)
    print("rah salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

#Caminhos para usar o diretório de saída
g_file_path = os.path.join(output_dir, 'G.tif')
rah_file_path = os.path.join(output_dir, 'rah.tif')
rn_file_path = os.path.join(output_dir, 'Rn.tif')

# Lendo valores dos rasters nas coordenadas do pixel quente
try:
    with rasterio.open(g_file_path) as src:
        # Convertendo coordenadas para índices de matriz de pixel
        row, col = src.index(*map(float, xy_Phot.split(',')))

        # Verifica se os índices estão dentro dos limites da imagem
        if 0 <= row < src.height and 0 <= col < src.width:
            z_GPhot = src.read(1)[row, col]
            print(f"Hot pixel G value: {z_GPhot}")
        else:
            raise IndexError("Coordenadas fornecidas estão fora dos limites do raster G.tif")

    with rasterio.open(rah_file_path) as src:
        z_rahPhot = src.read(1)[row, col] if 0 <= row < src.height and 0 <= col < src.width else None

    with rasterio.open(rn_file_path) as src:
        z_RnPhot = src.read(1)[row, col] if 0 <= row < src.height and 0 <= col < src.width else None

    if z_rahPhot is not None and z_RnPhot is not None:
        print("Aerodynamic resistance correction, be patient...")
        # Inclua aqui o restante das operações
    else:
        print("Algumas coordenadas estão fora dos limites para rah.tif ou Rn.tif")

except rasterio.errors.RasterioIOError as e:
    print(f"Erro ao abrir arquivo: {e}")
except Exception as e:
    print(f"Erro durante o processamento: {e}")

z_rahPhot_i = 0
i = 0
while abs(z_rahPhot_i - z_rahPhot) > 0.00001:
    print("Iteration number:", i)
    i += 1
    z_rahPhot_i = z_rahPhot
    print("Equation coefficients estimate >> dT = a * Ts + b")
    a = (z_RnPhot - z_GPhot) * z_rahPhot_i / ((z_TsPhot - z_TsPcold) * 1.25 * 1004)
    b = -a * z_TsPcold
    print('a:', a, 'b:', b)
    print("Calculating dT map - K...")
    
# Cálculo do mapa dT
dT = a * Ts + b
  
# Definir o caminho de saída para dT
dT_output_path = os.path.join(output_dir, 'dT.tif')

# Preparar os metadados para salvar dT
dT_meta = out_meta.copy()
dT_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': dT.shape[1],
    'height': dT.shape[0]
})

# Salvar dT usando rasterio
try:
    with rasterio.open(dT_output_path, 'w', **dT_meta) as dst:
        dst.write(np.float32(dT), 1)
    print("dT salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}") 

# Cálculo do Fluxo de Calor Sensível (H) 
  
# Definindo o caminho completo para o arquivo dT.tif e dT.tif

dt_file_path = os.path.join(output_dir, 'dT.tif')
rah_file_path = os.path.join(output_dir, 'rah.tif')

try:
    # Abrindo e lendo o raster dT
    with rasterio.open(dt_file_path) as src_dt:
        dT = src_dt.read(1)  # Lendo a primeira banda
    # Abrindo e lendo o raster dT
    with rasterio.open(rah_file_path) as src_rah:
        rah = src_rah.read(1)  # Lendo a primeira banda\a

    # Cálculo do Fluxo de Calor Sensível (H)
    H = (dT / rah) * 1.25 * 1004

    # Definir o caminho de saída para H
    H_output_path = os.path.join(output_dir, 'H.tif')

    # Preparar os metadados para salvar H
    H_meta = out_meta.copy()
    H_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': H.shape[1],
    'height': H.shape[0]
    })

    # Salvar H usando rasterio
    with rasterio.open(H_output_path, 'w', **H_meta) as dst:
        dst.write(np.float32(H), 1)
    print("H salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

print("Calculating sensible heat flux (H) - W/m2... Done!")

# Cálculo do Comprimento de Monin-Obukhov (L)
g = 9.81  # Aceleração devido à gravidade
L = -(1.25 * 1004 * Ts * u_astmap ** 3) / (0.41 * g * H)

# Definir o caminho de saída para L
L_output_path = os.path.join(output_dir, 'L.tif')

# Preparar os metadados para salvar L
L_meta = out_meta.copy()
L_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': L.shape[1],
    'height': L.shape[0]
})

# Salvar L usando rasterio
try:
    with rasterio.open(L_output_path, 'w', **L_meta) as dst:
        dst.write(np.float32(L), 1)
    print("L salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}") 
print("Calculating the Monin-Obukhov length map (L) - m... Done!")

# Calculating atmospheric stability correction (L200m, L2m, L01m)
L200m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (200 / L))) / 2), np.where(L > 0, -5 * (200 / L), 0))
L2m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (2 / L))) / 2), np.where(L > 0, -5 * (2 / L), 0))
L01m = np.where(L < 0, 2 * np.log((1 + np.sqrt(1 - 16 * (0.1 / L))) / 2), np.where(L > 0, -5 * (0.1 / L), 0))

# Definir o caminho de saída para L200m
L200m_output_path = os.path.join(output_dir, 'L200m.tif')
# Definir o caminho de saída para L2m
L2m_output_path = os.path.join(output_dir, 'L2m.tif')
# Definir o caminho de saída para L01m
L01m_output_path = os.path.join(output_dir, 'L01m.tif')

# Preparar os metadados para salvar L200m
L200m_meta = out_meta.copy()
L200m_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': L200m.shape[1],
    'height': L200m.shape[0]
})

# Salvar L200m usando rasterio
try:
    with rasterio.open(L200m_output_path, 'w', **L200m_meta) as dst:
        dst.write(np.float32(L200m), 1)
    print("L200m salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}") 
    
# Preparar os metadados para salvar L2m
L2m_meta = out_meta.copy()
L2m_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': L2m.shape[1],
    'height': L2m.shape[0]
})

# Salvar L2m usando rasterio
try:
    with rasterio.open(L2m_output_path, 'w', **L2m_meta) as dst:
        dst.write(np.float32(L2m), 1)
    print("L2m salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}") 

# Preparar os metadados para salvar L01m
L01m_meta = out_meta.copy()
L01m_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': L01m.shape[1],
    'height': L01m.shape[0]
})

# Salvar L01m usando rasterio
try:
    with rasterio.open(L01m_output_path, 'w', **L01m_meta) as dst:
        dst.write(np.float32(L01m), 1)
    print("L01m salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}") 
 
print("Calculating atmospheric stability correction (L200m, L2m, L01m)... Done!")

# Calculating corrected friction velocity map (u_astmap)
#with rasterio.open('Z0map.tif') as src:
    #Z0map = src.read(1)

#with rasterio.open('L200m.tif') as src:
    #L200m = src.read(1)

u_astmap = 0.41 * u_200m / (np.log(200 / Z0map) - L200m)

# Definir o caminho de saída para u_astmap
u_astmap_output_path = os.path.join(output_dir, 'u_astmap.tif')

# Preparar os metadados para salvar u_astmap
u_astmap_meta = out_meta.copy()
u_astmap_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': u_astmap.shape[1],
    'height': u_astmap.shape[0]
})

# Salvar u_astmap usando rasterio
try:
    with rasterio.open(u_astmap_output_path, 'w', **u_astmap_meta) as dst:
        dst.write(np.float32(u_astmap), 1)
    print("u_astmap salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

print("Calculating corrected friction velocity map (u_astmap) - m/s... Done!")

# Cálculo da Resistência Aerodinâmica Corrigida ao Transporte de Calor (rah)
#with rasterio.open('L2m.tif') as src:
    #L2m = src.read(1)

#with rasterio.open('L01m.tif') as src:
    #L01m = src.read(1)

rah = (np.log(2 / 0.1) - L2m + L01m) / (u_astmap * 0.41)

# Definir o caminho de saída para rah
rah_output_path = os.path.join(output_dir, 'rah.tif')

# Preparar os metadados para salvar rah
rah_meta = out_meta.copy()
rah_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': rah.shape[1],
    'height': rah.shape[0]
})

# Salvar rah usando rasterio
try:
    with rasterio.open(rah_output_path, 'w', **rah_meta) as dst:
        dst.write(np.float32(rah), 1)
    print("rah salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")
print("Calculating corrected aerodynamic resistance to heat transport (rah) - s/m... Done!")

# Definindo o caminho completo para o arquivo dT.tif e dT.tif

rah_file_path = os.path.join(output_dir, 'rah.tif')
H_file_path = os.path.join(output_dir, 'H.tif')  # Caminho para o arquivo H corretamente definido
dt_file_path = os.path.join(output_dir, 'dT.tif')
G_file_path = os.path.join(output_dir, 'G.tif')
Rn_file_path = os.path.join(output_dir, 'Rn.tif')

try:
    # Abrindo e lendo valores dos rasters nas coordenadas do pixel quente
    with rasterio.open(rah_file_path) as src:
        row, col = src.index(*map(float, xy_Phot.split(',')))
        z_rahPhot = src.read(1)[row, col]

    with rasterio.open(H_file_path) as src:
        z_HPhot = src.read(1)[row, col]

    with rasterio.open(dt_file_path) as src:
        z_dTPhot = src.read(1)[row, col]

    with rasterio.open(g_file_path) as src:
        z_GPhot = src.read(1)[row, col]

    with rasterio.open(rn_file_path) as src:
        z_RnPhot = src.read(1)[row, col]

    print("Hot pixel results, verify if Rhhot - Ghot = Hhot")
    print('Hhot:', z_HPhot)
    print('rahhot:', z_rahPhot)
    print('Ghot:', z_GPhot)
    print('Rnhot:', z_RnPhot, 'dT:', z_dTPhot)

except Exception as e:
    print("An error occurred while reading the raster files:", e)

# Cálculo do Fluxo de Calor Latente (LET)
# Define o caminho dos arquivos
Rn_file_path = os.path.join(output_dir, 'Rn.tif')
G_file_path = os.path.join(output_dir, 'G.tif')
H_file_path = os.path.join(output_dir, 'H.tif')

try:
    # Abrindo e lendo o raster Rn
    with rasterio.open(Rn_file_path) as src_Rn:
        Rn = src_Rn.read(1)  # Lendo a primeira banda

    # Abrindo e lendo o raster G
    with rasterio.open(G_file_path) as src_G:
        G = src_G.read(1)  # Lendo a primeira banda

    # Abrindo e lendo o raster H
    with rasterio.open(H_file_path) as src_H:
        H = src_H.read(1)  # Lendo a primeira banda

    # Cálculo do Fluxo de Calor Latente (LET)
    LET = Rn - G - H

    # Caminho para salvar o LET
    LET_output_path = os.path.join(output_dir, 'LET.tif')

    # Preparar os metadados para salvar o LET
    LET_meta = out_meta.copy()
    LET_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': LET.shape[1],
    'height': LET.shape[0]
    })

    # Salvar o LET usando rasterio
    with rasterio.open(LET_output_path, 'w', **LET_meta) as dst:
        dst.write(np.float32(LET), 1)
    print("LET salvo com sucesso.")

except Exception as e:
    print(f"Erro ao processar ou salvar os arquivos: {e}")

print("Calculating the latent heat flux (LET) - W/m2... Done!")
    
# Cálculo da Evapotranspiração Instantânea (ETi)

# Define o caminho dos arquivos
LET_file_path = os.path.join(output_dir, 'LET.tif')

try:
    # Abrindo e lendo o raster LET
    with rasterio.open(LET_file_path) as src_LET:
        LET = src_LET.read(1)  # Lendo a primeira banda

        # Cálculo da Evapotranspiração Instantânea (ETi)
        ETi = np.where(3600 * (LET / (2.45 * 10**6)) < 0, 0, 3600 * (LET / (2.45 * 10**6)))

    # Caminho para salvar o ETi
    ETi_output_path = os.path.join(output_dir, 'ETi.tif')

    # Preparar os metadados para salvar o ETi
    ETi_meta = out_meta.copy()
    ETi_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': ETi.shape[1],
    'height': ETi.shape[0]
    })

    # Salvar o ETi usando rasterio
    with rasterio.open(ETi_output_path, 'w', **ETi_meta) as dst:
        dst.write(np.float32(ETi), 1)
    print("ETi salvo com sucesso.")

except Exception as e:
    print(f"Erro ao processar ou salvar os arquivos: {e}")
print("Calculating instantaneous evapotranspiration (ETi) - mm/h... Done!")

# Suponha que ETi, EToi e ETo já foram calculados ou carregados

# Cálculo da Fração da Evapotranspiração de Referência (ETof)
#with rasterio.open('ETi.tif') as src:
    #ETi = src.read(1)

ETof = ETi / EToi

ETof_output_path = os.path.join(output_dir, 'ETof.tif')

# Preparar os metadados para salvar o ETof
ETof_meta = out_meta.copy()
ETof_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': ETof.shape[1],
    'height': ETof.shape[0]
})

# Salvar o ETof usando rasterio
try:
    with rasterio.open(ETof_output_path, 'w', **ETof_meta) as dst:
        dst.write(ETof, 1)
    print("ETof salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")
print("Calculating reference evapotranspiration fraction (ETof)... Done!")

# Cálculo da Evapotranspiração Diária (ETday)
ETday = ETof * ETo

ETday_output_path = os.path.join(output_dir, 'ETday.tif')

# Preparar os metadados para salvar o ETday
ETday_meta = out_meta.copy()
ETday_meta.update({
    'driver': 'GTiff',
    'count': 1,
    'dtype': 'float32',
    'width': ETday.shape[1],
    'height': ETday.shape[0]
})

# Salvar o ETday usando rasterio
try:
    with rasterio.open(ETday_output_path, 'w', **ETday_meta) as dst:
        dst.write(ETday, 1)
    print("ETday salvo com sucesso.")
except Exception as e:
    print(f"Erro ao escrever o arquivo TIFF: {e}")

print("Calculating daily evapotranspiration (ETday) - mm/dia... Done!")




