# -*- coding: utf-8 -*-

'''
/***************************************************************************
 DrainageBasinGeomorphology
                                 A QGIS plugin
 This plugin provides tools for geomorphological analysis in drainage basins.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-03-22
        copyright            : (C) 2025 by João Vitor Pimenta
        email                : jvpjoaopimenta@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''

__author__ = 'João Vitor Pimenta'
__date__ = '2025-03-22'
__copyright__ = '(C) 2025 by João Vitor Pimenta'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFolderDestination)
from .algorithms.fillCutDEMProcessing import calcFilledCuttedForParameter, verifyLibs

class filledCuttedCalc(QgsProcessingAlgorithm):
    '''
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    '''

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    DRAINAGE_BASINS = 'DRAINAGE_BASINS'
    DEM = 'DEM'
    INPUT_PARAMETER = 'INPUT_PARAMETER'
    HEIGHT_PARAMETER = 'HEIGHT (m)'
    ELEVATION_PARAMETER = 'ELEVATION (m)'
    AREA_PARAMETER = 'AREA (m2)'
    VOLUME_PARAMETER = 'VOLUME (m3)'
    NO_DATA_VALUE = 'NO_DATA_VALUE'
    FILLED_CUTTED_DEM = 'FILLED_CUTTED_DEM'
    MODIFIED_AREA = 'MODIFIED_AREA'
    OPEN_RASTERS = 'OPEN_RASTERS'
    OPEN_VECTORS = 'OPEN_VECTORS'


    def initAlgorithm(self, config):
        '''
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        '''

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.DRAINAGE_BASINS,
                self.tr('Drainage basins'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.DEM,
                self.tr('DEM'),
                [QgsProcessing.TypeRaster]
            )
        )

        self.addParameter(
            QgsProcessingParameterEnum(
                'SELECT_OPTION',
                'Parameter',
                options=[
                        self.HEIGHT_PARAMETER,
                        self.ELEVATION_PARAMETER,
                        self.AREA_PARAMETER,
                        self.VOLUME_PARAMETER
                        ],
                defaultValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.INPUT_PARAMETER,
                'Parameter value',
                type=QgsProcessingParameterNumber.Double,
                defaultValue='10.000000',
                maxValue=float('inf'),
                minValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.NO_DATA_VALUE,
                'Nodata value for output band',
                type=QgsProcessingParameterNumber.Double,
                defaultValue=None,
                optional=True
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.OPEN_RASTERS,
                self.tr('Open output files in raster folder after running algorithm'),
                defaultValue=False,
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.OPEN_VECTORS,
                self.tr('Open output files in vector folder after running algorithm'),
                defaultValue=False,
            )
        )

        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.FILLED_CUTTED_DEM,
                self.tr('Cutted/filled DEM')
            )
        )

        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.MODIFIED_AREA,
                self.tr('Modified area'))
            )

    def processAlgorithm(self, parameters, context, feedback):
        '''
        Here is where the processing itself takes place.
        '''

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        basinSource = self.parameterAsSource(parameters, self.DRAINAGE_BASINS, context)

        demLayer = self.parameterAsRasterLayer(parameters, self.DEM, context)

        parameterValue = self.parameterAsDouble(parameters, self.INPUT_PARAMETER, context)

        parameterNumber = self.parameterAsEnum(parameters, 'SELECT_OPTION', context)

        selectedParameter = [
                             self.HEIGHT_PARAMETER,
                             self.ELEVATION_PARAMETER,
                             self.AREA_PARAMETER,
                             self.VOLUME_PARAMETER,
                             ][parameterNumber]

        noDataValue = self.parameterAsDouble(parameters, self.NO_DATA_VALUE, context)

        openRasters = self.parameterAsBoolean(parameters, self.OPEN_RASTERS, context)

        openVectors = self.parameterAsBoolean(parameters, self.OPEN_VECTORS, context)

        inundationRasterFolder = self.parameterAsString(parameters, self.FILLED_CUTTED_DEM, context)

        inundationVectorFolder = self.parameterAsString(parameters, self.MODIFIED_AREA, context)

        verifyLibs()
        calcFilledCuttedForParameter(basinSource,demLayer,selectedParameter,parameterValue,inundationRasterFolder,inundationVectorFolder,noDataValue,openRasters,openVectors,feedback)
        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.FILLED_CUTTED_DEM: inundationRasterFolder,
                self.MODIFIED_AREA: inundationVectorFolder}

    def name(self):
        '''
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        '''
        return 'Fill and cut DEM'

    def displayName(self):
        '''
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        '''
        return self.tr(self.name())

    def groupId(self):
        return "basin_modifications"

    def group(self):
        '''
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        '''
        return self.tr("Basin modifications")

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(os.path.join(os.path.dirname(__file__), "icon.png"))

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("""
        <html>
            <body>
                <p>       
        This tool calculates the cutted/filled area as a raster, for each basin, and as a vector, for each basin, with the attribute table containing the height, elevation, area and volume values.               
                </p>
                <p>
        <strong>Drainage basins: </strong>Layer containing drainage basins as features.
        <strong>DEM: </strong>Raster containing the band with the altimetry of the drainage basins. 
        <strong>Parameter: </strong>It is the parameter of the EAV (above and below) curve to be used to calculate the inundated area.
        <strong>Parameter value: </strong>It is the numerical value of the parameter.
        <strong>Nodata value for output band: </strong>The value of nodata values ​​for the output layers.
        <strong>Cutted/filled DEM: </strong>Path to the folder containing the cutted/filled area rasters.
        <strong>Modified area: </strong>Path to the folder containing the cutted/filled area vectors, in the vector attribute table there is the height, elevation, area and volume value.     

        The use of a projected CRS is recommended.
                       
        If you need more information about how the plugin works, such as the calculations it performs, among other things, access: https://github.com/JoaoVitorPimenta/qgis-plugin-Drainage-Basin-Geomorphology
        If you have found any bugs, errors or have any requests to make, among other things, please acess: https://github.com/JoaoVitorPimenta/qgis-plugin-Drainage-Basin-Geomorphology/issues
        If you need training for the plugin, or want to contact the plugin author for any reason, send an email to: jvpjoaopimentadev@gmail.com                </p>
            </body>
        </html>
                    """)

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return filledCuttedCalc()
