# -*- coding: utf-8 -*-

'''
/***************************************************************************
 DrainageBasinGeomorphology
                                 A QGIS plugin
 This plugin provides tools for geomorphological analysis in drainage basins.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-03-22
        copyright            : (C) 2025 by João Vitor Pimenta
        email                : jvpjoaopimenta@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''

__author__ = 'João Vitor Pimenta'
__date__ = '2025-03-22'
__copyright__ = '(C) 2025 by João Vitor Pimenta'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.core import QgsProcessingException, QgsProject, QgsRasterLayer, QgsVectorLayer
from collections import Counter
from osgeo import gdal, ogr, osr
import numpy as np
import os

def verifyLibs():
        try:
            import numpy
        except ImportError:
            raise QgsProcessingException('Numpy library not found, please install it and try again.')

def loadDEM(demLayer):
    DEMpath = demLayer.dataProvider().dataSourceUri().split('|')[0]
    ds = gdal.Open(DEMpath)
    band = ds.GetRasterBand(1)
    demArray = band.ReadAsArray()
    noData = band.GetNoDataValue() if band.GetNoDataValue() is not None else -9999
    gt = ds.GetGeoTransform()
    proj = ds.GetProjection()
    rows, cols = demArray.shape
    ds = None
    return demArray, noData, gt, proj, rows, cols

def EAVprocessing(demArray,noData,gt,proj,cols,rows,basin,feedback):
    basinGeom = basin.geometry()
    wkb = basinGeom.asWkb()
    ogrGeom = ogr.CreateGeometryFromWkb(wkb)

    rasterDrive = gdal.GetDriverByName('MEM')
    vectorDrive = ogr.GetDriverByName('Memory')
    vectorDriveSource    = vectorDrive.CreateDataSource('wrk')
    vectorLayer = vectorDriveSource.CreateLayer('lyr', None, ogr.wkbUnknown)
    featureDef   = vectorLayer.GetLayerDefn()
    ogrFeat  = ogr.Feature(featureDef)
    ogrFeat.SetGeometry(ogrGeom)
    vectorLayer.CreateFeature(ogrFeat)
    ogrFeat = None

    maskDS = rasterDrive.Create('', cols, rows, 1, gdal.GDT_Byte)
    maskDS.SetGeoTransform(gt)
    maskDS.SetProjection(proj)
    gdal.RasterizeLayer(maskDS, [1], vectorLayer, burn_values=[1])

    mask = maskDS.GetRasterBand(1).ReadAsArray()
    validMask = (mask == 1) & (demArray != noData)
    validDataInsideBasin = demArray[validMask]
    validDataInsideBasinList = validDataInsideBasin.tolist()

    counterValues = Counter(validDataInsideBasinList)
    counterValuesOrdered = sorted(counterValues.items())

    elevations = [item[0] for item in counterValuesOrdered]
    countElevations = [item[1] for item in counterValuesOrdered]

    pixelWidth  = abs(gt[1])
    pixelHeight = abs(gt[5])
    areas = np.array(countElevations) * (pixelWidth * pixelHeight)
    cumulativeAreas = np.cumsum(areas)

    deltaElev = np.diff(elevations)
    volumes = ((cumulativeAreas[1:] + cumulativeAreas[:-1])/2) * deltaElev
    cumulativeVolumes = np.concatenate(([0], np.cumsum(volumes)))

    elevations = np.array(elevations)
    cumulativeAreasList = cumulativeAreas
    cumulativeVolumesList = cumulativeVolumes
    return elevations, cumulativeAreasList, cumulativeVolumesList, validMask

def calcFilledDEM(arrayInsideBasin, elevation, height, area, volume, folderRaster, folderVector, noDataValue, proj, gt, demLayer, openRasters, openVectors, basin, idx, step, feedback):
    feedback.setProgressText('Basin id '+str(basin.id())+' processing raster starting...')

    elevationsFilledPixels = np.where(arrayInsideBasin <= elevation, elevation, arrayInsideBasin)

    if np.isnan(elevationsFilledPixels).all():
        feedback.pushWarning('There are no DEM pixels within basin id '+str(basin.id())+' below respective elevation, so it is not possible to generate the output layers for this basin.')
        return

    mask = ~np.isnan(elevationsFilledPixels)
    rows, cols = np.where(mask)

    rowMin, rowMax = rows.min(), rows.max()
    colMin, colMax = cols.min(), cols.max()
    croppedDepthArray = elevationsFilledPixels[rowMin:rowMax+1, colMin:colMax+1]

    newOriginX = gt[0] + colMin * (gt[1])
    newOriginY = gt[3] + rowMin * (gt[5])

    newGt = (
        newOriginX,
        gt[1],
        0.0,
        newOriginY,
        0.0,
        gt[5]
    )

    DEMpath = demLayer.dataProvider().dataSourceUri().split('|')[0]
    rasterOriginal = gdal.Open(DEMpath)

    gt = rasterOriginal.GetGeoTransform()
    projecao = rasterOriginal.GetProjection()

    lines = croppedDepthArray.shape[0]
    columns = croppedDepthArray.shape[1]

    driver = gdal.GetDriverByName("GTiff")

    os.makedirs(folderRaster, exist_ok=True)
    pathRaster = os.path.join(folderRaster, 'FILLED_RASTER_BASIN_ID_'+str(basin.id())+'.tif')
    totalFilledRaster = driver.Create(pathRaster, columns, lines, 1, gdal.GDT_Float32)

    totalFilledRaster.SetGeoTransform(newGt)
    totalFilledRaster.SetProjection(projecao)

    totalFilledBand = totalFilledRaster.GetRasterBand(1)
    totalFilledBand.WriteArray(croppedDepthArray)

    totalFilledBand.SetNoDataValue(noDataValue)

    feedback.setProgressText('Basin id '+str(basin.id())+' processing raster completed')

    feedback.setProgressText('Basin id '+str(basin.id())+' processing vector starting...')
    filledPixels = np.where(arrayInsideBasin <= elevation, 1, 0)
    croppedFilledArray = filledPixels[rowMin:rowMax+1, colMin:colMax+1]

    filledRaster = driver.Create("MEM", columns, lines, 1, gdal.GDT_Float32)

    filledRaster.SetGeoTransform(newGt)
    filledRaster.SetProjection(projecao)

    filledBand = filledRaster.GetRasterBand(1)
    filledBand.WriteArray(croppedFilledArray)

    filledBand.SetNoDataValue(noDataValue)

    driver = ogr.GetDriverByName("Memory")
    shapefileDs = driver.CreateDataSource('memory')

    srs = osr.SpatialReference(wkt=proj)
    layer = shapefileDs.CreateLayer("vetor", srs=srs, geom_type=ogr.wkbPolygon)

    fieldDefn = ogr.FieldDefn("value", ogr.OFTInteger)
    layer.CreateField(fieldDefn)

    gdal.Polygonize(filledBand, filledBand, layer, 0, [], callback=None)

    geomType = layer.GetGeomType()
    mergedGeom = ogr.Geometry(geomType)

    for feature in layer:
        geom = feature.GetGeometryRef()
        mergedGeom = mergedGeom.Union(geom)

    driver = ogr.GetDriverByName("ESRI Shapefile")
    os.makedirs(folderVector, exist_ok=True)
    pathVector = os.path.join(folderVector, 'MODIFIED_AREA_BASIN_ID_'+str(basin.id())+'.shp')

    mergedDataSource = driver.CreateDataSource(pathVector)
    mergedLayer = mergedDataSource.CreateLayer("merged", srs=layer.GetSpatialRef(), geom_type=geomType)

    featureUnique = ogr.Feature(mergedLayer.GetLayerDefn())
    featureUnique.SetGeometry(mergedGeom)
    mergedLayer.CreateFeature(featureUnique)

    mergedLayer.CreateField(ogr.FieldDefn("Height", ogr.OFTReal))
    mergedLayer.CreateField(ogr.FieldDefn("Elevation", ogr.OFTReal))
    mergedLayer.CreateField(ogr.FieldDefn("Area", ogr.OFTReal))
    mergedLayer.CreateField(ogr.FieldDefn("Volume", ogr.OFTReal))

    feature = mergedLayer.GetNextFeature()

    feature.SetField("Height", height)
    feature.SetField("Elevation", elevation)
    feature.SetField("Area", area)
    feature.SetField("Volume", volume)
    mergedLayer.SetFeature(feature)

    filledBand.FlushCache()
    layer.SyncToDisk()
    mergedLayer.SyncToDisk()

    barProgress = int((idx + 1) * step)
    feedback.setProgress(barProgress)
    feedback.setProgressText('Basin id '+str(basin.id())+' processing vector completed')

    layer = None
    mergedLayer = None
    totalFilledRaster = None
    filledRaster = None
    filledBand = None
    totalFilledBand = None
    rasterOriginal = None

    if openRasters is True:
        rasterFilled = QgsRasterLayer(pathRaster,'FILLED_RASTER_BASIN_ID_'+str(basin.id()))
        QgsProject.instance().addMapLayer(rasterFilled)

    if openVectors is True:
        vectorFilled = QgsVectorLayer(pathVector, 'MODIFIED_AREA_BASIN_ID_'+str(basin.id()))
        QgsProject.instance().addMapLayer(vectorFilled)

def calcFilledForParameter(drainageBasinLayer,demLayer,parameter,parameterValue,pathRaster,pathVector,noDataValue,openRasters,openVectors,feedback):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    demArray,noData,gt,proj,rows,cols = loadDEM(demLayer)

    HEIGHT_PARAMETER = 'HEIGHT (m)'
    ELEVATION_PARAMETER = 'ELEVATION (m)'
    AREA_PARAMETER = 'AREA (m2)'
    VOLUME_PARAMETER = 'VOLUME (m3)'

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        if feedback.isCanceled():
            return

        elevations, cumulativeAreas, cumulativeVolumes, validMask = EAVprocessing(demArray,noData,gt,proj,cols,rows,basin,feedback)

        if len(elevations) == 0:
            feedback.pushWarning('There is no valid raster data in the basin of id '+str(basin.id())+' and therefore it is not possible to calculate the filled DEM of ​​this basin.')
            return

        arrayInsideBasin = np.where(validMask, demArray, np.nan)

        if parameter == HEIGHT_PARAMETER:
            if parameterValue > max(elevations) - min(elevations):
                feedback.pushWarning('The parameter value is greater than the maximum height of basin id '+str(basin.id())+' so it will be set as the maximum for that basin.')
                parameterValue = max(elevations) - min(elevations)

            heights = elevations - min(elevations)

            heightAreaInterpolation = np.interp(parameterValue, heights, cumulativeAreas)
            heightVolInterpolation = np.interp(parameterValue, heights, cumulativeVolumes)

            respectiveElevation = parameterValue+elevations[0]
            respectiveHeight = parameterValue
            respectiveArea = heightAreaInterpolation
            respectiveVolume = heightVolInterpolation

            calcFilledDEM(arrayInsideBasin, respectiveElevation, respectiveHeight, respectiveArea, respectiveVolume, pathRaster, pathVector, noDataValue, proj, gt, demLayer, openRasters, openVectors, basin, idx, step, feedback)

        if parameter == ELEVATION_PARAMETER:
            if parameterValue > max(elevations):
                feedback.pushWarning('The parameter value is greater than the maximum elevation of basin id '+str(basin.id())+' so it will be set as the maximum for that basin')
                parameterValue = max(elevations)

            if parameterValue < min(elevations):
                feedback.pushWarning('The parameter value is less than the minimum height of basin id '+str(basin.id())+' so it will be set as the minimum for that basin.')
                parameterValue = min(elevations)

            elevationAreaInterpolation = np.interp(parameterValue, elevations, cumulativeAreas)
            elevationVolInterpolation = np.interp(parameterValue, elevations, cumulativeVolumes)

            respectiveElevation = parameterValue
            respectiveHeight = respectiveElevation - elevations[0]
            respectiveArea = elevationAreaInterpolation
            respectiveVolume = elevationVolInterpolation

            calcFilledDEM(arrayInsideBasin, respectiveElevation, respectiveHeight, respectiveArea, respectiveVolume, pathRaster, pathVector, noDataValue, proj, gt, demLayer, openRasters, openVectors, basin, idx, step, feedback)

        if parameter == AREA_PARAMETER:
            if parameterValue > max(cumulativeAreas):
                feedback.pushWarning('The parameter value is greater than the maximum area of basin id '+str(basin.id())+' so it will be set as the maximum for that basin')
                parameterValue = max(cumulativeAreas)

            if parameterValue < min(cumulativeAreas):
                feedback.pushWarning('The parameter value is less than the minimum area of basin id '+str(basin.id())+' so it will be set as the minimum for that basin.')
                parameterValue = min(cumulativeAreas)

            areaVolInterpolation = np.interp(parameterValue, cumulativeAreas, cumulativeVolumes)
            areaElevationInterpolation = np.interp(parameterValue, cumulativeAreas, elevations)

            respectiveElevation = areaElevationInterpolation
            respectiveHeight = respectiveElevation - elevations[0]
            respectiveArea = parameterValue
            respectiveVolume = areaVolInterpolation

            calcFilledDEM(arrayInsideBasin, respectiveElevation, respectiveHeight, respectiveArea, respectiveVolume, pathRaster, pathVector, noDataValue, proj, gt, demLayer, openRasters, openVectors, basin, idx, step, feedback)

        if parameter == VOLUME_PARAMETER:
            if parameterValue > max(cumulativeVolumes):
                feedback.pushWarning('The parameter value is greater than the maximum volume of basin id '+str(basin.id())+' so it will be set as the maximum for that basin.')
                parameterValue = max(cumulativeVolumes)

            if parameterValue < min(cumulativeVolumes):
                feedback.pushWarning('The parameter value is less than the minimum volume of basin id '+str(basin.id())+' so it will be set as the minimum for that basin.')
                parameterValue = min(cumulativeVolumes)

            volAreaInterpolation = np.interp(parameterValue, cumulativeVolumes, cumulativeAreas)
            volElevationInterpolation = np.interp(parameterValue, cumulativeVolumes, elevations)

            respectiveElevation = volElevationInterpolation
            respectiveHeight = respectiveElevation - elevations[0]
            respectiveArea = volAreaInterpolation
            respectiveVolume = parameterValue
            
            calcFilledDEM(arrayInsideBasin, respectiveElevation, respectiveHeight, respectiveArea, respectiveVolume, pathRaster, pathVector, noDataValue, proj, gt, demLayer, openRasters, openVectors, basin, idx, step, feedback)
