# -*- coding: utf-8 -*-

'''
/***************************************************************************
 DrainageBasinGeomorphology
                                 A QGIS plugin
 This plugin provides tools for geomorphological analysis in drainage basins.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-03-22
        copyright            : (C) 2025 by João Vitor Pimenta
        email                : jvpjoaopimenta@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''

__author__ = 'João Vitor Pimenta'
__date__ = '2025-03-22'
__copyright__ = '(C) 2025 by João Vitor Pimenta'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.core import QgsProcessingException
from collections import Counter
import plotly.graph_objects as go
from plotly.subplots import make_subplots
from osgeo import gdal, ogr
import numpy as np
import csv
import itertools
import bisect
import os

def verifyLibs():
        try:
            import numpy
        except ImportError:
            raise QgsProcessingException('Numpy library not found, please install it and try again.')
        try:
            import plotly
        except ImportError:
            raise QgsProcessingException('Plotly library not found, please install it and try again.')

def loadDEM(demLayer):
    DEMpath = demLayer.dataProvider().dataSourceUri().split('|')[0]
    ds = gdal.Open(DEMpath)
    band = ds.GetRasterBand(1)
    demArray = band.ReadAsArray()
    noData = band.GetNoDataValue() if band.GetNoDataValue() is not None else -9999
    gt = ds.GetGeoTransform()
    proj = ds.GetProjection()
    rows, cols = demArray.shape
    ds = None
    return demArray, noData, gt, proj, rows, cols

def EAVAboveBelowProcessing(demArray,noData,gt,proj,cols,rows,basin,distanceContour,minimumLevel,maximumLevel,subtractsBelow,useOnlyDEMElev,useMinDEMElev,useMaxDEMElev,feedback):
    basinGeom = basin.geometry()
    wkb = basinGeom.asWkb()
    ogrGeom = ogr.CreateGeometryFromWkb(wkb)

    rasterDrive = gdal.GetDriverByName('MEM')
    vectorDrive = ogr.GetDriverByName('Memory')
    vectorDriveSource    = vectorDrive.CreateDataSource('wrk')
    vectorLayer = vectorDriveSource.CreateLayer('lyr', None, ogr.wkbUnknown)
    featureDef   = vectorLayer.GetLayerDefn()
    ogrFeat  = ogr.Feature(featureDef)
    ogrFeat.SetGeometry(ogrGeom)
    vectorLayer.CreateFeature(ogrFeat)
    ogrFeat = None

    maskDS = rasterDrive.Create('', cols, rows, 1, gdal.GDT_Byte)
    maskDS.SetGeoTransform(gt)
    maskDS.SetProjection(proj)
    gdal.RasterizeLayer(maskDS, [1], vectorLayer, burn_values=[1])

    mask = maskDS.GetRasterBand(1).ReadAsArray()
    validMask = (mask == 1) & (demArray != noData)
    validDataInsideBasin = demArray[validMask].tolist()

    if not validDataInsideBasin:
        feedback.pushWarning('There is no valid raster data in the basin of id '+str(basin.id())+' and therefore it is not possible to calculate the elevation - area - volume.')
        return None, None, None

    counterValues = Counter(validDataInsideBasin)
    counterValuesFillOrdered = sorted(counterValues.items())
    counterValuesCutOrdered = sorted(counterValues.items(),reverse=True)

    elevationsFill = [item[0] for item in counterValuesFillOrdered]
    originalElevationsFill = [item[0] for item in counterValuesFillOrdered]
    countElevations = [item[1] for item in counterValuesFillOrdered]

    elevationsCut = [item[0] for item in counterValuesCutOrdered]
    originalElevationsCut = [item[0] for item in counterValuesCutOrdered]
    countElevationsCut = [item[1] for item in counterValuesCutOrdered]

    pixelWidth  = abs(gt[1])
    pixelHeight = abs(gt[5])

    maxElevation = max(elevationsFill)
    minElevation = min(elevationsFill)

    areasFill = np.array(countElevations) * (pixelWidth * pixelHeight)
    originalCumulativeAreasFill = np.cumsum(areasFill)
    cumulativeAreasFill = np.cumsum(areasFill)

    areasCut = np.array(countElevationsCut) * (pixelWidth * pixelHeight)
    originalCumulativeAreasCut = np.cumsum(areasCut)
    cumulativeAreasCut = np.cumsum(areasCut)

    if (minimumLevel and maximumLevel) is not None:
        if minimumLevel > maximumLevel:
            raise QgsProcessingException('The minimum level value cannot be greater than the maximum level value.')

    deltaElevFill = np.diff(elevationsFill)
    volumesFill = ((cumulativeAreasFill[1:] + cumulativeAreasFill[:-1])/2) * deltaElevFill
    originalCumulativeVolumesFill = np.concatenate(([0], np.cumsum(volumesFill)))
    cumulativeVolumesFill = np.concatenate(([0], np.cumsum(volumesFill)))

    deltaElevCut = abs(np.diff(elevationsCut))
    volumesCut = ((cumulativeAreasCut[1:] + cumulativeAreasCut[:-1])/2) * deltaElevCut
    originalCumulativeVolumesCut = np.concatenate(([0], np.cumsum(volumesCut)))
    cumulativeVolumesCut = np.concatenate(([0], np.cumsum(volumesCut)))

    if useOnlyDEMElev is True:
        distanceContour = None
    if useMinDEMElev is True:
        minimumLevel = None
    if useMaxDEMElev is True:
        maximumLevel = None

    if distanceContour == 0:
        raise QgsProcessingException('The distance between contour lines cannot be 0.')

    indexConstantAreaFill = np.argmax(cumulativeAreasFill)
    constantAreaFill = cumulativeAreasFill[indexConstantAreaFill]

    indexConstantAreaCut = np.argmax(cumulativeAreasCut)
    constantAreaCut = cumulativeAreasCut[indexConstantAreaCut]

    if maximumLevel is not None:
        elevationsWithMaximumLevelFill = sorted(elevationsFill)
        elevationsWithMaximumLevelCut = sorted(elevationsCut, reverse=True)

        if maximumLevel not in elevationsFill:
            elevationsWithMaximumLevelFill = sorted(elevationsFill + [maximumLevel])
            elevationsWithMaximumLevelCut = sorted(elevationsCut + [maximumLevel], reverse=True)

            if distanceContour is None:
                cumulativeAreasFill = np.interp(elevationsWithMaximumLevelFill, originalElevationsFill, originalCumulativeAreasFill)
                cumulativeAreasCut = np.interp(elevationsWithMaximumLevelCut, originalElevationsCut[::-1], originalCumulativeAreasCut[::-1])
                cumulativeVolumesFill = np.interp(elevationsWithMaximumLevelFill, originalElevationsFill, originalCumulativeVolumesFill)
                cumulativeVolumesCut = np.interp(elevationsWithMaximumLevelCut, originalElevationsCut, originalCumulativeVolumesCut)

        indexCrescent = bisect.bisect_right(elevationsWithMaximumLevelFill, maximumLevel)

        negElevations = [-e for e in elevationsWithMaximumLevelCut]
        indexDecrescent = bisect.bisect_left(negElevations, -maximumLevel)

        elevationsFill = elevationsWithMaximumLevelFill[:indexCrescent]
        cumulativeAreasFill = cumulativeAreasFill[:indexCrescent]
        cumulativeVolumesFill = cumulativeVolumesFill[:indexCrescent]

        elevationsCut = elevationsWithMaximumLevelCut[indexDecrescent:]
        cumulativeAreasCut = cumulativeAreasCut[indexDecrescent:]
        cumulativeVolumesCut = cumulativeVolumesCut[:indexCrescent]

    if minimumLevel is not None:
        elevationsWithMaximumMinimumLevelCut = sorted(elevationsCut,reverse=True)
        elevationsWithMaximumMinimumLevelFill = sorted(elevationsFill)

        if minimumLevel not in elevationsCut:
            elevationsWithMaximumMinimumLevelCut = sorted(elevationsCut + [minimumLevel],reverse=True)
            elevationsWithMaximumMinimumLevelFill = sorted(elevationsFill + [minimumLevel])

            if distanceContour is None:
                cumulativeAreasCut = np.interp(elevationsWithMaximumMinimumLevelCut, originalElevationsCut[::-1], originalCumulativeAreasCut[::-1])
                cumulativeAreasFill = np.interp(elevationsWithMaximumMinimumLevelFill, originalElevationsFill, originalCumulativeAreasFill)
                cumulativeVolumesCut = np.interp(elevationsWithMaximumMinimumLevelCut, originalElevationsCut[::-1], originalCumulativeVolumesCut[::-1])
                cumulativeVolumesFill = np.interp(elevationsWithMaximumMinimumLevelFill, originalElevationsFill, originalCumulativeVolumesFill)

        indexCrescent = bisect.bisect_left(elevationsWithMaximumMinimumLevelFill, minimumLevel)

        negElevations = [-e for e in elevationsWithMaximumMinimumLevelCut]
        indexDecrescent = bisect.bisect_right(negElevations, -minimumLevel)

        elevationsCut = elevationsWithMaximumMinimumLevelCut[:indexDecrescent]
        cumulativeAreasCut = cumulativeAreasCut[:indexDecrescent]
        cumulativeVolumesCut = cumulativeVolumesCut [:indexDecrescent]

        elevationsFill = elevationsWithMaximumMinimumLevelFill[indexCrescent:]
        cumulativeAreasFill = cumulativeAreasFill[indexCrescent:]
        cumulativeVolumesFill = cumulativeVolumesFill[indexCrescent:]

    if distanceContour is not None:
        minElevation = min(elevationsFill)
        maxElevation = max(elevationsFill)

        elevationCurvesFill = np.arange(minElevation, maxElevation, distanceContour)
        if maxElevation not in elevationCurvesFill:
            elevationCurvesFill = np.append(elevationCurvesFill, maxElevation)
        interpAreasFill = np.interp(elevationCurvesFill, originalElevationsFill, originalCumulativeAreasFill)
        interpVolumesFill = np.interp(elevationCurvesFill, originalElevationsFill, originalCumulativeVolumesFill)

        elevationsFill = elevationCurvesFill.tolist()
        cumulativeAreasFill = interpAreasFill
        cumulativeVolumesFill = interpVolumesFill

        minElevation = min(elevationsCut)
        maxElevation = max(elevationsCut)

        elevationCurvesCut = np.arange(minElevation, maxElevation, distanceContour)
        if maxElevation not in elevationCurvesCut:
            elevationCurvesCut = np.append(elevationCurvesCut, maxElevation)
        interpAreasCut = np.interp(elevationCurvesCut, originalElevationsCut[::-1], originalCumulativeAreasCut[::-1])
        interpVolumesCut = np.interp(elevationCurvesCut, originalElevationsCut[::-1], originalCumulativeVolumesCut[::-1])

        elevationsCut = elevationCurvesCut[::-1].tolist()
        cumulativeAreasCut = interpAreasCut[::-1]
        cumulativeVolumesCut = interpVolumesCut[::-1]

    constantAreaFill = originalCumulativeAreasFill[-1]
    constantAreaCut = originalCumulativeAreasCut[-1]

    if minimumLevel is not None:
        if minimumLevel < min(originalElevationsFill):
            elevationsCutArray = np.array(elevationsCut)
            minElevationCutArray = np.min(originalElevationsCut)

            deltaH = minElevationCutArray - elevationsCutArray[elevationsCutArray < minElevationCutArray]
            volumeCut = originalCumulativeVolumesCut[-1] + constantAreaCut * deltaH
            lenVolumeCut = len(volumeCut)
            cumulativeVolumesCut[-lenVolumeCut:] = volumeCut

    if maximumLevel is not None:
        if maximumLevel > max(originalElevationsFill):
            elevationsFillArray = np.array(elevationsFill)
            maxElevationFillArray = np.max(originalElevationsFill)

            deltaH = elevationsFillArray[elevationsFillArray > maxElevationFillArray] - maxElevationFillArray
            volumeFill = originalCumulativeVolumesFill[-1] + constantAreaFill * deltaH
            lenVolumeFill = len(volumeFill)
            cumulativeVolumesFill[-lenVolumeFill:] = volumeFill

    cumulativeAreas = cumulativeAreasCut[::-1] - cumulativeAreasFill
    cumulativeVolumes = cumulativeVolumesCut[::-1] - cumulativeVolumesFill

    cumulativeVolumesToCalcZero = originalCumulativeVolumesCut[::-1] - originalCumulativeVolumesFill

    elevationWithVolumeZero = np.interp(0, cumulativeVolumesToCalcZero[::-1], originalElevationsFill[::-1])

    if subtractsBelow is True:
        cumulativeAreas = cumulativeAreasFill - cumulativeAreasCut[::-1]
        cumulativeVolumes = cumulativeVolumesFill - cumulativeVolumesCut[::-1]

        cumulativeVolumesToCalcZero = originalCumulativeVolumesFill - originalCumulativeVolumesCut[::-1]

        elevationWithVolumeZero = np.interp(0, cumulativeVolumesToCalcZero, originalElevationsFill)

    feedback.pushInfo('The elevation for the cut/fill volume to be 0 in basin '+str(basin.id())+' is: '+str(elevationWithVolumeZero))

    cumulativeAreasList = cumulativeAreas.tolist()
    cumulativeVolumesList = cumulativeVolumes.tolist()
    return elevationsFill, cumulativeAreasList, cumulativeVolumesList

def runEAVAboveBelow(drainageBasinLayer,demLayer,pathCsv,pathHtml,distanceContour,minimumLevel,maximumLevel,subtractsBelow,useOnlyDEMElev,useMinDEMElev,useMaxDEMElev,feedback,decimalPlaces,useAllDecimalPlaces):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    demArray,noData,gt,proj,rows,cols = loadDEM(demLayer)

    fig = go.Figure()
    listsWithData = []

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        if feedback.isCanceled():
            return

        feedback.setProgressText(f'Basin id {basin.id()} processing starting...')
        elevations, cumulativeAreas, cumulativeVolumes = EAVAboveBelowProcessing(
            demArray,noData,gt,proj,cols,rows,basin,
            distanceContour,minimumLevel,maximumLevel,subtractsBelow,
            useOnlyDEMElev,useMinDEMElev,useMaxDEMElev,feedback
            )

        if elevations is None and cumulativeAreas is None and cumulativeVolumes is None:
            continue

        if useAllDecimalPlaces is False:
            elevations = [round(num, decimalPlaces) for num in elevations]
            cumulativeAreas = [round(num, decimalPlaces) for num in cumulativeAreas]
            cumulativeVolumes = [round(num, decimalPlaces) for num in cumulativeVolumes]

        elevations.insert(0, f'Elevation basin id {basin.id()}')
        cumulativeAreas.insert(0, f'Area basin id {basin.id()}')
        cumulativeVolumes.insert(0, f'Volume basin id {basin.id()}')

        listsWithData.append(elevations)
        listsWithData.append(cumulativeAreas)
        listsWithData.append(cumulativeVolumes)

        feedback.setProgressText(f'Basin id {basin.id()} processing completed')

        if feedback.isCanceled():
            return

        feedback.setProgressText(f'Basin id {basin.id()} graph starting...')

        fig.add_trace(go.Scatter(
            x=cumulativeVolumes,
            y=elevations,
            mode='lines',
            name=f'Volume - Elevation basin id {basin.id()}',
            yaxis='y',
            xaxis='x'   # padrão
        ))

        fig.add_trace(go.Scatter(
            x=cumulativeAreas,
            y=elevations,
            mode='lines',
            name=f'Area - Elevation basin id {basin.id()}',
            yaxis='y2',   # eixo y secundário
            xaxis='x2'    # eixo x secundário
        ))

        barProgress = int((idx + 1) * step)
        feedback.setProgress(barProgress)
        feedback.setProgressText(f'Basin id {basin.id()} graph completed')

    # Configura layout com eixos secundários (x2 e y2)
    fig.update_layout(
        title='Elevation - Area - Volume graph',
        xaxis=dict(title='Volume (m³)'),
        yaxis=dict(title='Elevation (m)'),
        xaxis2=dict(
            title='Area (m²)',
            overlaying='x',
            side='top',
            autorange='reversed'
        ),
        yaxis2=dict(
            title='Elevation (m)',
            overlaying='y',
            side='right',
            position=1
        ),
    )

    fig.write_html(pathHtml)
    fig.show()

    if feedback.isCanceled():
        return

    feedback.setProgressText(f'Graph completed for all basins')

    with open(pathCsv, 'w', newline='') as archive:
        writer = csv.writer(archive)
        writer.writerows(
            itertools.zip_longest(*[
                [col[0]] + [f"{v:.{decimalPlaces}f}" for v in col[1:]]
                for col in listsWithData
            ])
        )

