# -*- coding: utf-8 -*-

'''
/***************************************************************************
 DrainageBasinGeomorphology
                                 A QGIS plugin
 This plugin provides tools for geomorphological analysis in drainage basins.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-03-22
        copyright            : (C) 2025 by João Vitor Pimenta
        email                : jvpjoaopimenta@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''

__author__ = 'João Vitor Pimenta'
__date__ = '2025-03-22'
__copyright__ = '(C) 2025 by João Vitor Pimenta'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from math import pi
from qgis.core import QgsPointXY, QgsRaster, QgsProcessingException, QgsGeometry
import geopandas as gpd
from osgeo import gdal, ogr

def verifyLibs():
        try:
            import numpy
        except ImportError:
            raise QgsProcessingException('Numpy library not found, please install it and try again.')
        
        try:
            import geopandas
        except ImportError:
            raise QgsProcessingException('Geopandas library not found, please install it and try again.')
def getStreamsInsideBasin(streamLayer, drainageBasin,feedback):
    streamsWithin = []

    basinGeom = drainageBasin.geometry()

    streamWithin = [
        stream for stream in streamLayer.getFeatures()
        if stream.geometry().within(basinGeom)
    ]

    streamsWithin.extend(streamWithin)

    if not streamsWithin:
        feedback.pushWarning('There are no channels entirely within the basin of id '+str(drainageBasin.id())+' and the calculation of some parameters for this basin may be compromised')

    return streamsWithin
def createGdfStream(streams):
    geometries2d = []

    for feat in streams:
        geom = feat.geometry()

        if geom.constGet().is3D():
            geom.get().dropZValue()
        if geom.constGet().isMeasure():
            geom.get().dropMValue()

        if not geom.isMultipart():
            geometries2d.append(geom)
        else:
            multiGeom = geom.asGeometryCollection()
            if multiGeom:
                geometries2d.append(multiGeom[0])

    gdfStream = gpd.GeoDataFrame(
        geometry=geometries2d
    )
    return gdfStream
def obtainFirstAndLastPoint(gdfStream):
    if gdfStream.empty:
        return
    coords = gdfStream.geometry.apply(lambda geom: [geom.coords[0], geom.coords[-1]])
    gdfStream['first'] = coords.str[0]
    gdfStream['last'] = coords.str[1]
    return

def createOrderColumn(gdfStream):
    gdfStream['order'] = None
    return

def findOrder(gdf, streamMap, index):
    if gdf.at[index, 'order'] is not None:
        return

    stream = gdf.loc[index]
    first = str(stream['first'])

    if first not in streamMap:
        gdf.at[index, 'order'] = 1
        return

    tributariesIndexes = streamMap[first]
    maxOrder = 1
    maxOrderCount = 0

    for tributaryIndex in tributariesIndexes:
        if gdf.at[tributaryIndex, 'order'] is None:
            findOrder(gdf, streamMap, tributaryIndex)

        tributary_order = gdf.at[tributaryIndex, 'order']

        if maxOrder == tributary_order:
            maxOrderCount += 1
        elif maxOrder < tributary_order:
            maxOrder = tributary_order
            maxOrderCount = 1

    gdf.at[index, 'order'] = maxOrder + 1 if maxOrderCount > 1 else maxOrder

def createStreamMap(gdfStream):
    streamMap = {}
    for index, row in gdfStream.iterrows():
        last = str(row['last'])

        if last not in streamMap:
            streamMap[last] = []

        streamMap[last].append(index)
    return streamMap

def fillOrder(gdfStream):
    gdfStream['order'] = None
    streamMap = createStreamMap(gdfStream)

    for index in gdfStream.index:
        findOrder(gdfStream, streamMap, index)

def mergeStreams(gdf):
    if gdf.empty:
        return

    gdf['processed'] = False
    newRows = []
    indicesToRemove = []

    for idx in gdf.index:
        if gdf.loc[idx, 'processed']:
            continue

        currentIdx = idx
        chain = [currentIdx]
        gdf.loc[currentIdx, 'processed'] = True
        currentOrder = gdf.loc[currentIdx, 'order']

        while True:
            currentStart = gdf.loc[currentIdx, 'first']
            candidates = gdf[
                (gdf['last'] == currentStart) & 
                (gdf['order'] == currentOrder) & 
                (~gdf['processed'])
            ]

            if len(candidates) != 1:
                break

            nextIdx = candidates.index[0]
            chain.insert(0, nextIdx)
            gdf.loc[nextIdx, 'processed'] = True
            currentIdx = nextIdx

        currentIdx = idx

        while True:
            currentEnd = gdf.loc[currentIdx, 'last']
            candidates = gdf[
                (gdf['first'] == currentEnd) & 
                (gdf['order'] == currentOrder) & 
                (~gdf['processed'])
            ]

            if len(candidates) != 1:
                break

            nextIdx = candidates.index[0]
            chain.append(nextIdx)
            gdf.loc[nextIdx, 'processed'] = True
            currentIdx = nextIdx

        if len(chain) > 1:
            geoms = gdf.loc[chain, 'geometry'].tolist()

            allCoords = []
            for i, geom in enumerate(geoms):
                if i == 0:
                    allCoords.extend(geom.coords[:])
                else:
                    allCoords.extend(geom.coords[1:])
            
            newGeom = geoms[0].__class__(allCoords)
            
            newRows.append({
                'geometry': newGeom,
                'first': allCoords[0],
                'last': allCoords[-1],
                'order': currentOrder
            })
            
            indicesToRemove.extend(chain)

    gdf.drop(indicesToRemove, inplace=True)
    
    for row in newRows:
        newIndex = gdf.index.max() + 1 if not gdf.empty else 0
        gdf.loc[newIndex] = row
    
    gdf.drop(columns=['processed'], inplace=True, errors='ignore')
    
    if not gdf.index.is_unique:
        gdf.reset_index(drop=True, inplace=True)
    return

def calculateStreamLength(gdfStream):
    gdfStream['length'] = gdfStream.geometry.length
    return

def createGdfLinear(gdfStream):
    if gdfStream.empty:
        gdfLinear = gpd.GeoDataFrame({'Stream Order': [1]})
        gdfLinear.loc[len(gdfLinear)] = [None]
        return gdfLinear

    maxStreamOrder = gdfStream['order'].max()
    gdfLinear = gpd.GeoDataFrame({'Stream Order': range(1, maxStreamOrder + 1)})
    gdfLinear.loc[len(gdfLinear)] = [None]
    return gdfLinear

def calculateStreamNumber (gdfStream,gdfLinear):
    streamNumber = gdfStream['order'].value_counts().sort_index()
    gdfLinear['Stream Number'] = gdfLinear['Stream Order'].map(streamNumber)
    return

def calculateTotalStreamLength(gdfStream,gdfLinear):
    gdfWithSumLength = gdfStream.groupby('order')['length'].sum().reset_index()
    gdfLinear['Stream Length Total (km)'] = gdfWithSumLength['length']/1000
    return

def calculateMeanStreamLength(gdfStream,gdfLinear):
    gdfWithMeansLenghts = gdfStream.groupby('order')['length'].mean().reset_index()
    gdfLinear['Stream Length Mean (km)'] = gdfWithMeansLenghts['length']/1000
    return

def calculateStreamLengthRatio (gdfLinear):
    lengthGroupedByOrder = gdfLinear.set_index('Stream Order')['Stream Length Mean (km)']
    streamLengthRatio = (lengthGroupedByOrder / lengthGroupedByOrder.shift(+1))
    gdfLinear['Stream Length Ratio'] = gdfLinear['Stream Order'].map(streamLengthRatio)
    return

def calculateStreamLengthRatioMean (gdfLinear):
    streamLengthRatioMean = gdfLinear['Stream Length Ratio'].mean()
    gdfLinear.loc[gdfLinear.index[-1], 'Mean Stream Length Ratio'] = streamLengthRatioMean
    return

def calculateBifurcationRatio (gdfLinear):
    streamNumber = gdfLinear.set_index('Stream Order')['Stream Number']
    bifurcationRatio = streamNumber / streamNumber.shift(-1)
    gdfLinear['Bifurcation Ratio'] = gdfLinear['Stream Order'].map(bifurcationRatio)
    return

def calculateBifurcationRatioMean (gdfLinear):
    bifurcationRatioMean = gdfLinear['Bifurcation Ratio'].mean()
    gdfLinear.loc[gdfLinear.index[-1], 'Mean Bifurcation Ratio'] = bifurcationRatioMean
    return

def calculateRhoCoefficient (gdfLinear):
    Rlm = gdfLinear['Stream Length Ratio'].mean()
    Rbm = gdfLinear['Bifurcation Ratio'].mean()

    RhoCoefficient = Rlm/Rbm
    gdfLinear.loc[gdfLinear.index[-1], 'RHO Coefficient'] = RhoCoefficient
    return

def calculateSinuosityIndex(gdfStream,gdfLinear):
    if gdfStream.empty:
        gdfLinear['Main channel sinuosity index'] = None
        return

    maxOrder = gdfStream['order'].max()
    filterMaxOrder = gdfStream[gdfStream['order'] == maxOrder]

    firstPoint = filterMaxOrder.iloc[-1]['last']
    lastPoint = filterMaxOrder.iloc[0]['first']

    straightLine = QgsGeometry.fromPolylineXY([
        QgsPointXY(firstPoint[0], firstPoint[1]),
        QgsPointXY(lastPoint[0], lastPoint[1])
    ])

    straightLineLength = straightLine.length()/1000

    totalLength = gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Stream Length Total (km)'].values[0]
    sinuosityIndex = totalLength/straightLineLength
    gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Main channel sinuosity index'] = sinuosityIndex
    return

def createGdfShape(basin):
    basinGeom = basin.geometry()
    gdfShape = gpd.GeoDataFrame(geometry=[basinGeom])
    return gdfShape

def calculateAreaPerimeter(gdfShape):
    gdfShape['Area (km2)'] = [gdfShape.geometry.area/1000000]
    gdfShape['Perimeter (km)'] = [gdfShape.geometry.length/1000]
    return

def calculateFitnessRatio(gdfShape,gdfLinear):
    maxOrder = gdfLinear['Stream Order'].max()
    totalLength = gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Stream Length Total (km)'].values[0]
    fitnessRatio = (totalLength/gdfShape['Perimeter (km)'])
    gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Fitness Ratio (Rf)'] = fitnessRatio.iloc[0][0]
    return

def calculateBasinLength(gdfStream,gdfShape,basin,feedback):
    if gdfStream.empty:
        gdfShape['Basin Length (Lg) (km)'] = None
        return

    maxOrder = gdfStream['order'].max()
    filterMaxOrder = gdfStream[gdfStream['order'] == maxOrder]

    firstPoint = filterMaxOrder.iloc[-1]['last']
    lastPoint = filterMaxOrder.iloc[0]['first']

    dx = lastPoint[0] - firstPoint[0]
    dy = lastPoint[1] - firstPoint[1]

    minX, minY, maxX, maxY = gdfShape.total_bounds
    maxDimBasin = max(maxX - minX, maxY - minY)
    mult = maxDimBasin*10

    firstExtended = QgsPointXY(firstPoint[0] - dx * mult, firstPoint[1] - dy * mult)
    lastExtended = QgsPointXY(lastPoint[0] + dx * mult, lastPoint[1] + dy * mult)

    parallelLine = QgsGeometry.fromPolylineXY([firstExtended, lastExtended])
    intersection = parallelLine.intersection(QgsGeometry.fromWkt(gdfShape.boundary.iloc[0].wkt))

    if intersection.isEmpty():
        feedback.pushWarning('A parallel line between higher order channels does not intersect the perimeter of the basin of id'+str(basin.id())+', the drainage network has errors!')
        gdfShape['Basin Length (Lg) (km)'] = None
        return

    pointsIntersection = intersection.asMultiPoint()
    pointsOrdered = sorted(pointsIntersection, key=lambda p: (p.y(), p.x()))

    basinLengthLine = QgsGeometry.fromPolylineXY(pointsOrdered)
    basinLength = basinLengthLine.length()
    gdfShape['Basin Length (Lg) (km)'] = basinLength/1000
    return

def calculateWanderingRatio(gdfShape,gdfLinear):
    maxOrder = gdfLinear['Stream Order'].max()
    totalLength = gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Stream Length Total (km)'].values[0]
    wanderingRatio = (totalLength/gdfShape['Basin Length (Lg) (km)'])
    gdfLinear.loc[gdfLinear['Stream Order'] == maxOrder, 'Wandering Ratio (Rw)'] = wanderingRatio.iloc[0]
    return

def calculateStreamFrequency(gdfShape,gdfLinear):
    numStreams = gdfLinear['Stream Number'].sum()
    streamFrequency = gdfShape['Area (km2)']/numStreams
    gdfLinear.loc[gdfLinear.index[-1], 'Stream Frequency (Fs) (1/km2)'] = streamFrequency.iloc[0][0]
    return

def calculateDrainageDensity(gdfShape,gdfLinear):
    sumLengths = gdfLinear['Stream Length Total (km)'].sum()
    drainageDensity = sumLengths/gdfShape['Area (km2)']
    gdfLinear.loc[gdfLinear.index[-1], 'Drainage Density (Dd) (km/km2)'] = drainageDensity.iloc[0][0]
    return

def calculateDrainageTexture(gdfShape,gdfLinear):
    sumStreams = gdfLinear['Stream Number'].sum()
    perimeter = gdfShape['Perimeter (km)']
    drainageTexture = sumStreams/perimeter
    gdfLinear.loc[gdfLinear.index[-1], 'Drainage texture (Dt) (1/km)'] = drainageTexture.iloc[0][0]
    return

def calculateLengthOverlandFlow(gdfLinear):
    lengthOverlandFlow = 1/(2*gdfLinear['Drainage Density (Dd) (km/km2)'])
    gdfLinear['Length of overland flow (Lg) (km)'] = lengthOverlandFlow
    return

def calculateDrainageIntensity(gdfLinear):
    streamFrequency = gdfLinear['Stream Frequency (Fs) (1/km2)']
    drainageDensity = gdfLinear['Drainage Density (Dd) (km/km2)']
    drainageIntensity = streamFrequency/drainageDensity
    gdfLinear['Drainage intensity (Di) (1/km)'] = drainageIntensity
    return

def calculateInfiltrationNumber(gdfLinear):
    streamFrequency = gdfLinear['Stream Frequency (Fs) (1/km2)']
    drainageDensity = gdfLinear['Drainage Density (Dd) (km/km2)']
    drainageIntensity = streamFrequency*drainageDensity
    gdfLinear['Infiltration number (If) (km/km4)'] = drainageIntensity
    return

def calculateConstantChannel(gdfLinear):
    constantOfChannel = 1/gdfLinear['Drainage Density (Dd) (km/km2)']
    gdfLinear['Constant of channel maintenance (Ccm) (km2/km)'] = constantOfChannel
    return

def calculateCirculatoryRatio(gdfShape):
    area = gdfShape['Area (km2)']
    perimeter = gdfShape['Perimeter (km)']
    gdfShape['Circulatory Ratio (Rc)'] = 4*pi*area/(perimeter**2)
    return

def calculateElongationRatio(gdfShape):
    area = gdfShape['Area (km2)']
    length = gdfShape['Basin Length (Lg) (km)']
    gdfShape['Elongation Ratio (Re)'] = (2*(area/pi)**0.5)/length
    return

def calculateFormFactor(gdfShape):
    area = gdfShape['Area (km2)']
    length = gdfShape['Basin Length (Lg) (km)']
    gdfShape['Form Factor (Ff)'] = area/(length**2)
    return

def calculateLemniscateRatio(gdfShape):
    area = gdfShape['Area (km2)']
    length = gdfShape['Basin Length (Lg) (km)']
    gdfShape['Lemniscate Ratio (K)'] = (length**2)/(4*area)
    return

def calculateShapeIndex(gdfShape):
    area = gdfShape['Area (km2)']
    length = gdfShape['Basin Length (Lg) (km)']
    gdfShape['Shape Index (Sb)'] = (length**2)/area
    return

def calculateCompactnessCoefficient(gdfShape):
    area = gdfShape['Area (km2)']
    perimeter = gdfShape['Perimeter (km)']
    gdfShape['Compactness coefficient (Cc)'] = perimeter/(2*(pi*area)**0.5)
    return

def createGdfRelief():
    gdfRelief = gpd.GeoDataFrame(index=[0])
    return gdfRelief

def loadDEM(demLayer):
    DEMpath = demLayer.dataProvider().dataSourceUri().split('|')[0]
    ds = gdal.Open(DEMpath)
    band = ds.GetRasterBand(1)
    demArray = band.ReadAsArray()
    noData = band.GetNoDataValue() if band.GetNoDataValue() is not None else -9999
    gt = ds.GetGeoTransform()
    proj = ds.GetProjection()
    rows, cols = demArray.shape
    ds = None
    return demArray, noData, gt, proj, rows, cols

def calculateMinMaxMeanElevation(demArray,noData,gt,proj,rows,cols,basin,gdfRelief,feedback):
    basinGeom = basin.geometry()
    wkb = basinGeom.asWkb()
    ogrGeom = ogr.CreateGeometryFromWkb(wkb)

    rasterDrive = gdal.GetDriverByName('MEM')
    vectorDrive = ogr.GetDriverByName('Memory')
    vectorDriveSource    = vectorDrive.CreateDataSource('wrk')
    vectorLayer = vectorDriveSource.CreateLayer('lyr', None, ogr.wkbUnknown)
    featureDef   = vectorLayer.GetLayerDefn()
    ogrFeat  = ogr.Feature(featureDef)
    ogrFeat.SetGeometry(ogrGeom)
    vectorLayer.CreateFeature(ogrFeat)
    ogrFeat = None

    maskDS = rasterDrive.Create('', cols, rows, 1, gdal.GDT_Byte)
    maskDS.SetGeoTransform(gt)
    maskDS.SetProjection(proj)
    gdal.RasterizeLayer(maskDS, [1], vectorLayer, burn_values=[1])

    mask = maskDS.GetRasterBand(1).ReadAsArray()
    validMask = (mask == 1) & (demArray != noData)
    validDataInsideBasin = demArray[validMask].tolist()

    if not validDataInsideBasin:
        feedback.pushWarning('There is no valid raster data in the basin of id '+str(basin.id())+' and the calculation of some parameters for this basin may be compromised')
        gdfRelief['Minimum Elevation (m)'] = None
        gdfRelief['Maximum Elevation (m)'] = None
        gdfRelief['Mean Elevation (m)'] = None
        return

    minElev = min(validDataInsideBasin)
    maxElev = max(validDataInsideBasin)
    meanElev = ((sum(validDataInsideBasin))/(len(validDataInsideBasin)))

    gdfRelief['Minimum Elevation (m)'] = minElev
    gdfRelief['Maximum Elevation (m)'] = maxElev
    gdfRelief['Mean Elevation (m)'] = meanElev
    return

def calculateRelief(gdfRelief):
    minElevation = gdfRelief['Minimum Elevation (m)']
    maxElevation = gdfRelief['Maximum Elevation (m)']
    gdfRelief['Relief (Bh) (m)'] = maxElevation - minElevation
    return

def calculateReliefRatio(gdfRelief,gdfShape):
    relief = gdfRelief['Relief (Bh) (m)']
    reliefInKm = relief/1000
    length = gdfShape['Basin Length (Lg) (km)']
    gdfRelief['Relief Ratio (Rh)'] = reliefInKm/length
    return

def calculateRelativeRelief (gdfRelief,gdfShape):
    maxElev = gdfRelief['Maximum Elevation (m)']
    maxElevInKm = maxElev/1000
    perimeter = gdfShape['Perimeter (km)']
    gdfRelief['Relative Relief (Rhp)'] = maxElevInKm*100/perimeter
    return

def calculateRuggednessNumber (gdfRelief,gdfLinear):
    relief = gdfRelief['Relief (Bh) (m)']
    reliefInKm = relief/1000
    drainageDensity = gdfLinear['Drainage Density (Dd) (km/km2)'].iloc[-1]
    gdfRelief['Ruggedness number (Rn)'] = reliefInKm * drainageDensity
    return

def calculateDissectionIndex (gdfRelief):
    minElev = gdfRelief['Minimum Elevation (m)']
    maxElev = gdfRelief['Maximum Elevation (m)']
    gdfRelief['Dissection index (Di)'] = (maxElev - minElev)/maxElev
    return

def calculateGradientRatio(gdfStream,gdfLinear,dem,gdfRelief):
    if gdfStream.empty:
        gdfRelief['Gradient Ratio (Gr)'] = None
        return

    maxOrder = gdfStream['order'].max()
    filterMaxOrder = gdfStream[gdfStream['order'] == maxOrder]

    firstPoint = filterMaxOrder.iloc[-1]['last']
    lastPoint = filterMaxOrder.iloc[0]['first']
    
    firstPointQgs = QgsPointXY(firstPoint[0], firstPoint[1])
    lastPointQgs = QgsPointXY(lastPoint[0], lastPoint[1])

    identificatorFirst = dem.dataProvider().identify(firstPointQgs, QgsRaster.IdentifyFormatValue)
    identificatorLast = dem.dataProvider().identify(lastPointQgs, QgsRaster.IdentifyFormatValue)

    lowestPoint = identificatorFirst.results()[1]
    hightestPoint = identificatorLast.results()[1]

    if lowestPoint is None or hightestPoint is None:
        gdfRelief['Gradient Ratio (Gr)'] = None
        return

    maxOrder = gdfLinear['Stream Order'].max()
    filteredMaxOrder = gdfLinear[gdfLinear['Stream Order'] == maxOrder]
    
    ls = filteredMaxOrder['Stream Length Total (km)'].values
    gdfRelief['Gradient Ratio (Gr)'] = (hightestPoint - lowestPoint)/ls
    return

def createGdfConcatenated(gdfLinear,gdfShape,gdfRelief,basin):
    gdfLinear.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfLinear.columns]
    gdfShape.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfShape.columns]
    gdfRelief.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfRelief.columns]

    gdfLinearFloat = gdfLinear.astype(float)
    gdfReliefFloat = gdfRelief.astype(float)

    gdfLinearFloat.loc[-1] = gdfLinear.columns
    gdfLinearFloat.index = gdfLinearFloat.index + 1
    gdfLinearFloat.sort_index(inplace=True)
    gdfLinearFloat.loc[gdfLinearFloat.index[-1], 'Stream Order basin id ' + str(basin.id())] = 'General'
    gdfLinearFloat.columns = range(gdfLinearFloat.shape[1])

    gdfShape.drop(columns='geometry basin id ' + str(basin.id()),inplace=True)
    gdfShapeFloat = gdfShape.astype(float)
    gdfShapeFloat.loc[-1] = gdfShapeFloat.columns
    gdfShapeFloat.index = gdfShapeFloat.index + 1
    gdfShapeFloat.sort_index(inplace=True)
    gdfShapeFloat.columns = range(gdfShapeFloat.shape[1])

    gdfReliefFloat.loc[-1] = gdfReliefFloat.columns
    gdfReliefFloat.index = gdfReliefFloat.index + 1
    gdfReliefFloat.sort_index(inplace=True)
    gdfReliefFloat.columns = range(gdfReliefFloat.shape[1])
    
    gdfConcatenated = gpd.pd.concat([gdfLinearFloat,gdfShapeFloat,gdfReliefFloat])
    return gdfConcatenated

def formatGdfLinear(gdfLinear,basin):
    gdfLinear.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfLinear.columns]
    gdfLinearFloat = gdfLinear.astype(float)
    gdfLinearFloat.loc[-1] = gdfLinear.columns
    gdfLinearFloat.index = gdfLinearFloat.index + 1
    gdfLinearFloat.sort_index(inplace=True)
    gdfLinearFloat.loc[gdfLinearFloat.index[-1], 'Stream Order basin id ' + str(basin.id())] = 'General'
    gdfLinearFloat.columns = range(gdfLinearFloat.shape[1])
    return gdfLinearFloat

def formatGdfShape(gdfShape,basin):
    gdfShape.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfShape.columns]

    gdfShape.drop(columns='geometry basin id ' + str(basin.id()),inplace=True)
    gdfShapeFloat = gdfShape.astype(float)
    gdfShapeFloat.loc[-1] = gdfShapeFloat.columns
    gdfShapeFloat.index = gdfShapeFloat.index + 1
    gdfShapeFloat.sort_index(inplace=True)
    gdfShapeFloat.columns = range(gdfShapeFloat.shape[1])
    return gdfShapeFloat

def formatGdfRelief(gdfRelief,basin):
    gdfRelief.columns = [f'{col} basin id ' + str(basin.id()) for col in gdfRelief.columns]
    gdfReliefFloat = gdfRelief.astype(float)

    gdfReliefFloat.loc[-1] = gdfReliefFloat.columns
    gdfReliefFloat.index = gdfReliefFloat.index + 1
    gdfReliefFloat.sort_index(inplace=True)
    gdfReliefFloat.columns = range(gdfReliefFloat.shape[1])
    return gdfReliefFloat

def calculateMorphometrics(demArray,noData,gt,proj,rows,cols,drainageBasinLayer,streamLayer,demLayer,path,feedback):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    gdfConcatenateds = []

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        feedback.setProgressText('Basin id '+str(basin.id())+' processing starting...')
        streamsInside = getStreamsInsideBasin(streamLayer, basin, feedback)
        if feedback.isCanceled():
            return
        gdfStream = createGdfStream(streamsInside)
        obtainFirstAndLastPoint(gdfStream)
        createOrderColumn(gdfStream)
        fillOrder(gdfStream)
        mergeStreams(gdfStream)
        calculateStreamLength(gdfStream)
        if feedback.isCanceled():
            return
        gdfLinear = createGdfLinear(gdfStream)
        calculateStreamNumber(gdfStream,gdfLinear)
        calculateTotalStreamLength(gdfStream,gdfLinear)
        calculateMeanStreamLength(gdfStream,gdfLinear)
        calculateStreamLengthRatio(gdfLinear)
        calculateStreamLengthRatioMean(gdfLinear)
        calculateBifurcationRatio(gdfLinear)
        calculateBifurcationRatioMean(gdfLinear)
        calculateRhoCoefficient(gdfLinear)
        calculateSinuosityIndex(gdfStream,gdfLinear)
        if feedback.isCanceled():
            return
        gdfShape = createGdfShape(basin)
        calculateAreaPerimeter(gdfShape)
        calculateFitnessRatio(gdfShape,gdfLinear)
        calculateBasinLength(gdfStream,gdfShape,basin,feedback)
        calculateWanderingRatio(gdfShape,gdfLinear)
        calculateDrainageDensity(gdfShape,gdfLinear)
        calculateStreamFrequency(gdfShape,gdfLinear)
        calculateDrainageTexture(gdfShape,gdfLinear)
        calculateLengthOverlandFlow(gdfLinear)
        calculateConstantChannel(gdfLinear)
        calculateDrainageIntensity(gdfLinear)
        calculateInfiltrationNumber(gdfLinear)
        calculateCirculatoryRatio(gdfShape)
        calculateElongationRatio(gdfShape)
        calculateFormFactor(gdfShape)
        calculateLemniscateRatio(gdfShape)
        calculateShapeIndex(gdfShape)
        calculateCompactnessCoefficient(gdfShape)
        if feedback.isCanceled():
            return
        gdfRelief = createGdfRelief()
        calculateMinMaxMeanElevation(demArray,noData,gt,proj,rows,cols,basin,gdfRelief,feedback)
        calculateRelief(gdfRelief)
        calculateReliefRatio(gdfRelief,gdfShape)
        calculateRelativeRelief(gdfRelief,gdfShape)
        calculateRuggednessNumber(gdfRelief,gdfLinear)
        calculateDissectionIndex(gdfRelief)
        calculateGradientRatio(gdfStream,gdfLinear,demLayer,gdfRelief)
        if feedback.isCanceled():
            return
        gdfConcatenated = createGdfConcatenated(gdfLinear,gdfShape,gdfRelief,basin)
        gdfConcatenateds.append(gdfConcatenated)

        barProgress = int((idx + 1) * step)
        feedback.setProgress(barProgress)
        feedback.setProgressText('Basin id '+str(basin.id())+' processing completed')

    if feedback.isCanceled():
        return
    gdfFinal = gpd.GeoDataFrame(gpd.pd.concat(gdfConcatenateds, ignore_index=True))
    gdfFinal.to_csv(path, index=False, header=False)

    return

def calculateLinearParameters(drainageBasinLayer,streamLayer,path,feedback):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    gdfsLinear = []

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        feedback.setProgressText('Basin id '+str(basin.id())+' processing starting...')
        streamsInside = getStreamsInsideBasin(streamLayer, basin, feedback)
        if feedback.isCanceled():
            return
        gdfStream = createGdfStream(streamsInside)
        obtainFirstAndLastPoint(gdfStream)
        createOrderColumn(gdfStream)
        fillOrder(gdfStream)
        mergeStreams(gdfStream)
        calculateStreamLength(gdfStream)
        if feedback.isCanceled():
            return
        gdfLinear = createGdfLinear(gdfStream)
        calculateStreamNumber(gdfStream,gdfLinear)
        calculateTotalStreamLength(gdfStream,gdfLinear)
        calculateMeanStreamLength(gdfStream,gdfLinear)
        calculateStreamLengthRatio(gdfLinear)
        calculateStreamLengthRatioMean(gdfLinear)
        calculateBifurcationRatio(gdfLinear)
        calculateBifurcationRatioMean(gdfLinear)
        calculateRhoCoefficient(gdfLinear)
        calculateSinuosityIndex(gdfStream,gdfLinear)
        if feedback.isCanceled():
            return
        gdfShape = createGdfShape(basin)
        calculateAreaPerimeter(gdfShape)
        calculateFitnessRatio(gdfShape,gdfLinear)
        calculateBasinLength(gdfStream,gdfShape,basin,feedback)
        calculateWanderingRatio(gdfShape,gdfLinear)
        calculateDrainageDensity(gdfShape,gdfLinear)
        calculateStreamFrequency(gdfShape,gdfLinear)
        calculateDrainageTexture(gdfShape,gdfLinear)
        calculateLengthOverlandFlow(gdfLinear)
        calculateConstantChannel(gdfLinear)
        calculateDrainageIntensity(gdfLinear)
        calculateInfiltrationNumber(gdfLinear)
        if feedback.isCanceled():
            return
        gdfFormated = formatGdfLinear(gdfLinear,basin)
        gdfsLinear.append(gdfFormated)

        barProgress = int((idx + 1) * step)
        feedback.setProgress(barProgress)
        feedback.setProgressText('Basin id '+str(basin.id())+' processing completed')

    if feedback.isCanceled():
        return
    gdfFinal = gpd.GeoDataFrame(gpd.pd.concat(gdfsLinear, ignore_index=True))
    gdfFinal.to_csv(path, index=False, header=False)

    return

def calculateShapeParameters(drainageBasinLayer,streamLayer,path, feedback):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    gdfsShape = []

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        feedback.setProgressText('Basin id '+str(basin.id())+' processing starting...')
        streamsInside = getStreamsInsideBasin(streamLayer, basin, feedback)
        if feedback.isCanceled():
            return
        gdfStream = createGdfStream(streamsInside)
        obtainFirstAndLastPoint(gdfStream)
        createOrderColumn(gdfStream)
        fillOrder(gdfStream)
        mergeStreams(gdfStream)
        calculateStreamLength(gdfStream)
        if feedback.isCanceled():
            return
        gdfShape = createGdfShape(basin)
        calculateAreaPerimeter(gdfShape)
        calculateBasinLength(gdfStream,gdfShape,basin,feedback)
        calculateCirculatoryRatio(gdfShape)
        calculateElongationRatio(gdfShape)
        calculateFormFactor(gdfShape)
        calculateLemniscateRatio(gdfShape)
        calculateShapeIndex(gdfShape)
        calculateCompactnessCoefficient(gdfShape)
        if feedback.isCanceled():
            return
        gdfFormated = formatGdfShape(gdfShape,basin)
        gdfsShape.append(gdfFormated)

        barProgress = int((idx + 1) * step)
        feedback.setProgress(barProgress)
        feedback.setProgressText('Basin id '+str(basin.id())+' processing completed')

    if feedback.isCanceled():
        return
    gdfFinal = gpd.GeoDataFrame(gpd.pd.concat(gdfsShape, ignore_index=True))
    gdfFinal.to_csv(path, index=False, header=False)

    return

def calculateReliefParameters(demArray,noData,gt,proj,rows,cols,drainageBasinLayer,streamLayer,demLayer,path, feedback):
    feedback.setProgress(0)
    total = drainageBasinLayer.featureCount()
    step = 100.0 / total if total else 0

    gdfsRelief = []

    for idx, basin in enumerate(drainageBasinLayer.getFeatures()):
        feedback.setProgressText('Basin id '+str(basin.id())+' processing starting...')
        streamsInside = getStreamsInsideBasin(streamLayer, basin, feedback)
        if feedback.isCanceled():
            return
        gdfStream = createGdfStream(streamsInside)
        obtainFirstAndLastPoint(gdfStream)
        createOrderColumn(gdfStream)
        fillOrder(gdfStream)
        mergeStreams(gdfStream)
        calculateStreamLength(gdfStream)
        if feedback.isCanceled():
            return
        gdfLinear = createGdfLinear(gdfStream)
        calculateStreamNumber(gdfStream,gdfLinear)
        calculateTotalStreamLength(gdfStream,gdfLinear)
        if feedback.isCanceled():
            return
        gdfShape = createGdfShape(basin)
        calculateAreaPerimeter(gdfShape)
        calculateBasinLength(gdfStream,gdfShape,basin,feedback)
        calculateDrainageDensity(gdfShape,gdfLinear)
        if feedback.isCanceled():
            return
        gdfRelief = createGdfRelief()
        calculateMinMaxMeanElevation(demArray,noData,gt,proj,rows,cols,basin,gdfRelief,feedback)
        calculateRelief(gdfRelief)
        calculateReliefRatio(gdfRelief,gdfShape)
        calculateRelativeRelief(gdfRelief,gdfShape)
        calculateRuggednessNumber(gdfRelief,gdfLinear)
        calculateDissectionIndex(gdfRelief)
        calculateGradientRatio(gdfStream,gdfLinear,demLayer,gdfRelief)
        if feedback.isCanceled():
            return
        gdfFormated = formatGdfRelief(gdfRelief,basin)
        gdfsRelief.append(gdfFormated)

        barProgress = int((idx + 1) * step)
        feedback.setProgress(barProgress)
        feedback.setProgressText('Basin id '+str(basin.id())+' processing completed')

    if feedback.isCanceled():
        return    
    gdfFinal = gpd.GeoDataFrame(gpd.pd.concat(gdfsRelief, ignore_index=True))
    gdfFinal.to_csv(path, index=False, header=False)

    return

def runAllMorphometricParameters(drainageBasinLayer,streamLayer,demLayer,path,feedback):
    demArray,noData,gt,proj,rows,cols = loadDEM(demLayer)

    calculateMorphometrics(demArray,noData,gt,proj,rows,cols,drainageBasinLayer,streamLayer,demLayer,path,feedback)

def runReliefParameters(drainageBasinLayer,streamLayer,demLayer,path,feedback):
    demArray,noData,gt,proj,rows,cols = loadDEM(demLayer)

    calculateReliefParameters(demArray,noData,gt,proj,rows,cols,drainageBasinLayer,streamLayer,demLayer,path, feedback)
