# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DotMap
                                 A QGIS plugin
 Create density dot layer from polygons
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-06-04
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Jose Manuel Mira
        email                : josema.mira@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.core import *
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *

from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QApplication
from PyQt5.QtWidgets import *
from .resources import *
from .dot_map_dialog import DotMapDialog
import os.path
import random
import math

PLACEHOLDER_TEXT = "Select a layer"

class DotMap:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DotMap_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        self.dlg = DotMapDialog()
        self.actions = []
        self.menu = self.tr(u'&Dot Map')
        self.toolbar = self.iface.addToolBar(u'DotMap')
        self.toolbar.setObjectName(u'DotMap')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        return QCoreApplication.translate('DotMap', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)
        if whats_this is not None:
            action.setWhatsThis(whats_this)
        if add_to_toolbar:
            self.toolbar.addAction(action)
        if add_to_menu:
            self.iface.addPluginToVectorMenu(self.menu, action)

        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = ':/plugins/DotMap/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Dot density layer'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        for action in self.actions:
            self.iface.removePluginVectorMenu(self.tr(u'&Dot Map'), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def run(self):
        # clear form
        self.dlg.layerComboBox.clear()
        self.dlg.fieldComboBox.clear()
        self.dlg.minValBox.clear()
        self.dlg.maxValBox.clear()
        self.dlg.valForDot.clear()
        self.dlg.simMinValBox.clear()
        self.dlg.simMaxValBox.clear()
        self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)
        self.dlg.simulationButton.setEnabled(False)

        # placeholder
        self.dlg.layerComboBox.addItem("Select a layer")

        # list polygon layers
        layer_list = []
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                layer_list.append(layer.name())
                self.dlg.layerComboBox.addItem(layer.name())

        # connect layer change once (disconnect old if any)
        try:
            self.dlg.layerComboBox.activated.disconnect()
        except Exception:
            pass
        self.dlg.layerComboBox.activated.connect(self.onLayerChanged)

        if not layer_list:
            QMessageBox.about(self.iface.mainWindow(), 'Message', 'Polygon layers not found')
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)
        else:
            # ensure placeholder selected
            self.dlg.layerComboBox.setCurrentIndex(0)
            self.dlg.show()

        # dialog loop
        result = self.dlg.exec_()
        if result:
            dialog = QProgressDialog()
            dialog.setWindowTitle("Dot progress")
            bar = QProgressBar(dialog)
            bar.setRange(0, 100)
            bar.setValue(0)
            dialog.setBar(bar)
            dialog.setMinimumWidth(300)
            dialog.show()

            try:
                # guard: do nothing if placeholder selected (shouldn't happen because OK disabled)
                if self.dlg.layerComboBox.currentIndex() <= 0:
                    return

                name = self.dlg.layerComboBox.currentText()
                layer = QgsProject.instance().mapLayersByName(name)[0]
                crs = layer.crs().authid()

                divisor = float(self.dlg.valForDot.text())
                if divisor <= 0:
                    raise ValueError("Divisor must be > 0")

                count_features = max(1, layer.featureCount())  # avoid div by zero

                dotLyr = QgsVectorLayer('Point?crs=' + crs, f"{self.dlg.fieldComboBox.currentText()} (1 dot = {divisor:g})", "memory")
                vpr = dotLyr.dataProvider()
                symbol = QgsMarkerSymbol.createSimple({'name': 'circle', 'color': 'black', 'size': '1'})
                dotLyr.renderer().setSymbol(symbol)
                dotFeatures = []
                j = 0

                field_name = self.dlg.fieldComboBox.currentText()
                for feature in layer.getFeatures():
                    try:
                        pop_val = feature[field_name]
                        if pop_val is None:
                            continue
                        pop = float(pop_val)
                    except Exception:
                        continue

                    if pop > 0:
                        j += 1
                        percent = int((j / float(count_features)) * 100)
                        bar.setValue(percent)
                        QApplication.processEvents()

                        # number of dots for the feature
                        ratio = pop / divisor
                        base = int(ratio)
                        frac = ratio - base
                        total_dots = base + (1 if random.random() < frac else 0)
                        if total_dots <= 0:
                            continue

                        g = feature.geometry()
                        if not g.isMultipart():
                            rect = g.boundingBox()
                            minx, miny, maxx, maxy = rect.xMinimum(), rect.yMinimum(), rect.xMaximum(), rect.yMaximum()
                            found = 0
                            pts = []
                            while found < total_dots:
                                x = random.uniform(minx, maxx)
                                y = random.uniform(miny, maxy)
                                pnt = QgsPointXY(x, y)
                                if g.contains(QgsGeometry.fromPointXY(pnt)):
                                    pts.append(pnt)
                                    found += 1
                            if pts:
                                f = QgsFeature()
                                f.setGeometry(QgsGeometry.fromMultiPointXY(pts))
                                dotFeatures.append(f)
                        else:
                            # Multipolygon: split into parts, allocate dots by area
                            parts = []
                            try:
                                for part in g.constParts():
                                    parts.append(QgsGeometry(part.clone()))
                            except Exception:
                                for part in g.asGeometryCollection():
                                    parts.append(part)

                            if not parts:
                                continue

                            areas = [max(0.0, p.area()) for p in parts]
                            total_area = sum(areas)
                            if total_area <= 0:
                                continue

                            quotas = [ (total_dots * (a / total_area)) for a in areas ]
                            base_counts = [ int(math.floor(q)) for q in quotas ]
                            remainder = total_dots - sum(base_counts)
                            fracs = [ (q - b, idx) for idx, (q, b) in enumerate(zip(quotas, base_counts)) ]
                            fracs.sort(reverse=True)
                            for k in range(remainder):
                                base_counts[fracs[k][1]] += 1

                            pts_all = []
                            for part_geom, npts in zip(parts, base_counts):
                                if npts <= 0:
                                    continue
                                rect = part_geom.boundingBox()
                                minx, miny, maxx, maxy = rect.xMinimum(), rect.yMinimum(), rect.xMaximum(), rect.yMaximum()
                                found = 0
                                tries = 0
                                while found < npts:
                                    x = random.uniform(minx, maxx)
                                    y = random.uniform(miny, maxy)
                                    pnt = QgsPointXY(x, y)
                                    if part_geom.contains(QgsGeometry.fromPointXY(pnt)):
                                        pts_all.append(pnt)
                                        found += 1
                                    else:
                                        tries += 1
                                        if tries % 1000 == 0:
                                            QApplication.processEvents()
                            if pts_all:
                                f = QgsFeature()
                                f.setGeometry(QgsGeometry.fromMultiPointXY(pts_all))
                                dotFeatures.append(f)

                if dotFeatures:
                    vpr.addFeatures(dotFeatures)
                    dotLyr.updateExtents()
                    QgsProject.instance().addMapLayers([dotLyr])

                # clear
                self.dlg.layerComboBox.clear()
                self.dlg.fieldComboBox.clear()
                self.dlg.minValBox.clear()
                self.dlg.maxValBox.clear()
                self.dlg.valForDot.clear()

            except Exception as e:
                QMessageBox.about(self.iface.mainWindow(), 'Error', f'Invalid numeric input: {e}')

    def onLayerChanged(self, idx):
        # If placeholder, reset and do nothing
        if idx <= 0 or self.dlg.layerComboBox.currentText() == PLACEHOLDER_TEXT:
            self.dlg.fieldComboBox.clear()
            self.dlg.simulationButton.setEnabled(False)
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)
            return
        # otherwise continue plugin logic
        self.getNumFields()

    def getNumFields(self):
        # guard against placeholder
        if self.dlg.layerComboBox.currentIndex() <= 0:
            return
        self.dlg.fieldComboBox.clear()
        name = self.dlg.layerComboBox.currentText()
        layer = QgsProject.instance().mapLayersByName(name)[0]
        fields = layer.fields()
        for field in fields:
            try:
                if hasattr(field, "isNumeric") and field.isNumeric():
                    self.dlg.fieldComboBox.addItem(field.name())
                else:
                    if field.typeName() in ('Integer', 'Integer64', 'Real', 'Double', 'Float', 'Numeric', 'Decimal'):
                        self.dlg.fieldComboBox.addItem(field.name())
            except Exception:
                continue

        # connect stats and simulation (disconnect old to avoid duplicates)
        try:
            self.dlg.simulationButton.clicked.disconnect()
        except Exception:
            pass
        self.dlg.fieldComboBox.activated.connect(self.getStats)
        self.dlg.simulationButton.setEnabled(True)
        self.dlg.simulationButton.clicked.connect(self.simulation)

    def getStats(self):
        if self.dlg.layerComboBox.currentIndex() <= 0:
            return
        fieldSel = self.dlg.fieldComboBox.currentText()
        name = self.dlg.layerComboBox.currentText()
        layer = QgsProject.instance().mapLayersByName(name)[0]

        values = []
        for feature in layer.getFeatures():
            try:
                value = feature[fieldSel]
                if value is None:
                    continue
                v = float(value)
                if v > 0:
                    values.append(v)
            except Exception:
                continue

        if not values:
            self.dlg.minValBox.setText('0')
            self.dlg.maxValBox.setText('0')
            return

        maxVal = max(values)
        minVal = min(values)
        self.dlg.minValBox.setText(f"{minVal:g}")
        self.dlg.maxValBox.setText(f"{maxVal:g}")

    def simulation(self):
        try:
            divisor = float(self.dlg.valForDot.text())
            max_val = float(self.dlg.maxValBox.text())
            min_val = float(self.dlg.minValBox.text())

            if divisor <= 0 or divisor > max_val:
                QMessageBox.information(self.iface.mainWindow(), QCoreApplication.translate('ERROR', "ERROR"), "Divisor must be > 0 and ≤ max value")
                return

            simMaxVal = int(max_val / divisor)
            simMinVal = int(min_val / divisor)
            self.dlg.simMaxValBox.setText(str(simMaxVal))
            self.dlg.simMinValBox.setText(str(simMinVal))
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(True)
        except Exception:
            QMessageBox.information(self.iface.mainWindow(), QCoreApplication.translate('ERROR', "ERROR"), "Divisor vacío o no numérico, o mayor que el valor máximo")
