# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Dn2Nbr
                                 A QGIS plugin
 Convert Digital Number Landsat7 images to NBR
                              -------------------
        begin                : 2013-11-13
        copyright            : (C) 2013 by Daniel Sánchez Pillot Gutiérrez
        email                : dspillotg@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
# Initialize Qt resources from file resources.py
import resources
#Import raster calculations, dates and stuff
from qgis.analysis import QgsRasterCalculator, QgsRasterCalculatorEntry
import math
import time
import datetime
# Import the code for the dialog
from dn2nbrdialog import Dn2NbrDialog
import os.path
import qgis.core

def busca_capa(nombre_capa):

	for (nombre, capa) in QgsMapLayerRegistry.instance().mapLayers().iteritems():
		if capa.name() == nombre_capa:
			return capa

	return None

class RasterBanda:
	'Clase de base para las escenas'
	def __init__(self, raster, banda, mask, usaMask):
		self.banda = banda
		self.nombre = raster.name() + "@1"
		self.nomCorto = raster.name()
		self.raster = raster
		
		if usaMask:
			self.maskNombre = mask.name() + "@1"
			self.rasterMask = mask
			self.usaMask = usaMask
		else:
			self.usaMask = usaMask
	
	#Función para buscar valores en un archivo MTL de Landsat7
	def obtenValor(self, texto_buscado, texto_busqueda):
		#Buscamos el texto indicado y obtenemos el valor buscando el salto de linea 
		pos_texto = texto_busqueda.find(texto_buscado)
		
		if pos_texto != -1:
			pos_dato = pos_texto + len(texto_buscado) + 3
			cont = 1
			while(texto_busqueda.endswith('\n',pos_dato,pos_dato + cont) == False):
				valor = texto_busqueda[pos_dato:pos_dato + cont]
				cont = cont + 1
		else:
			#Regresa vacío
			valor = ""
		
		#Regresa el valor encontrado
		return valor
	
	#Función para calcular el día Juliano
	def dia_juliano(self, AA, MM, DD, HR, Min, Sec, UTcor): 
		
		fecha = datetime.date(AA, MM, DD)
		tupla_fecha = fecha.timetuple()
		dia = time.strftime("%j", tupla_fecha)
		
		return dia
	
	#Función para pasar una imagen de niveles digitales a niveles de radiación utilizando mapcalc	
	def dig2Rad(self, ruta_header, raster_entrada, raster_salida):
		
		#Leemos el header
		with open(ruta_header, 'r') as header:
			texto_header = header.read()
		header.close()
		
		#Obtenemos los valores necesarios con el método obtenValor
		#cambiando el texto de búsquda según el tipo de archivo
		lmax = self.obtenValor('RADIANCE_MAXIMUM_BAND_' + self.banda, texto_header)
		if lmax != "":
			lmin = self.obtenValor('RADIANCE_MINIMUM_BAND_' + self.banda, texto_header)
			qcalmax = self.obtenValor('QUANTIZE_CAL_MAX_BAND_' + self.banda, texto_header)
			qcalmin = self.obtenValor('QUANTIZE_CAL_MIN_BAND_' + self.banda, texto_header)
		else:
			lmax = self.obtenValor('LMAX_BAND' + self.banda, texto_header)
			lmin = self.obtenValor('LMIN_BAND' + self.banda, texto_header)
			qcalmax = self.obtenValor('QCALMAX_BAND' + self.banda, texto_header)
			qcalmin = self.obtenValor('QCALMIN_BAND' + self.banda, texto_header)
		
		#Redefinimos el nombre del raster de salida incluyendo la ruta
		ruta = os.path.dirname(unicode(self.raster.dataProvider().dataSourceUri()))
		raster_salida = ruta + os.path.sep + raster_salida + ".tif"
		
		#Asigna rasters al vector de la calculadora
		raster = []
		
		#raster 01
		raster_calc_01 = QgsRasterCalculatorEntry()
		raster_calc_01.ref = raster_entrada
		raster_calc_01.raster = self.raster
		raster_calc_01.bandNumber = 1
		raster.append(raster_calc_01)
		
		#Genera la fórmula que usaremos en el mapcalc dependiendo de la máscara
		if self.usaMask:
			
			#raster 02
			raster_calc_02 = QgsRasterCalculatorEntry()
			raster_calc_02.ref = self.maskNombre
			raster_calc_02.raster = self.rasterMask
			raster_calc_02.bandNumber = 1
			raster.append(raster_calc_02)
			
			formula_dig2rad = "((((" + lmax + "-(" + lmin + "))/(" + qcalmax + "-(" + qcalmin + ")))*(" + raster_entrada + "-(" + qcalmin + ")))+(" + lmin + "))*(" + self.maskNombre + " > 0)"
			
		else:
			formula_dig2rad = "((((" + lmax + "-(" + lmin + "))/(" + qcalmax + "-(" + qcalmin + ")))*(" + raster_entrada + "-(" + qcalmin + ")))+(" + lmin + "))"
			
		#Define parámetros y realiza el cálculo
		calc = QgsRasterCalculator( formula_dig2rad, raster_salida, "GTiff", self.raster.extent(), self.raster.width(), self.raster.height(), raster )
		calc.processCalculation()

		#Carga la capa recién creada
		infoRaster = QFileInfo(raster_salida)
		nombreBase = infoRaster.baseName()
		self.rasterRad = QgsRasterLayer(raster_salida, nombreBase)
		#Y agrégala al registro para que se refleje en el listado de capas
		QgsMapLayerRegistry.instance().addMapLayer(self.rasterRad)
		
		#Guarda el nombre para el futuro
		self.rasterRadNombre = nombreBase
		
	#Función para pasar de niveles de radiación a reflectancia utilizando mapcalc
	def rad2Ref(self, ruta_header, raster_entrada, raster_salida):
		#Leemos el header
		with open(ruta_header, 'r') as header:
			texto_header = header.read()
		header.close()
		
		#Obtenemos los valores necesarios
		#Obtenemos fecha y hora de toma y convertimos a enteros par obtener el día Juliano
		fecha_toma = self.obtenValor('DATE_ACQUIRED', texto_header)
		if fecha_toma != "":
			hora_toma = self.obtenValor('SCENE_CENTER_TIME', texto_header)
		else:
			fecha_toma = self.obtenValor('ACQUISITION_DATE', texto_header)
			hora_toma = self.obtenValor('SCENE_CENTER_SCAN_TIME', texto_header)
			
		(anio, mes, dia) = fecha_toma.split('-')
		anio = int(anio)
		mes = int(mes)
		dia = int(dia)
		
		(hora, min, seg) = hora_toma.split(':')
		hora = int(hora)
		min = int(min)
		seg = float(seg[0:6])
		
		dia_jul = int(self.dia_juliano(anio, mes, dia, hora, min, seg, 0))
		
		#Calculamos la distancia del Sol a la Tierra en el día Juliano obtenido
		d = (1 - 0.01672 * math.cos(math.radians(0.9856 * (dia_jul - 4))))
		
		#Obtenemos la ESUNi (Radiación solar exoatmosférica)
		if self.banda == '1':
			ESUNi = 1997
		elif self.banda == '2':
			ESUNi = 1812
		elif self.banda == '3':
			ESUNi = 1533
		elif self.banda == '4':
			ESUNi = 1039
		elif self.banda == '5':
			ESUNi = 230.8
		elif self.banda == '7':
			ESUNi = 84.90
		elif self.banda == '8':
			ESUNi = 1362
		
		#Calculamos el ángulo solar del cénit
		z = 90 - float(self.obtenValor('SUN_ELEVATION', texto_header))
		
		#Genera la fórmula que usaremos en el mapcalc
		formula_rad2ref = '(' + str(math.pi) + '*' + raster_entrada + '*(' + str(d) + '^2))/(' + str(ESUNi) + '*cos(' + str(z) + '))'
		
		#Asigna rasters al vector de la calculadora
		raster = []
		
		#raster 01
		raster_calc_01 = QgsRasterCalculatorEntry()
		raster_calc_01.ref = raster_entrada
		raster_calc_01.raster = self.rasterRad
		raster_calc_01.bandNumber = 1
		raster.append(raster_calc_01)

		#Redefinimos el nombre del raster de salida incluyendo la ruta
		ruta = os.path.dirname(unicode(self.raster.dataProvider().dataSourceUri()))
		raster_salida = ruta + os.path.sep + raster_salida + ".tif"
				
		#Define parámetros y realiza el cálculo
		calc = QgsRasterCalculator( formula_rad2ref, raster_salida, "GTiff", self.raster.extent(), self.raster.width(), self.raster.height(), raster )
		calc.processCalculation()
		
		#Carga la capa recién creada
		infoRaster = QFileInfo(raster_salida)
		nombreBase = infoRaster.baseName()
		self.rasterRef = QgsRasterLayer(raster_salida, nombreBase)
		#Y agrégala al registro para que se refleje en el listado de capas
		QgsMapLayerRegistry.instance().addMapLayer(self.rasterRef)
		
		#Guarda el nombre para el futuro
		self.rasterRefNombre = nombreBase
		
	#Función para pasar de niveles digitales a reflectancia utilizando mapcalc
	def dig2Ref(self, ruta_header, nombre_rad, nombre_ref):
	
		#Llama el procedimiento dig2Rad para pasar de niveles digitales a radiación de la banda apropiada
		self.dig2Rad(ruta_header, self.nombre, nombre_rad)

		#Llama el procedimiento rad2Ref para pasar de niveles de radiación a reflectancia  de la banda apropiada
		self.rad2Ref(ruta_header, nombre_rad + '@1', nombre_ref)

class Dn2Nbr:

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
		# initialize plugin directory
		self.plugin_dir = os.path.dirname(__file__)
		# initialize locale
		locale = QSettings().value("locale/userLocale")[0:2]
		localePath = os.path.join(self.plugin_dir, 'i18n', 'dn2nbr_{}.qm'.format(locale))
		
		if os.path.exists(localePath):
			self.translator = QTranslator()
			self.translator.load(localePath)

			if qVersion() > '4.3.3':
				QCoreApplication.installTranslator(self.translator)
		
		# Create the dialog (after translation) and keep reference
		self.dlg = Dn2NbrDialog()
		
		#Connect browse buttons
		QObject.connect(self.dlg.ui.pb_name_nbr, SIGNAL("clicked()"), self.select_nbr)
		QObject.connect(self.dlg.ui.pb_path_mtl, SIGNAL("clicked()"), self.select_mtl)
		
		#Connect checkboxes
		QObject.connect(self.dlg.ui.chb_mask_b4, SIGNAL("toggled(bool)"), self.use_mask_b4)
		QObject.connect(self.dlg.ui.chb_mask_b7, SIGNAL("toggled(bool)"), self.use_mask_b7)
		
	def use_mask_b4(self):
		self.dlg.ui.cb_mask_b4.setEnabled(not self.dlg.ui.chb_mask_b4.isChecked())
	
	def use_mask_b7(self):
		self.dlg.ui.cb_mask_b7.setEnabled(not self.dlg.ui.chb_mask_b7.isChecked())
	
	def select_mtl(self):
		archivo_mtl = QFileDialog.getOpenFileName(None, "Select MTL file", "", "MTL text File (*.txt)")
		if archivo_mtl != None: self.dlg.ui.ln_path_mtl.setText(archivo_mtl)
		
	def select_nbr(self):
		archivo_nbr = QFileDialog.getSaveFileName(None, "Save output GeoTIFF file", "", "GeoTIFF (*.tif)")
		if archivo_nbr != None: self.dlg.ui.ln_name_nbr.setText(archivo_nbr)
		
	def initGui(self):
		# Create action that will start plugin configuration
		self.action = QAction(
				QIcon(":/plugins/dn2nbr/icon.png"),
				u"DN2NBR", self.iface.mainWindow())
		# connect the action to the run method
		self.action.triggered.connect(self.run)

		# Add toolbar button and menu item
		self.iface.addToolBarIcon(self.action)
		self.iface.addPluginToMenu(u"&Digital Number 2 NBR", self.action)

	def unload(self):
		# Remove the plugin menu item and icon
		self.iface.removePluginMenu(u"&Digital Number 2 NBR", self.action)
		self.iface.removeToolBarIcon(self.action)

	# run method that performs all the real work
	def run(self):
		
		#Nos conectamos al map Canvas para obtener el listado de capas
		canvas = qgis.utils.iface.mapCanvas()
		allLayers = canvas.layers()
			
		#Agregamos listado de capas a los comboboxes
		self.dlg.ui.cb_raster_b4.clear()
		self.dlg.ui.cb_raster_b7.clear()
		self.dlg.ui.cb_mask_b4.clear()
		self.dlg.ui.cb_mask_b7.clear()
		
		for i in allLayers:
			self.dlg.ui.cb_raster_b4.addItem(i.name())
			self.dlg.ui.cb_raster_b7.addItem(i.name())
			self.dlg.ui.cb_mask_b4.addItem(i.name())
			self.dlg.ui.cb_mask_b7.addItem(i.name())

		# show the dialog
		self.dlg.show()
		# Run the dialog event loop
		result = self.dlg.exec_()
		# See if OK was pressed
		if result == 1:
			
			#Obtenemos los valores del diálogo para el procesamiento
			raster_b4 = busca_capa(unicode(self.dlg.ui.cb_raster_b4.currentText()))
			raster_b7 = busca_capa(unicode(self.dlg.ui.cb_raster_b7.currentText()))
			mask_b4 = busca_capa(unicode(self.dlg.ui.cb_mask_b4.currentText()))
			mask_b7 = busca_capa(unicode(self.dlg.ui.cb_mask_b7.currentText()))
			ruta_mtl = str(self.dlg.ui.ln_path_mtl.text())
			nombre_nbr = unicode(self.dlg.ui.ln_name_nbr.text())
			
			tiene_mask_b4 = (not self.dlg.ui.chb_mask_b4.isChecked())
			tiene_mask_b7 = (not self.dlg.ui.chb_mask_b7.isChecked())
			
			#Instancia banda 4
			banda4 = RasterBanda(raster_b4,'4',mask_b4, tiene_mask_b4)
					
			#Llama el procedimiento dig2Ref para pasar de niveles digitales a reflectancia de la banda 4
			banda4.dig2Ref(ruta_mtl, banda4.nomCorto + '_RAD', banda4.nomCorto + '_REF')
			
			#Instancia banda 7 
			banda7 = RasterBanda(raster_b7,'7',mask_b7, tiene_mask_b7)
			
			#Llama el procedimiento dig2Ref para pasar de niveles digitales a reflectancia de la banda 7
			banda7.dig2Ref(ruta_mtl, banda7.nomCorto + '_RAD', banda7.nomCorto + '_REF')
			
			#Prepara datos para NBR
			raster_nbr = []			
			
			#Banda 4
			raster_calc_b4 = QgsRasterCalculatorEntry()
			raster_calc_b4.ref = banda4.rasterRefNombre + '@1'
			raster_calc_b4.raster = banda4.rasterRef
			raster_calc_b4.bandNumber = 1
			raster_nbr.append(raster_calc_b4)
			
			#Banda 7
			raster_calc_b7 = QgsRasterCalculatorEntry()
			raster_calc_b7.ref = banda7.rasterRefNombre + '@1'
			raster_calc_b7.raster = banda7.rasterRef
			raster_calc_b7.bandNumber = 1
			raster_nbr.append(raster_calc_b7)
			
			
			#Agrega máscaras según las opciones marcadas por el usuario
			
			if tiene_mask_b4:
				#Máscara Banda 4
				raster_calc_maskb4 = QgsRasterCalculatorEntry()
				raster_calc_maskb4.ref = banda4.maskNombre
				raster_calc_maskb4.raster = banda4.rasterMask
				raster_calc_maskb4.bandNumber = 1
				raster_nbr.append(raster_calc_maskb4)
			
			if tiene_mask_b7:
				#Máscara Banda 7
				raster_calc_maskb7 = QgsRasterCalculatorEntry()
				raster_calc_maskb7.ref = banda7.maskNombre
				raster_calc_maskb7.raster = banda7.rasterMask
				raster_calc_maskb7.bandNumber = 1
				raster_nbr.append(raster_calc_maskb7)
			
			
			#Prepara fórmula NBR de acuerdo a las máscaras aplicadas
			formula_nbr = "((" + raster_calc_b4.ref + " - " + raster_calc_b7.ref + ") / (" + raster_calc_b4.ref + " + " + raster_calc_b7.ref + "))"
			
			if tiene_mask_b4 and tiene_mask_b7:
				formula_nbr += " * ((" + raster_calc_maskb4.ref + " * " + raster_calc_maskb7.ref + ") > 0)"
			elif tiene_mask_b4 and not tiene_mask_b7:
				formula_nbr += " * (" + raster_calc_maskb4.ref + " > 0)"
			elif not tiene_mask_b4 and tiene_mask_b7:
				formula_nbr += " * (" + raster_calc_maskb7.ref + " > 0)"
			
			calc = QgsRasterCalculator( formula_nbr, nombre_nbr, "GTiff", banda4.raster.extent(), banda4.raster.width(), banda4.raster.height(), raster_nbr )			
			calc.processCalculation()

			#Carga la capa recién creada
			infoRaster = QFileInfo(nombre_nbr)
			nombreBase = infoRaster.baseName()
			rasterNbr = QgsRasterLayer(nombre_nbr, nombreBase)
			
			#Y agrégala al registro para que se refleje en el listado de capas
			QgsMapLayerRegistry.instance().addMapLayer(rasterNbr)
			
			QMessageBox.information(self.iface.mainWindow(),"Success!", "Raster image " + nombre_nbr + " created successfully")