# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DissolveWithStats
                                 A QGIS plugin
 Group entities with same value for one field, calculate statistics on the other fields
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-02-19
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Julie Pierson, UMR 5319 Passages, CNRS
        email                : julie.pierson@cnrs.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction
#from PyQt5 import QtCore
from qgis.core import QgsMessageLog, QgsVectorLayer, QgsVectorDataProvider, NULL
import os
import processing
from statistics import median, stdev

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .dissolve_stats_dialog import DissolveWithStatsDialog


class DissolveWithStats:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DissolveWithStats_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Dissolve with stats')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('DissolveWithStats', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/dissolve_stats/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Dissolve with stats'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Dissolve with stats'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = DissolveWithStatsDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # get selected layer in combo box comboLayerList
            selectedLayer = self.dlg.mMapLayerComboBox.currentLayer()
            # get selected field in combo box comboFieldList
            selectedField = self.dlg.mFieldComboBox.currentField()
            # get fields to keep and stats to calculate
            listKeep = []
            listStats = []
            for row in range(self.dlg.tableWidget.rowCount()):
                listKeep.append(self.dlg.tableWidget.cellWidget(row, 2).checkState())
                listStats.append(self.dlg.tableWidget.cellWidget(row,3).currentText())
            # get output path and name
            outFile = self.dlg.outLayerName.text()
            # run qgis:dissolve algorithm from processing module ('OUTPUT':'memory' for temporary layer)
            processing.run("native:dissolve", {'INPUT':selectedLayer,'FIELD':[selectedField],'OUTPUT':outFile})
            # calculate new field values with calculateFields function
            dicNewValues = self.calculateFields(selectedLayer, selectedField, listKeep, listStats, outFile)
            # integrates these new values in the output attribute table, and remove fields if necessary with setAttributes function
            self.setAttributes(selectedLayer, listKeep, outFile, dicNewValues, selectedField)
            # add layer to map with addFile function
            self.addFile(outFile)


    # once the dissolve output layer is created, calculates its new attributes values
    def calculateFields(self, selectedLayer, selectedField, listKeep, listStats, outFile):
        # iterates over features to get attributes as list of lists : [['33','Aquitaine'],['40','Aquitaine'],['44','Pays de la Loire']]
        features = selectedLayer.getFeatures()
        attrs = [feature.attributes() for feature in features]
        # get list of values for dissolve field, i.e. ['Aquitaine', 'Aquitaine', 'Pays de la Loire']
        indexDissolveField = selectedLayer.fields().indexFromName(selectedField)
        valuesDissolveField = [feature[indexDissolveField] for feature in attrs]
        # get list of fields names
        listFieldNames = selectedLayer.fields().names()
        #listFieldNames = [field for field in listFieldNames if field != 'fid']
        # trick for dissolve field
        listStats[indexDissolveField] = 'First'
        # create empty dictionary, keys are dissolve field values, values are empty dictionaries (dict of dict)
        dicNewValues = dict((k, {}) for k in valuesDissolveField)
        # values are now dictionaries with kept field for keys and empty lists for values
        for key in dicNewValues.keys():
            dicNewValues[key] = dict((k, []) for k in listFieldNames)
        # for each kept field
        for i,field in enumerate(listFieldNames):
            if listKeep[i] == 2:
                # get field values, i.e. ['44','33','40']
                valuesField = [feature[i] for feature in attrs]
                # populates dicNewValues -> {'Aquitaine':{'CODE_DEPT':['33','40'], 'NOM_DEPT':['Gironde','Landes']}, 'Pays de la Loire':{...}}
                for x,y in zip(valuesField, valuesDissolveField):
                    dicNewValues[y][field].append(x)
                # calculates aggregation values for each new feature, i.e. for 'count' : [2,1]
                for k in dicNewValues.keys():
                    newValues = self.aggregateValues(listStats[i], dicNewValues[k][field])
                    dicNewValues[k][field] = newValues
        return dicNewValues
    
    # takes function name (count, mean, max...) and list of values as input, output one value as result
    def aggregateValues(self, stat, l):
        switcher = {
                # by alphabetical order
                "Concatenation": lambda l: ",".join(l) if l else NULL,
                "Count": lambda l: len(l) if l else NULL,
                "First": lambda l: l[0] if l else NULL,
                "Last": lambda l: l[-1] if l else NULL,
                "Max": lambda l: max(l) if l else NULL,
                "Mean": lambda l: sum(l)/len(l) if l else NULL,
                "Median": lambda l: median(l) if l else NULL,
                "Min": lambda l: min(l) if l else NULL,
                "Standard deviation": lambda l: stdev(l) if l else NULL,
                "Sum": lambda l: sum(l) if l else NULL,
                "Uniquification": lambda l: ",".join(set(l)) if l else NULL
                }
        return switcher[stat](l)
    
    # removes fields from the output which mustn't be kept, and set the other field values
    def setAttributes(self, selectedLayer, listKeep, outFile, dicNewValues, dissolveField):
        # get indexes of fields to be deleted
        listIndexesDel = [i for i in range(len(listKeep)) if listKeep[i] == 0]
        # get layer, provider and provider capabilities
        outputLayer = QgsVectorLayer(outFile, "name", "ogr")
        provider = outputLayer.dataProvider()
        caps = provider.capabilities()
        # delete fields to be deleted
        if caps & QgsVectorDataProvider.DeleteAttributes:
            provider.deleteAttributes(listIndexesDel)
            outputLayer.updateFields()
        # get remaining fields names
        fields = outputLayer.fields().names()
        # get input layer fields names
        inputFields = selectedLayer.fields().names()
        # for each remaining field
        outputLayer.startEditing()
        for field in fields:
            # check if field was present in input layer (useful if new field was created automatically like fid for gpkg)
            if field in inputFields:
                # for each feature
                iterator = outputLayer.getFeatures()
                for feature in iterator:
                    # set feature attribute for this field
                    dissolveValue = feature[dissolveField]
                    feature[field] = dicNewValues[dissolveValue][field]
                    outputLayer.updateFeature(feature)
        outputLayer.commitChanges()
    
    # add output layer to the map
    def addFile(self, output):
        # get file name (for ex. 'result') from complete path (for ex. '/mnt/temp/result.shp')
        filename = output.split('/')[-1]
        filename = filename.split('.')[0]
        # load layer in QGIS
        vlayer = self.iface.addVectorLayer(output, filename, "ogr")
        if not vlayer:
            QgsMessageLog.logMessage('Layer failed to load')
        return
