# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ImportDialog
                                 A QGIS plugin
 -This plugin is designed for Dameng Database. It can be used to connect to 
Dameng Database and import the selected spatial data tables into QGIS (Quantum 
Geographic Information System) for display as temporary layers. Additionally, the 
plugin allows setting filter conditions for the data in the selected spatial data 
tables, such as SRID (Spatial Reference Identifier), data type, and ROWNUM.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-07-16
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Dameng
        email                : yangzhenyu@dameng.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import asyncio
import dmPython
import uuid
import numpy as np
import re

from qgis.PyQt.QtGui import QRegExpValidator
from PyQt5.QtCore import QVariant, QRegExp ,QTime, QDate, QDateTime
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtCore import pyqtSignal, Qt, QCoreApplication, QThread, pyqtSlot
from qgis.utils import iface
from PyQt5.QtWidgets import QPushButton, QComboBox, QLineEdit, QMessageBox, QProgressBar, QLabel, QVBoxLayout, QWidget
from qgis.core import (
    QgsVectorLayer, QgsGeometry, QgsFeature, QgsProject,
    QgsFields, QgsField, Qgis
)
from .comboFilter import PopupFilter
from .importWorker import ImportWorker
from decimal import Decimal
from datetime import datetime, date, time
# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'import.ui'))


class ImportDialog(QtWidgets.QDialog, FORM_CLASS):
    closingPlugin = pyqtSignal()
    loop: asyncio.AbstractEventLoop

    BackButton: QPushButton

    databaseBox: QComboBox
    tableBox: QComboBox
    columnBox: QComboBox
    operateBox: QComboBox
    typeBox: QComboBox
    limitEdit: QLineEdit
    limitMinEdit: QLineEdit
    limitMaxEdit: QLineEdit
    addLayerButton: QPushButton
    layernameEdit: QLineEdit
    sridEdit: QLineEdit
    progress_bar: QProgressBar
    worker_thread: QThread
    container: QWidget
    
    conn: object
    schema_name: str
    table_name: str
    column_name: str
    operator: str
    type_name: str
    wkt: str
    limit: int
    limit_min: int
    limit_max: int
    layer_name: str
    srid: int
    cursor: object

    def __init__(self, plugin, db_connection):
        """Constructor."""
        super().__init__(plugin.iface.mainWindow())
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.conn = db_connection
        self.plugin = plugin
        self.setupUi(self)
        self.init_layout()
        self.init_schema_box()
        self.progress_bar = None
        self.worker_thread = None
        self.container = None
        # create cursor
        self.cursor = self.conn.cursor()


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ImportDialog', message)


    #Init import ui layout:
    #AddLayer bottun: Add layer to QGIS
    #Back bottun: Back to connect layout
    #database Box: Choose schema box
    #table Box: Choose table box
    #column Box: Choose column box
    #operate Box: Choose operate box
    #type Box: Choose type box
    def init_layout(self):
        # Associate buttons, drop-down lists with functions, and trigger the functions after operation
        self.databaseBox.activated[str].connect(self.schema_box_change)
        self.tableBox.activated[str].connect(self.table_box_change)
        self.columnBox.activated[str].connect(self.column_box_change)
        self.operateBox.activated[str].connect(self.operate_box_change)
        self.typeBox.activated[str].connect(self.type_box_change)
        self.addLayerButton.clicked.connect(self.add_layer_button_clicked)
        self.BackButton.clicked.connect(self.back_button_clicked)
        # Regular Expression: Only allow input of numbers (integers)
        # ^[0-9]+$ indicates that from the start to the end of the input, there must be one or more digits.
        int_validator = QRegExpValidator(QRegExp("^[0-9]+$"), self.limitEdit)
        self.limitEdit.setValidator(int_validator)
        int_validator = QRegExpValidator(QRegExp("^[0-9]+$"), self.limitMinEdit)
        self.limitMinEdit.setValidator(int_validator)
        int_validator = QRegExpValidator(QRegExp("^[0-9]+$"), self.limitMaxEdit)
        self.limitMaxEdit.setValidator(int_validator)
        int_validator = QRegExpValidator(QRegExp("^[0-9]+$"), self.sridEdit)
        self.sridEdit.setValidator(int_validator)

        self.databaseBox.setEnabled(False)
        self.databaseBox.clear()
        self.tableBox.setEnabled(False)
        self.tableBox.clear()
        self.columnBox.setEnabled(False)
        self.columnBox.clear()
        self.operateBox.setEnabled(False)
        self.operateBox.clear()
        self.operateBox.addItems(["ALL","=", ">", "<", ">=", "<=", ">=v1 and <=v2"])
        self.typeBox.setEnabled(False)
        self.typeBox.clear()
        self.typeBox.addItems(["Point", "LineString", "Polygon", "MultiPoint", "MultiLineString", "MultiPolygon"])
        self.addLayerButton.setEnabled(False)

        self.reset_operateBox()
        self.reset_typeBox()
        self.reset_limitEdit()
        self.reset_sridEdit()


    # Reset the operateBox: it should be unselectable and display "ALL" by default.
    def reset_operateBox(self):
        self.operateBox.setEnabled(False)
        self.operateBox.setCurrentIndex(0)


    # Reset the typeBox: it should be unselectable.
    def reset_typeBox(self):
        self.typeBox.setEnabled(False)
        self.typeBox.setCurrentIndex(-1) 


    # Reset the limitEdit: limitEdit should be unselectable and visible, while limitMinEdit and limitMaxEdit should be unselectable and invisible.
    def reset_limitEdit(self):
        self.limitEdit.setEnabled(False)
        self.limitEdit.clear()
        self.limitEdit.show()
        self.limitMinEdit.setEnabled(False)
        self.limitMinEdit.clear()
        self.limitMinEdit.hide()
        self.limitMaxEdit.setEnabled(False)
        self.limitMaxEdit.clear()
        self.limitMaxEdit.hide()


    # Reset the sridEdit to be unselectable.
    def reset_sridEdit(self):
        self.sridEdit.setEnabled(False)
        self.sridEdit.clear()
        self.sridEdit.show()


    def back_button_clicked(self):
        self.hide()
        self.plugin.show_connect_page()

    
    def is_valid_identifier(self, name: str) -> bool:
        if name.startswith('"') and name.endswith('"'):
            inner = name[1:-1]
            if re.search(r'(?<!")"(?!")', inner):
                return False
            if re.search(r"[';--/*\\]", inner):
                return False
            if len(inner) < 1 or len(inner) > 126:
                return False
            return True
        else:
            if not re.match(r'^[a-zA-Z_$#][a-zA-Z0-9_$#]*$', name):
                return False
            if len(name) < 1 or len(name) > 128:
                return False
            return True

    
    def escape_identifier(self, name: str) -> str:
        if name.startswith('"') and name.endswith('"'):
            inner = name[1:-1]
            escaped_name = inner.replace('"', '""')
            return f'"{escaped_name}"'
        else:
            escaped_name = name.replace('"', '""')
            return f'"{escaped_name}"'


    # Initialize the content of the schema box, including all items accessible to the current user.
    def init_schema_box(self):
        # create cursor
        cursor = self.conn.cursor()
        try:
            # Construct a query SQL
            # Query all schemas currently accessible
            #sql = """ SELECT USERNAME FROM ALL_USERS ORDER BY USERNAME;"""
            sql = """SELECT * FROM SYSOBJECTS where  TYPE$ ='SCH';"""
            # Execute SQL
            cursor.execute(sql)
                
            # Get all schema names (extract the first column of the query results)
            dbs = sorted([row[0] for row in cursor.fetchall()])
            if dbs:
                self.databaseBox.addItems(dbs)
                self.databaseBox.setCurrentIndex(-1)
                self.databaseBox.setEnabled(True)
            for i in range(self.databaseBox.count()):
                self.databaseBox.setItemData(i, self.databaseBox.itemText(i), Qt.ToolTipRole)
            self.databaseBox.view().installEventFilter(PopupFilter(self.databaseBox))
        except Exception as e:
        # Capture other unknown errors
            iface.messageBar().pushMessage(
                "init schemaBox Error:", 
                sql, 
                level=Qgis.Critical)
        finally:
            if cursor:
                cursor.close()
        

    # The content of the schema drop-down list changes
    def schema_box_change(self):
        # Clear the table drop-down list and set it to an unclickable state
        self.tableBox.setEnabled(False)
        self.tableBox.clear()
        # Clear the spatial column drop-down list and set it to an unclickable state
        self.columnBox.setEnabled(False)
        self.columnBox.clear()
        # Reset the operateBox, typeBox, limitEdit, and sridEdit
        self.reset_operateBox()
        self.reset_typeBox()
        self.reset_limitEdit()
        self.reset_sridEdit()
        # Set the "Add Layer" button to an unclickable state
        self.addLayerButton.setEnabled(False)

        self.schema_name = self.databaseBox.currentText()
        cursor = self.conn.cursor()
        try:
            # Construct a query SQL
            # Query all tables in the current schema
            sql = """ SELECT TABLE_NAME FROM ALL_TABLES WHERE OWNER = ? ORDER BY TABLE_NAME;"""# nosec B608

            # Execute SQL
            cursor.execute(sql, (self.schema_name))
                
            # Get all table names (extract the first column of the query results)
            tables = sorted([row[0] for row in cursor.fetchall()])
            if tables:
                self.tableBox.addItems(tables)
                self.tableBox.setCurrentIndex(-1)
                self.tableBox.setEnabled(True)
            for i in range(self.tableBox.count()):
                self.tableBox.setItemData(i, self.tableBox.itemText(i), Qt.ToolTipRole)
            self.tableBox.view().installEventFilter(PopupFilter(self.tableBox))
        except ValueError as e:
            QMessageBox.critical(self, "Input error", str(e))
        except Exception as e:
        # Capture other unknown errors
            iface.messageBar().pushMessage(
                "Change schemaBox Error:", 
                sql, 
                level=Qgis.Critical)
        finally:
            if cursor:
                cursor.close()
        

    # The content of the table drop-down list changes
    def table_box_change(self):
        # Clear the spatial column drop-down list and set it to an unclickable state
        self.columnBox.setEnabled(False)
        self.columnBox.clear()
        # Reset the operateBox, typeBox, limitEdit, and sridEdit
        self.reset_operateBox()
        self.reset_typeBox()
        self.reset_limitEdit()
        self.reset_sridEdit()
        # Set the "Add Layer" button to an unclickable state
        self.addLayerButton.setEnabled(False)

        self.table_name = self.tableBox.currentText()
        cursor = self.conn.cursor()
        try:
            # Construct a query SQL
            # Query all geometry columns in the current table
            sql = """select F_GEOMETRY_COLUMN from sysgeo2.geometry_columns where F_TABLE_SCHEMA = ? and F_TABLE_NAME = ?;"""# nosec B608

            # Execute SQL
            cursor.execute(sql, (self.schema_name), (self.table_name))
                
            # Get all column names (extract the first column of the query results)
            columns = sorted([row[0] for row in cursor.fetchall()])
            if columns:
                self.columnBox.addItems(columns)
                self.columnBox.setCurrentIndex(-1)
                self.columnBox.setEnabled(True)
            for i in range(self.columnBox.count()):
                self.columnBox.setItemData(i, self.columnBox.itemText(i), Qt.ToolTipRole)
            self.columnBox.view().installEventFilter(PopupFilter(self.columnBox))
        except ValueError as e:
            QMessageBox.critical(self, "Input error", str(e))
        except Exception as e:
        # Capture other unknown errors
            iface.messageBar().pushMessage(
                "Change TableBox Error:", 
                sql, 
                level=Qgis.Critical)
        finally:
            if cursor:
                cursor.close()


    # The content of the column drop-down list changes (displays spatial data in WKT format)
    def column_box_change_with_all(self):
        # Reset the operateBox, typeBox, limitEdit, and sridEdit
        self.reset_operateBox()
        self.reset_typeBox()
        self.reset_limitEdit()
        self.reset_sridEdit()
        # Set the "Add Layer" button to an unclickable state
        self.addLayerButton.setEnabled(False)

        self.column_name = self.columnBox.currentText()
        self.column_name = self.column_name.upper()
        self.schema_name = self.schema_name.upper()
        self.table_name = self.table_name.upper()
        cursor = self.conn.cursor()
        try:
            # Construct a query SQL
            # Query the WKT format of all spatial data under the current column
            sql = "select dmgeo2.st_astext(?) from  ?.?;" # nosec B608
            
            # Execute SQL
            cursor.execute(sql, (self.column_name), (self.schema_name), (self.table_name))

            # Get all data (extract the first column of the query results)
            datas = [row[0] for row in cursor.fetchall()]
            if datas:
                self.operateBox.setCurrentIndex(-1)
                self.operateBox.setEnabled(True)
        except Exception as e:
        # Capture other unknown errors
            iface.messageBar().pushMessage(
                "Change ColumnBox Error:", 
                sql, 
                level=Qgis.Critical)
        finally:
            if cursor:
                cursor.close()


    # The content of the column drop-down list changes
    def column_box_change(self):
        # Reset the operateBox, typeBox, limitEdit, and sridEdit
        self.reset_operateBox()
        self.reset_typeBox()
        self.reset_limitEdit()
        self.reset_sridEdit()
        # Set the "Add Layer" button to an unclickable state
        self.addLayerButton.setEnabled(False)

        self.column_name = self.columnBox.currentText()

        self.operateBox.setEnabled(True)
        self.typeBox.setEnabled(True)
        self.sridEdit.setEnabled(True)


    # The content of the operate drop-down list changes
    def operate_box_change(self):
        self.operator = self.operateBox.currentText()
        # Determine the state of limitEdit based on the operator
        if self.operateBox.currentText() == 'ALL':
            self.reset_limitEdit()
        elif self.operateBox.currentText() == '>=v1 and <=v2':
            self.reset_limitEdit()
            self.limitEdit.hide()
            self.limitMinEdit.setEnabled(True)
            self.limitMinEdit.show()
            self.limitMaxEdit.setEnabled(True)
            self.limitMaxEdit.show()
        else:
            self.reset_limitEdit()
            self.limitEdit.setEnabled(True)


    # When the content of the type drop-down list changes, set the "Add Layer" button to a clickable state
    def type_box_change(self):
        self.type_name = self.typeBox.currentText()
        self.addLayerButton.setEnabled(True)


    # Click the "Add Layer" button
    def add_layer_button_clicked(self):
        self.type_name = self.typeBox.currentText()
        # The default value of operator is ALL
        self.operator = self.operateBox.currentText()
        if not self.operator:
            self.operator = 'ALL'
        # The default value of srid is 4326
        self.srid = self.sridEdit.text()
        if not self.srid:
            self.srid = 4326
        # The default value of limit is 1000
        self.limit = self.limitEdit.text()
        if not self.limit:
            self.limit = 1000
        # The default value of limit_min is 1000
        self.limit_min = self.limitMinEdit.text()
        if not self.limit_min:
            self.limit_min = 1000
        # The default value of limit_max is 1000
        self.limit_max = self.limitMaxEdit.text()
        if not self.limit_max:
            self.limit_max = 1000
        # The default value of layer_name is table name + '_' + four random numbers or letters
        self.layer_name = self.layernameEdit.text()
        if not self.layer_name:
            self.layer_name = self.table_name + '-' + str(uuid.uuid4())[0:4]

        # Disable the button to prevent repeated clicks
        self.addLayerButton.setEnabled(False)
        # Loading Progress Bar
        self.show_progreass_bar()

        # Create a background thread
        self.worker_thread = ImportWorker(self, self.layer_name)
        self.worker_thread.progress_updated.connect(self.update_progress)
        self.worker_thread.finished.connect(self.on_import_finished)
        self.worker_thread.error_occurred.connect(self.on_import_error)
        self.worker_thread.start()

        #layer = self.create_dynamic_layer(self.layer_name)
        #QgsProject.instance().addMapLayer(layer, False)
        #QgsProject.instance().layerTreeRoot().insertLayer(0, layer)


    def cancel_import(self):
        """Used after the "Cancel Import" button is clicked"""
        if hasattr(self, 'worker_thread') and self.worker_thread:
            # Disconnect the Import Operation Button Connection
            self.worker_thread.progress_updated.disconnect(self.update_progress)
            self.worker_thread.finished.disconnect(self.on_import_finished)
            self.worker_thread.error_occurred.disconnect(self.on_import_error)
            
            # Request thread interruption and stop
            self.worker_thread.requestInterruption()
            self.worker_thread.quit()
            self.worker_thread.wait()
            self.worker_thread = None
        
        # Destroy the component, and clear the references to the progress bar and the container
        if hasattr(self, 'container'):
            iface.messageBar().clearWidgets()
            self.container.deleteLater()
            self.container = None
        self.progress_bar = None

        # Restore the Import Button
        self.addLayerButton.setEnabled(True)
        
        iface.messageBar().pushMessage(
            "import operation has been interrupted",
            level = Qgis.Warning, 
            duration = 3
        )


    def on_container_destroyed(self):
        """Used to listen to the X button on the right side of the progress bar; it is used to destroy the component by default"""
        if hasattr(self, 'worker_thread') and self.worker_thread:
            # Disconnect the Import Operation Button connection
            self.worker_thread.progress_updated.disconnect(self.update_progress)
            self.worker_thread.finished.disconnect(self.on_import_finished)
            self.worker_thread.error_occurred.disconnect(self.on_import_error)
            
            # Request thread interruption and stop
            self.worker_thread.requestInterruption()
            self.worker_thread.quit()
            self.worker_thread.wait()
            self.worker_thread = None

        # Component Destruction: Clear Progress Bar and Container References
        self.progress_bar = None
        self.container = None
        
        # Restore the Import Button
        self.addLayerButton.setEnabled(True)


    def cleanup_import(self):
        """通用清理函数,导入完成/导入过程报错时调用，用于清理线程与组件"""
        # Clean up the thread
        if hasattr(self, 'worker_thread') and self.worker_thread:
            self.worker_thread.requestInterruption()
            self.worker_thread.quit()
            self.worker_thread.wait()
            self.worker_thread = None
        
        # Clean up the component
        if hasattr(self, 'container') and self.container:
            iface.messageBar().clearWidgets()
            self.container.deleteLater()
            self.container = None
        self.progress_bar = None

        # Restore the Import Button
        self.addLayerButton.setEnabled(True)


    def show_progreass_bar(self):
        """在QGIS中显示导入进度条"""
        # Clear existing messages
        iface.messageBar().clearWidgets()
        # Create the progress bar and label
        self.progress_bar = QProgressBar()
        self.progress_bar.setRange(0, 100)
        self.progress_bar.setValue(0)
        cancel_btn = QPushButton("cancel")
        cancel_btn.clicked.connect(self.cancel_import)
        label = QLabel("正在导入数据...")
        # Create a container
        self.container = QWidget()
        layout = QVBoxLayout(self.container)
        layout.addWidget(label)
        layout.addWidget(self.progress_bar)
        layout.addWidget(cancel_btn)

        # Listen for the container's destruction event
        self.container.destroyed.connect(self.on_container_destroyed)

        # Add to the message bar (display continuously, do not close automatically)
        iface.messageBar().pushWidget(
            self.container,
            level=Qgis.Info,
            duration=0
        )
        
    
    @pyqtSlot(int)
    def update_progress(self, value):
        """Update the progress bar"""
        if self.progress_bar is not None:
            self.progress_bar.setValue(value)


    @pyqtSlot(object)
    def on_import_finished(self, layer):
        """Restore the state after the import is completed"""
        self.cleanup_import()

        if layer:
            # If successful, add the layer to the project
            QgsProject.instance().addMapLayer(layer, False)
            QgsProject.instance().layerTreeRoot().insertLayer(0, layer)
            # Show "Success" message
            iface.messageBar().pushSuccess(
                "Success",
                f"import {layer.featureCount()} features"
            )
        else:
            # Show "Fail" message
            iface.messageBar().pushCritical(
                "Critical",
                f"import features failed!"
            )


    @pyqtSlot(str)
    def on_import_error(self, error_msg):
        """Handling Thread Errors"""
        self.cleanup_import()

        # Show "Error" message
        iface.messageBar().pushCritical(
            "Critical",
            f"import fail: {error_msg}"
        )


    # Map the types of typeBox (Point, LineString, ...) to the spatial types of DM (ST_Point, ST_LineString, ...)
    def typebox_type_to_dm(self, box_type):
        if box_type == 'Point':
            return 'ST_Point'
        elif box_type == 'LineString':
            return 'ST_LineString'
        elif box_type == 'Polygon':
            return 'ST_Polygon'
        elif box_type == 'MultiPoint':
            return 'ST_MultiPoint'
        elif box_type == 'MultiLineString':
            return 'ST_MultiLineString'
        elif box_type == 'MultiPolygon':
            return 'ST_MultiPolygon'
        else:
            iface.messageBar().pushMessage(
                self.tr("typeBox has Error"),
                level = Qgis.Critical)
            return ''


    # Map Dameng data types to QVariant types in QGIS
    def dm_type_to_qgis(self, dm_type):
        dm_type = dm_type.upper()
        if "INT" in dm_type or "NUMBER" in dm_type and "DECIMAL" not in dm_type:
            return QVariant.Int
        elif "FLOAT" in dm_type or "DOUBLE" in dm_type or "DECIMAL" in dm_type:
            return QVariant.Double
        elif "TIME" in dm_type and "DATETIME" not in dm_type and "TIMESTAMP" not in dm_type:
            return QVariant.Time
        elif "DATE" in dm_type and "TIME" not in dm_type:
            return QVariant.Date
        elif "DATETIME" in dm_type or "TIMESTAMP" in dm_type:
            return QVariant.DateTime
        else:  # Strings, character types, and the like use String by default
            return QVariant.String


    def get_table_fields(self):
        # Dynamically obtain the non-spatial field metadata of the table (field name, data type)
        cursor = self.conn.cursor()
        try:
            # Query all fields of the table (excluding spatial columns)
            sql = """
                SELECT COLUMN_NAME, DATA_TYPE 
                FROM ALL_TAB_COLUMNS 
                WHERE OWNER = ? 
                  AND TABLE_NAME = ?
                  AND COLUMN_NAME != ?
                ORDER BY COLUMN_ID
            """# nosec B608
            cursor.execute(sql, (self.schema_name), (self.table_name), (self.column_name))
            fields = cursor.fetchall()  # Format: [(Field Name 1, Type 1), (Field Name 2, Type 2), ...]
            return fields
        except dmPython.DatabaseError as e:
            iface.messageBar().pushCritical(self.tr("Could not find data"), f"{e.message}")
            return []
        finally:
            if cursor:
                cursor.close()


    # Import the memory layer
    def import_memory_layer(self, wkt, layer_name="memory layer", crs="EPSG:4326"):
        # Determine the geometry type and create the corresponding memory layer
        if wkt.startswith("POINT"):
            layer_type = "Point"
        elif wkt.startswith("LINESTRING"):
            layer_type = "LineString"
        elif wkt.startswith("POLYGON"):
            layer_type = "Polygon"
        else:
            raise ValueError(self.tr("Unsupported type"))

        uri = f"{layer_type}?crs={crs}"
        layer = QgsVectorLayer(uri, layer_name, "memory")
        if not layer.isValid():
            raise RuntimeError(self.tr("Failed to add leyer "))

        # Add attribute fields (Example: id, integer)
        fields = QgsFields()
        fields.append(QgsField("id", 2))  #
        layer.dataProvider().addAttributes(fields)
        layer.updateFields()

        # Create geometry from WKT and add features
        geom = QgsGeometry.fromWkt(wkt)
        feat = QgsFeature()
        feat.setGeometry(geom)
        feat.setAttributes([1])
        layer.dataProvider().addFeatures([feat])
        layer.updateExtents()

        # Perform the operation outside the project
        return layer
    

    def get_limit_data(self, fields):
        """查询所有属性字段和空间列WKT"""
        try:
            #cursor = self.conn.cursor()

            escaped_attr_fields = []
            # 1.Validate and escape all attribute field names
            for f in fields:
                field_name = f[0]
                escaped_attr_fields.append(self.escape_identifier(field_name))
            
            # Concatenate all attribute field names (for SQL query)
            attr_fields = ", ".join(escaped_attr_fields)  # field name
            int64_max = np.iinfo(np.int64).max
            escaped_schema_name = self.escape_identifier(self.schema_name)
            escaped_table_name = self.escape_identifier(self.table_name)
            escaped_column_name = self.escape_identifier(self.column_name)
            if self.operator == 'ALL':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM {escaped_schema_name}.{escaped_table_name}
                    WHERE dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (self.srid), (self.typebox_type_to_dm(self.type_name)))
            elif self.operator == '>=v1 and <=v2':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn >= ? and rn <= ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit_min), (self.limit_max), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            elif self.operator == '>=':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn >= ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            elif self.operator == '>':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn > ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            elif self.operator == '=':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn = ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            elif self.operator == '<=':
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn <= ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            else:
                # Construct a query SQL
                sql = f"""
                    SELECT {attr_fields}, dmgeo2.st_astext({escaped_column_name}) FROM 
                    (
                        SELECT rownum AS rn, {attr_fields}, {escaped_column_name}
                        FROM {escaped_schema_name}.{escaped_table_name}
                        WHERE rownum <= ?
                    )
                    WHERE rn < ? and dmgeo2.st_srid({escaped_column_name}) = ? and dmgeo2.st_geometrytype({escaped_column_name}) = ?;
                """# nosec B608
                self.cursor.execute(sql, (int64_max), (self.limit), (self.srid), (self.typebox_type_to_dm(self.type_name)))
            return
        except dmPython.DatabaseError as e:
            iface.messageBar().pushMessage(
                "get_limit_data Error:", 
                sql,
                level=Qgis.Critical)
            return
        except ValueError as e:
            QMessageBox.critical(self, "Input error", str(e))
        

    def get_total_size(self):
        cursor = self.conn.cursor()
        try:
            escaped_schema_name = self.escape_identifier(self.schema_name)
            escaped_table_name = self.escape_identifier(self.table_name)
            # Query all fields of the table (excluding spatial columns)
            sql = f"""
                SELECT count(*)
                FROM {escaped_schema_name}.{escaped_table_name};
            """# nosec B608
            cursor.execute(sql)
            return cursor.fetchone()
        except dmPython.DatabaseError as e:
            iface.messageBar().pushCritical(self.tr("Could not get total size"), f"{e.message}")
            return []
        finally:
            if cursor:
                cursor.close()

        
    def closeEvent(self, event):
        self.plugin.quit_plugin()
        event.accept()