"""Verify metadata in ZIP package"""
import zipfile
import os
import re

zip_path = os.path.join(os.path.dirname(__file__), '..', 'ready_to_install', 'DEM_Downscaling.zip')

if not os.path.exists(zip_path):
    print(f"[X] ZIP file not found: {zip_path}")
    exit(1)

with zipfile.ZipFile(zip_path, 'r') as zipf:
    # Find metadata.txt
    metadata_entry = None
    for name in zipf.namelist():
        if name.endswith('metadata.txt'):
            metadata_entry = name
            break
    
    if not metadata_entry:
        print("[X] metadata.txt not found in ZIP")
        exit(1)
    
    # Read metadata
    content = zipf.read(metadata_entry).decode('utf-8')
    
    # Extract URLs
    repo_match = re.search(r'repository=(.+)', content)
    tracker_match = re.search(r'tracker=(.+)', content)
    license_match = re.search(r'license=(.+)', content)
    
    print("=== Metadata in ZIP Package ===")
    if repo_match:
        repo_url = repo_match.group(1).strip()
        print(f"Repository: {repo_url}")
        if 'github.com/nqm12a02/DEM-Downscaling' in repo_url:
            print("  [OK] GitHub repository URL is correct!")
        else:
            print("  [X] Repository URL needs update")
    else:
        print("[X] Repository URL not found")
    
    if tracker_match:
        tracker_url = tracker_match.group(1).strip()
        print(f"Tracker: {tracker_url}")
        if 'github.com/nqm12a02/DEM-Downscaling/issues' in tracker_url:
            print("  [OK] GitHub tracker URL is correct!")
        else:
            print("  [X] Tracker URL needs update")
    else:
        print("[X] Tracker URL not found")
    
    if license_match:
        license_val = license_match.group(1).strip()
        print(f"License: {license_val}")
        if license_val == 'GPLv2':
            print("  [OK] License is correct!")
        else:
            print("  [X] License needs update")
    else:
        print("[X] License not found")
    
    print(f"\nZIP File: {zip_path}")
    print(f"Size: {os.path.getsize(zip_path) / 1024:.2f} KB")
    
    # Final check
    if (repo_match and 'github.com/nqm12a02' in repo_match.group(1) and
        tracker_match and 'github.com/nqm12a02' in tracker_match.group(1) and
        license_match and license_match.group(1).strip() == 'GPLv2'):
        print("\n[OK] All metadata is correct! Ready for QGIS repository upload.")
    else:
        print("\n[X] Some metadata fields need to be corrected.")
