# -*- coding: utf-8 -*-
"""
TXT → Vector
将坐标 TXT 导入为多环多边形图层（通过第二字段区分环）。
新增：
1. 自动检测文件编码（UTF-8 / UTF-8-SIG / GBK / ANSI 等）
2. 增加参数：是否交换 XY（测量坐标 X=北,Y=东 → GIS 坐标 X=东,Y=北）
3. 增加参数：是否自动识别坐标顺序（基于样本统计的启发式判断）
4. 支持解析摘要行中扩展字段：地块编号、地块名称、面积、环数等
5. 输出图层增加字段：parcel_id, parcel_name, area_ha, area_m2, rings, vertex_cnt, filename
联系人：槐中路科科 996517087@qq.com
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterFile,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsProcessing,          # ← ★★★★★ 修复关键：加入这个 import
    QgsVectorLayer,
    QgsFields,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsPointXY,
    QgsProject,
)
from PyQt5.QtCore import QVariant
import os
import re


# ---------------------------------------------------------
# 自动检测编码（UTF-8 / UTF-8-SIG / GBK / ANSI / Latin-1）
# ---------------------------------------------------------
def detect_encoding(path):
    encodings = ("utf-8-sig", "utf-8", "gb18030", "gbk", "cp936", "latin-1")
    for enc in encodings:
        try:
            with open(path, "r", encoding=enc) as f:
                for _ in range(10):
                    if not f.readline():
                        break
            return enc
        except UnicodeDecodeError:
            continue
    return "utf-8"


# ---------------------------------------------------------
# 自动判断是否需要交换 XY （启发式）
# ---------------------------------------------------------
def scan_xy_stats(path, enc, vertex_re, header_tags, summary_line, max_samples=500):
    xs, ys = [], []
    with open(path, "r", encoding=enc) as f:
        for raw in f:
            line = raw.strip()
            if not line:
                continue
            if any(line.startswith(tag) for tag in header_tags):
                continue
            if summary_line.match(line):
                continue
            m = vertex_re.match(line)
            if m:
                x_raw = float(m.group(2))
                y_raw = float(m.group(3))
                xs.append(x_raw)
                ys.append(y_raw)
                if len(xs) >= max_samples:
                    break
    if not xs or not ys:
        return False
    avg_x = sum(xs) / len(xs)
    avg_y = sum(ys) / len(ys)
    return avg_x > avg_y


class TxtToVectorImporter(QgsProcessingAlgorithm):
    INPUT_TXT = "INPUT_TXT"
    OUTPUT = "OUTPUT"
    SWITCH_XY = "SWITCH_XY"
    AUTO_DETECT_XY = "AUTO_DETECT_XY"

    def name(self):
        return "txt_to_vector"

    def displayName(self):
        return self.tr("TXT → Vector")

    def group(self):
        return ""

    def groupId(self):
        return ""

    def shortHelpString(self):
        return self.tr(
            "解析坐标 TXT，外环第二字段=1，内环依次 2,3,…；支持自动识别编码，可选自动判断坐标顺序并交换 XY；"
            "输出地块编号、名称、面积、环数等属性字段。\n"
            "摘要行建议格式：顶点数,面积(ha),地块编号,类型,地块名称,环数,面积(m2),@\n"
            "联系人：槐中路科科 996517087@qq.com"
        )

    def createInstance(self):
        return TxtToVectorImporter()

    def tr(self, t):
        return QCoreApplication.translate("TxtToVectorImporter", t)

    # ---------------------------------------------------------
    # 参数
    # ---------------------------------------------------------
    def initAlgorithm(self, _config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_TXT,
                self.tr("输入 TXT 文件"),
                extension="txt",
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.AUTO_DETECT_XY,
                self.tr("自动识别坐标顺序（推荐谨慎使用，有歧义时请关闭并手动控制交换 XY）"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.SWITCH_XY,
                self.tr("交换 XY（测量坐标 X=北,Y=东 → GIS 坐标 X=东,Y=北）"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("输出矢量图层"),
                type=QgsProcessing.TypeVectorPolygon,
            )
        )

    # ---------------------------------------------------------
    # 主体逻辑
    # ---------------------------------------------------------
    def processAlgorithm(self, params, context, feedback):
        txt = self.parameterAsFile(params, self.INPUT_TXT, context)
        auto_detect = self.parameterAsBool(params, self.AUTO_DETECT_XY, context)
        switch_xy_manual = self.parameterAsBool(params, self.SWITCH_XY, context)

        if not os.path.isfile(txt):
            raise QgsProcessingException(self.tr("TXT 文件不存在。"))

        enc = detect_encoding(txt)

        vertex_re = re.compile(
            r"^J\d+,\s*(\d+),\s*([+-]?\d+\.\d+),\s*([+-]?\d+\.\d+)"
        )
        summary_line = re.compile(r"^\d+,")
        header_tags = ("[属性描述", "[地块坐标]")

        # ---------------------------------------------------------
        # 是否自动判断 XY
        # ---------------------------------------------------------
        if auto_detect:
            detected_switch = scan_xy_stats(
                txt, enc, vertex_re, header_tags, summary_line
            )
            do_switch = detected_switch
        else:
            do_switch = switch_xy_manual

        # ---------------------------------------------------------
        # 解析 TXT
        # ---------------------------------------------------------
        features = []
        current_rings = {}
        current_meta = {}

        def finalize_feature():
            if current_rings:
                features.append(
                    {"rings": dict(current_rings), "meta": dict(current_meta)}
                )

        with open(txt, "r", encoding=enc) as fp:
            for raw in fp:
                line = raw.strip()
                if not line:
                    continue

                # 跳过头部
                if any(line.startswith(tag) for tag in header_tags):
                    continue

                # 摘要行
                if summary_line.match(line):
                    finalize_feature()
                    current_rings.clear()
                    current_meta.clear()

                    parts = [p.strip() for p in line.split(",")]
                    meta = {}

                    if len(parts) > 0:
                        try:
                            meta["vertex_count"] = int(parts[0])
                        except:
                            pass
                    if len(parts) > 1:
                        try:
                            meta["area_ha"] = float(parts[1])
                        except:
                            pass
                    if len(parts) > 2:
                        meta["parcel_id"] = parts[2]
                    if len(parts) > 4:
                        meta["parcel_name"] = parts[4]
                    if len(parts) > 5:
                        try:
                            meta["rings"] = int(parts[5])
                        except:
                            pass
                    if len(parts) > 6:
                        try:
                            meta["area_m2"] = float(parts[6])
                        except:
                            pass

                    current_meta.update(meta)
                    continue

                # 坐标行
                m = vertex_re.match(line)
                if m:
                    ring_id = int(m.group(1))
                    x_raw = float(m.group(2))
                    y_raw = float(m.group(3))

                    if do_switch:
                        x = y_raw
                        y = x_raw
                    else:
                        x = x_raw
                        y = y_raw

                    current_rings.setdefault(ring_id, []).append(QgsPointXY(x, y))

        finalize_feature()

        if not features:
            raise QgsProcessingException(self.tr("未解析到有效坐标。"))

        # ---------------------------------------------------------
        # 输出图层
        # ---------------------------------------------------------
        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))
        fields.append(QgsField("parcel_id", QVariant.String))
        fields.append(QgsField("parcel_name", QVariant.String))
        fields.append(QgsField("area_ha", QVariant.Double))
        fields.append(QgsField("area_m2", QVariant.Double))
        fields.append(QgsField("rings", QVariant.Int))
        fields.append(QgsField("vertex_cnt", QVariant.Int))
        fields.append(QgsField("filename", QVariant.String))

        layer_name = os.path.basename(txt)
        mem = QgsVectorLayer("Polygon?crs=EPSG:4490", layer_name, "memory")
        mem.dataProvider().addAttributes(fields)
        mem.updateFields()

        feats = []
        for fid, info in enumerate(features, 1):
            ring_map = info["rings"]
            meta = info.get("meta", {})

            if 1 not in ring_map:
                continue

            outer = list(ring_map[1])
            if len(outer) < 3:
                continue
            if outer[0] != outer[-1]:
                outer.append(outer[0])

            rings_xy = [outer]

            # 内环
            for lab in sorted(k for k in ring_map.keys() if k > 1):
                ring =  list(ring_map[lab])
                if len(ring) < 3:
                    continue
                if ring[0] != ring[-1]:
                    ring.append(ring[0])
                rings_xy.append(ring)

            geom = QgsGeometry.fromPolygonXY(rings_xy)

            area_m2 = geom.area()
            area_ha = area_m2 / 10000.0
            rings_num = len(rings_xy)
            vertex_cnt = sum(len(r) for r in rings_xy)

            # 若 TXT 提供摘要值，则覆盖
            if "area_ha" in meta:
                try:
                    area_ha = float(meta["area_ha"])
                except:
                    pass
            if "area_m2" in meta:
                try:
                    area_m2 = float(meta["area_m2"])
                except:
                    pass
            if "rings" in meta:
                try:
                    rings_num = int(meta["rings"])
                except:
                    pass
            if "vertex_count" in meta:
                try:
                    vertex_cnt = int(meta["vertex_count"])
                except:
                    pass

            parcel_id = meta.get("parcel_id", "")
            parcel_name = meta.get("parcel_name", parcel_id)

            feat = QgsFeature(fields)
            feat.setAttributes(
                [
                    fid,
                    parcel_id,
                    parcel_name,
                    area_ha,
                    area_m2,
                    rings_num,
                    vertex_cnt,
                    os.path.basename(txt),
                ]
            )
            feat.setGeometry(geom)
            feats.append(feat)

        mem.dataProvider().addFeatures(feats)
        mem.updateExtents()
        QgsProject.instance().addMapLayer(mem)

        return {self.OUTPUT: mem}
