
# -*- coding: utf-8 -*-
"""
Vector → TXT
导出矢量要素为坐标 TXT（支持多环/空洞）。
新增：
1. 导出时可选择是否交换 XY（GIS 坐标 X=东,Y=北 → 测量坐标 X=北,Y=东）
2. 摘要行增加地块编号、地块名称、环数、面积（平方米/公顷）等信息
联系人：槐中路科科 996517087@qq.com
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterString,
    QgsProcessingParameterField,
    QgsProcessingParameterNumber,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsProcessing,
    QgsWkbTypes,
)


class CoordinateTxtExporter(QgsProcessingAlgorithm):
    INPUT = "INPUT"
    OUTPUT = "OUTPUT"
    HEADER = "HEADER"
    PARCEL_FIELD = "PARCEL_FIELD"
    PRECISION = "PRECISION"
    SWITCH_XY = "SWITCH_XY"

    # ------------- metadata -------------
    def name(self):
        return "vector_to_txt"

    def displayName(self):
        return self.tr("Vector → TXT")

    def group(self):
        return ""

    def groupId(self):
        return ""

    def shortHelpString(self):
        return self.tr(
            "将面/线/点图层导出为坐标 TXT 文件；若多环/空洞，第二字段依次 1=外环, 2=第1空洞…\n"
            "新增：导出时可选择是否交换 XY（GIS 坐标 → 测量坐标），摘要行增加地块编号、名称、面积、环数等。\n"
            "联系人：槐中路科科 996517087@qq.com"
        )

    def createInstance(self):
        return CoordinateTxtExporter()

    def tr(self, t):
        return QCoreApplication.translate("CoordinateTxtExporter", t)

    # ------------- parameters -------------
    def initAlgorithm(self, _config=None):
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT,
                self.tr("输入图层 (面/线/点)"),
                [QgsProcessing.TypeVectorPolygon,
                 QgsProcessing.TypeVectorLine,
                 QgsProcessing.TypeVectorPoint],
            )
        )
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr("输出 TXT 文件"),
                fileFilter="文本文件 (*.txt)",
            )
        )
        header_default = (
            "[属性描述]\n"
            "坐标系=2000国家大地坐标系\n"
            "几度分带=3\n"
            "投影类型=高斯克吕格\n"
            "计量单位=米\n"
            "带号=38\n"
            "精度=0.001\n"
            "转换参数=,,,,,,\n"
            "[地块坐标]"
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.HEADER,
                self.tr("[属性描述] 内容"),
                multiLine=True,
                defaultValue=header_default,
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.PARCEL_FIELD,
                self.tr("地块编号字段 (可选)"),
                type=QgsProcessingParameterField.Any,
                parentLayerParameterName=self.INPUT,
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.PRECISION,
                self.tr("坐标小数位数"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=3,
                minValue=0,
                maxValue=6,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.SWITCH_XY,
                self.tr("导出时交换 XY（GIS 坐标 X=东,Y=北 → 测量坐标 X=北,Y=东）"),
                defaultValue=False,
            )
        )

    # ------------- core -------------
    def processAlgorithm(self, params, context, feedback):
        layer = self.parameterAsVectorLayer(params, self.INPUT, context)
        if layer is None:
            raise QgsProcessingException(self.invalidSourceError(params, self.INPUT))

        out_path = self.parameterAsFileOutput(params, self.OUTPUT, context)
        prec = int(self.parameterAsInt(params, self.PRECISION, context))
        fmt = f"%.{prec}f"
        header = self.parameterAsString(params, self.HEADER, context)
        parcel_field = self.parameterAsString(params, self.PARCEL_FIELD, context)
        switch_xy = self.parameterAsBool(params, self.SWITCH_XY, context)

        with open(out_path, "w", encoding="utf-8") as f:
            f.write(header + "\n")

            for feat_idx, feat in enumerate(layer.getFeatures(), 1):
                geom = feat.geometry()
                if not geom or geom.isEmpty():
                    continue

                # Build list of rings for polygons
                all_polys = []
                if geom.isMultipart():
                    all_polys = geom.asMultiPolygon()
                else:
                    poly = geom.asPolygon()
                    if poly:
                        all_polys = [poly]

                # 顶点数（不计闭合点）
                if all_polys:
                    vertex_total = sum(len(r) - 1 for p in all_polys for r in p)
                else:
                    vertex_total = len(list(geom.vertices()))

                # 环数（面要素），线/点则为 0
                if all_polys:
                    ring_count = sum(len(p) for p in all_polys)
                else:
                    ring_count = 0

                # 面积信息
                is_polygon = (
                    QgsWkbTypes.geometryType(geom.wkbType())
                    == QgsWkbTypes.PolygonGeometry
                )
                area_m2 = geom.area() if is_polygon else 0.0
                area_ha = area_m2 / 10000.0 if is_polygon else 0.0

                type_label = {
                    QgsWkbTypes.PolygonGeometry: "面",
                    QgsWkbTypes.LineGeometry: "线",
                }.get(QgsWkbTypes.geometryType(geom.wkbType()), "点")

                # 地块编号与名称：目前名称=编号
                if parcel_field and parcel_field in feat.fields().names():
                    parcel_id = str(feat[parcel_field])
                else:
                    parcel_id = f"地块{feat_idx}"
                parcel_name = parcel_id

                # 摘要行格式：
                # 顶点数,面积(ha),地块编号,类型,地块名称,环数,面积(m2),@
                f.write(
                    f"{vertex_total},"
                    f"{round(area_ha, 4)},"
                    f"{parcel_id},"
                    f"{type_label},"
                    f"{parcel_name},"
                    f"{ring_count},"
                    f"{round(area_m2, 3)},"
                    f"@\n"
                )

                # 坐标行：Jxx,ring_no,X,Y
                if all_polys:  # polygon
                    for poly in all_polys:
                        for ring_idx, ring in enumerate(poly):
                            ring_no = ring_idx + 1  # 1 outer, 2 inner ...
                            pts = list(ring)
                            if not pts:
                                continue
                            if pts[0] != pts[-1]:
                                pts.append(pts[0])
                            for vid, pt in enumerate(pts[:-1], 1):
                                if switch_xy:
                                    x_out, y_out = pt.y(), pt.x()
                                else:
                                    x_out, y_out = pt.x(), pt.y()
                                f.write(
                                    f"J{vid:02d},{ring_no},"
                                    f"{fmt % x_out},{fmt % y_out}\n"
                                )
                            first = pts[0]
                            if switch_xy:
                                x0, y0 = first.y(), first.x()
                            else:
                                x0, y0 = first.x(), first.y()
                            f.write(
                                f"J01,{ring_no},"
                                f"{fmt % x0},{fmt % y0}\n"
                            )
                else:  # line/point
                    pts = list(geom.vertices())
                    for vid, pt in enumerate(pts, 1):
                        if switch_xy:
                            x_out, y_out = pt.y(), pt.x()
                        else:
                            x_out, y_out = pt.x(), pt.y()
                        f.write(
                            f"J{vid:02d},1,{fmt % x_out},{fmt % y_out}\n"
                        )

        return {self.OUTPUT: out_path}
