# -*- coding: utf-8 -*-
"""
Batch TXT → Vector
批量导入文件夹中的多个坐标 TXT 文件并合并为一个图层。
兼容 QGIS 3.40（不使用 QgsProcessingParameterFolder）。
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterString,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsProcessing,      # ← ★★★★★ 修复关键：加入这个 import
    QgsVectorLayer,
    QgsFields,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsPointXY,
    QgsProject,
)
from PyQt5.QtCore import QVariant
import os
import re

from .txt_to_vector_importer import detect_encoding, scan_xy_stats


class BatchTxtToVectorImporter(QgsProcessingAlgorithm):
    INPUT_FOLDER = "INPUT_FOLDER"
    OUTPUT = "OUTPUT"
    SWITCH_XY = "SWITCH_XY"
    AUTO_DETECT_XY = "AUTO_DETECT_XY"

    def name(self):
        return "batch_txt_to_vector"

    def displayName(self):
        return self.tr("Batch TXT → Vector")

    def group(self):
        return ""

    def groupId(self):
        return ""

    def tr(self, t):
        return QCoreApplication.translate("BatchTxtToVectorImporter", t)

    def shortHelpString(self):
        return self.tr(
            "批量导入指定文件夹中的多个 TXT 坐标文件，将所有地块合并至一个多边形图层。\n"
            "为兼容当前 QGIS 版本，文件夹路径通过字符串参数输入。\n"
            "支持：自动识别编码、自动判断坐标顺序（可选）、XY 交换、摘要字段解析等。"
        )

    def createInstance(self):
        return BatchTxtToVectorImporter()

    # ---------------------------------------------------------
    def initAlgorithm(self, _config=None):
        self.addParameter(
            QgsProcessingParameterString(
                self.INPUT_FOLDER,
                self.tr("输入 TXT 文件夹路径（例如 C:\\\\Users\\\\hebei\\\\Desktop\\\\txtfolder）"),
                defaultValue="",
                multiLine=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.AUTO_DETECT_XY,
                self.tr("自动识别坐标顺序（建议配合人工检查）"),
                defaultValue=False,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.SWITCH_XY,
                self.tr("交换 XY（测量坐标 → GIS 坐标）"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("输出矢量图层"),
                type=QgsProcessing.TypeVectorPolygon,
            )
        )

    # ---------------------------------------------------------
    @staticmethod
    def parse_single_txt(path, auto_detect, switch_xy_manual):
        enc = detect_encoding(path)

        vertex_re = re.compile(r"^J\d+,\s*(\d+),\s*([+-]?\d+\.\d+),\s*([+-]?\d+\.\d+)")
        summary_line = re.compile(r"^\d+,")
        header_tags = ("[属性描述", "[地块坐标]")

        # 自动识别 XY
        if auto_detect:
            detected_switch = scan_xy_stats(
                path, enc, vertex_re, header_tags, summary_line
            )
            do_switch = detected_switch
        else:
            do_switch = switch_xy_manual

        features = []
        current_rings = {}
        current_meta = {}

        def finalize_feature():
            if current_rings:
                features.append({"rings": dict(current_rings), "meta": dict(current_meta)})

        with open(path, "r", encoding=enc) as fp:
            for raw in fp:
                line = raw.strip()
                if not line:
                    continue

                if any(line.startswith(tag) for tag in header_tags):
                    continue

                if summary_line.match(line):
                    finalize_feature()
                    current_rings.clear()
                    current_meta.clear()

                    parts = [p.strip() for p in line.split(",")]
                    meta = {}

                    if len(parts) > 0:
                        try:
                            meta["vertex_count"] = int(parts[0])
                        except:
                            pass
                    if len(parts) > 1:
                        try:
                            meta["area_ha"] = float(parts[1])
                        except:
                            pass
                    if len(parts) > 2:
                        meta["parcel_id"] = parts[2]
                    if len(parts) > 4:
                        meta["parcel_name"] = parts[4]
                    if len(parts) > 5:
                        try:
                            meta["rings"] = int(parts[5])
                        except:
                            pass
                    if len(parts) > 6:
                        try:
                            meta["area_m2"] = float(parts[6])
                        except:
                            pass

                    current_meta.update(meta)
                    continue

                m = vertex_re.match(line)
                if m:
                    ring_id = int(m.group(1))
                    x_raw = float(m.group(2))
                    y_raw = float(m.group(3))

                    if do_switch:
                        x = y_raw
                        y = x_raw
                    else:
                        x = x_raw
                        y = y_raw

                    current_rings.setdefault(ring_id, []).append(QgsPointXY(x, y))

        finalize_feature()
        return features

    # ---------------------------------------------------------
    def processAlgorithm(self, params, context, feedback):
        folder = self.parameterAsString(params, self.INPUT_FOLDER, context).strip()
        auto_detect = self.parameterAsBool(params, self.AUTO_DETECT_XY, context)
        switch_xy_manual = self.parameterAsBool(params, self.SWITCH_XY, context)

        if not folder or not os.path.isdir(folder):
            raise QgsProcessingException(self.tr("输入文件夹不存在或路径为空。"))

        txt_files = [
            os.path.join(folder, fn)
            for fn in sorted(os.listdir(folder))
            if fn.lower().endswith(".txt")
        ]

        if not txt_files:
            raise QgsProcessingException(self.tr("指定文件夹中未找到 TXT 文件。"))

        # 创建输出图层
        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))
        fields.append(QgsField("parcel_id", QVariant.String))
        fields.append(QgsField("parcel_name", QVariant.String))
        fields.append(QgsField("area_ha", QVariant.Double))
        fields.append(QgsField("area_m2", QVariant.Double))
        fields.append(QgsField("rings", QVariant.Int))
        fields.append(QgsField("vertex_cnt", QVariant.Int))
        fields.append(QgsField("filename", QVariant.String))

        layer_name = os.path.basename(folder.rstrip("\\/"))
        mem = QgsVectorLayer("Polygon?crs=EPSG:4490", layer_name, "memory")
        mem.dataProvider().addAttributes(fields)
        mem.updateFields()

        feats = []
        fid_global = 1

        total = len(txt_files)
        for idx, path in enumerate(txt_files, 1):
            if feedback:
                feedback.pushInfo(f"解析：{os.path.basename(path)}")
                feedback.setProgress(int(idx * 100 / total))

            file_features = self.parse_single_txt(
                path, auto_detect, switch_xy_manual
            )

            for info in file_features:
                ring_map = info["rings"]
                meta = info.get("meta", {})

                if 1 not in ring_map:
                    continue

                outer = list(ring_map[1])
                if len(outer) < 3:
                    continue
                if outer[0] != outer[-1]:
                    outer.append(outer[0])

                rings_xy = [outer]

                # 内环
                for lab in sorted(k for k in ring_map.keys() if k > 1):
                    ring = list(ring_map[lab])
                    if len(ring) < 3:
                        continue
                    if ring[0] != ring[-1]:
                        ring.append(ring[0])
                    rings_xy.append(ring)

                geom = QgsGeometry.fromPolygonXY(rings_xy)
                area_m2 = geom.area()
                area_ha = area_m2 / 10000.0
                rings_num = len(rings_xy)
                vertex_cnt = sum(len(r) for r in rings_xy)

                # 覆盖摘要
                if "area_ha" in meta:
                    try: area_ha = float(meta["area_ha"])
                    except: pass
                if "area_m2" in meta:
                    try: area_m2 = float(meta["area_m2"])
                    except: pass
                if "rings" in meta:
                    try: rings_num = int(meta["rings"])
                    except: pass
                if "vertex_count" in meta:
                    try: vertex_cnt = int(meta["vertex_count"])
                    except: pass

                parcel_id = meta.get("parcel_id", "")
                parcel_name = meta.get("parcel_name", parcel_id)

                feat = QgsFeature(fields)
                feat.setAttributes(
                    [
                        fid_global,
                        parcel_id,
                        parcel_name,
                        area_ha,
                        area_m2,
                        rings_num,
                        vertex_cnt,
                        os.path.basename(path),
                    ]
                )
                feat.setGeometry(geom)

                feats.append(feat)
                fid_global += 1

        mem.dataProvider().addFeatures(feats)
        mem.updateExtents()
        QgsProject.instance().addMapLayer(mem)

        return {self.OUTPUT: mem}
