
# -*- coding: utf-8 -*-
"""
Vector → TXT
导出矢量要素为坐标 TXT（支持多环/空洞）。
联系人：槐中路科科 996517087@qq.com
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterString,
    QgsProcessingParameterField,
    QgsProcessingParameterNumber,
    QgsProcessingException,
    QgsProcessing,
    QgsWkbTypes,
)
from datetime import datetime

class CoordinateTxtExporter(QgsProcessingAlgorithm):
    INPUT = "INPUT"
    OUTPUT = "OUTPUT"
    HEADER = "HEADER"
    PARCEL_FIELD = "PARCEL_FIELD"
    PRECISION = "PRECISION"

    # ------------- metadata -------------
    def name(self): return "vector_to_txt"
    def displayName(self): return self.tr("Vector → TXT")
    def group(self): return ""
    def groupId(self): return ""
    def shortHelpString(self):
        return self.tr(
            "将面/线/点图层导出为坐标 TXT 文件；若多环/空洞，第二字段依次 1=外环, 2=第1空洞…\n"
            "联系人：槐中路科科 996517087@qq.com"
        )
    def createInstance(self): return CoordinateTxtExporter()
    def tr(self, t): return QCoreApplication.translate("CoordinateTxtExporter", t)

    # ------------- parameters -------------
    def initAlgorithm(self, _config=None):
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT, self.tr("输入图层 (面/线/点)"),
                [QgsProcessing.TypeVectorPolygon,
                 QgsProcessing.TypeVectorLine,
                 QgsProcessing.TypeVectorPoint]
            )
        )
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT, self.tr("输出 TXT 文件"), fileFilter="文本文件 (*.txt)"
            )
        )
        header_default = (
            "[属性描述]\n"
            "坐标系=2000国家大地坐标系\n"
            "几度分带=3\n"
            "投影类型=高斯克吕格\n"
            "计量单位=米\n"
            "带号=38\n"
            "精度=0.001\n"
            "转换参数=,,,,,,\n"
            "[地块坐标]"
        )
        self.addParameter(
            QgsProcessingParameterString(self.HEADER, self.tr("[属性描述] 内容"),
                                         multiLine=True, defaultValue=header_default)
        )
        self.addParameter(
            QgsProcessingParameterField(self.PARCEL_FIELD, self.tr("地块编号字段 (可选)"),
                                        type=QgsProcessingParameterField.Any,
                                        parentLayerParameterName=self.INPUT,
                                        optional=True)
        )
        self.addParameter(
            QgsProcessingParameterNumber(self.PRECISION, self.tr("坐标小数位数"),
                                         type=QgsProcessingParameterNumber.Integer,
                                         defaultValue=3, minValue=0, maxValue=6)
        )

    # ------------- core -------------
    def processAlgorithm(self, params, context, feedback):
        layer = self.parameterAsVectorLayer(params, self.INPUT, context)
        if layer is None:
            raise QgsProcessingException(self.invalidSourceError(params, self.INPUT))

        out_path = self.parameterAsFileOutput(params, self.OUTPUT, context)
        prec = int(self.parameterAsInt(params, self.PRECISION, context))
        fmt = f"%.{prec}f"
        header = self.parameterAsString(params, self.HEADER, context)
        parcel_field = self.parameterAsString(params, self.PARCEL_FIELD, context)

        with open(out_path, "w", encoding="utf-8") as f:
            f.write(header + "\n")

            for feat_idx, feat in enumerate(layer.getFeatures(), 1):
                geom = feat.geometry()
                if not geom or geom.isEmpty():
                    continue

                # Build list of rings
                all_polys = []
                if geom.isMultipart():
                    all_polys = geom.asMultiPolygon()
                else:
                    poly = geom.asPolygon()
                    if poly:
                        all_polys = [poly]

                # compute vertex count (exclude duplicate closure)
                vertex_total = sum(len(r) - 1 for p in all_polys for r in p) if all_polys else len(list(geom.vertices()))
                area_ha = geom.area() / 10000 if QgsWkbTypes.geometryType(geom.wkbType()) == QgsWkbTypes.PolygonGeometry else 0
                type_label = {QgsWkbTypes.PolygonGeometry:"面",
                              QgsWkbTypes.LineGeometry:"线"}.get(QgsWkbTypes.geometryType(geom.wkbType()), "点")
                parcel_label = (str(feat[parcel_field])
                                if parcel_field and parcel_field in feat.fields().names()
                                else f"地块{feat_idx}")

                f.write(f"{vertex_total},{round(area_ha,4)},{parcel_label},{type_label},,,,@\n")

                if all_polys:  # polygon
                    for poly in all_polys:
                        for ring_idx, ring in enumerate(poly):
                            ring_no = ring_idx + 1  # 1 outer, 2 inner ...
                            pts = ring
                            if pts[0] != pts[-1]:
                                pts.append(pts[0])
                            for vid, pt in enumerate(pts[:-1], 1):
                                f.write(f"J{vid:02d},{ring_no},{fmt % pt.x()},{fmt % pt.y()}\n")
                            first = pts[0]
                            f.write(f"J01,{ring_no},{fmt % first.x()},{fmt % first.y()}\n")
                else:  # line/point fallback
                    for vid, pt in enumerate(geom.vertices(), 1):
                        f.write(f"J{vid:02d},1,{fmt % pt.x()},{fmt % pt.y()}\n")

        return {self.OUTPUT: out_path}
