
# -*- coding: utf-8 -*-
"""
TXT → Vector
将坐标 TXT 导入为多环多边形图层（通过第二字段区分环）。
联系人：槐中路科科 996517087@qq.com
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm, QgsProcessingParameterFile,
    QgsProcessingParameterVectorDestination, QgsProcessingException,
    QgsProcessing, QgsVectorLayer, QgsFields, QgsField, QgsFeature,
    QgsGeometry, QgsPointXY, QgsProject
)
from PyQt5.QtCore import QVariant
import os, re

class TxtToVectorImporter(QgsProcessingAlgorithm):
    INPUT_TXT = "INPUT_TXT"
    OUTPUT = "OUTPUT"

    def name(self): return "txt_to_vector"
    def displayName(self): return self.tr("TXT → Vector")
    def group(self): return ""
    def groupId(self): return ""
    def shortHelpString(self):
        return self.tr(
            "解析坐标 TXT，外环第二字段=1，内环依次 2,3,…；生成多边形图层。\n"
            "联系人：槐中路科科 996517087@qq.com"
        )
    def createInstance(self): return TxtToVectorImporter()
    def tr(self, t): return QCoreApplication.translate("TxtToVectorImporter", t)

    def initAlgorithm(self, _config=None):
        self.addParameter(
            QgsProcessingParameterFile(self.INPUT_TXT, self.tr("输入 TXT 文件"), extension="txt")
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT, self.tr("输出矢量图层"), type=QgsProcessing.TypeVectorPolygon
            )
        )

    def processAlgorithm(self, params, context, feedback):
        txt = self.parameterAsFile(params, self.INPUT_TXT, context)
        if not os.path.isfile(txt):
            raise QgsProcessingException(self.tr("TXT 文件不存在。"))

        vertex_re = re.compile(r"^J\d+,\s*(\d+),\s*([+-]?\d+\.\d+),\s*([+-]?\d+\.\d+)")
        summary_line = re.compile(r"^\d+,")
        header_tags = ("[属性描述", "[地块坐标]")

        features_rings = []  # list of dict(label -> list[pt])
        rings = {}

        with open(txt, "r", encoding="utf-8") as fp:
            for line in fp:
                line = line.strip()
                if not line or any(line.startswith(tag) for tag in header_tags):
                    continue
                if summary_line.match(line):
                    if rings:
                        features_rings.append(rings)
                        rings = {}
                    continue
                m = vertex_re.match(line)
                if m:
                    label = int(m.group(1))
                    x, y = float(m.group(2)), float(m.group(3))
                    rings.setdefault(label, []).append(QgsPointXY(x, y))
        if rings:
            features_rings.append(rings)

        if not features_rings:
            raise QgsProcessingException(self.tr("未解析到坐标。"))

        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))
        mem = QgsVectorLayer("Polygon?crs=EPSG:4490", os.path.basename(txt), "memory")
        mem.dataProvider().addAttributes(fields)
        mem.updateFields()

        feats = []
        for fid, ring_map in enumerate(features_rings, 1):
            if 1 not in ring_map:
                continue
            outer = ring_map[1]
            if outer[0] != outer[-1]:
                outer.append(outer[0])
            rings_xy = [outer]
            for lab in sorted(k for k in ring_map if k > 1):
                ring = ring_map[lab]
                if ring[0] != ring[-1]:
                    ring.append(ring[0])
                rings_xy.append(ring)
            geom = QgsGeometry.fromPolygonXY(rings_xy)
            feat = QgsFeature(fields)
            feat.setAttributes([fid])
            feat.setGeometry(geom)
            feats.append(feat)

        mem.dataProvider().addFeatures(feats)
        mem.updateExtents()
        QgsProject.instance().addMapLayer(mem)
        return {self.OUTPUT: mem}
