# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CompareClassA
                                 A QGIS plugin
 Compare two datasets of GPKG pointZ geometry based on french legislation formatting "class A".
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-08-13
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Vincent Bénet
        email                : vincent.benet@outlook.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis._core import QgsExpression, QgsExpressionContext, QgsExpressionContextUtils, QgsCoordinateTransform, \
    QgsPointXY, QgsUnitTypes, QgsCoordinateReferenceSystem
from qgis.core import QgsProject

import numpy
from scipy import spatial
import matplotlib.patheffects as pe
from matplotlib import pyplot as plt, colors

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .compare_class_a_dialog import CompareClassADialog
import os.path


class CompareClassA:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CompareClassA_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Compare Class A')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CompareClassA', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/compare_class_a/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Compare'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Compare Class A'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = CompareClassADialog()
            self.dlg.reference.layerChanged.connect(lambda layer: self.dlg.expression_reference.setLayer(layer))
            self.dlg.result.layerChanged.connect(lambda layer: self.dlg.expression_result.setLayer(layer))

        self.dlg.result.setFilters(4)
        self.dlg.reference.setFilters(4)
        self.dlg.expression_reference.setLayer(self.dlg.reference.currentLayer())
        self.dlg.expression_result.setLayer(self.dlg.result.currentLayer())

        self.dlg.show()
        result = self.dlg.exec_()
        if result:
            elevation_conversions = {  # https://en.wikipedia.org/wiki/Foot_(unit)
                "Meter": 1,
                "International foot (1/3 yard)": 0.3048,
                "US Survey foot (12 inches)": 1200 / 3937,
                "Indian Survey foot (0.3047996 m)": 0.3047996,
                "Metric foot (1/3 m)": 1 / 3,
                "Hesse foot (1/4 m)": 1 / 4,
                "Baden foot (30 cm)": 0.3,
            }
            pos_reference = []
            layer_reference = self.dlg.reference.currentLayer()
            crs_reference = layer_reference.crs()
            unit_xy_reference = crs_reference.mapUnits()
            expression_reference = QgsExpression(self.dlg.expression_reference.expression())
            for feature in layer_reference.getFeatures():
                context = QgsExpressionContext()
                context.appendScopes(QgsExpressionContextUtils.globalProjectLayerScopes(layer_reference))
                context.setFeature(feature)
                x, y = feature.geometry().asPoint()
                z = expression_reference.evaluate(context) * elevation_conversions[self.dlg.reference_unit.currentText()]
                pos_reference.append([x, y, z])
            pos_reference = numpy.array(pos_reference)
            if unit_xy_reference != QgsUnitTypes.DistanceMeters:
                new_crs_reference = QgsCoordinateReferenceSystem("EPSG:3857")
                transform = QgsCoordinateTransform(crs_reference, new_crs_reference, QgsProject.instance().transformContext())
                crs_reference = new_crs_reference
                for i in range(len(pos_reference)):
                    x_new, y_new = transform.transform(QgsPointXY(pos_reference[i][0], pos_reference[i][1]))
                    pos_reference[i][0] = x_new
                    pos_reference[i][1] = y_new
            pos_result = []
            layer_result = self.dlg.result.currentLayer()
            crs_result = layer_result.crs()
            transform = QgsCoordinateTransform(crs_result, crs_reference, QgsProject.instance().transformContext())
            expression_result = QgsExpression(self.dlg.expression_result.expression())
            for feature in layer_result.getFeatures():
                context = QgsExpressionContext()
                context.appendScopes(QgsExpressionContextUtils.globalProjectLayerScopes(layer_result))
                context.setFeature(feature)
                pos = feature.geometry().asPoint()
                x, y = transform.transform(QgsPointXY(pos.x(), pos.y()))
                z = expression_result.evaluate(context) * elevation_conversions[self.dlg.result_unit.currentText()]
                pos_result.append([x, y, z])
            pos_result = numpy.array(pos_result)
            dists_xy, dists_z, pos_interp = compare_curves(pos_reference, pos_result, self.dlg.max_dist.value())
            scores = compare_pipe_to_pipe(dists_xy, dists_z)
            draw(scores, dists_xy, dists_z, pos_reference, pos_interp)


def draw(scores, diff_xy, diff_z, reference, res):
    nb = min(len(diff_xy), 50)
    ref_x, ref_y, ref_z = reference.T
    ref_abc = curvilinear_abs(reference)
    res_x, res_y, res_z, res_abc = res.T
    indexes = numpy.around(numpy.linspace(0, len(res), 10))
    mid_x = numpy.median(ref_x)
    mid_y = numpy.median(ref_y)
    fig, axs = plt.subplots(2, 2)
    axs[0][0].plot(ref_x - mid_x, ref_y - mid_y, color="black")
    axs[0][0].fill_between(
        ref_x - mid_x,
        ref_y - mid_y - 1.5,
        ref_y - mid_y + 1.5,
        color="red",
        interpolate=True,
        alpha=1,
        # edgecolor="white",
        label=f"XY 100% = {(scores['XY 100%'] * 100):.2f} / 150 cm ({(scores['XY 100% 150cm']*100):.2f} %)",
    )
    fig.legend().get_texts()[-1].set_color(["red", "green"][scores['XY 100% 150cm'] > 0])
    axs[0][0].fill_between(
        ref_x - mid_x,
        ref_y - mid_y - 0.4,
        ref_y - mid_y + 0.4,
        color="blue",
        interpolate=True,
        alpha=1,
        # edgecolor="white",
        label=f"XY 90%  = {(scores['XY 90%'] * 100):.2f} /  40 cm ({(scores['XY 90% 40cm']*100):.2f} %)",
    )
    axs[0][0].fill_between(
        ref_x - mid_x,
        ref_y - mid_y - 0.2,
        ref_y - mid_y + 0.2,
        color="green",
        interpolate=True,
        alpha=1,
        # edgecolor="white",
        label=f"XY 60%  = {(scores['XY 60%'] * 100):.2f} /  20 cm ({(scores['XY 60% 20cm']*100):.2f} %)",
    )
    axs[0][0].scatter(
        res_x - mid_x,
        res_y - mid_y,
        c=diff_xy,
        cmap=plt.cm.turbo,
        edgecolor="white"
    )
    for i, point in enumerate(numpy.array([res_x, res_y]).T):
        if i not in indexes:
            continue
        distance = diff_xy[i]
        x, y = point
        txt = f"{i} : {(distance * 100):.2f} cm"
        axs[0][0].annotate(
            text=txt,
            xy=(x - mid_x, y - mid_y),
            xytext=(5, 0),
            textcoords='offset pixels',
            path_effects=[pe.withStroke(linewidth=4, foreground="white")]
        )
    axs[1][0].axvspan(0, 1.5, color="red", alpha=1)
    axs[1][0].axvline(scores["XY 100%"], color="red", linewidth=1, path_effects=[pe.withStroke(linewidth=3, foreground="white")])
    axs[1][0].axvspan(0, 0.4, color="blue", alpha=1)
    axs[1][0].axvline(scores["XY 90%"], color="blue", linewidth=1, path_effects=[pe.withStroke(linewidth=3, foreground="white")])
    axs[1][0].axvspan(0, 0.2, color="green", alpha=1)
    axs[1][0].axvline(scores["XY 60%"], color="green", linewidth=1, path_effects=[pe.withStroke(linewidth=3, foreground="white")])
    n_xy, bins_xy, patches_xy = axs[1][0].hist(diff_xy, bins=nb, edgecolor='black')
    fracs = numpy.array([v for v in abs(bins_xy[:-1])])
    for thisfrac, thispatch in zip(fracs, patches_xy):
        thispatch.set_facecolor(plt.cm.turbo(colors.Normalize(fracs.min(), fracs.max())(thisfrac)))
    axs[1][1].axvspan(0, 0.7, color="red", alpha=1)
    axs[1][1].axvline(scores["Z 100%"], color="red", linewidth=1, path_effects=[pe.withStroke(linewidth=3, foreground="white")])
    axs[1][1].axvspan(0, 0.4, color="blue", alpha=1)
    axs[1][1].axvline(scores["Z 90%"], color="blue", linewidth=1, path_effects=[pe.withStroke(linewidth=3, foreground="white")])
    axs[0][1].plot(ref_abc, ref_z, color="black")
    axs[0][1].fill_between(
        ref_abc,
        ref_z - 0.7,
        ref_z + 0.7,
        color="red",
        interpolate=True,
        alpha=1,
        # edgecolor="white",
        label=f"Z 100%  = {(scores['Z 100%'] * 100):.2f} /  70 cm ({(scores['Z 100% 70cm']*100):.2f} %)",
    )
    axs[0][1].fill_between(
        ref_abc,
        ref_z - 0.4,
        ref_z + 0.4,
        color="blue",
        interpolate=True,
        alpha=1,
        # edgecolor="white",
        label=f"Z 90%   = {(scores['Z 90%'] * 100):.2f} /  40 cm ({(scores['Z 90% 40cm']*100):.2f} %)",
    )
    n_z, bins_z, patches_z = axs[1][1].hist(diff_z, bins=nb, edgecolor='black')
    fracs = numpy.array([v for v in abs(bins_z[:-1])])
    for thisfrac, thispatch in zip(fracs, patches_z):
        thispatch.set_facecolor(plt.cm.turbo(colors.Normalize(fracs.min(), fracs.max())(thisfrac)))
    axs[0][1].scatter(
        res_abc,
        res_z,
        c=diff_z,
        cmap=plt.cm.turbo,
        edgecolor="white"
    )
    for i, point in enumerate(numpy.array([res_abc, res_z]).T):
        if i not in indexes:
            continue
        distance = diff_z[i]
        a, z = point
        txt = f"{i} : {(distance * 100):.2f} cm"
        axs[0][1].annotate(
            text=txt,
            xy=(a, z),
            xytext=(0, 10),
            ha='left',
            rotation=90,
            textcoords='offset pixels',
            path_effects=[pe.withStroke(linewidth=4, foreground="white")]
        )
    axs[0][0].set_title('XY View')
    axs[0][0].set_xlabel('Coordinates X (meter)')
    axs[0][0].set_ylabel('Coordinates Y (meter)')
    axs[1][0].set_title(f'XY Distances : Score = {scores["percent_xy"]:.2f} %', color=["red", "green"][scores['ok_xy']])
    axs[1][0].set_xlabel('Distance XY (m)')
    axs[1][0].set_ylabel('Population')
    axs[1][0].set_xlim(left=0, right=max(diff_xy))
    axs[0][1].set_title('Z View')
    axs[0][1].set_xlabel('Curvilign Abcisse (m)')
    axs[0][1].set_ylabel('Elevation (m)')
    axs[1][1].set_title(f'Z Distances : Score = {scores["percent_z"]:.2f} %', color=["red", "green"][scores['ok_z']])
    axs[1][1].set_xlabel('Distance Z (m)')
    axs[1][1].set_ylabel('Population')
    axs[1][1].set_xlim(left=0, right=max(diff_z))
    txt = f""
    if not scores['ok']:
        txt += f"OUT OF CLASS A\n"
    txt += f"Score = {scores['percent']:.2f} %"
    fig.suptitle(txt, fontsize=16, color=["red", "green"][scores['ok']])
    fig.legend()
    plt.show()


def compare_pipe_to_pipe(dists_xy, dists_z):
    if not len(dists_xy):
        print(f"Exit because not len(dists_XY)")
    scores = {
        "XY 60%": numpy.percentile(dists_xy, 60),
        "XY 90%": numpy.percentile(dists_xy, 90),
        "XY 100%": numpy.percentile(dists_xy, 100),
        "Z 90%": numpy.percentile(dists_z, 90),
        "Z 100%": numpy.percentile(dists_z, 100),
    }
    scores["XY 60% 20cm"] = 1 - scores['XY 60%'] / 0.2
    scores["XY 90% 40cm"] = 1 - scores['XY 90%'] / 0.4
    scores["XY 100% 150cm"] = 1 - scores['XY 100%'] / 1.5
    scores["Z 90% 40cm"] = 1 - scores['Z 90%'] / 0.4
    scores["Z 100% 70cm"] = 1 - scores['Z 100%'] / 0.7
    scores['score_xy'] = scores['XY 60% 20cm'] + scores['XY 90% 40cm'] + scores['XY 100% 150cm']
    scores['score_z'] = scores['Z 90% 40cm'] + scores['Z 100% 70cm']
    scores['score'] = scores['score_xy'] + scores['score_z']
    scores["percent"] = scores['score'] / 5 * 100
    scores["percent_xy"] = scores['score_xy'] / 3 * 100
    scores["percent_z"] = scores['score_z'] / 2 * 100
    scores['ok_xy'] = not (
            scores['XY 60% 20cm'] < 0 or
            scores['XY 90% 40cm'] < 0 or
            scores['XY 100% 150cm'] < 0
    )
    scores["ok_z"] = not (
            scores['Z 90% 40cm'] < 0 or
            scores['Z 100% 70cm'] < 0
    )
    scores["ok"] = scores['ok_xy'] * scores["ok_z"]
    return scores


def compare_curves(reference, computed, max_dist, step=0.01):
    interp_reference = interp1d_curve_step(reference, step)
    abc_interp_reference = curvilinear_abs(interp_reference)
    tree = spatial.KDTree(interp_reference[..., :2])
    min_dist, index_min_dist = tree.query(computed[..., :2], k=1)
    axis_min_dist = interp_reference[index_min_dist, 2]
    dist_axis = numpy.abs(computed[..., 2] - axis_min_dist)
    mask = ~numpy.logical_or(index_min_dist == 0, index_min_dist == len(interp_reference) - 1)
    diff_xy = min_dist[mask]
    diff_z = dist_axis[mask]
    abc = abc_interp_reference[index_min_dist][mask]
    x, y, z = computed[mask].T
    res = numpy.array([x, y, z, abc]).T
    if max_dist > 0:
        mask = diff_xy < max_dist
        res = res[mask]
        diff_xy = diff_xy[mask]
        diff_z = diff_z[mask]
    return diff_xy, diff_z, res


def interp1d_curve_step(points, step):
    abs_cur = curvilinear_abs(points)
    interp_abs_cur = numpy.linspace(abs_cur[0], abs_cur[-1], int(abs_cur[-1] / step) + 1)
    interp_points = numpy.empty((interp_abs_cur.size, points.shape[1]))
    for i in range(points.shape[1]):
        interp_points[:, i] = numpy.interp(interp_abs_cur, abs_cur, points[:, i])
    return interp_points


def curvilinear_abs(points):
    return numpy.insert(numpy.cumsum(numpy.nansum((points[:-1] - points[1:]) ** 2, axis=1) ** 0.5), 0, 0)
