# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Clustering
								 A QGIS plugin
 Receives polygon shapefile as input and applies clustering methods for the generation of choropleth maps 
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
							  -------------------
		begin                : 2020-03-18
		git sha              : $Format:%H$
		copyright            : (C) 2020 by Prudencio T. and Maia B.
		email                : tiagoprudencio16@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from PyQt5.QtGui import QColor
from qgis.core import (QgsProcessing,
					   QgsFeatureSink,
					   QgsProcessingException,
					   QgsProcessingAlgorithm,
					   QgsProcessingParameterFeatureSource,
					   QgsProcessingParameterDefinition,
					   QgsProcessingParameterFeatureSink,
					   QgsProcessingParameterField,
					   QgsProcessingParameterNumber,
					   QgsField,
					   QgsVectorLayerUtils,
					   QgsFeature,
					   QgsSymbol,
					   QgsRendererCategory,
					   QgsCategorizedSymbolRenderer,
					   QgsProcessingUtils
					   )
#from qgis import processing
import os
import inspect
from qgis.PyQt.QtGui import QIcon
from sklearn.cluster import KMeans
from processing.gui.wrappers import WidgetWrapper
from ClusterMap.gui.ProcessingUI.kmeansWrapper import kmeansWrapper
from ClusterMap.classification.classification import classification
from sklearn.metrics import  silhouette_score, silhouette_samples
import numpy as np



class KMeansClusteringAlgorithm(QgsProcessingAlgorithm):
	"""
	This is an example algorithm that takes a vector layer and
	creates a new identical one.

	It is meant to be used as an example of how to create your own
	algorithms and explain methods and variables used to do it. An
	algorithm like this will be available in all elements, and there
	is not need for additional work.

	All Processing algorithms should extend the QgsProcessingAlgorithm
	class.
	"""

	# Constants used to refer to parameters and outputs. They will be
	# used when calling the algorithm from another algorithm, or when
	# calling from the QGIS console.

	PARAMETERS = 'PARAMETERS'
	N_CLUSTERS = 'N_CLUSTERS'
	OUTPUT = 'OUTPUT'
	
	def initAlgorithm(self, config=None):
		"""
		Here we define the inputs and output of the algorithm, along
		with some other properties.
		"""

		# We add the input vector features source. It can have any kind of
		# geometry.

		slot = ParameterLayer(
			self.PARAMETERS,
			description =self.tr(''),
			)

		slot.setMetadata({
			'widget_wrapper': kmeansWrapper 
		})

		self.addParameter(slot)
		
		
		self.addParameter(
			QgsProcessingParameterNumber(
				self.N_CLUSTERS,
				self.tr('Number of clusters'),
				type=QgsProcessingParameterNumber.Integer,
				defaultValue=2,
				minValue=2,
				maxValue=10
			)
		)

		# We add a feature sink in which to store our processed features (this
		# usually takes the form of a newly created vector layer when the
		# algorithm is run in QGIS).
		self.addParameter(
			QgsProcessingParameterFeatureSink(
				self.OUTPUT,
				self.tr('Output layer')
			)
		)

	def parameterAsClustering(self, parameters, name, context):
		return parameters[name]

		
	def processAlgorithm(self, parameters, context, feedback):
		"""
		Here is where the processing itself takes place.
		"""

		# Retrieve the feature source and sink. The 'dest_id' variable is used
		# to uniquely identify the feature sink, and must be included in the
		# dictionary returned by the processAlgorithm function.

		param = self.parameterAsClustering(
			parameters,
			self.PARAMETERS,
			context
		)

		source = param['layer']
		X = param['dataset']
		excluded = param['id']
		attr = param['attributes'] 

		n_clusters = self.parameterAsInt(
			parameters,
			self.N_CLUSTERS,
			context
		)

		# If source was not found, throw an exception to indicate that the algorithm
		# encountered a fatal error. The exception text can be any string, but in this
		# case we use the pre-built invalidSourceError method to return a standard
		# helper text for when a source cannot be evaluated
		if source is None:
			raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
		fields = source.fields()
		fields.append(QgsField("cluster", QVariant.Int))

		(sink, dest_id) = self.parameterAsSink(
			parameters,
			self.OUTPUT,
			context,
			fields,
			source.wkbType(),
			source.sourceCrs()
		)

		# If sink was not created, throw an exception to indicate that the algorithm
		# encountered a fatal error. The exception text can be any string, but in this
		# case we use the pre-built invalidSinkError method to return a standard
		# helper text for when a sink cannot be evaluated
		if sink is None:
			raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

		# Compute the number of steps to display within the progress bar and
		# get features from source
		total = 100.0 / source.featureCount() if source.featureCount() else 0
		features = source.getFeatures()

		kmeans = KMeans(
			n_clusters=n_clusters,
			random_state= 0).fit(X)

		current = 0
		X_ = list()
		for feature in (features):
			# Stop the algorithm if cancel button has been clicked
			if feedback.isCanceled():
				break
			newFeat = QgsFeature(fields)
			for field in source.fields():
				newFeat[field.name()] = feature[field.name()]
			newFeat.setGeometry(feature.geometry())
			# Add a feature in the sink
			if feature.id() in excluded:
				pass
			else:
				newFeat['cluster'] = int(kmeans.labels_[current])
				current =current+1
				aux = [feature[field] for field in attr ]
				X_.append(aux)

			sink.addFeature(newFeat, QgsFeatureSink.FastInsert)

			# Update the progress bar
			feedback.setProgress(int(current * total))

		feedback.pushInfo('\n####### THE AVERAGE SILLHOUETTE SCORE EACH CLUSTER ####### \n')
		sample_silhouette_values = silhouette_samples(X, kmeans.labels_, metric = 'euclidean')	
		for i in range(n_clusters):
			value = np.mean(sample_silhouette_values[kmeans.labels_ == i])
			feedback.pushInfo('Cluster ' + str(i) + ' the average silhouette_score is: '+ str(value) +'\n')

		score  = silhouette_score(X, kmeans.labels_, metric='euclidean')
		feedback.pushInfo('\n####### THE AVERAGE TOTAL SILLHOUETTE SCORE ####### \n')
		feedback.pushInfo('The average total silhouette_score is: '+ str(score)+'\n')

		self.legends = classification(X_,kmeans.labels_,attr).decisionTree()

		feedback.pushInfo('\n####### RULES OF A DECISION TREE #######'+'\n')

		for i in sorted(self.legends.keys()):
			feedback.pushInfo('Class ' + i + ':\n ' + self.legends[i] + '\n')
	
		self.dest_id=dest_id
		return {self.OUTPUT: dest_id}

	# Create Symbol Renderer
	def postProcessAlgorithm(self, context, feedback):
		output = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)
	
		my_classes = {0: list([[43,131,186,255]]),
			  1: list([[241,96,93,255]]),
			  2: list([[157,211,167,255]]),
			  3: list([[237,248,185,255]]),
			  4: list([[232,221,58,255]]),
			  5: list([[249,158,89,255]]),
			  6: list([[108,206,89,255]]),
			  7: list([[137,107,178,255]]),
			  8: list([[205,63,113,255]]),
			  9: list([[215,25,28,255]])
			  }

		rules = dict()
		for i in range(len(self.legends.keys())):
			my_classes[i].append(self.legends[str(i)])
			rules[i] = my_classes[i]

		categories = []
		for class_value, (symbol_property, label_text) in rules.items():

			# get default symbol for this geometry type
			symbol = QgsSymbol.defaultSymbol(output.geometryType())

			# symbol.set*(symbol_property)
			symbol.setColor(QColor(symbol_property[0],symbol_property[1],symbol_property[2],symbol_property[3]))

			# create a category with these properties
			category = QgsRendererCategory(class_value, symbol, label_text)
			categories.append(category)

		mi_rend = QgsCategorizedSymbolRenderer('cluster', categories) 
		output.setRenderer(mi_rend)
		output.triggerRepaint()
		return {self.OUTPUT: self.dest_id}

	def tr(self, string):
		"""
		Returns a translatable string with the self.tr() function.
		"""
		return QCoreApplication.translate('Processing', string)

	def createInstance(self):
		return KMeansClusteringAlgorithm()

	def name(self):
		"""
		Returns the algorithm name, used for identifying the algorithm. This
		string should be fixed for the algorithm, and must not be localised.
		The name should be unique within each provider. Names should contain
		lowercase alphanumeric characters only and no spaces or other
		formatting characters.
		"""
		return 'kmeansclustering'

	def displayName(self):
		"""
		Returns the translated algorithm name, which should be used for any
		user-visible display of the algorithm name.
		"""
		return self.tr('K-Means Clustering')

	def group(self):
		"""
		Returns the name of the group this algorithm belongs to. This string
		should be localised.
		"""
		return self.tr('Clustering Methods')

	def groupId(self):
		"""
		Returns the unique ID of the group this algorithm belongs to. This
		string should be fixed for the algorithm, and must not be localised.
		The group id should be unique within each provider. Group id should
		contain lowercase alphanumeric characters only and no spaces or other
		formatting characters.
		"""
		return 'clusteringmethods'

	def icon(self):
		"""
		Should return a QIcon which is used for your provider inside
		the Processing toolbox.
		"""
		cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
		icon = QIcon(os.path.join(os.path.join(cmd_folder, 'cluster2.png')))
		return icon


	def shortHelpString(self):
		"""
		Returns a localised short helper string for the algorithm. This string
		should provide a basic description about what the algorithm does and the
		parameters and outputs associated with it..
		"""
		return self.tr("The k-means method finds k different clusters in the data set. The center of each cluster will be called a centroid and will have the average of the values in this cluster. The task of the algorithm is to find the nearest centroid (using euclidian distance) and assign the point found to that cluster. Centroids are updated always taking the average value of all points in that cluster. For this method, numerical values are needed to calculate the distance, the nominal values can then be mapped into binary values for the same calculation. In case of success, the data are separated organically and can thus be labeled and centroid become a reference to classify new dataares")

class ParameterLayer(QgsProcessingParameterDefinition):

	def __init__(self, name, description='', parentLayerParameterName='INPUT'):
		super().__init__(name, description)
		self._parentLayerParameter = parentLayerParameterName

	def clone(self):
		copy = ParameterLayer(self.name(), self.description(), self._parentLayerParameter)
		return copy

	def type(self):
			return self.typeName()

	@staticmethod
	def typeName():
		return 'method_graph'
	def checkValueIsAcceptable(self, value, context=None):
		#if not isinstance(value, list):
		#   return False
		#for field_def in value:
		#    if not isinstance(field_def, dict):
		#        return False
		#    if 'name' not in field_def.keys():
		#         return False
		#    if 'type' not in field_def.keys():
		#        return False
		#    if 'expression' not in field_def.keys():
		#        return False
		return True

	def valueAsPythonString(self, value, context):
		return str(value)

	def asScriptCode(self):
		raise NotImplementedError()

	@classmethod
	def fromScriptCode(cls, name, description, isOptional, definition):
		raise NotImplementedError()

	def parentLayerParameter(self):
		return self._parentLayerParameter