# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ClusteringMainDialog
                                 A QGIS plugin
 Perform clustering on vector layers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2021-03-08
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Andrea Folini
        email                : andreafolini96@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
from qgis.PyQt import QtGui, uic, QtCore
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtCore import Qt, QVariant
from qgis.core import QgsField, QgsVectorLayer, QgsProject, QgsVectorLayerExporter
from PyQt5.QtWidgets import QFileDialog
from PyQt5 import QtGui
import numpy as np
import pandas as pd
import json
from sklearn.preprocessing import StandardScaler, MinMaxScaler
from sklearn.feature_selection import VarianceThreshold
from sklearn import metrics
import matplotlib.pyplot as plt
from sklearn.cluster import AgglomerativeClustering, KMeans
from scipy.cluster.hierarchy import dendrogram, linkage, fcluster
from scipy.spatial.distance import cdist, pdist
from scipy import stats
from qgis.core import QgsMapLayerProxyModel, QgsFieldProxyModel

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'cluster_analysis_dialog_base.ui'))


class ClusteringMainDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, iface, parent=None):
        """Constructor."""
        super(ClusteringMainDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.iface = iface
        self.experiments = []
        self.configuration = self.config()
        self.setWindowTitle('Cluster analysis')
        # Connect UI widgets' signals to functions
        self.layerBox.currentIndexChanged.connect(self.populate_features)
        self.addButton.clicked.connect(self.add_features)
        self.addAllButton.clicked.connect(self.add_all)
        self.delSelButton.clicked.connect(self.delete_selected)
        self.delAllButton.clicked.connect(self.delete_all)
        self.runButton.clicked.connect(self.run_clustering)
        self.plotButton.clicked.connect(self.plot_clusters)
        self.featureButton.clicked.connect(self.automated_features)
        self.expListBox.currentIndexChanged.connect(self.select_experiment)
        self.plotGraph.clicked.connect(self.plot_graph)
        self.loadButton.clicked.connect(self.load_experiment)
        self.saveButton.clicked.connect(self.save_experiment)
        self.addExpButton.clicked.connect(self.add_score_exp)
        self.delSelExpButton.clicked.connect(self.delete_selected_score_exp)
        self.delAllExpButton.clicked.connect(self.delete_all_score_exp)
        self.scoreButton.clicked.connect(self.compute_score)
        self.correlationCheck.stateChanged.connect(self.corr_enable)
        self.standData.clicked.connect(self.norm_uncheck)
        self.normData.clicked.connect(self.stand_uncheck)
        self.clustAlgo.currentIndexChanged.connect(self.plot_enable)
        self.cleaningButton.clicked.connect(self.run_selection)
        self.populate_algorithm()
        self.delete_all()
        self.layerBox.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.cleanLayerBox.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.featureBox.setLayer(self.layerBox.currentLayer())
        self.featureBox.setFilters(QgsFieldProxyModel.Numeric)
        self.domCheck.clicked.connect(self.dom_enable)
        self.quasiCheck.clicked.connect(self.quasiClicked)
        self.constantCheck.clicked.connect(self.constantClicked)
        self.constantCheck.clicked.connect(self.constantClicked)
        self.firstCheck.clicked.connect(self.dom_enable)
        self.distCheck.clicked.connect(self.dom_enable)
        # Regular expression used to allow only numeric inputs in a text box
        regex = QtCore.QRegExp("^(-?)(0|([1-9][0-9]*))(\\.[0-9]+)?$")
        validator = QtGui.QRegExpValidator(regex)
        self.minDom.setValidator(validator)
        self.minDom.setValidator(validator)

    # Function called when the window is closed
    def closeEvent(self, event):
        self.reset_form()

    # Load configuration file
    def config(self):
        dir_path = os.path.dirname(os.path.realpath(__file__))
        config_path = dir_path + "/Configuration.json"
        with open(config_path) as jsonFile:
            json_object = json.load(jsonFile)
            jsonFile.close()
        freq_cut = json_object['frequency_cut']
        unique_cut = json_object['unique_cut']
        ent_iter = json_object['entropy_iterations']
        sample_size = json_object['sample_size']
        max_cluster = json_object['graph_max_cluster']
        distance = json_object['distance']
        return Configuration(freq_cut, unique_cut, ent_iter, sample_size, max_cluster, distance)

    # Clean everything after closing the window
    def reset_form(self):
        self.correlationCheck.setChecked(False)
        self.constantCheck.setChecked(False)
        self.quasiCheck.setChecked(False)
        self.correlationThreshold.setValue(0.80)
        self.firstCheck.setChecked(True)
        self.minDom.clear()
        self.maxDom.clear()
        self.selMessageBox.clear()
        self.featureList.clear()
        self.clustAlgo.setCurrentIndex(0)
        self.algoBox.setCurrentIndex(0)
        self.standData.setChecked(True)
        self.normData.setChecked(False)
        self.expName.clear()
        self.newFieldName.clear()
        self.numClusterBox.setValue(0)
        self.messageBox.clear()
        self.expListBox.clear()
        self.experiments = []
        self.experimentFeatures.clear()
        self.expClustAlgo.clear()
        self.experimentNCluster.clear()
        self.expSil.clear()
        self.expDB.clear()
        self.compExpBox.clear()
        self.scoreExpList.clear()
        self.scoreBox.clear()
        self.resMessageBox.clear()

    def dom_enable(self):
        if self.domCheck.isChecked():
            self.minDom.setEnabled(True)
            self.maxDom.setEnabled(True)
        else:
            self.minDom.setEnabled(False)
            self.maxDom.setEnabled(False)

    def norm_uncheck(self):
        if self.normData.isChecked:
            self.normData.setChecked(False)

    def stand_uncheck(self):
        if self.standData.isChecked:
            self.standData.setChecked(False)

    def plot_enable(self):
        if self.clustAlgo.currentText() == "Hierarchical":
            self.plotButton.setEnabled(True)
        else:
            self.plotButton.setEnabled(False)

    def quasiClicked(self):
        if self.quasiCheck.isChecked():
            self.constantCheck.setChecked(True)

    def constantClicked(self):
        if not self.constantCheck.isChecked():
            self.quasiCheck.setChecked(False)

    def corr_enable(self):
        if self.correlationCheck.isChecked():
            self.correlationThreshold.setEnabled(True)
            self.firstCheck.setEnabled(True)
            self.lowCorrCheck.setEnabled(True)
            self.distCheck.setEnabled(True)
            self.domCheck.setEnabled(True)
            if self.domCheck.isChecked():
                self.minDom.setEnabled(True)
                self.maxDom.setEnabled(True)
        else:
            self.correlationThreshold.setEnabled(False)
            self.firstCheck.setEnabled(False)
            self.lowCorrCheck.setEnabled(False)
            self.distCheck.setEnabled(False)
            self.domCheck.setEnabled(False)
            self.minDom.setEnabled(False)
            self.maxDom.setEnabled(False)

    def populate_algorithm(self):
        self.algoBox.clear()
        self.algoBox.addItem('Entropy')
        self.algoBox.addItem('Sampling Entropy')
        self.clustAlgo.clear()
        self.clustAlgo.addItem('Hierarchical')
        self.clustAlgo.addItem('K-Means')

    def populate_features(self):
        self.delete_all()
        self.featureBox.setLayer(self.layerBox.currentLayer())
        self.featureBox.setFilters(QgsFieldProxyModel.Numeric)

    def add_features(self):
        feature = self.featureBox.currentText()
        # avoid adding empty item
        if feature == "":
            return
        if not self.featureList.findItems(feature, Qt.MatchFixedString | Qt.MatchCaseSensitive):
            self.featureList.addItem(feature)

    def add_all(self):
        for i in range(self.featureBox.count()):
            feature = self.featureBox.itemText(i)
            if not self.featureList.findItems(feature, Qt.MatchFixedString | Qt.MatchCaseSensitive):
                self.featureList.addItem(feature)

    def delete_selected(self):
        selected = self.featureList.selectedItems()
        for item in selected:
            row = self.featureList.row(item)
            self.featureList.takeItem(row)

    def delete_all(self):
        self.featureList.clear()

    # Main function for clustering. Called after pressing "Run" button in the clustering window
    def run_clustering(self):
        selectedfeatures = self.select_features()
        numcluster = int(self.numClusterBox.text())
        exp_name = self.expName.toPlainText()
        # Check that every field is filled out correctly. If not output an error message and exit the function
        if not selectedfeatures:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No features selected')
            return
        full_dataset = self.create_table("clustering")
        n_samples = full_dataset.shape[0]
        if numcluster < 2:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('Number of clusters must be higher than 1')
            return
        elif numcluster >= n_samples:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('Number of clusters must be lower than the number of samples')
            return
        if not exp_name:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No experiment name selected')
            return
        if not self.newFieldName.toPlainText():
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No new layer field name selected')
            return
        # Check for duplicate experiment name
        for exp in self.experiments:
            if exp.get_name() == exp_name:
                self.messageBox.setStyleSheet("QLabel {  color : red; }")
                self.messageBox.setText('The experiment name is already used. Please enter a different one')
                return
        # Check if the data needs to be scaled
        scaling = "no"
        if self.standData.isChecked():
            scaling = "stand"
        if self.normData.isChecked():
            scaling = "norm"
        selected_data = full_dataset[selectedfeatures]
        if scaling == "stand":
            scaler = StandardScaler()
            selected_data = pd.DataFrame(data=scaler.fit_transform(X=selected_data), columns=selected_data.columns)
        elif scaling == "norm":
            scaler = MinMaxScaler()
            selected_data = pd.DataFrame(data=scaler.fit_transform(X=selected_data), columns=selected_data.columns)
        # Calculate all the parameters of the experiment and create the new object
        clust_algo = self.clustAlgo.currentText()
        clustering_labels = self.calc_clusters(selected_data, clust_algo, numcluster)
        sil_index = round(metrics.silhouette_score(X=selected_data, labels=clustering_labels), 7)
        db_index = round(metrics.davies_bouldin_score(X=selected_data, labels=clustering_labels), 7)
        experiment = Experiment(features=selectedfeatures, num_cluster=numcluster, name=exp_name,
                                cluster_method=clust_algo, clusters=clustering_labels, sil_index=sil_index,
                                db_index=db_index)
        self.experiments.append(experiment)
        self.expListBox.addItem(experiment.get_name())
        self.add_new_field(experiment)
        self.messageBox.setStyleSheet("QLabel {  color : green; }")
        self.messageBox.setText('Experiment ' + exp_name + ' complete')
        # Reload the compatible experiments for the selected one
        current_exp_index = self.expListBox.currentIndex()
        current_experiment = self.experiments[current_exp_index]
        self.select_compatible_experiments(current_experiment)

    # Return cluster labels for the specified dataset, algorithm for clustering and selected number of clusters
    def calc_clusters(self, dataset, clust_algo, num_cluster):
        if clust_algo == "Hierarchical":
            clustering = AgglomerativeClustering(n_clusters=num_cluster, affinity=self.configuration.get_distance(),
                                                 linkage='complete')
        elif clust_algo == "K-Means":
            clustering = KMeans(n_clusters=num_cluster)
        return clustering.fit_predict(dataset)

    # Get the informations about the experiment selected in the Results window
    def select_experiment(self):
        exp_index = self.expListBox.currentIndex()
        experiment = self.experiments[exp_index]
        self.experimentFeatures.clear()
        self.experimentNCluster.clear()
        self.expSil.clear()
        self.expDB.clear()
        self.expClustAlgo.clear()
        for feature in experiment.get_features():
            self.experimentFeatures.appendPlainText(feature)
        self.experimentNCluster.appendPlainText(str(experiment.get_num_cluster()))
        self.expSil.appendPlainText(str(experiment.get_sil_index()))
        self.expDB.appendPlainText(str(experiment.get_db_index()))
        self.expClustAlgo.appendPlainText(experiment.get_cluster_method())
        self.select_compatible_experiments(experiment)

    # Check which experiments have the same number of samples of the selected experiment (the experiments that can be
    # used to calculate the score with the comparison matrix)
    def select_compatible_experiments(self, current_exp):
        self.compExpBox.clear()
        self.scoreExpList.clear()
        self.scoreBox.clear()
        exp_instances = len(current_exp.get_clusters())
        compatible_exp = []
        for exp in self.experiments:
            if len(exp.get_clusters()) == exp_instances and (not current_exp.get_name() == exp.get_name()):
                compatible_exp.append(exp)
        for exp in compatible_exp:
            self.compExpBox.addItem(exp.get_name())
        self.scoreExpList.addItem(current_exp.get_name())

    def add_score_exp(self):
        exp = self.compExpBox.currentText()
        if not self.scoreExpList.findItems(exp, Qt.MatchFixedString | Qt.MatchCaseSensitive):
            self.scoreExpList.addItem(exp)



    def delete_selected_score_exp(self):
        selected = self.scoreExpList.currentItem()
        if not selected.text() == self.expListBox.currentText():
            row = self.scoreExpList.row(selected)
            self.scoreExpList.takeItem(row)

    def delete_all_score_exp(self):
        self.scoreExpList.clear()
        exp = self.expListBox.currentText()
        self.scoreExpList.addItem(exp)

    # Add the clustering labels to the vector file
    def add_new_field(self, experiment):
        layer = self.layerBox.currentLayer()
        new_field_name = self.newFieldName.toPlainText()
        original_name = new_field_name
        # Check for fields with the same name and assign a new one if a duplicate is found
        name_prefix = 1
        while layer.fields().indexFromName(new_field_name) != -1:
            new_field_name = str(name_prefix) + '_' + original_name
            name_prefix += 1
        layer.dataProvider().addAttributes([QgsField(new_field_name, QVariant.Int, len=6)])
        layer.updateFields()
        layer.startEditing()
        field_index = layer.fields().indexFromName(new_field_name)
        instances = layer.getFeatures()
        i = 1
        cluster_list = experiment.get_clusters()
        for instance in instances:
            layer.changeAttributeValue(instance.id(), field_index, int(cluster_list[i-1]))
            i += 1
        layer.updateFields()
        layer.commitChanges()

    # Automatic feature selection for clustering
    def automated_features(self):
        if not self.layerBox.currentText():
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No layer selected')
            return
        dataset = self.create_table("clustering")
        algorithm = self.algoBox.currentText()
        n_samples = self.configuration.get_sample_size()
        if algorithm == 'Sampling Entropy' and dataset.shape[0] <= n_samples:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('The selected dataset does not have enough instances to use Sampling Entropy')
            return
        self.delete_all()

        # If new feature selection algorithms needs to be added, add conditions to this if statement
        if algorithm == 'Entropy':
            selected_features = self.calc_entropy_features(dataset)
        elif algorithm == 'Sampling Entropy':

            selected_features = self.calc_sampling_entropy_features(dataset)
        for feature in selected_features:
            self.featureList.addItem(feature)

    # Load a previously saved experiment and add it to the Results window
    def load_experiment(self):
        dir_path = os.path.dirname(os.path.realpath(__file__))
        exp_dir = dir_path + "/Experiments"
        filename, _filter = QFileDialog.getOpenFileName(self, "Select experiment", exp_dir, "*.txt")
        try:
            file = open(filename, "r")
        except FileNotFoundError:
            return
        exp_features = []
        exp_clusters = []
        exp_name = file.readline().rstrip('\n')
        exp_nclust = file.readline().rstrip('\n')
        exp_clust_algo = file.readline().rstrip('\n')
        exp_sil_index = file.readline().rstrip('\n')
        exp_db_index = file.readline().rstrip('\n')
        file.readline()
        current_read = file.readline().rstrip('\n')
        while not(current_read == '[Clusters]:'):
            exp_features.append(current_read)
            current_read = file.readline().rstrip('\n')
        current_read = file.readline().rstrip('\n')
        while not (current_read == ''):
            exp_clusters.append(current_read)
            current_read = file.readline().rstrip('\n')
        file.close()
        name_dup = 1
        new_exp_name = exp_name
        while self.expListBox.findText(new_exp_name) != -1:
            new_exp_name = exp_name + "(" + str(name_dup) + ")"
            name_dup += 1
        experiment = Experiment(name=new_exp_name, num_cluster=exp_nclust, features=exp_features, clusters=exp_clusters,
                                cluster_method=exp_clust_algo, sil_index=exp_sil_index, db_index=exp_db_index)
        self.experiments.append(experiment)
        self.expListBox.addItem(experiment.get_name())
        # Reload the compatible experiments for the selected one
        current_exp_index = self.expListBox.currentIndex()
        current_experiment = self.experiments[current_exp_index]
        self.select_compatible_experiments(current_experiment)
        self.resMessageBox.setStyleSheet("QLabel {  color : green; }")
        self.resMessageBox.setText(new_exp_name + ' loaded correctly')

    # Save the selected experiment
    def save_experiment(self):
        dir_path = os.path.dirname(os.path.realpath(__file__))
        exp_index = self.expListBox.currentIndex()
        experiment = self.experiments[exp_index]
        exp_name = experiment.get_name()
        exp_nclust = experiment.get_num_cluster()
        exp_features = experiment.get_features()
        exp_clusters = experiment.get_clusters()
        exp_clust_algo = experiment.get_cluster_method()
        exp_sil_index = experiment.get_sil_index()
        exp_db_index = experiment.get_db_index()
        name_dup = 1
        new_name = exp_name
        while os.path.exists(dir_path + "/Experiments/" + new_name + ".txt"):
            new_name = exp_name + "(" + str(name_dup) + ")"
            name_dup += 1
        file = open(dir_path + "/Experiments/" + new_name + ".txt", "x")
        file.write(exp_name + '\n')
        file.write(str(exp_nclust) + '\n')
        file.write(exp_clust_algo + '\n')
        file.write(str(exp_sil_index) + '\n')
        file.write(str(exp_db_index) + '\n')
        file.write('[Features]:\n')
        for feature in exp_features:
            file.write(feature + '\n')
        file.write('[Clusters]:\n')
        for cluster in exp_clusters:
            file.write(str(cluster) + '\n')
        file.close()
        self.resMessageBox.setStyleSheet("QLabel {  color : green; }")
        self.resMessageBox.setText(exp_name + ' saved correctly')

    # Plot the dendrogram from hierarchical clustering
    def plot_clusters(self):
        selected_features = self.select_features()
        if not selected_features:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No features selected')
            return
        data = self.create_table("clustering")[selected_features]
        scaling = "no"
        if self.standData.isChecked():
            scaling = "stand"
        if self.normData.isChecked():
            scaling = "norm"
        if scaling == "stand":
            scaler = StandardScaler()
            data = pd.DataFrame(data=scaler.fit_transform(X=data), columns=data.columns)
        elif scaling == "norm":
            scaler = MinMaxScaler()
            data = pd.DataFrame(data=scaler.fit_transform(X=data), columns=data.columns)
        distance = self.configuration.get_distance()
        # in scipy library manhattan is called cityblock
        if distance == "manhattan":
            distance = "cityblock"
        cluster_tree = linkage(y=data, method='complete', metric=distance)
        dendrogram(cluster_tree)
        plt.title('Hierarchical Clustering Dendrogram')
        plt.xlabel('Sample index')
        plt.ylabel('Distance')
        plt.show()

    # Plot the WSS and BSS graph from both clustering algorithms
    def plot_graph(self):
        selected_features = self.select_features()
        if not selected_features:
            self.messageBox.setStyleSheet("QLabel {  color : red; }")
            self.messageBox.setText('No features selected')
            return
        wss_values = []
        bss_values = []
        max_cluster = self.configuration.get_max_cluster()
        k_values = range(1, max_cluster + 1)
        data = self.create_table("clustering")[selected_features]
        scaling = "no"
        if self.standData.isChecked():
            scaling = "stand"
        if self.normData.isChecked():
            scaling = "norm"
        if scaling == "stand":
            scaler = StandardScaler()
            data = pd.DataFrame(data=scaler.fit_transform(X=data), columns=data.columns)
        elif scaling == "norm":
            scaler = MinMaxScaler()
            data = pd.DataFrame(data=scaler.fit_transform(X=data), columns=data.columns)

        cluster_method = self.clustAlgo.currentText()
        data = data.to_numpy()

        for k in k_values:
            if cluster_method == "Hierarchical":
                clustering = AgglomerativeClustering(n_clusters=k, affinity=self.configuration.get_distance(),
                                                     linkage='complete')
            elif cluster_method == "K-Means":
                clustering = KMeans(n_clusters=k)

            clusters = clustering.fit_predict(data)
            centroids = [np.mean(data[clusters == c], axis=0) for c in range(0, k)]
            D = cdist(data, centroids, 'euclidean')
            d = np.min(D, axis=1)
            wss = sum(d**2)
            tss = sum(pdist(data)**2) / data.shape[0]
            bss = tss - wss
            wss_values.append(wss)
            bss_values.append(bss)
        fig = plt.figure(figsize=(8, 6))
        font = {'family': 'sans', 'size': 16}
        plt.rc('font', **font)
        plt.plot(k_values, wss_values, 'bo-', color='red', label='WSS')
        plt.plot(k_values, bss_values, 'bo-', color='blue', label='BSS')
        plt.grid(True)
        plt.xlabel('Number of clusters')
        plt.ylabel('BSS & WSS')
        plt.title('BSS & WSS trends')
        plt.xticks(k_values)
        plt.legend()
        plt.show()

    # Get the features selected for clustering
    def select_features(self):
        selectedfeatures = [str(self.featureList.item(i).text()) for i in range(self.featureList.count())]
        return selectedfeatures

    # Calculate the score among experiments with the comparison matrix
    def compute_score(self):
        self.scoreBox.clear()
        score_experiments = []
        base_exp_index = self.expListBox.currentIndex()
        base_experiment = self.experiments[base_exp_index]
        score_experiments.append(base_experiment)
        for i in range(self.scoreExpList.count()):
            exp_name = str(self.scoreExpList.item(i).text())
            for item in self.experiments:
                if exp_name == item.get_name():
                    score_experiments.append(item)
        cluster_list = []
        for exp in score_experiments:
            clustering = exp.get_clusters()
            cluster_list.append(clustering)

        exp_num = len(score_experiments)
        num_inst = len(cluster_list[0])
        comp_matrix = self.comparison_matrix(cluster_list, num_inst)
        score_sum = 0
        # Add 1 to score only if two samples are never in the same cluster or if they are in the same cluster for
        # every experiment
        for i in range(num_inst):
            for j in range(num_inst):
                if comp_matrix[i][j] == 0 or comp_matrix[i][j] == exp_num:
                    score_sum += 1
        score = score_sum/(num_inst**2)
        score = round(score, 5)
        self.scoreBox.appendPlainText(str(score))

    # Return the comparison matrix
    def comparison_matrix(self, cluster_list, num_inst):
        comp_matrix = np.zeros((num_inst, num_inst))
        # Add 1 to the matrix cell i,j if the samples i and j are in the same cluster for two experiments
        # Only iterate on half matrix since it's symmetric
        for i in range(num_inst):
            for j in range(i, num_inst):
                for clusters in cluster_list:
                    if clusters[i] == clusters[j]:
                        comp_matrix[i][j] += 1
                        if not i == j:
                            comp_matrix[j][i] += 1
        return comp_matrix

    # Main function for the feature selection window
    def run_selection(self):
        # Check that every field is filled out correctly. If not output an error message and exit the function
        if not self.cleanLayerBox.currentLayer():
            self.selMessageBox.setStyleSheet("QLabel {  color : red; }")
            self.selMessageBox.setText('Select a layer')
            return
        if (not self.correlationCheck.isChecked()) and (not self.constantCheck.isChecked()) and\
                (not self.quasiCheck.isChecked()):
            self.selMessageBox.setStyleSheet("QLabel {  color : red; }")
            self.selMessageBox.setText('Select at least one operation')
            return
        data = self.create_table("selection")
        if self.correlationCheck.isChecked():
            if self.domCheck.isChecked():
                if self.minDom.text() == "" or self.maxDom.text() == "":
                    self.selMessageBox.setStyleSheet("QLabel {  color : red; }")
                    self.selMessageBox.setText('Enter min and max domain values')
                    return
                if float(self.maxDom.text()) <= float(self.minDom.text()):
                    self.selMessageBox.setStyleSheet("QLabel {  color : red; }")
                    self.selMessageBox.setText('Max domain value is smaller than min value')
                    return
            corr_threshold = self.correlationThreshold.value()
            data = self.cross_correlation(data, corr_threshold)
        if self.constantCheck.isChecked():
            # VarianceThreshold raises an exception if all the features are below the selected threshold
            try:
                data = self.low_variance(data, 0)
            except ValueError:
                self.selMessageBox.setStyleSheet("QLabel {  color : red; }")
                self.selMessageBox.setText('There are no features that fulfill the requirements')
        if self.quasiCheck.isChecked():
            freq_threshold = self.configuration.get_frequency_cut()
            unique_values_threshold = self.configuration.get_unique_cut()
            data = self.near_zero_var(data, freq_threshold, unique_values_threshold)
        layer = self.cleanLayerBox.currentLayer()
        # Create a new vector layer with the same geometry type of the selected layer and the same fields and
        # instances
        geometry = ""
        change_geometry = True
        crs = layer.crs().authid()
        if layer.geometryType() == 0:
            geometry = "Point" + "?crs=" + crs
        elif layer.geometryType() == 1:
            geometry = "Line" + "?crs=" + crs
        elif layer.geometryType() == 2:
            geometry = "Polygon" + "?crs=" + crs
        else:
            change_geometry = False
            geometry = "None"
        new_layer = QgsVectorLayer(geometry, layer.name() + "_mod", "memory")
        new_layer_data = new_layer.dataProvider()
        attr = layer.dataProvider().fields().toList()
        new_layer_data.addAttributes(attr)
        new_layer.updateFields()
        feats = layer.getFeatures()
        new_layer_data.addFeatures(feats)
        if change_geometry:
            self.update_geometry(layer, new_layer)
        # Keep in the new layer only the features selected from the process
        fld_names = new_layer_data.fields().names()
        new_fields = new_layer_data.fields()
        keep_fields = data.columns.values.tolist()
        remove_fields = []
        for field in fld_names:
            if field not in keep_fields:
                f_index = new_fields.indexFromName(field)
                remove_fields.append(f_index)
        new_layer_data.deleteAttributes(remove_fields)
        new_layer.updateFields()
        QgsProject.instance().addMapLayer(new_layer)
        self.selMessageBox.setStyleSheet("QLabel {  color : green; }")
        self.selMessageBox.setText('Selection complete. New vector layer ' + layer.name() + "_mod" + " created")

    # Copy the geometry from the selected layer to the new one
    def update_geometry(self, layer, new_layer):
        new_features = new_layer.getFeatures()
        new_layer_data = new_layer.dataProvider()
        for f in new_features:
            fid = f.id()
            geom = layer.getFeature(fid).geometry()
            new_layer_data.changeGeometryValues({fid+1: geom})

    # Remove the features with zero variance
    def low_variance(self, data, threshold):
        variance_threshold = VarianceThreshold(threshold)
        variance_threshold.fit(data)
        columns = data.columns[variance_threshold.get_support(indices=True)]
        return data[columns]

    def near_zero_var(self, data, freq_threshold, u_value_threshold):
        drop_columns = []
        for c in data.columns:
            freq_rank = data[c].value_counts()
            freq_ratio = freq_rank.iloc[0]/freq_rank.iloc[1]
            unique_number = data[c].nunique()
            unique_ratio = unique_number/data.shape[0]
            if freq_ratio > freq_threshold and unique_ratio < u_value_threshold:
                drop_columns.append(c)
        return data.drop(drop_columns, axis=1)


    # Remove the features with high correlation
    def cross_correlation(self, data, threshold):
        corr_matrix = data.corr().abs()
        bool_features = np.full(corr_matrix.shape[0], True, dtype=bool)
        corr_threshold = threshold
        feature_values = np.arange(start=corr_matrix.shape[0], stop=0, step=-1.0, dtype=float)
        if self.lowCorrCheck.isChecked():
            for i in range(feature_values.shape[0]):
                ave_corr = corr_matrix.mean()
                feature_values[i] = 1 - ave_corr[i]
        if self.distCheck.isChecked():
            for i in range(feature_values.shape[0]):
                shapiro_value = stats.shapiro(data.iloc[:, i]).statistic
                feature_values[i] = shapiro_value
        if self.domCheck.isChecked():
            dom = float(self.maxDom.text()) - float(self.minDom.text())
            for i in range(feature_values.shape[0]):
                interval = np.max(data.iloc[:, i]) - np.min(data.iloc[:, i])
                dom_coverage = interval/dom
                feature_values[i] = dom_coverage
        for i in range(corr_matrix.shape[0]):
            if not bool_features[i]:
                continue
            bad_features = []
            best_feat = i
            for j in range(i+1, corr_matrix.shape[0]):
                if not bool_features[j]:
                    continue
                if corr_matrix.iloc[i][j] >= corr_threshold:
                    if feature_values[best_feat] >= feature_values[j]:
                        bad_features.append(j)
                    else:
                        bad_features.append(best_feat)
                        best_feat = j
            for f in bad_features:
                if bool_features[f]:
                    bool_features[f] = False
        corr_features = data.columns[bool_features]
        corr_data = data[corr_features]
        return corr_data

    # Return the dataset of the selected vector layer (in feature selection or clustering section)
    def create_table(self, section):
        if section == "selection":
            selectedlayer = self.cleanLayerBox.currentLayer()
        else:
            selectedlayer = self.layerBox.currentLayer()
        data_features = selectedlayer.fields()
        remove_fields = []
        # Keep only numeric features and remove the others
        for feature in data_features:
            if not feature.isNumeric():
                remove_fields.append(feature.name())
        featurenames = [feature.name() for feature in data_features]
        for to_remove in remove_fields:
            featurenames.remove(to_remove)
        instances = selectedlayer.getFeatures()
        fullTable = []
        for instance in instances:
            fullTable.append([])
            for feature in featurenames:
                # Transform NULL values in 0
                if instance[feature]:
                    fullTable[len(fullTable) - 1].append(instance[feature])
                else:
                    fullTable[len(fullTable) - 1].append(0)
        dataTable = pd.DataFrame(fullTable, index=np.arange(len(fullTable)), columns=featurenames)
        return dataTable

    # Get the best features selected with the entropy algorithm
    def calc_entropy_features(self, data):
        ranked_features = self.entropy_ranking(data)
        base_entropy = self.calc_base_entropy(data)
        ranked_features = ranked_features - base_entropy
        selected_features = ranked_features[(ranked_features.Entropy > 0)]
        selected_features.sort_values('Entropy')
        return selected_features.index.values.tolist()

    def calc_sampling_entropy_features(self, data):
        total_rank = pd.DataFrame(0, index=data.columns, columns=['Entropy'])
        base_entropy = 0.0
        iterations = self.configuration.get_entropy_iterations()
        n_samples = self.configuration.get_sample_size()
        for i in range(0, iterations):
            sample_data = data.sample(n_samples)
            partial_rank = self.entropy_ranking(sample_data)
            partial_base = self.calc_base_entropy(sample_data)
            total_rank = total_rank + partial_rank
            base_entropy = base_entropy + partial_base
        ranked_features = total_rank - base_entropy
        selected_features = ranked_features[(ranked_features.Entropy > 0)]
        selected_features.sort_values(['Entropy'])
        return selected_features.index.values.tolist()

    def calc_base_entropy(self, data):
        b_dist = self.calc_distance(data)
        b_sim = self.calc_similarity(b_dist)
        base_entropy = self.calc_entropy(b_sim)
        return base_entropy

    def entropy_ranking(self, data):
        rank = pd.DataFrame(0, index=data.columns, columns=['Entropy'])
        for c in data.columns:
            temp_data = data.drop(c, axis=1)
            dist = self.calc_distance(temp_data)
            sim = self.calc_similarity(dist)
            rank.loc[c] = self.calc_entropy(sim)

        return rank

    def calc_distance(self, data):
        #distance = pd.DataFrame(0.0, index=np.arange(data.shape[0]), columns=np.arange(data.shape[0]))
        data = data.to_numpy()
        distance = np.zeros((data.shape[0], data.shape[0]))
        norm_vect = np.array([data[:, c].max() - data[:, c].min() for c in range(data.shape[1])])
        norm_vect = np.where(norm_vect == 0, 1, norm_vect)
        for i in range(data.shape[0]):
            for j in range(i, data.shape[0]):
                diff = data[i] - data[j]
                norm_diff = diff/norm_vect
                distance[i][j] = np.sqrt(np.power(norm_diff, 2).sum())
                distance[j][i] = distance[i][j]
        return distance

    def calc_similarity(self, dist):
        #similarity = pd.DataFrame(0.0, index=np.arange(dist.shape[0]), columns=np.arange(dist.shape[0]))
        similarity = np.zeros((dist.shape[0], dist.shape[0]))
        alpha = -np.log(0.5) / dist.mean().mean()
        for i in range(dist.shape[0]):
            for j in range(i, dist.shape[0]):
                similarity[i][j] = np.exp(-alpha*dist[i][j])
                similarity[j][i] = similarity[i][j]
        return similarity

    def calc_entropy(self, sim):
        entropy = 0.0
        for i in range(sim.shape[0]):
            for j in range(i+1, sim.shape[0]):
                s = sim[i][j]
                if not (s == 1):
                    entropy += s * np.log2(s) + (1 - s) * np.log2(1 - s)
        return -entropy


class Experiment:
    def __init__(self, name, features, num_cluster, cluster_method, clusters, sil_index, db_index):
        self.__name = name
        self.__num_cluster = num_cluster
        self.__features = features
        self.__cluster_method = cluster_method
        self.__clusters = clusters
        self.__sil_index = sil_index
        self.__db_index = db_index

    def get_name(self):
        return self.__name

    def get_features(self):
        return self.__features

    def get_num_cluster(self):
        return self.__num_cluster

    def get_clusters(self):
        return self.__clusters

    def get_cluster_method(self):
        return self.__cluster_method

    def get_sil_index(self):
        return self.__sil_index

    def get_db_index(self):
        return self.__db_index


class Configuration:
    def __init__(self, frequency_cut, unique_cut, entropy_iterations, sample_size, graph_max_cluster, distance):
        self.__frequency_cut = frequency_cut
        self.__unique_cut = unique_cut
        self.__entropy_iterations = entropy_iterations
        self.__sample_size = sample_size
        self.__max_cluster = graph_max_cluster
        self.__distance = distance

    def get_frequency_cut(self):
        return self.__frequency_cut

    def get_unique_cut(self):
        return self.__unique_cut

    def get_entropy_iterations(self):
        return self.__entropy_iterations

    def get_sample_size(self):
        return self.__sample_size

    def get_max_cluster(self):
        return self.__max_cluster

    def get_distance(self):
        return self.__distance
