# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CTDQ
                                 A QGIS plugin
 A suite of handy QGIS tools for Civil/Geo Engineers and Designers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-08-18
        copyright            : (C) 2025 by CeeThreeDee
        email                : info@ceethreedee.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'CeeThreeDee'
__date__ = '2025-08-18'
__copyright__ = '(C) 2025 by CeeThreeDee'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import sys
import inspect

from qgis.core import QgsProcessingAlgorithm, QgsApplication, Qgis
from .ctdq_provider import CTDQProvider
from PyQt5.QtWidgets import QAction, QMenu
from PyQt5.QtGui import QIcon, QDesktopServices
from PyQt5.QtCore import QUrl, QCoreApplication, QSettings, QTranslator, QLocale, Qt

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]

if cmd_folder not in sys.path:
    sys.path.insert(0, cmd_folder)


class CTDQPlugin(object):

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.provider = CTDQProvider(self.iface)
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale (robust to QVariant / different return types)
        raw_locale = QSettings().value("locale/userLocale", QLocale.system().name())
        try:
            # ensure we have a Python string and take the language part
            locale = str(raw_locale)[0:2]
        except Exception:
            locale = QLocale.system().name()[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", f"CeeThreeDeeQTOOLS_{locale}.qm")

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if Qgis.QGIS_VERSION_INT > 40303:
                QCoreApplication.installTranslator(self.translator)

    def initProcessing(self):
        """Init Processing provider for QGIS >= 3.8."""
        self.provider = CTDQProvider(self.iface)
        QgsApplication.processingRegistry().addProvider(self.provider)

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("CeeThreeDeeQTools", message)

    def initGui(self):
        self.initProcessing()

        # Add "CeeThreeDee Qtools" as a submenu under the existing "Plugins" menu
        plugins_menu = None
        for action in self.iface.mainWindow().menuBar().actions():
            if action.text() == self.tr("&Plugins"):
                plugins_menu = action.menu()
                break

        if plugins_menu is not None:
            self.menu = QMenu(self.tr("CeeThreeDee Qtools"), plugins_menu)
            plugins_menu.addMenu(self.menu)
        else:
            # Fallback: create menu at top level if Plugins menu not found
            self.menu = QMenu(self.tr("CeeThreeDee Qtools"), self.iface.mainWindow())
            self.iface.mainWindow().menuBar().addMenu(self.menu)

        # Create action for the validation dialog
        self.validateAction = QAction(
            QIcon(os.path.join(os.path.dirname(__file__), "./Assets/img/CTD_logo.png")),
            self.tr("Validate Project Report"),
            self.iface.mainWindow(),
        )
        self.validateAction.triggered.connect(self.openValidationDialog)
        self.menu.addAction(self.validateAction)

        # Create action for the Mirror Project tool
        self.mirrorProjectAction = QAction(
            QIcon(os.path.join(os.path.dirname(__file__), "./Assets/img/CTD_logo.png")),
            self.tr("Mirror Project"),
            self.iface.mainWindow(),
        )
        self.mirrorProjectAction.triggered.connect(self.openMirrorProjectDialog)
        self.menu.addAction(self.mirrorProjectAction)

        # Create action for the Package Layer Updater tool
        self.packageLayerUpdaterAction = QAction(
            QIcon(os.path.join(os.path.dirname(__file__), "./Assets/img/CTD_logo.png")),
            self.tr("Package Layer Updater"),
            self.iface.mainWindow(),
        )
        self.packageLayerUpdaterAction.triggered.connect(self.openPackageLayerUpdaterDialog)
        self.menu.addAction(self.packageLayerUpdaterAction)

        # Create action for the Layers Advanced dock widget
        self.layersAdvancedAction = QAction(
            QIcon(os.path.join(os.path.dirname(__file__), "./Assets/img/CTD_logo.png")),
            self.tr("Layers Advanced"),
            self.iface.mainWindow(),
        )
        self.layersAdvancedAction.setCheckable(True)
        self.layersAdvancedAction.triggered.connect(self.toggleLayersAdvancedDock)
        self.menu.addAction(self.layersAdvancedAction)

        # Initialize dock widget reference (created on first toggle)
        self.layersAdvancedDock = None

        # Create action that will start plugin help
        self.helpAction = QAction(
            QIcon(os.path.join(os.path.dirname(__file__), "./Assets/img/CTD_logo.png")),
            self.tr("CeeThreeDee Qtools Help"),
            self.iface.mainWindow(),
        )
        self.helpAction.triggered.connect(
            lambda: QDesktopServices.openUrl(
                QUrl("https://github.com/Kapanther/CeeThreeDeeQTools")
            )
        )
        self.menu.addAction(self.helpAction)

    def openValidationDialog(self):
        from .Tools.ValidateProjectReport.ctdq_ValidateProjectReportDialog import ValidateProjectReportDialog
        dialog = ValidateProjectReportDialog()
        dialog.exec_()

    def openMirrorProjectDialog(self):
        """Open the Mirror Project dialog."""
        from .Tools.MirrorProject.ctdq_MirrorProjectDialog import MirrorProjectDialog
        from .Tools.MirrorProject.ctdq_MirrorProjectLogic import MirrorProjectLogic
        from qgis.PyQt.QtWidgets import QMessageBox, QProgressDialog
        from qgis.PyQt.QtCore import Qt
        from qgis.core import QgsProject
        
        # Check if a project is open
        if not QgsProject.instance().fileName():
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("No Project Open"),
                self.tr("Please open a QGIS project before using the Mirror Project tool.")
            )
            return
        
        # Create dialog non-modally and register export callback so the dialog remains open after run
        dialog = MirrorProjectDialog(self.iface.mainWindow())
        # Define the export callback to be registered with the dialog.
        def export_callback(progress_cb):
            # Collect selections and options at time of export
            selected_layers = dialog.get_selected_layers()
            selected_themes = dialog.get_selected_themes()
            selected_layouts = dialog.get_selected_layouts()
            target_projects = dialog.get_target_projects()
            replace_data_source = dialog.get_replace_data_source()
            update_symbology = dialog.get_update_symbology()
            fix_layer_order = dialog.get_fix_layer_order()
            create_backups = dialog.get_create_backups()
            add_layer_groups = dialog.get_add_layer_groups()
            preserve_layer_filters = dialog.get_preserve_layer_filters()
            preserve_auxiliary_tables = dialog.get_preserve_auxiliary_tables()  # NEW

            # Create progress dialog (modal) to show progress bar; console will contain textual log
            progress = QProgressDialog(
                "Exporting layers to target projects...",
                "Cancel",
                0,
                100,
                self.iface.mainWindow()
            )
            progress.setWindowModality(Qt.WindowModal)
            progress.setWindowTitle("Mirror Project")
            progress.show()

            def update_progress(message, percent):
                progress.setLabelText(message)
                progress.setValue(percent)
                if progress.wasCanceled():
                    raise Exception("Operation cancelled by user")
                # Also forward to dialog console via progress_cb
                try:
                    progress_cb(message, percent)
                except Exception:
                    pass

            try:
                # Execute the export
                results = MirrorProjectLogic.export_layers_to_projects(
                    selected_layers,
                    target_projects,
                    False,  # skip_same_name removed from UI; do not skip
                    replace_data_source,
                    update_symbology,
                    fix_layer_order,
                    update_progress,
                    create_backups=create_backups,
                    add_layer_groups=add_layer_groups,
                    selected_themes=selected_themes,
                    selected_layouts=selected_layouts,
                    preserve_layer_filters=preserve_layer_filters,
                    preserve_auxiliary_tables=preserve_auxiliary_tables  # NEW
                )
                progress.close()
                # Display results in the dialog console
                try:
                    dialog.display_results(results)
                except Exception:
                    pass
            except Exception as e:
                progress.close()
                try:
                    dialog.append_console(f"Error during export: {str(e)}")
                except Exception:
                    pass
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    self.tr("Mirror Project - Error"),
                    self.tr(f"An error occurred during export:\n{str(e)}")
                )

        # Register callback and show dialog (keep it open)
        dialog.set_export_callback(export_callback)
        dialog.show()

    def openPackageLayerUpdaterDialog(self):
        """Open the Package Layer Updater dialog."""
        from .Tools.PackageLayerUpdater.ctdq_PackageLayerUpdaterDialog import PackageLayerUpdaterDialog
        from .Tools.PackageLayerUpdater.ctdq_PackageLayerUpdaterLogic import PackageLayerUpdaterLogic
        from qgis.PyQt.QtWidgets import QMessageBox, QProgressDialog
        from qgis.PyQt.QtCore import Qt
        from qgis.core import QgsProject
        
        # Check if a project is open
        if not QgsProject.instance().fileName():
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("No Project Open"),
                self.tr("Please open a QGIS project before using the Package Layer Updater tool.")
            )
            return
        
        # Create dialog non-modally and register update callback
        dialog = PackageLayerUpdaterDialog(self.iface.mainWindow())
        
        # Define the update callback
        def update_callback(progress_cb):
            # Collect selections at time of update
            selected_layers = dialog.get_selected_layers()
            target_geopackages = dialog.get_target_geopackages()
            update_new_only = dialog.get_update_new_only()
            fix_fids = dialog.get_fix_fids()
            preserve_fid = dialog.get_preserve_fid()
            
            # Create progress dialog
            progress = QProgressDialog(
                "Updating geopackage layers...",
                "Cancel",
                0,
                100,
                self.iface.mainWindow()
            )
            progress.setWindowModality(Qt.WindowModal)
            progress.setWindowTitle("Package Layer Updater")
            progress.show()
            
            def update_progress(message, percent):
                progress.setLabelText(message)
                progress.setValue(percent)
                if progress.wasCanceled():
                    raise Exception("Operation cancelled by user")
                try:
                    progress_cb(message, percent)
                except Exception:
                    pass
            
            try:
                # Execute the update
                results = PackageLayerUpdaterLogic.update_geopackage_layers(
                    selected_layers,
                    target_geopackages,
                    update_progress,
                    update_new_only=update_new_only,
                    fix_fids=fix_fids,
                    preserve_fid=preserve_fid
                )
                progress.close()
                
                # Display results in console
                try:
                    dialog.display_results(results)
                except Exception:
                    pass
            except Exception as e:
                progress.close()
                try:
                    dialog.append_console(f"Error during update: {str(e)}")
                except Exception:
                    pass
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    self.tr("Package Layer Updater - Error"),
                    self.tr(f"An error occurred during update:\n{str(e)}")
                )

        # Register callback and show dialog
        dialog.set_update_callback(update_callback)
        dialog.show()

    def toggleLayersAdvancedDock(self, checked):
        """Toggle the Layers Advanced dock widget."""
        # Create the dock widget on first toggle if it doesn't exist
        if self.layersAdvancedDock is None:
            from .Tools.LayersAdvanced.LayersAdvancedDialog import LayersAdvancedDialog
            self.layersAdvancedDock = LayersAdvancedDialog(self.iface, self.iface.mainWindow())
            self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.layersAdvancedDock)
            
            # Connect visibility changed to update action state
            def update_action_state(visible):
                if hasattr(self, 'layersAdvancedAction'):
                    self.layersAdvancedAction.setChecked(visible)
            
            self.layersAdvancedDock.visibilityChanged.connect(update_action_state)
        
        # Toggle visibility based on checked state
        if checked:
            self.layersAdvancedDock.show()
        else:
            self.layersAdvancedDock.hide()

    def unload(self):
        """
        Unloads the plugin and removes the provider from the processing registry.
        """
        # Remove provider safely - check if it still exists
        try:
            if hasattr(self, 'provider') and self.provider is not None:
                QgsApplication.processingRegistry().removeProvider(self.provider)
        except RuntimeError:
            # Provider already deleted, ignore
            pass

        # Remove the "CeeThreeDee Qtools" menu and its actions
        if self.menu:
            self.menu.clear()
            self.iface.mainWindow().menuBar().removeAction(self.menu.menuAction())
            self.menu = None

        # Remove the Layers Advanced dock widget
        if self.layersAdvancedDock:
            self.iface.removeDockWidget(self.layersAdvancedDock)
            self.layersAdvancedDock = None

        del self.validateAction
        del self.mirrorProjectAction
        del self.packageLayerUpdaterAction
        del self.layersAdvancedAction
        del self.helpAction