# -*- coding: utf-8 -*-
"""
/***************************************************************************
 A QGIS plugin cadastral_classifier
 
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-10-27
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Nikolai Shurupov (Universidad de 
                                                             Alcalá de 
                                                             Henares)
        email                : nikolai.shurupov@uah.es

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

#------------------------------------------------------------------------------

# Imports for qgis core functions
from qgis.PyQt.QtCore import (QSettings, QTranslator, QCoreApplication, Qt, QVariant,
                              pyqtSignal)

from qgis.PyQt.QtGui import QIcon, QPixmap
from qgis.PyQt.QtWidgets import (QAction, QFileDialog, QTableWidgetItem,
                                QHeaderView, QMessageBox, QListWidgetItem, QMenu)

# Initialize  from file resources.py
from .resources import *
# Import the code for the dialog
from .cadastral_classifier_dialog import cadastral_classifierDialog

# Import qgis utils
from qgis.core import (Qgis, QgsVectorLayer, QgsField, QgsProject, QgsMessageLog, 
                      QgsCategorizedSymbolRenderer, QgsCoordinateReferenceSystem,
                      QgsCoordinateTransform, QgsSpatialIndex, QgsFeatureRequest,
                      QgsTask, QgsApplication)

from qgis.utils import iface

# Global imports for processing
#import os.path
from os import chdir, getcwd
from os.path import dirname, basename, join, isfile, exists, isdir
from pandas import read_csv, merge, DataFrame, isna
from time import time
from copy import deepcopy
from glob import glob
from zipfile import ZipFile
import processing
from pathlib import Path

# Local imports for processing
from .utils.table_type_generator import table_type_generator
from .utils.shp_utils import unzip_shp, merge_qgis, shp_copy, unzip_files
from .utils.palette_generator import palette_generator, random_color
from .utils.use_definer import use_function_definer
from .utils.get_OSM_green_zones import get_green_zones
from .utils.input_check import input_check

#------------------------------------------------------------------------------

# Global label for task messages
MESSAGE_CATEGORY = 'Cadastral Classifier'

class setup_data_Task(QgsTask):
    ''' Task for preprocessing of raw data'''
    
    taskStarted = pyqtSignal()
    taskEnded = pyqtSignal()
    
    def __init__(self, dlg, d_codes, selected_muns):
        """
		* Initialise the thread
		"""
		# call constructor for QsgThread
        super().__init__("Cadastral Classifier", QgsTask.CanCancel)

		# load arguments into class variables
        self.dlg = dlg
        self.d_codes = d_codes
        self.selected_muns_codes = selected_muns
        self.t_process = None
        self.exception = None

    def finished(self, result):
        
        self.taskEnded.emit()  # Emit signal at the end of the task
        
        # success
        if result:
               
            # message to show on emergent window       
            finish_message = 'Tiempo total de ejecución: ' + self.t_process + ' segundos.'

   			# Notify User
            QgsMessageLog.logMessage(finish_message, MESSAGE_CATEGORY, Qgis.Success)
            iface.messageBar().pushMessage("¡Tratamiento de datos finalizado con éxito!", finish_message, level=Qgis.Success, duration=10)
            
        else:
   			# failed without exception (user cancel)
            if self.exception is None:
                QgsMessageLog.logMessage('Cancelado sin excepciones', MESSAGE_CATEGORY, Qgis.Warning)
                iface.messageBar().pushMessage("Atención:", 'Proceso cancelado', level=Qgis.Warning, duration=10)

   			# failed with exception (raise exception)
            else:
                QgsMessageLog.logMessage(f'Calcelado con la excepción: {self.exception}', MESSAGE_CATEGORY, Qgis.Critical)
                iface.messageBar().pushMessage("Error:", f'Calcelado con la excepción: {str(self.exception)}', level=Qgis.Critical)
   				# raise self.exception
                   
    def cancel(self):
   		"""
   		* This function is called when the task is cancelled
   		"""

   		# log that the task was cancelled
   		QgsMessageLog.logMessage("Proceso de preprocesamiento cancelado manualmente por el usuario", MESSAGE_CATEGORY, Qgis.Info)

   		# run the cancel method from QgsTask
   		super().cancel()

    def run(self):
        ''' Main function of the first process. Using the CAT and SHP files 
        from an input municipality, it performs the analysis of the developed
        area of each present use.
        
        - CAT (.cat): This file is the raw data that stores all the data for each
        cadastral parcel. It is necessary to extract the information to different
        tables, using structured dictionaries (based on positions and length of
        each field). This can be consulted in the 'cadastral_structure.py' file)
        
        - SHP (.shp): This is the spatial file that stores the parcels geometries.
        It has a cadastral reference number for each parcel. This field is used
        determine the area of each use of each parcel, using the CAT tables.
        
        User can select only the urban CAT and SHP, or also include the rural 
        data. In this case, a merge will be performed, for both, CAT and SHP
        separately, and after that the analysis will be carried out.
        
        CAT file and the tables generated are managed using pandas dataframe 
        methods, while SHP is managed with QGIS native method (layers). This is 
        due to convienence of geometries management, since geopandas is too
        difficult to add to the QGIS Python environment.
        
        As well as adding each use area, each parcel will also have a field that 
        determine the total built area, the built area on the surface level
        (floor 0 or floor 1, to ensure avoiding some multifamily buildings issues),
        the highest floors, the year of the oldest construction 
        present and the number of residential bulidings on the parcels that have
        any residential building.
        '''
                
        self.taskStarted.emit()  # Emit signal at start of the task

        try: 
                        
			# setup for progress bar ad message
            QgsMessageLog.logMessage("Iniciado el tratamiento inicial de los datos", MESSAGE_CATEGORY, Qgis.Info)
        
            # save time before starting the processing
            t_start = time()
                    
            # set the progress value of the progress bar to 0
            progress = 0
            self.setProgress(progress)
            
            # create a list to store all the type-tables that user wants to generate
            list_of_interest = []        
            
            # the tyble-type-14 will be the only one that is checked by default and 
            # can't be unchecked since it is necessary for all the later process,
            # all other are optional to generate
            if self.dlg.checkBox_type_01.isChecked():
                list_of_interest.append(1)
                
            if self.dlg.checkBox_type_11.isChecked():
                list_of_interest.append(11)
                
            if self.dlg.checkBox_type_13.isChecked():
                list_of_interest.append(13)
                
            if self.dlg.checkBox_type_14.isChecked():
                list_of_interest.append(14)
                
            if self.dlg.checkBox_type_15.isChecked():
                list_of_interest.append(15)
                
            if self.dlg.checkBox_type_16.isChecked():
                list_of_interest.append(16)
                
            if self.dlg.checkBox_type_17.isChecked():
                list_of_interest.append(17)
                
            if self.dlg.checkBox_type_90.isChecked():
                list_of_interest.append(90)
            
            # get the directry path to save the outputs
            dest_filepath = self.dlg.dir_output_result_p1.text()
        
            #------------------------------------------------------------------
            
            if self.dlg.option_multiple_mun_p1.isChecked() == True:
                
                # get the paths of each input file
                urban_SHP_file_path = self.dlg.dir_shp_urb_multi.text()
                urban_CAT_file_path = self.dlg.dir_cat_urb_multi.text()
        
                # if users wants the rural parcel to be included, get the paths
                if self.dlg.Option_include_rural.isChecked():
                    rural_SHP_file_path = self.dlg.dir_shp_rust_multi.text()
                    rural_CAT_file_path = self.dlg.dir_cat_rust_multi.text()
                
                else:
                    rural_SHP_file_path = None
                    rural_CAT_file_path = None
            
                selected_muns_codes = self.selected_muns_codes
            
            #------------------------------------------------------------------
            
            elif self.dlg.option_single_mun_p1.isChecked() == True:
                
                # get the paths of each input file
                urban_SHP_file_path = self.dlg.dir_shp_urb_unique.text()
                urban_CAT_file_path = self.dlg.dir_cat_urb_unique.text()
        
                # if users wants the rural parcel to be included, get the paths
                if self.dlg.Option_include_rural.isChecked():
                    rural_SHP_file_path = self.dlg.dir_shp_rust_unique.text()
                    rural_CAT_file_path = self.dlg.dir_cat_rust_unique.text()
                
                else:
                    rural_SHP_file_path = None
                    rural_CAT_file_path = None
                
                selected_muns_codes = [basename(dest_filepath)] # simulate a selection
                
            #------------------------------------------------------------------
            
            progress_each_mun = 100/len(selected_muns_codes) # state the progress bar evolution caps
 
            # list of the fileds that are interesting to extraxt from the .shp
            SHP_fields_of_interest = ["REFCAT", "AREA", "COORX", "COORY", "TIPO"]
                        
            i = 0 # store numerically the initialization
            for selected in selected_muns_codes:
                
                if self.isCanceled():
                    QgsMessageLog.logMessage("User cancelled the operation!", MESSAGE_CATEGORY, Qgis.Warning)
                    return False  # Task cancellation acknowledged here
                
                progress_start = i * progress_each_mun
                # state the cap that has to be reached once all the operations
                # in the mun are performed
                progress_cap = progress_each_mun + i * progress_each_mun
                
                progress_total_increase = progress_cap - progress_start

                progress = progress_start
                
                proceed = False
                
                if self.dlg.option_multiple_mun_p1.isChecked() == True:
    
                    selected_code = selected[1:6] # get the code of the mun
                    selected_name = selected_code + selected[7:] # get the name
                    selected_name = selected_name.replace(" ", "_") # remove spaces to avoid later issues
                    
                    # get the internal paths of each of the components
                    path_P_U = self.d_codes[selected_code]['SHP_path_U']
                    path_P_R = self.d_codes[selected_code]['SHP_path_R']
                    path_CAT_U = self.d_codes[selected_code]['CAT_path_U']
                    path_CAT_R = self.d_codes[selected_code]['CAT_path_R']
                    
                    # create a path to store all the municipality files
                    mun_path = join(dest_filepath, selected_name)
                    
                    if self.dlg.Option_omit_previous.isChecked() == False or (self.dlg.Option_omit_previous.isChecked() == True and exists(mun_path) == False):
                        
                        proceed = True
                        P_U = None
                        urb_path = None
                        P_R = None 
                        rural_path = None

                        # path to save the table tipe generations
                        output_csv = join(mun_path, "tablas_tipo")
                        
                        if path_P_U:
                            # unzip urban parcels file and save the path of the results
                            P_U = unzip_files(urban_SHP_file_path, join(mun_path, "SHP_URBANO"), path_P_U, 
                                              "parcelario")
                        
                        if path_CAT_U:
                            # unzip urban CAT file and save the path of the results
                            CAT_U = unzip_files(urban_CAT_file_path, output_csv, path_CAT_U, 
                                              "cat")
                            # generate the file(s) and table-type(s) using a function included
                            # in utils folder
                            urb_path = table_type_generator(CAT_U, list_of_interest, output_csv)            

                        # if the rural option has been selected unzip in the same way 
                        # the files
                        if self.dlg.Option_include_rural.isChecked() == True:
                            
                            if path_P_R:
                                P_R = unzip_files(rural_SHP_file_path, join(mun_path, "SHP_RUSTICO"), path_P_R, 
                                                  "parcelario")
                            
                            if path_CAT_R:
                                CAT_R = unzip_files(rural_CAT_file_path, output_csv, path_CAT_R, 
                                                  "cat")
                                # generate the file(s) in the same way as previously
                                rural_path = table_type_generator(CAT_R, list_of_interest, output_csv)

                        #------------------------------------------------------
                        
                        # itera sobre las tablas tipo que se ha querido generar y las une una a una
                        # iterate over all the table-types selected
                        for table_type in list_of_interest:
                                                        
                            if urb_path:
                            # read each table-type file (.csv, urban and rural)
                                df_type_urban = read_csv(join(urb_path, ('Tipo_' + str(table_type)\
                                                          + '.csv')), dtype = str)
                            if rural_path:
                                df_type_rural = read_csv(join(rural_path, ('Tipo_' + str(table_type)\
                                                           + '.csv')), dtype = str)
                            
                            # due to a possible differences between the column types of the 
                            # data that each table-type has (rural an urban), it is needed
                            # to perform an unification of the formats. Iterate over the 
                            # urban one and checks the data types. If they differ try to 
                            # change the types in one direction, and in the other one if 
                            # it fails
                            
                            if urb_path and rural_path:
                                for column in df_type_urban.columns:
                                    if df_type_urban[column].dtype != df_type_rural[column].dtype:
                                        
                                        try:
                                            df_type_urban = df_type_urban.astype(
                                                {column: df_type_rural[column].dtype})
                                            
                                        except:
                                            df_type_rural = df_type_rural.astype(
                                                {column: df_type_urban[column].dtype})
                    
                                # use pandas method merge function to perform the union of both
                                # CAT files (urban and rural)
                                df_type_merge = merge(df_type_urban, df_type_rural, how = "outer")
                    
                                # Save the file as .csv format
                                df_type_merge.to_csv(join(output_csv, ('Tipo_' + str(table_type) + "_U_R.csv")))
                                
                                # add the table-type-14 path to the variable used on next
                                # processes
                                if table_type == 14:
                                    table_type_14_path = join(output_csv, ('Tipo_' + str(table_type) + "_U_R.csv"))
                                    
                            elif urb_path and not rural_path:
                                table_type_14_path = (join(urb_path, "Tipo_14.csv"))
                                
                            elif rural_path and not urb_path:
                                table_type_14_path = (join(rural_path, "Tipo_14.csv"))
                                
                        #------------------------------------------------------

                        # if user also included rural parcels to the analysis
                        if P_U and P_R:
                                
                            # use qgis merge function (util folder) to perform the union of 
                            # both SHP files (urban and rural), that generates a new file
                            wd_shp = merge_qgis(P_R, P_U, join(mun_path, "procesado_" + basename(mun_path) + ".shp"))
                            
                
                        elif P_U and not P_R:
                            # copy the .shp of urban parcels
                            wd_shp = shp_copy(P_U, join(mun_path, "procesado_" + basename(mun_path) + ".shp"))
                        
                        elif P_R and not P_U:
                            wd_shp = shp_copy(P_R, join(mun_path, "procesado_" + basename(mun_path) + ".shp"))
                        
                        #------------------------------------------------------
                        
                    else:
                        wd_shp = join(mun_path, "procesado_" + basename(mun_path) + ".shp")
                        
                if self.dlg.option_single_mun_p1.isChecked() == True:
                    
                    # get the path of the result file to be saved later
                    result_file_path = join(dest_filepath, "procesado_SHP.shp")
                                                                
                    proceed = True

                    # generates a new path to save the mid-process files (the
                    # unzipped CAT files and the table-type files)
                    output_csv = join(dirname(result_file_path), "tablas_tipo")
                        
                    #----------------------------------------------------------
                    
                    # if a route of urban CAT has been selected
                    if urban_CAT_file_path:
                        
                        # generate the file(s) and table-type(s) using a function included
                        # in utils folder
                        urb_path = table_type_generator(urban_CAT_file_path, list_of_interest, output_csv)
                        table_type_14_path = join(urb_path, "Tipo_14.csv")
                    
                    #----------------------------------------------------------
                    
                    # if user also selected the rural file option
                    if rural_CAT_file_path:
                        
                        # generate the file(s) in the same way as previously
                        rural_path = table_type_generator(rural_CAT_file_path, list_of_interest, output_csv)
                                    
                        # itera sobre las tablas tipo que se ha querido generar y las une una a una
                        # iterate over all the table-types selected
                        for table_type in list_of_interest:
                            
                            # read each table-type file (.csv, urban and rural)
                            df_type_urban = read_csv(join(urb_path, ('Tipo_' + str(table_type)\
                                                      + '.csv')), dtype = str)
                
                            df_type_rural = read_csv(join(rural_path, ('Tipo_' + str(table_type)\
                                                       + '.csv')), dtype = str) 
                            
                            # due to a possible differences between the column types of the 
                            # data that each table-type has (rural an urban), it is needed
                            # to perform an unification of the formats. Iterate over the 
                            # urban one and checks the data types. If they differ try to 
                            # change the types in one direction, and in the other one if 
                            # it fails
                            
                            for column in df_type_urban.columns:
                                if df_type_urban[column].dtype != df_type_rural[column].dtype:
                                    
                                    try:
                                        df_type_urban = df_type_urban.astype(
                                            {column: df_type_rural[column].dtype})
                                        
                                    except:
                                        df_type_rural = df_type_rural.astype(
                                            {column: df_type_urban[column].dtype})
                
                            # use pandas method merge function to perform the union of both
                            # CAT files (urban and rural)
                            df_type_merge = merge(df_type_urban, df_type_rural, how = "outer")
                
                            # Save the file as .csv format
                            df_type_merge.to_csv(join(output_csv, ('Tipo_' + str(table_type) + "_U_R.csv")))
                            
                            # add the table-type-14 path to the variable used on next
                            # processes
                            if table_type == 14:
                                table_type_14_path = join(output_csv, ('Tipo_' + str(table_type) + "_U_R.csv"))
                                
                    #----------------------------------------------------------
                    
                    # if the input SHP file is compressed in zip format, unzip it and 
                    # store the row of the new .shp file
                    
                    if urban_SHP_file_path:

                        if urban_SHP_file_path.lower().endswith("zip"):
                            wd_urb = unzip_shp(urban_SHP_file_path)
                            
                        else:
                            wd_urb = urban_SHP_file_path
                   
                    else:
                        wd_urb = None

                    #----------------------------------------------------------
                    # if user also included rural parcels to the analysis
                    if rural_SHP_file_path:
                        
                        # follow same process as for urban parcels
                        if rural_SHP_file_path[-4:].lower().endswith("zip"): 
                            wd_rus = unzip_shp(rural_SHP_file_path)
                            
                        else:
                            wd_rus = rural_SHP_file_path
                    
                    else:
                        wd_rus = None
                        
                    if wd_rus and wd_urb:
                        # use qgis merge function (util folder) to perform the union of 
                        # both SHP files (urban and rural), that generates a new file
                        wd_shp = merge_qgis(wd_rus, wd_urb, result_file_path)
                                
                    elif wd_rus and not wd_urb:
                        # copy the .shp of urban parcels
                        wd_shp = shp_copy(wd_rus, result_file_path)
                        
                    elif wd_urb and not wd_rus:
                        wd_shp = shp_copy(wd_urb, result_file_path)

                # update progress
                progress += 0.01 * progress_total_increase
                self.setProgress(progress)
                
                if proceed == True:
            
                    if isfile(wd_shp) == True:
        
                        # open the shp as QGIS layer, get its provider and start the editing
                        layer = QgsVectorLayer(wd_shp, "", "ogr")
                        layer_provider = layer.dataProvider()
                        layer.startEditing()
                                                
                        #------------------------------------------------------
                        
                        # if the rural parcels were included, it is needed to remove the parcels
                        # with type "X"
                                        
                        listOfIds = [feat.id() for feat in layer.getFeatures() if feat['TIPO'] == 'X']     
                        layer.deleteFeatures(listOfIds)
                            
                        #------------------------------------------------------
                        
                        # get the fields names of the layer
                        fields_name = layer.fields().names()
                                                            
                        # list to store the names of the fields to remove
                        remove_indexers = []
                
                        # iterate over all the fields 
                        for field in fields_name:
                            
                            # if it is not in the list of fields of interest
                            if field not in SHP_fields_of_interest:
                                
                                # appendt it to remove list
                                remove_indexers.append(fields_name.index(field))
                        
                        # remove the unnecesary fields and update the layer
                        layer_provider.deleteAttributes(remove_indexers)
                        layer.updateFields()
                        
                        #------------------------------------------------------
                        
                        # get the new fields
                        fields_name = layer.fields().names()
                        
                        # get the index (numerical position) of the area attribute, and cadastral
                        # reference
                        area_index = fields_name.index("AREA")
                        
                        # get all the features (rows) of the layer
                        features = layer.getFeatures()
                        
                        #create a dictionary to save all unique cadastral references
                        unique_ref_cat = dict()
                            
                        # iterate over all features
                        for f in features:
                            
                            # store the id of the feature (index column of each feature)
                            f_id = f.id()
                            
                            # store its cadastral reference number
                            f_refcat = f["REFCAT"]
                                
                            # to each unique cadastral reference in the dictionary add the id
                            # of the parcel that has it
                            if f_refcat not in unique_ref_cat.keys():
                                unique_ref_cat[f_refcat] = [f_id]
                            
                            # if the key already exist just add the id 
                            else:
                                unique_ref_cat[f_refcat].append(f_id)
                        
                        # iterate over all the cadastral references keys
                        for key in unique_ref_cat.keys():
                            
                            # store the ids that have it
                            ls_parcels = unique_ref_cat[key]
                            
                            # for the ones that has multiple ids (multipart geometry)
                            if len(ls_parcels) > 1:
                                
                                area_fix = 0
                                
                                # summarize the area of all the ids
                                for parcel in ls_parcels:
                                    parcel_area = layer.getFeature(parcel).attribute(area_index)
                                    area_fix += parcel_area
                                    
                                # change the area of the multipart geometries. Now they have the
                                # area of the whole geometries of the multipart entity, so 
                                # the following use asignments take in account them as a single
                                # entity
                                for parcel in ls_parcels:
                                    # generate a tuple with the index of field and it's new value
                                    attr_value = {area_index:area_fix}
                                    layer_provider.changeAttributeValues({parcel:attr_value})
                            
                        #----------------------------------------------------------

                        # update progress
                        progress += 0.01 * progress_total_increase
                        self.setProgress(progress)
                        #print (table_type_14_path)
                        # read the table-type-14 csv as pandas dataframe. All data is read as string
                        df_type_14 = read_csv(table_type_14_path, dtype = str, encoding='latin-1')

                        # delete the fifth element of the use typology codification, since it 
                        # dont give any relevant information for the classification
                        
                        df_type_14['TIPOLOGIA'] = df_type_14['105_tip'].astype(str).str[0:4]

                        # remove any row that has code '0000' since its a 'non edified category
                        # and just introduces disturbances in future processes
                        if "0000" in list(set(df_type_14["TIPOLOGIA"])):
                            rows_to_remove = list(df_type_14[df_type_14["105_tip"] == "0000"].index)
                            df_type_14 = df_type_14.drop(rows_to_remove)
                            
                        #----------------------------------------------------------
                        ls_floor_values = df_type_14["65_pt"].unique()
                        ls_excluded_floors = []
                        
                        for item in ls_floor_values:
                            try:
                                int(item)
                            except:
                                ls_excluded_floors.append(item)
                        
                        # transform the floor data into an integer whenever possible. For 
                        # those values that are not set a -99 value
                        
                        df_type_14["65_pt"] = df_type_14["65_pt"].apply(lambda x: int(x) if x not in ls_excluded_floors and isna(x) == False else -99)
                        
                        #----------------------------------------------------------
                        ls_year_values = df_type_14["79_aec"].unique()
                        ls_excluded_years = []
                        
                        for item in ls_year_values:
                            try:
                                int(item)
                            except:
                                ls_excluded_years.append(item)
            
                        # transform building date into integer whenever possible. For those
                        # that are no possible set a value o 0
                        
                        df_type_14["79_aec"] = df_type_14["79_aec"].apply(lambda x: int(x) if x not in ls_excluded_years and isna(x) == False else 0)
                        
                        # list with all the cadastral reference numbers present in the 
                        # table-type-14 dataframe
                        ref_CAT_type_14 = list(df_type_14["31_pc"])
                    
                        # merge both (table-type-14 and SHP) cadastral reference number lists 
                        # and get the set (unique) values
                        ref_CAT_type_14.extend(list(unique_ref_cat.keys()))
                        ref_CAT = set(ref_CAT_type_14)
                                
                        # create a dictionary that will contain as keys each of the unique 
                        # cadastral reference numbers, and set the value to 0
                        d_base_num = dict.fromkeys(ref_CAT, 0)
                        #d_base_list = {key: [] for key in ref_CAT}
                        
                        # create a list with the fields to be added to the resulting shp
                        list_uses = ["A_TOT_EDIF", "SUP_ED_0", "SUP_ED_1", "ANTIGUEDAD",
                                     "N_PLANTAS", "NUM_INM_R"]
                        
                        # add to the previous list all the possible codification of categorical
                        # use present in the dataframe
                        list_tp  = list(set(df_type_14["TIPOLOGIA"]))
                        list_tp = [x for x in list_tp if x not in ["nan", "NaN", "Null", "null", None]]
                        list_uses.extend(list_tp)
                        
                        # add all the list element to the layer as a new field with integet type
                        for use in list_uses:
                            layer_provider.addAttributes([QgsField(use, QVariant.Int)])
                         
                        # update the fields and layer provider
                        layer.updateFields()
                        
                        # create a second dictionary with all the new fields as keys 
                        d_evaluation = dict.fromkeys(list_uses)
                        
                        # for each key of the second one, add as its value a copy of the first
                        # dictionary
                        for key in d_evaluation.keys():
                            d_evaluation[key] = deepcopy(d_base_num)

                        #----------------------------------------------------------------------
                        
                        # update progress
                        progress += 0.01 * progress_total_increase
                        self.setProgress(progress)
                        
                        max_iter_p1 = len(df_type_14)
                        
                        # get the max number of iterations performable by the main processes
                        # (used to update the progress bar values)
                        progress_1st_process_increase = ((progress_cap - progress_start)*0.29)/max_iter_p1
                        
                        # iterate over all the table-type-14 rows (cadastral registers)
                        for index, row in df_type_14.iterrows():
                            
                            if self.isCanceled():
                                QgsMessageLog.logMessage("User cancelled the operation!", MESSAGE_CATEGORY, Qgis.Warning)
                                return False  # Task cancellation acknowledged here
                            
                            cad_ref = row["31_pc"] # cadastral reference number
                            tp_parc = row["TIPOLOGIA"] # edification typology
                            area = float(row["84_stl"]) # area of the edification
                            floor = int(row["65_pt"]) # floor of the edification
                            building_date = int(row["79_aec"]) # year of construcion building
                        
                            # iterate over all the uses (new fields that were added)
                            for use in list_uses:
                                
                                # if the use is the floor number, update the value of parcel
                                # only iif it is greater from the one save. So, this variable 
                                # will start from 0 an will always save the highest floor
                                if use == "N_PLANTAS":
                                    if floor > d_evaluation[use][cad_ref]:
                                        d_evaluation[use][cad_ref] = floor
                                        
                                # if the use is the year when the construction was built, 
                                # set it to 0(no data) if there is no value or update it 
                                # with the most old one if the year value is, at least, 
                                # superior to 1000
                                                                        
                                if use == "ANTIGUEDAD":
                                    if (d_evaluation[use][cad_ref] == 0 and building_date > 1000):
                                        d_evaluation[use][cad_ref] = building_date
                                        
                                    elif building_date < d_evaluation[use][cad_ref]:
                                        d_evaluation[use][cad_ref] = building_date
                                
                                # the number of residential buildings is evaluated one by on if
                                # it is any of the codes below. Each edification of that typology
                                # adds 1 to the variable value
                                if use == "NUM_INM_R":
                                    if tp_parc in ["0111", "0112"]:
                                        d_evaluation[use][cad_ref] += 1
                                    elif tp_parc in ["0121", "0122", "0131"]:
                                        d_evaluation[use][cad_ref] = 1
                                
                                # the parcel surface edified is evaluated using the 0 and 1 floor
                                # adding the edification area to each variable. There are cases
                                # that multifamiliar edification doesnt have 0 floor edification
                                # so the first one needs to be used
                                elif use == "SUP_ED_0":
                                    if floor == 0:
                                        d_evaluation[use][cad_ref] += area
                                        
                                elif use == "SUP_ED_1":
                                    if floor == 1:
                                        d_evaluation[use][cad_ref] += area
                                
                                # for the rest of the uses (codes of edification uses) add it up
                                # to each code, adding it to the dictionary
                                elif tp_parc == use:
                                    d_evaluation[use][cad_ref] += area
                                    d_evaluation['A_TOT_EDIF'][cad_ref] += area
                            
                            # update the progress bar value, using the previous value and the 
                            # currect index and maximum number of iterations
                            progress += progress_1st_process_increase
                            self.setProgress(progress)
                        
                        #------------------------------------------------------
                        
                        # get the new updated features, and layer field names        
                        features = layer.getFeatures()
                        fields_name = layer.fields().names()

                        max_iter_p2 = layer.featureCount()
                        progress_2nd_process_increase = ((progress_cap - progress_start)*0.68)/max_iter_p2
    
                        # transform the names of the variables in their index
                        # value inside the layer, for later attribute change
                        d_name_index_relation = {}
                        for key in d_evaluation.keys():
                            d_name_index_relation[key] = fields_name.index(key)
                        
                        d_transposed = {}
                        
                        # Iterate through the outer dictionary
                        for var, id_dict in d_evaluation.items():
                            for identifier, value in id_dict.items():
                                
                                # Initialize the inner dictionary if the identifier is not already a key
                                if identifier not in d_transposed:
                                    d_transposed[identifier] = {}
                                    
                                # Assign the value to the corresponding variable in the inner dictionary
                                d_transposed[identifier][d_name_index_relation[var]] = value
                        
                        d_translated = {}
                        
                        # Apply the key mapping
                        for old_key, new_key in unique_ref_cat.items():
                            if old_key in d_transposed:
                                for nk in new_key:
                                    d_translated[nk] = d_transposed[old_key]

                        #------------------------------------------------------
                        for key in d_translated.keys():
                            
                            layer_provider.changeAttributeValues({key:d_translated[key]})
                            
                            # update the progress bar
                            progress += progress_2nd_process_increase
                            self.setProgress(progress)
                                
                        # commit the changed made to the features
                        layer.commitChanges()
                
                        del layer
                        
                        i += 1
                    else:
                        print('Se ha producido un error a la hora de generar el ' +\
                              'fichero procesado. Asegúrese de haber introducido ' +\
                              'correctamente cada fichero según las indicaciones.')
                        break
                else:
                    self.setProgress(progress + progress_total_increase)
                    
            # set the progress bar to 100
            self.setProgress(100)
            
            # update the input required for the next process (clasification)
            if self.dlg.option_single_mun_p1.isChecked() == True:
                self.dlg.dir_edif_unique_p2.setText(result_file_path)
                self.dlg.dir_edif_unique_p2.setStyleSheet("background-color: rgb(200, 255, 200)")
    
            elif self.dlg.option_multiple_mun_p1.isChecked() == True:
                self.dlg.dir_root_multi_p2.setText(dest_filepath)
                self.dlg.dir_root_multi_p2.setStyleSheet("background-color: rgb(200, 255, 200)")
    
            # save time after running all the processing
            t_finish = time()
            
            # calculate the time needed to perform the processing
            t_process = (t_finish - t_start)
            
            self.t_process = str(round(t_process, 2))
            
            if self.isCanceled():
				# raise UserAbortedNotification('USER Killed')
                return False
            
            layer = QgsVectorLayer(wd_shp, '', 'ogr')
            
            # check if results are valid, and raise successful finish
            if layer.isValid():
				# return success
                return True
            else:
                self.exception = Exception("Output Dataset Invalid")
                return False
            
		# catch-all to prevent exceptions being buried by QGIS
        except Exception as e:
            self.exception = e
            return False

#------------------------------------------------------------------------------

class clasif_Task(QgsTask):
    ''' Task for classification process'''
    
    taskStarted = pyqtSignal()
    taskEnded = pyqtSignal()
    
    def __init__(self, dlg):
        """
		* Initialise the thread
		"""
		# call constructor for QsgThread
        super().__init__("Cadastral Classifier", QgsTask.CanCancel)

		# load arguments into class variables
        self.dlg = dlg
        self.t_process = None
        self.exception = None

    def finished(self, result):
        
        self.taskEnded.emit()  # Emit signal at the end of the task
        
        # success
        if result:
               
            # message to show on emergent window       
            finish_message = 'Tiempo total de ejecución: ' + self.t_process + ' segundos.'
            
   			# Notify User
            QgsMessageLog.logMessage(finish_message, MESSAGE_CATEGORY, Qgis.Success)
            iface.messageBar().pushMessage("¡Clasificación finalizada con éxito!", finish_message, level=Qgis.Success, duration=10)
            
        else:
   			# failed without exception (user cancel)
            if self.exception is None:
                QgsMessageLog.logMessage('Cancelado sin excepciones', MESSAGE_CATEGORY, Qgis.Warning)
                iface.messageBar().pushMessage("Atención:", 'Proceso cancelado', level=Qgis.Warning, duration=10)

   			# failed with exception (raise exception)
            else:
                QgsMessageLog.logMessage(f'Calcelado con la excepción: {self.exception}', MESSAGE_CATEGORY, Qgis.Critical)
                iface.messageBar().pushMessage("Error:", f'Calcelado con la excepción: {str(self.exception)}', level=Qgis.Critical)
   				# raise self.exception
                   
    def cancel(self):
   		"""
   		* This function is called when the task is cancelled
   		"""

   		# log that the task was cancelled
   		QgsMessageLog.logMessage("Proceso clasificación cancelado manualmente por el usuario", MESSAGE_CATEGORY, Qgis.Info)

   		# run the cancel method from QgsTask
   		super().cancel()
    
    def run(self):
        
        ''' Main function of the second process. Using the previously generated
        edification SHP, performs a clasification process of all the parcels.
        
        - Input SHP (.shp): shapefile of the previous process, that has the area
        of each possible code of the zone for each unique parcel.
        
        - Output SHP (.shp): resluting shapefile with the classification of the
        all the parcels of the studied area. As fileds it will have the 
        percentage for each of the selected classes, the total built area,
        the built area over surface, the proportion of built area over 
        surface, the max floor, year of oldest construction and 
        the number of residential buildings and finally the given use to the
        parcel.
        
        User can select 3 types of classification processes: basic, intermediate
        and advanced. A tab is given for each of them.
        
        - Basic: user can select different uses to include in the classification.
        Any use that was not selected will be classified as 'ohers'.
        
        - Intermediate: user can slect different uses to include in the 
        classification. In this case there are more uses, and for 3 of them, 
        it is possible to define mixed ones. For Single-family reditentials,
        Multi-family residentials and Industrial, user can define it mixed 
        class should be evaluated. For each of this 3 it is possible to define
        the percentage of minimum amount of developed area to define a 'pure'
        use. Below that value, mixed uses will be evaluated. All others follow
        max amount criteria.
        
        Along with these presets, there is an option to include a method to 
        perform a classification os non-built parcels to green zone using
        OpenStreetMaps. This has its limitations so take te results of the
        classification of these parcels with caution.
        
        Result is added to QGIS map and displayed with the selected color
        palette, depending on the classification method selected.
        '''
        
        self.taskStarted.emit()  # Emit signal at start of the task
        
        try:
                        
			# setup for progress bar ad message
            QgsMessageLog.logMessage("Iniciada la clasificación", MESSAGE_CATEGORY, Qgis.Info)
            
            # save time before starting the processing
            t_start = time()
            
            # structure the data input and outputs based on the option selecte 
            # by the user
            if self.dlg.option_single_mun_p2.isChecked() == True:
                
                # get the result file path
                dir_clasif_result = self.dlg.dir_clasif_result_unique_p2.text()
                output_filename_p2 = basename(dir_clasif_result) # get the name of the file
                
                # create a list to appen it, simulating as if it was a multiple option
                ls_mun = []
                ls_mun.append(basename(self.dlg.dir_edif_unique_p2.text()))
                
                root_path = dirname(self.dlg.dir_edif_unique_p2.text())
                                        
            elif self.dlg.option_multiple_mun_p2.isChecked() == True:
                
                # get the result file path
                output_filename_p2 = self.dlg.string_suffix_multi_p2.text()
                
                root_path = self.dlg.dir_root_multi_p2.text()
                
                chdir(root_path)
                
                # create a list with all the folder contained in the directory provided
                ls_mun = glob('*')
            
            # if no name has been declared assign one
            if not output_filename_p2:
                output_filename_p2 = "clasificacion.shp"
                
            elif output_filename_p2.lower().endswith('.shp') == False:
                output_filename_p2 = output_filename_p2 + ".shp"
            
            # set the progress value of the progress bar to 0
            progress = 0
            self.setProgress(progress)
            
            # get the caps for each mun execution
            progress_each_mun = 100/len(ls_mun)
            
            z = 0
                    
            # iterate over all municipalities that the selected directory has
            for mun in ls_mun:
                
                if self.isCanceled():
                    QgsMessageLog.logMessage("User cancelled the operation!", MESSAGE_CATEGORY, Qgis.Warning)
                    return False  # Task cancellation acknowledged here
                
                progress_start = z * progress_each_mun
                # state the cap that has to be reached once all the operations
                # in the mun are performed
                progress_cap = progress_each_mun + z * progress_each_mun
                
                progress_total_increase = progress_cap - progress_start
                
                progress = progress_start
                
                # generate a filepath of the mun to contain the results
                out_dir = join(root_path, mun)
                                        
                # get the input/output paths            
                if self.dlg.option_single_mun_p2.isChecked() == True:
                    filepath_edif_SHP_input = self.dlg.dir_edif_unique_p2.text()
                    filepath_clasif_SHP_output = dir_clasif_result
        
                elif self.dlg.option_multiple_mun_p2.isChecked() == True:
                    filepath_edif_SHP_input = join(out_dir, "procesado_" + mun + ".shp")
                    
                    if isfile(filepath_edif_SHP_input) == False:
                        filepath_edif_SHP_input = join(out_dir, "edif.shp")
                        
                        if isfile(filepath_edif_SHP_input) == False:
                            filepath_edif_SHP_input = join(out_dir, "procesado_SHP.shp")
        
                    
                    filepath_clasif_SHP_output = join(out_dir, output_filename_p2)
                    
                # if the directory has the file of the previous process continue 
                # with the rest of operations
                
                if isfile(filepath_edif_SHP_input) == True:
        
                    # based on the user selection of the option of classification, open the 
                    # csv files of the basic csv file as a dataframe
                    if self.dlg.option_basic_clasif.isChecked():
                        df_clasif_parameters_CSV = read_csv(join(dirname(__file__),
                                             "clasif","clasificacion_basica.csv"))
                            
                    # if it was selected open the intermediate csv as dataframe
                    elif self.dlg.option_intermediate_clasif.isChecked():
                        df_clasif_parameters_CSV = read_csv(join(dirname(__file__),
                                             "clasif", "clasificacion_intermedia.csv"))
                    
                    # for the advanced option get the dataframe from the table present in 
                    # the tab 
                    elif self.dlg.option_advanced_clasif.isChecked():
                        
                        # create a new dataframe
                        df_clasif_parameters_CSV = DataFrame()
                        
                        # count the number of rows and columns of the table
                        n_rows = self.dlg.table_advanced_option.rowCount()        
                        n_cols = self.dlg.table_advanced_option.columnCount()
                        
                        # declare a variable to use as index in the dataframe
                        identifier = 0
                        
                        # iterate over the rows and columns
                        for r in range (0, n_rows):
                            for c in range (0, n_cols):
                                
                                # get the column name
                                col_name = self.dlg.table_advanced_option.horizontalHeaderItem(c).text()
                                
                                # get the text saved in the cell
                                item = self.dlg.table_advanced_option.item(r,c)
                                
                                # if there is any value, save the cell value to the item
                                # with the same position in the dataframe
                                if item:                
                                    df_clasif_parameters_CSV.loc[int(identifier), col_name] = item.text()
                                    
                                else:
                                    df_clasif_parameters_CSV.loc[int(identifier), col_name] = ""
                            
                            # updata index value
                            identifier += 1
                                    
                    #--------------------------------------------------------------
                    
                    # get a list with the classes (clasification uses) of the dataframe
                    ls_keys = list(df_clasif_parameters_CSV["Clase"])
                    
                    # get a list with the categories (string format) that each class has 
                    categories = list(df_clasif_parameters_CSV["Categorias"])
                    
                    # create a list to store the lists of each category code
                    ls_cateogories = []
                    
                    # iterate over each cateogry cluster
                    for category in categories:
                        
                        # split the string using the comma
                        categories_list = list(category.split(","))
                        
                        # add the list with the codes to the list with all categories by class
                        ls_cateogories.append(categories_list)
                
                    # get the labels and colors of each lass
                    labels = list(df_clasif_parameters_CSV["Etiqueta"])
                    colors = list(df_clasif_parameters_CSV["Color"])
                     
                    # create dictionaries with the classes 
                    d_evaluate = dict.fromkeys(ls_keys)
                    d_palettes = dict.fromkeys(ls_keys)
                    d_labels = dict.fromkeys(ls_keys)
                    
                    # index to use on list values extraction
                    i = 0
                    
                    # iterate over one of the dictionaries
                    for key in d_evaluate.keys():
                        
                        # fill the evaluation dictionary with the list of categories codes
                        d_evaluate[key] = ls_cateogories[i]  
                        
                        # fill the class dictionary with its labels 
                        d_labels[key] = labels[i]
                        
                        # if a color is present, fill it 
                        if colors[i]:
                            d_palettes[key] = colors[i]
                   
                        # in other case generate a random one
                        else:
                            d_palettes[key] = random_color()
                        
                        i+=1
                    
                    # add a colo and label to a static uses that always will be present
                    d_palettes["SIN_EDIF"] = '#D5D8DC'
                    d_palettes["OTROS"] = '#999999'
                    d_palettes["ZON_VER"] = '#72CF72'
                    
                    d_labels["SIN_EDIF"] = 'Sin edificar'
                    d_labels["OTROS"] = 'Otros' 
                    d_labels["ZON_VER"] = 'Zonas verdes'
                        
                    # in the intermediate clase add label and color to the mixed values
                    if self.dlg.option_intermediate_clasif.isChecked():
                        
                        d_palettes["RES_UNI_MX"] = '#D98880'
                        d_palettes["RES_PLU_MX"] = '#F1948A'
                        d_palettes["IND_MX"] = '#BB8FCE'
                        
                        d_labels["RES_UNI_MX"] = 'Residencial unifamiliar mixto'
                        d_labels["RES_PLU_MX"] = 'Residencial plurifamiliar mixto'
                        d_labels["IND_MX"] = 'Industrial mixto'
                    
                    # for the advanced option, for tha cases where mixed use posibility is
                    # given , generate a label and random color for that class
                    elif self.dlg.option_advanced_clasif.isChecked():
                        
                        for index, row in df_clasif_parameters_CSV.iterrows():
                            try:
                                if float(row["%"]) > 0:
                                    
                                    new_class = row["Clase"] + "_MX"
                                    
                                    d_labels[new_class] = row["Etiqueta"] + " mixto"
                                    d_palettes[new_class] = random_color()
                            except:
                                pass
                           
                    #--------------------------------------------------------------
                    
                    # generate a list to store the selected classes to classify
                    ls_selected = []
                    
                    # dictionary to store the mixed values
                    d_mixed = dict()
                    
                    # declaring a list variable to store differente variables to pass to 
                    # later flexible function
                    variables = []
                    
                    # if basic option was selected, add the classes if they are checked
                    # set the threshold of non_edified parcel to 0 (only is the parcel has 0
                    # are aedified will be considered as non edified), and select the option
                    # basic on the use definir function
                    if self.dlg.option_basic_clasif.isChecked():
                        
                        if self.dlg.Checkbox_Resi.isChecked() == True:
                            ls_selected.append("RES")
                        if self.dlg.Checkbox_Indus.isChecked() == True:
                            ls_selected.append("IND")
                        if self.dlg.Checkbox_Equip.isChecked() == True:
                            ls_selected.append("EQUIP")
                        if self.dlg.Checkbox_Com_Ofi.isChecked() == True:
                            ls_selected.append("COM_OFI")
                        if self.dlg.Checkbox_Ed_Sing.isChecked() == True:
                            ls_selected.append("ED_SING")
                        if self.dlg.Checkbox_Oc_Res.isChecked() == True:
                            ls_selected.append("OC_RES")
                        if self.dlg.Checkbox_Ed_Sing.isChecked() == True:
                            ls_selected.append("ED_SING")
                            
                        value_threshold_not_built = 0
                        
                        use_definer = use_function_definer(selection = "basic")
                            
                    #--------------------------------------------------------------
                    
                    # if intermediate option was selected, add the checked classes, set threshold 
                    # to user defined or 0 if it was not and append variable values to the
                    # list that later will be passes to use definer function. In this case,
                    # mixed uses are present, and their definition is based on percentages.
                    elif self.dlg.option_intermediate_clasif.isChecked():
                                            
                        if self.dlg.Checkbox_resi_unif.isChecked() == True:
                            ls_selected.append("RES_UNI")
                        if self.dlg.Checkbox_resi_plu.isChecked() == True:
                            ls_selected.append("RES_PLU")
                        if self.dlg.Checkbox_Indus_2.isChecked() == True:
                            ls_selected.append("IND")
                        if self.dlg.Checkbox_ofi.isChecked() == True:
                            ls_selected.append("OFI")                
                        if self.dlg.Checkbox_equip_edu.isChecked() == True:
                            ls_selected.append("EQUIP_EDU")
                        if self.dlg.Checkbox_equip_sanit.isChecked() == True:
                            ls_selected.append("EQUIP_SANI")
                        if self.dlg.Checkbox_equip_otros.isChecked() == True:
                            ls_selected.append("EQUIP_OTR")
                        if self.dlg.Checkbox_comercio.isChecked() == True:
                            ls_selected.append("COM")
                        if self.dlg.Checkbox_hos_resta.isChecked() == True:
                            ls_selected.append("HOS_REST")
                        if self.dlg.Checkbox_ocio_esp.isChecked() == True:
                            ls_selected.append("OCIO_ESP")
                        if self.dlg.Checkbox_edif_sing.isChecked() == True:
                            ls_selected.append("ED_SING")
                            
                        #----------------------------------------------------------  
                        # Mixed uses. Get the slider values to define them and create a cluster
                        # of codes that will also be counted as pure use, independently of 
                        # the percentages
                        
                        # list with specific codes that will be used as cluster to define
                        # pure residential use as a combination with this ones
                        conglomerate_include = ["0221", "0213", "0511", "0521", "0512", "0522"]
                        
                        # create a list with the codes of the unifamily residentials 
                        # alongside the conglomerates to work as a pure cluster
                        congregated_list_resi_uni = deepcopy(d_evaluate["RES_UNI"])
                        congregated_list_resi_uni.extend(conglomerate_include)
                        
                        # create a list with the codes of the multifamily residentials 
                        # alongside the conglomerates to work as a pure cluster
                        congregated_list_resi_plu = deepcopy(d_evaluate["RES_PLU"])
                        congregated_list_resi_plu.extend(conglomerate_include)
                        
                        if self.dlg.Checkbox_resi_uni_mix.isChecked() == True and\
                            self.dlg.Checkbox_resi_uni_mix.isEnabled() == True:
                                
                            d_mixed["RES_UNI"] = self.dlg.slider_threshold_uni.value()
                            
                        #----------------------------------------------------------    
                        if self.dlg.Checkbox_resi_plu_mix.isChecked() == True and\
                            self.dlg.Checkbox_resi_plu_mix.isEnabled() == True:
                                
                            d_mixed["RES_PLU"] = self.dlg.slider_threshold_plu.value()
                            
                        #----------------------------------------------------------   
                        if self.dlg.Checkbox_Indus_mix.isChecked() == True and\
                            self.dlg.Checkbox_Indus_mix.isEnabled() == True:
                                
                            d_mixed["IND"] = self.dlg.slider_threshold_ind.value()
                        
                        #----------------------------------------------------------
                        
                        if self.dlg.Checkbox_threshold_not_built.isChecked() == True:
                            value_threshold_not_built = float(self.dlg.slider_threshold_not_built.value()/100)
                            
                        else:
                            value_threshold_not_built = 0
                        
                        #----------------------------------------------------------
                        
                        use_definer = use_function_definer(selection = "intermediate")
                        
                        variables.append(d_mixed)
                        variables.append(congregated_list_resi_uni)
                        variables.append(congregated_list_resi_plu)
                        
                    #--------------------------------------------------------------
                    
                    # if advanced option was selected extract the information of the tab 
                    # table 
                    elif self.dlg.option_advanced_clasif.isChecked():
                        
                        # get the rows (number of classes)
                        n_rows = self.dlg.table_advanced_option.rowCount()        
                        
                        for r in range (0, n_rows):
                            
                            # get the item related to the check/uncheck state
                            item = self.dlg.table_advanced_option.item(r, 0)
                                 
                            # if it was checked
                            if item.checkState() == 2:
                                
                                # get the class name
                                class_item = self.dlg.table_advanced_option.item(r, 1).text()
                                
                                # append it to the list of selected ones to perform the analysis
                                ls_selected.append(class_item)
                                
                                # get the purity required for that class
                                try:
                                    purity =  float(self.dlg.table_advanced_option.item(r, 4).text())
                                    
                                except:
                                    purity = 0
                                
                                # if the purity is greater thatn 0, means user wants to 
                                # include a mixed class additional to 'pure' one
                                if purity > 0:
                                    
                                    d_mixed[class_item] = purity
            
                        # same as on inremediate, get the non edified minimum value
                        if self.dlg.Checkbox_threshold_not_built_2.isChecked() == True:
                            value_threshold_not_built = float(self.dlg.slider_threshold_not_built_2.value()/100)
                            
                        else:
                            value_threshold_not_built = 0
                        
                        # select the advanced method con the user definer function
                        use_definer = use_function_definer(selection = "advanced")
                        
                        variables.append(d_mixed)
                    
                    #--------------------------------------------------------------
                    
                    # copy the edfication shapefile (the one first process generates)
                    ruta_dbf = shp_copy(filepath_edif_SHP_input, filepath_clasif_SHP_output)
                    
                    # get the raw name of the coppied file, to give it to QGIS layer 
                    new_file_name = basename(ruta_dbf)[:-4]
                    
                    # read the copy as a QGIS layer
                    layer = QgsVectorLayer(ruta_dbf, new_file_name, "ogr")
                    layer_provider = layer.dataProvider()
                    
                    #--------------------------------------------------------------
                
                    # save the fields names of the layer
                    columns_1 = layer.fields().names()
                    
                    # get the index of the last field 
                    index_last_field = columns_1.index("NUM_INM_R")
            
                    # get the use list by selecting the ones from the last one that is not
                    # a codification one
                    list_uses = columns_1[index_last_field + 1:len(columns_1)]
                    
                    # get a list with the keys of the selected classes
                    list_equiv = list(d_evaluate.keys())
                      
                    #--------------------------------------------------------------
                    
                    # add new the columns of use and the proportion of edified area to the
                    # classification (result) shpaefile
                    layer_provider.addAttributes([QgsField("USO", QVariant.String)])
                    layer_provider.addAttributes([QgsField("PROP_AE_AT", QVariant.Double)])
            
                    # add the classification uses fields to the new shapefile
                    for use in list_equiv:
                        layer_provider.addAttributes([QgsField(use, QVariant.Int)])
                    
                    # update fields and layer provider
                    layer.updateFields()
                    layer_provider = layer.dataProvider()
                        
                    progress += 0.01 * progress_total_increase
                    self.setProgress(progress)
                    
                    # get maximum iteration number for the progress bar
                    max_iter = layer.featureCount()
                    
                    progress_increase = ((progress_cap - progress_start)*0.99)/max_iter
             
                    # get all the features        
                    features = layer.getFeatures()
                    
                    # get the new column name names
                    columns_2 = layer.fields().names()
                    
                    # append the list of the columns to the variables to pass to the use 
                    # definer function
                    variables.append(columns_2)
                    
                    # save the index of the fields of use and proportion
                    use_column_index = columns_2.index("USO")
                    prop_column_index = columns_2.index("PROP_AE_AT")
                    
                    # start the editing of the layer
                    layer.startEditing()
            
                    # iterate over all features
                    for f in features:
                        
                        if self.isCanceled():
                            QgsMessageLog.logMessage("User cancelled the operation!", MESSAGE_CATEGORY, Qgis.Warning)
                            return False  # Task cancellation acknowledged here
                        
                        # get the id
                        f_id = f.id()
                        
                        # get the feature(parcel) total area
                        total_parcel_Area = f["AREA"]
                        
                        # get its surface built area
                        built_area = max(f["SUP_ED_0"], f["SUP_ED_1"])
                        
                        # get the total built area
                        built_area_2 = f["A_TOT_EDIF"]
                                   
                        #----------------------------------------------------------
                        
                        # the proportion value depends on if the floor 0 or 1 are relevat 
                        # to dermine it. If it is not (cause floor codification could not 
                        # be transformed to int) total edified area will be used
                        if total_parcel_Area > 0:
                            proportion = (built_area/total_parcel_Area)
                            proportion_2 = (built_area_2/total_parcel_Area)
                            
                        else:
                            proportion = 0
                            proportion_2 = 0
                     
                        if proportion > 0:
                            proportion_value = proportion
                            
                        else:
                            proportion_value = proportion_2
                            
                        #----------------------------------------------------------
                        
                        # create a variable to store the use of the parcel
                        use = None
                        
                        # if the proportion of edified area is equal or lesser to the 
                        # threshold given, its use will be 'non_edified'
                        if (proportion_value <= value_threshold_not_built):
                            use = "SIN_EDIF" 
                                
                        # create a dictionary to store the percentages for each class
                        d_feature_values = dict.fromkeys(list_equiv, 0)
                        
                        # if it doesn't have already an use ('non_edified')
                        if not use:
                        
                            # iterate over the possible classes that are declared
                            for use2 in list_equiv:
                                
                                # create a variable for that use
                                use_area = 0 
                                
                                # get the codes that belong to that class
                                equiv = d_evaluate[use2] 
                                
                                # itera over all the codes of that class
                                for e in equiv:
                                    
                                    # if the use is present in the list of possible uses
                                    # that are present in the shapefile
                                    if e in list_uses:
                                        
                                        # add up the built are of that use of that feature
                                        use_area += f[e]
                                
                                # the built proportion of that use is calculated
                                if use_area > 0:
                                    value = (use_area/f["A_TOT_EDIF"])*100
                                    
                                else:
                                    value = 0
                                
                                # store the proportion
                                d_feature_values[use2] = value
                                    
                            # list to store the percentages
                            lv = []
                            
                            # iterate over the uses and append its percentage
                            for use in list_equiv:
                                lv.append(d_feature_values[use])
                         
                            # copy the list with the percentages and sort it from max to min
                            lv_s = deepcopy(lv)
                            lv_s.sort(reverse = True)
                            
                            # create a list to store the names of the class
                            lv_s_names = []
                         
                            # fill the name list with the name of the max-to-min list, if 
                            # it is grater than 0
                            for i in range (0, len(lv_s)): 
                                if lv_s[i] > 0:           
                                    lv_s_names.append(list_equiv[lv.index(lv_s[i])])
                            
                            # if there is at least one use that have a percentage greater
                            # than 0
                            if len(lv_s_names) > 0:
                                
                                # execute the use definer function to get the use the parcel
                                use = use_definer(f, d_feature_values, lv_s_names,
                                                  ls_selected, variables)
                            else:
                                # any other case, put it in 'others'
                                use = "OTROS"
                        
                        # use the keys of the dictionary used for each feature to store the
                        # percentage values to the feature, using the indexers and layer
                        # provider
                        for key in d_feature_values.keys():
                            
                            use_index = columns_2.index(key)
                            use_percentage_adjudication = {use_index:d_feature_values[key]}
                            layer_provider.changeAttributeValues({f_id:use_percentage_adjudication})
                        
                        # save the use to the feature field
                        use_adjudication = {use_column_index:use}
                        layer_provider.changeAttributeValues({f_id:use_adjudication})
                        
                        # save the proportion value to the feature field
                        proportion_value_adjudication = {prop_column_index:proportion_value}
                        layer_provider.changeAttributeValues({f_id:proportion_value_adjudication})
                            
                        # update progress bar                        
                        progress += progress_increase

                        self.setProgress(progress)
            
                    # save the changes made by calling the commit changes to layer
                    layer.commitChanges()
            
                    # delete the unnecesary fields (areas of each code)
                    use_list_indexers = [i for i in range(index_last_field + 1, len(columns_1))]
                    layer_provider.deleteAttributes(use_list_indexers)        
                    layer.updateFields()
                    
                    # lastly, check if green zones have to be included
                    if self.dlg.checkbox_zonas_verdes.isChecked():
        
                        # Define the CRS of the OSM Overpass query
                        crs_OSM = QgsCoordinateReferenceSystem('EPSG:4326')
                                            
                        # Get the bounding box of the municipality layer in the 
                        # right CRS
                        bbox_layer = QgsCoordinateTransform(layer.crs(), crs_OSM, QgsProject.instance()).transformBoundingBox(layer.extent())
        
                        # get the bbox in an apropriate format
                        bbox_QUERY = (bbox_layer.yMinimum(), bbox_layer.xMinimum(), bbox_layer.yMaximum(), bbox_layer.xMaximum())
                        
                        # get the green using the bbox
                        green_zones_layer = get_green_zones(bbox_QUERY)
                        
                        layer_crs = layer.crs()
                        
                        if green_zones_layer:
                        
                            # reproject it to 25830 since it is in 4326
                            gz_layer_reproj = processing.run("native:reprojectlayer",
                                                             {"INPUT": green_zones_layer,
                                                             "TARGET_CRS": layer_crs,
                                                             "OUTPUT": "memory:"})["OUTPUT"]
                            # start eiting both layers
                            layer.startEditing()
                            gz_layer_reproj.startEditing()
                            
                            # add ID and AR_OL_ZV (Overlay Green Zone Area) to the
                            # classification layer
                            layer.addExpressionField('$id', QgsField('ID', QVariant.Int))
                            layer_provider.addAttributes([QgsField('AR_OL_ZV', QVariant.Double)])
                            
                            # update the fiels
                            layer.updateFields()
                                           
                            # get the patial inex of the green zones layer
                            green_zones_index = QgsSpatialIndex(gz_layer_reproj.getFeatures())
                        
                            # get all the features that has been classified as 'non built'
                            # in the previous operations
                            feats_sin_edif = []
                            for feature in layer.getFeatures(QgsFeatureRequest().setFilterExpression('"USO"=\'SIN_EDIF\'')):
                                feats_sin_edif.append(feature)
                        
                            # create a list to store all the features that will change 
                            # their use to 'green zones'
                            feats_selected = []
                            for feature in feats_sin_edif: # iterate over all the 'non edified features'
                            
                                # get the IDs of the green zones that intersects with 
                                # the feature, using spatial indexing to improve performance
                                intersected_ids = green_zones_index.intersects(feature.geometry().boundingBox())
                                                            
                                # iterate over the IDs of intersecting green zones 
                                for intersected_id in intersected_ids:
                                    
                                    # get the intersected feature with the green zone layer
                                    intersected_feature = gz_layer_reproj.getFeature(intersected_id)
                                    
                                    # get reat intersection between both geometries,
                                    # and store the pair if they intersect
                                    if intersected_feature.geometry().intersects(feature.geometry()):
                                        feats_selected.append((feature, intersected_feature))
                            
                            # create a dictionary to store the intersection areas
                            d_ZV_IDs = {}
                            # iterate over the intersection pairs (layer feature - green zone feature)
                            for feature, intersected_feature in feats_selected:
                                
                                # create a variable to sotre the intersection area
                                overlayed_area = 0
                                
                                # compute the intersetcion
                                intersection = feature.geometry().intersection(intersected_feature.geometry())
                                overlayed_area += intersection.area() #store the area
                                
                                # add the value to the areas dictionary
                                if feature.id() in d_ZV_IDs:
                                    d_ZV_IDs[feature.id()] += overlayed_area
                                else:
                                    d_ZV_IDs[feature.id()] = overlayed_area
                                    
                            # Get the index of the 'AR_OL_ZV' field
                            ar_ol_zv_index = layer.fields().indexFromName('AR_OL_ZV')
                                                
                            # Loop over the features in the layer and update the 'AR_OL_ZV' attribute
                            for feature in layer.getFeatures():
                                feature_id = feature.id()
                                if feature_id in d_ZV_IDs:
                                    layer.changeAttributeValue(feature_id, ar_ol_zv_index, d_ZV_IDs[feature_id])
        
                            # add a new field base on the expression
                            layer.addExpressionField('CASE WHEN "AREA" > 0 THEN "AR_OL_ZV" / "AREA" ELSE 0 END', QgsField('RATIO_ZV_AR', QVariant.Double))
                        
                            # create a last list to get the parcels which overlayed area
                            # is superior to a 10% of its own area
                            feats_selected = []
                            for feature in layer.getFeatures(QgsFeatureRequest().setFilterExpression('"RATIO_ZV_AR" > 0.10')):
                                feats_selected.append(feature.id())
                        
                            for feature_id in feats_selected:
                                layer.changeAttributeValue(feature_id, layer.fields().lookupField('USO'), 'ZON_VER')
                            layer.commitChanges() # commit changes to the classification layer
                            
                            del gz_layer_reproj, green_zones_layer
                            
                        else:
                            print('No se han podido obtener las zonas verdes de ' +\
                                  'OpenStreetMaps.')
                            pass
                        
                    if self.dlg.checkbox_add_result_to_project.isChecked() == True:
                        
                        # add the layer to the QGIS map
                        QgsProject.instance().addMapLayer(layer)
                        
                        # get the index of use field
                        fni = layer.fields().indexFromName('USO')
                        
                        # get its unique values
                        unique_values = list(layer.uniqueValues(fni))
                        
                        # generate a palette and label for each use, using the dictionaries
                        categories = palette_generator(layer, unique_values, d_palettes, d_labels)   
                        
                        # create a renderer for each class
                        renderer = QgsCategorizedSymbolRenderer('USO', categories)
                        
                        # assign the created renderer to the layer
                        if renderer is not None:
                            layer.setRenderer(renderer)
                        
                        # paint the layer
                        layer.triggerRepaint()
                    
                    z += 1
                    
                else:
                    print("El directorio: '" + dirname(filepath_edif_SHP_input) +\
                          "' no contiene el fichero preprocesado del módulo de " +\
                          "tratamiento de datos.")
                    
            # set progress bar to 100
            self.setProgress(100)
            
            #----------------------------------------------------------------------
            
            # save time after running all the processing
            t_finish = time()
            
            # calculate the time needed to perform the processing
            t_process = (t_finish - t_start)
            
            self.t_process = str(round(t_process, 2))
            
            if self.isCanceled():
				# raise UserAbortedNotification('USER Killed')
                return False
                        
            # check if results are valid, and raise successful finish
            if layer.isValid():
				# return success
                return True
            else:
                self.exception = Exception("Output Dataset Invalid")
                return False
            
		# catch-all to prevent exceptions being buried by QGIS
        except Exception as e:
            self.exception = e
            return False
            
#------------------------------------------------------------------------------

class cadastral_classifier:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = join(
            self.plugin_dir,
            'i18n',
            'cadastral_classifier_{}.qm'.format(locale))

        if exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&cadastral_classifier')
        
        self.toolbar = self.iface.addToolBar(u'Cadastral Classifier')
        self.toolbar.setObjectName(u'Cadastral Classifier')

        # Initialize the interface
        self.dlg = cadastral_classifierDialog()
        
        # Task manager to process in the background
        self.tm = QgsApplication.taskManager()

        # First starter checker
        self.first_start = False
        
        # d_codes variable declaration
        self.d_codes = None

        #----------------------------------------------------------------------

        ### Create the connections with the bottons and their functions
        ## Buttons related to input and output files routes/names
        
        self.dlg.button_cat_urb_multi_directory.clicked.connect(self.input_cat_urb_multi_directory)
        self.dlg.button_shp_urb_multi_directory.clicked.connect(self.input_shp_urb_multi_directory)
        self.dlg.button_cat_urb_multi.clicked.connect(self.input_cat_urb_multi)
        self.dlg.button_shp_urb_multi.clicked.connect(self.input_shp_urb_multi)
        self.dlg.button_cat_urb_unique.clicked.connect(self.input_cat_urb_unique)
        self.dlg.button_shp_urb_unique.clicked.connect(self.input_shp_urb_unique)
        
        self.dlg.button_cat_rust_multi_directory.clicked.connect(self.input_cat_rust_multi_directory)
        self.dlg.button_shp_rust_multi_directory.clicked.connect(self.input_shp_rust_multi_directory)
        self.dlg.button_cat_rust_multi.clicked.connect(self.input_cat_rust_multi)
        self.dlg.button_shp_rust_multi.clicked.connect(self.input_shp_rust_multi)
        self.dlg.button_cat_rust_unique.clicked.connect(self.input_cat_rust_unique)
        self.dlg.button_shp_rust_unique.clicked.connect(self.input_shp_rust_unique)
        
        self.dlg.button_output_result_edif.clicked.connect(self.output_result_p1)
        self.dlg.button_edif_unique_p2.clicked.connect(self.input_edif_unique_p2)

        self.dlg.button_clasif_result_unique_p2.clicked.connect(self.output_clasif_result_unique_p2)
        self.dlg.button_root_multi_p2.clicked.connect(self.input_dir_multi_p2)
        
        self.dlg.filtro_mun.textChanged.connect(self.filter_items)
        self.dlg.filtro_mun.setPlaceholderText('Filtrar municipios (separar con una coma para varios)...')
        
        self.dlg.filtro_mun.textChanged.connect(self.filter_items)

        self.dlg.mun_list.itemClicked.connect(self.update_items)
        self.dlg.button_selec_todos_mun.clicked.connect(self.select_all_mun)
        self.dlg.button_deselec_todos_mun.clicked.connect(self.deselect_all_mun)
        self.dlg.Option_include_rural.toggled.connect(self.check_all_inputs)
                
        #----------------------------------------------------------------------
        
        # Create the connections with the buttons that runs the main functions
        self.dlg.button_create_result_shp.clicked.connect(self.setup_data) 
        self.dlg.button_classifier.clicked.connect(self.clasif)

        # Create the connections with the checkable items
        self.dlg.button_check_basic_clasif.clicked.connect(self.check_basic)
        self.dlg.button_uncheck_bsaic_clasif.clicked.connect(self.uncheck_basic) 
        self.dlg.button_check_intermediate.clicked.connect(self.check_intermediate)
        self.dlg.button_uncheck_intermediate.clicked.connect(self.uncheck_intermediate)   
        self.dlg.button_include_mx.clicked.connect(self.include_mx)
        self.dlg.button_exclude_mx.clicked.connect(self.exclude_mx)
        
        ## Connections of the addvanced clasification method
        
        # Add and delete row functions 
        self.dlg.button_add_row.clicked.connect(self.add_row)  
        self.dlg.button_delete_row.clicked.connect(self.delete_row)

        # Add and delete categories
        self.dlg.button_add_categories.clicked.connect(self.add_categories)  
        self.dlg.button_delete_categories.clicked.connect(self.deselect_all_categories)  
    
        # Import and save the clasifications
        self.dlg.button_import.clicked.connect(self.import_clasif)  
        self.dlg.button_save.clicked.connect(self.save_clasif)  
        
        # Set the column widths
        self.set_column_widths()
        
        #----------------------------------------------------------------------
        
        # Buttons related to show and hide optional panels
        self.dlg.help_panel_1.setHidden(True)   
        self.dlg.help_panel_2.setHidden(True)
        self.dlg.Box_intermediate_clasif.setHidden(True)   
        self.dlg.Box_advanced_clasif.setHidden(True)        
        self.dlg.widget_opcion_U_unique.setHidden(True)
        #self.dlg.Frame_single_mun_p2.setHidden(True)
        self.dlg.Widget_single_mun_p2.setHidden(True)
        self.dlg.Group_table_selection.setHidden(True)
        self.dlg.tab_info_intermediate_categories.setHidden(True)
        self.dlg.frame_threshold_not_built.setHidden(True)
        self.dlg.frame_threshold_not_built_2.setHidden(True)
        self.dlg.frame_3.setHidden(True)
        self.dlg.frame_R_multi.setHidden(True)

        # Setting the routes to the icon files
        folder_icon = join(dirname(__file__), 'icons', 'folder_icon.png')
        zip_icon = join(dirname(__file__), 'icons', 'zip_icon.png')
        file_icon = join(dirname(__file__), 'icons', 'file_icon.png')
        file_gz_icon = join(dirname(__file__), 'icons', 'file_gz_icon.png')
        arrow_left_icon = join(dirname(__file__), 'icons', 'arrow_left_icon.png')
        arrow_right_icon = join(dirname(__file__), 'icons', 'arrow_right_icon.png')
             
        # Set the icons for each button that has a custom one
        self.dlg.button_cat_urb_multi_directory.setIcon(QIcon(folder_icon))
        self.dlg.button_shp_urb_multi_directory.setIcon(QIcon(folder_icon))
        self.dlg.button_cat_urb_multi.setIcon(QIcon(zip_icon))
        self.dlg.button_shp_urb_multi.setIcon(QIcon(zip_icon))
        self.dlg.button_cat_urb_unique.setIcon(QIcon(file_gz_icon))
        self.dlg.button_shp_urb_unique.setIcon(QIcon(file_gz_icon))
        
        self.dlg.button_cat_rust_multi_directory.setIcon(QIcon(folder_icon))
        self.dlg.button_shp_rust_multi_directory.setIcon(QIcon(folder_icon))
        self.dlg.button_cat_rust_multi.setIcon(QIcon(zip_icon))
        self.dlg.button_shp_rust_multi.setIcon(QIcon(zip_icon))
        self.dlg.button_cat_rust_unique.setIcon(QIcon(file_gz_icon))
        self.dlg.button_shp_rust_unique.setIcon(QIcon(file_gz_icon))
        
        self.dlg.button_output_result_edif.setIcon(QIcon(folder_icon))
        self.dlg.button_edif_unique_p2.setIcon(QIcon(file_icon))

        self.dlg.button_clasif_result_unique_p2.setIcon(QIcon(file_icon))
        self.dlg.button_root_multi_p2.setIcon(QIcon(folder_icon))
        
        self.dlg.button_add_categories.setIcon(QIcon(arrow_right_icon))
        self.dlg.button_delete_categories.setIcon(QIcon(arrow_left_icon))
        
        #----------------------------------------------------------------------
        
        # Setting the routes to the funding logos
        logo_EU_NG = join(dirname(__file__), 'icons', 'logo_EU_NG.jpg')
        logo_AEI = join(dirname(__file__), 'icons', 'logo_AEI.jpg')
        logo_PRTR = join(dirname(__file__), 'icons', 'logo_PRTR.png')
        logo_min_innov = join(dirname(__file__), 'icons', 'logo_ministerio_innovacion.png')

        # Set the logos of the interface
        self.dlg.label_logo_EU_NG.setPixmap(QPixmap(logo_EU_NG))
        self.dlg.label_logo_AEI.setPixmap(QPixmap(logo_AEI))
        self.dlg.label_logo_PRTR.setPixmap(QPixmap(logo_PRTR))
        self.dlg.label_logo_min_innov.setPixmap(QPixmap(logo_min_innov))

        # Set the connections to the sliders
        self.dlg.slider_threshold_not_built.valueChanged[int].connect(self.update_double_value)
        self.dlg.slider_threshold_not_built_2.valueChanged[int].connect(self.update_double_value_2)
        self.dlg.slider_threshold_uni.valueChanged[int].connect(self.update_uni_mx_value)
        self.dlg.slider_threshold_plu.valueChanged[int].connect(self.update_plu_mx_value)
        self.dlg.slider_threshold_ind.valueChanged[int].connect(self.update_ind_mx_value)

        # Delete the option to deselect the type 14 table
        self.dlg.checkBox_type_14.setEnabled(False)
        
        self.dlg.input_CAT_URB_multi_type = None
        self.dlg.input_SHP_URB_multi_type = None
        self.dlg.input_CAT_RUS_multi_type = None
        self.dlg.input_SHP_RUS_multi_type = None
    
    #--------------------------------------------------------------------------
    
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('cadastral_classifier', message)

    #--------------------------------------------------------------------------
    
    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """
        
        # Initialize the icon of the plugin
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds the plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action
    
    #--------------------------------------------------------------------------

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = join(dirname(__file__), 'icon.jpg')
        self.add_action(
            icon_path,
            text=self.tr(u'cadastral_classifier'),
            callback=self.run,
            parent=self.iface.mainWindow())
    
        # will be set False in run()
        self.first_start = True
        
    #--------------------------------------------------------------------------
    
    def disableButtons(self):
        '''function to disable execute task button while a task is started'''
        self.dlg.button_create_result_shp.setDisabled(True)
        self.dlg.button_classifier.setDisabled(True)

    #--------------------------------------------------------------------------
    
    def enableButtons(self):
        '''function to enable execute task button once the task has finished'''

        self.dlg.button_create_result_shp.setDisabled(False)
        self.dlg.button_classifier.setDisabled(False)
        
    #--------------------------------------------------------------------------
    
    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&cadastral_classifier'),
                action)
            self.iface.removeToolBarIcon(action)
        
        del self.toolbar
        
    #--------------------------------------------------------------------------
    
    def select_all_mun(self):
        ''' Allows to select all visible elements of the municipalities list '''
        
        for i in range(self.dlg.mun_list.count()):
            item = self.dlg.mun_list.item(i)
            if not item.isHidden():
                item.setCheckState(Qt.Checked)
        self.update_items()
    
    #--------------------------------------------------------------------------

    def deselect_all_mun(self):
        ''' Allows to deselect all visible elements of the municipalities
        list '''
        
        for i in range(self.dlg.mun_list.count()):
            item = self.dlg.mun_list.item(i)
            if not item.isHidden():
                item.setCheckState(Qt.Unchecked)
        self.update_items()
    
    #--------------------------------------------------------------------------

    def filter_items(self, filter_txt):
        ''' Allows to filter items of the municipalities list in a more agile 
        way, letting to filter multiple ones by separating them with a comma'''
        
        filter_text_split = filter_txt.split(",") # split all the filtering text
        filtered_mun = []
        for text in filter_text_split:
            for i in range(self.dlg.mun_list.count()):
                item = self.dlg.mun_list.item(i)
                
                # update the visible items basing on coincidences of each splited
                # item separately
                if text.lower() not in item.text().lower() and item.text().lower() not in filtered_mun:
                    self.dlg.mun_list.setRowHidden(i, True)
                else:
                    self.dlg.mun_list.setRowHidden(i, False)
                    filtered_mun.append(item.text().lower())
                    
    #--------------------------------------------------------------------------
    
    def update_items(self):
        ''' Updates the list of the selected municipalities liiking for what
        items in the selection has been checked'''
        self.dlg.mun_list_selected.clear() #clears the list
        for i in range(self.dlg.mun_list.count()): # titerate over the mun list
            item = self.dlg.mun_list.item(i)
            if item.checkState() == Qt.Checked: # if they are checked
                self.dlg.mun_list_selected.addItem(item.text()) # add item to second list
    
    #--------------------------------------------------------------------------
    
    def get_selected_mun(self):
        ''' Gets the municipalities that has been selected '''
        
        selection = []
        for i in range(self.dlg.mun_list_selected.count()): # iterate over the items
            item = self.dlg.mun_list_selected.item(i)
            selection.append(item.text()) # add them to a list
        selection.sort()
        return selection
    
    #--------------------------------------------------------------------------
    
    def update_mun_list(self):
        ''' Updates the list of municipalities that a classification can be
        performed. Ir requieres the pair of paths CAT-SHP to be introduced. The
        pair of the urban parcels or, if the it has been selected, the rural ones
        aswell. It is executed anytime user selects any of the paths or checks/
        unchecks the rural option'''
        
        #----------------------------------------------------------------------
        
        def available_multi_mun(input_CAT, input_CAT_type, input_SHP, input_SHP_type,
                                parcel_type = None):
            
            if parcel_type:
                
                d_info = {"Name": None,
                          
                          "SHP_path_U": None,
                          "SHP_path_R": None,

                          "CAT_path_U": None,
                          "CAT_path_R": None}
                
                if input_CAT_type == 'zip':
                    
                    # read the zip file of the CAT file
                    zip_obj_cat = ZipFile(input_CAT, 'r')
                    
                    # get a list of the paths that it contains
                    files_list_cat = zip_obj_cat.namelist() # list with the files in it
                                    
                elif input_CAT_type == 'directory':
                    
                    # get current working directory
                    get_current_cwd = getcwd()
                    
                    # set as working directory the introduced by user
                    chdir(input_CAT)
                    
                    # get the list of the files inside the introduced directory
                    files_list_cat = glob('*')
                    
                    # get back to previous directory
                    chdir(get_current_cwd)
                                
                # defines adictionary that will store for each municipality code (key)
                # its relevant infomration
                d_codes = {}
                
                # iterate over all the files inside the zip
                for file in files_list_cat:
                    
                    # copy the structured dictionaty
                    d_info_copy = deepcopy(d_info)
                    d_info_copy["CAT_path_" + parcel_type] = file # add the internal path of the file
                                    
                    # get the code of the municipality
                    cat_code = basename(file)[:5]
        
                    # if it is not an empty string (possible routes to itself)
                    if cat_code != '':
                        
                        # assign the route
                        d_codes[cat_code] = d_info_copy
                
                if input_SHP_type == 'zip':
                    
                    # read the zip file of the SHP file
                    with ZipFile(input_SHP, 'r') as zip_obj_shp:
                        files_list_shp = [info.filename for info in zip_obj_shp.infolist() if not info.is_dir()]

                elif input_SHP_type == 'directory':
                    
                    input_SHP_path = Path(input_SHP)
                    files_list_shp = [str(p) for p in input_SHP_path.rglob('*') if p.is_file()]
                    
                # iterate over all the files that it contains
                for file in files_list_shp:
                    
                    file_path = Path(file)
        
                    # if it is not a reference to itself
                    if  file_path.parts and file_path.name != "":

                        # get different components to deconstruct the  filename
                        split_names = file_path.parent.name
                        split_raw_files = file_path.name
                        split_names_spaces = split_names.split(" ")
                        split_names_code = split_names[:5]
                        split_names_mun = ' '.join(split_names_spaces[2:])
                        split_file_comp = split_raw_files.split("_")
                        component = split_file_comp[3][:-4] # get the component (parcel, block, etc)
                        
                        # save the route to the parcel file
                        if component.lower() == 'parcela':
                                            
                            d_codes[split_names_code]["SHP_path_" + parcel_type] = file # save internal path
                            
                            # name is saved here because CAT only use municipality codes
                            d_codes[split_names_code]["Name"] = split_names_mun
                            
                return d_codes
            
        #----------------------------------------------------------------------
        
        def merge_dictionaries(dict1, dict2):
            
            if not dict2:
                return dict1.copy()
            
            else:
                
                merged_dict = {}
    
                # Get all unique keys from both dictionaries
                all_keys = set(dict1.keys()).union(dict2.keys())
                
                for key in all_keys:
                    # Retrieve nested dictionaries, defaulting to empty dict if key not found
                    nested_dict1 = dict1.get(key, {})
                    nested_dict2 = dict2.get(key, {})
                    
                    all_nested_keys = set(nested_dict1.keys()).union(nested_dict2.keys())
                    
                    # Merge the nested dictionaries
                    merged_nested = {}
                    
                    for nested_key in all_nested_keys:
                        
                        value1 = nested_dict1.get(nested_key)
                        value2 = nested_dict2.get(nested_key)
                        
                        # Choose the value that is not None, preferring value2 over value1
                        if value1 is not None:
                            merged_nested[nested_key] = value1
                            
                        elif value2 is not None:
                            merged_nested[nested_key] = value2
                            
                        else:
                            merged_nested[nested_key] = None  # Both are None or missing
                    
                    merged_dict[key] = merged_nested
                    
                return merged_dict

        #----------------------------------------------------------------------
        
        # clear the current mun list object
        self.dlg.mun_list.clear()
        self.dlg.mun_list_selected.clear()
        
        # define a dictionary to save the internals paths of the zip filess 
        # with a given structure
        
        # get each path
        text_cat_urb_multi = self.dlg.dir_cat_urb_multi.text()
        text_shp_urb_multi = self.dlg.dir_shp_urb_multi.text()
        text_cat_rust_multi = self.dlg.dir_cat_rust_multi.text()
        text_shp_rust_multi= self.dlg.dir_shp_rust_multi.text()
        
        d_codes_urb = {}
        d_codes_rust = {}
        
        # check if all path conditions are me to show the list of municipalitiess
        if text_cat_urb_multi and text_shp_urb_multi:
                        
            d_codes_urb = available_multi_mun(input_CAT = text_cat_urb_multi,
                                          input_CAT_type = self.dlg.input_CAT_URB_multi_type,
                                          input_SHP = text_shp_urb_multi,
                                          input_SHP_type = self.dlg.input_SHP_URB_multi_type,
                                          parcel_type = "U")
            
        if self.dlg.Option_include_rural.isChecked() == True and\
            text_cat_rust_multi and text_shp_rust_multi:
                            
            d_codes_rust = available_multi_mun(input_CAT = text_cat_rust_multi,
                                          input_CAT_type = self.dlg.input_CAT_RUS_multi_type,
                                          input_SHP = text_shp_rust_multi,
                                          input_SHP_type = self.dlg.input_SHP_RUS_multi_type,
                                          parcel_type = "R")
                      
        d_codes =  merge_dictionaries(d_codes_urb, d_codes_rust)

        keys_to_delete = []
        
        for key in d_codes.keys():
            code = d_codes[key]
            
            if not code["SHP_path_U"] and not code["SHP_path_R"]:
               keys_to_delete.append(key)

        for key in set(keys_to_delete):
            d_codes.pop(key)
        
        # once all the data has been sorted add the items to the list
        # one by one using the dictionary keys and the municipality name
        for mun_key in sorted(d_codes):
            
            if d_codes[mun_key]["Name"]:
                mun_item = QListWidgetItem("[" + str(mun_key) + "] " + d_codes[mun_key]["Name"])
                
            else:
                mun_item = QListWidgetItem("[" + str(mun_key) + "]")

            mun_item.setFlags(Qt.ItemIsUserCheckable | Qt.ItemIsEnabled)
            mun_item.setCheckState(Qt.Unchecked)

            self.dlg.mun_list.addItem(mun_item)
        
        # save the variable
        self.d_codes = d_codes
    
    #--------------------------------------------------------------------------
    ''' All these functions aim to manage the input and output file paths/names'''
    
    def input_cat_urb_multi(self):
        
        # get the name from the dialog generated to select it 
        filename, _filter = QFileDialog.getOpenFileName(
            
            # set the string to display on dialog and the files that can be 
            # selected (* indicates all, since it can be a .CAT file or .gzip)
            self.dlg, "Seleccionar comprimido con ficheros CAT urbanos de los municipios","", '*.zip')
        
        # store the name to the string variable
        self.dlg.dir_cat_urb_multi.setText(filename)
        
        self.dlg.input_CAT_URB_multi_type = 'zip'
        
        # call the check function to ensure that the correct file was selected
        self.check_all_inputs()
        
    #--------------------------------------------------------------------------
    def input_shp_urb_multi(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar comprimido con los acrhivos vectoriales(SHP) urbanos de los municipios","", '*.zip')
        self.dlg.dir_shp_urb_multi.setText(filename)
        self.dlg.input_SHP_URB_multi_type = 'zip'
        self.check_all_inputs()
        
    #--------------------------------------------------------------------------    
    def input_cat_rust_multi(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar comprimido con ficheros CAT rústicos de los municipios","", '*.zip')
        self.dlg.dir_cat_rust_multi.setText(filename)
        self.dlg.input_CAT_RUS_multi_type = 'zip'
        self.check_all_inputs()
        
    #--------------------------------------------------------------------------        
    def input_shp_rust_multi(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar comprimido con los acrhivos vectoriales(SHP) rústicos de los municipios","", '*.zip')
        self.dlg.dir_shp_rust_multi.setText(filename)
        self.dlg.input_SHP_RUS_multi_type = 'zip'
        self.check_all_inputs()

    #--------------------------------------------------------------------------
    ''' All these functions aim to manage the input and output file paths/names'''
    
    def input_cat_urb_multi_directory(self):
        
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar comprimido con ficheros CAT urbanos de los municipios","")
        self.dlg.dir_cat_urb_multi.setText(filename)
        self.dlg.input_CAT_URB_multi_type = 'directory'
        self.check_all_inputs()

    #--------------------------------------------------------------------------
    def input_shp_urb_multi_directory(self):
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar comprimido con los acrhivos vectoriales(SHP) urbanos de los municipios","")
        self.dlg.dir_shp_urb_multi.setText(filename)
        self.dlg.input_SHP_URB_multi_type = 'directory'
        self.check_all_inputs()
        
    #--------------------------------------------------------------------------    
    def input_cat_rust_multi_directory(self):
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar comprimido con ficheros CAT rústicos de los municipios","")
        self.dlg.dir_cat_rust_multi.setText(filename)
        self.dlg.input_CAT_RUS_multi_type = 'directory'
        self.check_all_inputs()

    #--------------------------------------------------------------------------        
    def input_shp_rust_multi_directory(self):
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar comprimido con los acrhivos vectoriales(SHP) rústicos de los municipios","")
        self.dlg.dir_shp_rust_multi.setText(filename)
        self.dlg.input_SHP_RUS_multi_type = 'directory'
        self.check_all_inputs()

    #--------------------------------------------------------------------------    
    def input_cat_urb_unique(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar el archivo CAT (comprimido o descomprimido) urbano del municipio","", '*')
        self.dlg.dir_cat_urb_unique.setText(filename)

        input_correct = input_check (filename, component = 'urban',
                                     typology = "cat", form = 'unique')
        
        if input_correct == True:
            self.dlg.dir_cat_urb_unique.setStyleSheet("background-color: rgb(200, 255, 200)")
            
        else:
            self.dlg.dir_cat_urb_unique.setStyleSheet("background-color: rgb(255, 200, 200)")
        
    #--------------------------------------------------------------------------    
    def input_shp_urb_unique(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar el archivo vectorial (SHP comprimido o descomprimido) urbano del municipio","", '*')
        self.dlg.dir_shp_urb_unique.setText(filename)
        
        input_correct = input_check (filename, component = 'urban',
                                     typology = "shp", form = 'unique')
        
        if input_correct == True:
            self.dlg.dir_shp_urb_unique.setStyleSheet("background-color: rgb(200, 255, 200)")
            
        else:
            self.dlg.dir_shp_urb_unique.setStyleSheet("background-color: rgb(255, 200, 200)")  
            
    #--------------------------------------------------------------------------    
    def input_cat_rust_unique(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar el archivo CAT (comprimido o descomprimido) rústico del municipio","", '*')
        self.dlg.dir_cat_rust_unique.setText(filename)
        
        input_correct = input_check (filename, component = 'rustic',
                                     typology = "cat", form = 'unique')
        
        if input_correct == True:
            self.dlg.dir_cat_rust_unique.setStyleSheet("background-color: rgb(200, 255, 200)")
            
        else:
            self.dlg.dir_cat_rust_unique.setStyleSheet("background-color: rgb(255, 200, 200)")
            
    #--------------------------------------------------------------------------    
    def input_shp_rust_unique(self):  
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar el archivo vectorial (SHP comprimido o descomprimido) rústico del municipio","", '*')
        self.dlg.dir_shp_rust_unique.setText(filename)
        
        input_correct = input_check (filename, component = 'rustic',
                                     typology = "shp", form = 'unique')
        
        if input_correct == True:
            self.dlg.dir_shp_rust_unique.setStyleSheet("background-color: rgb(200, 255, 200)")
            
        else:
            self.dlg.dir_shp_rust_unique.setStyleSheet("background-color: rgb(255, 200, 200)") 
            
    #--------------------------------------------------------------------------
    def output_result_p1(self):  
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar el directorio en el que se guardarán todos los ficheros de salida","")
        self.dlg.dir_output_result_p1.setText(filename)
        if exists(filename):
            self.dlg.dir_output_result_p1.setStyleSheet("background-color: rgb(200, 255, 200)") 
        else:
            self.dlg.dir_output_result_p1.setStyleSheet("background-color: rgb(255, 200, 200)") 

        
    #--------------------------------------------------------------------------
    def input_edif_unique_p2(self):  
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Seleccionar fichero resultante del proceso de tratamiento de datos","", '*')
        self.dlg.dir_edif_unique_p2.setText(filename)
        
        if filename:
            if basename(filename).split(".")[-1].lower() == "shp":
                self.dlg.dir_edif_unique_p2.setStyleSheet("background-color: rgb(200, 255, 200)") 
            else:
                self.dlg.dir_edif_unique_p2.setStyleSheet("background-color: rgb(255, 200, 200)") 
        else:
            self.dlg.dir_edif_unique_p2.setStyleSheet("background-color: rgb(255, 200, 200)")
            
    #--------------------------------------------------------------------------    
    def output_clasif_result_unique_p2(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Seleccionar ruta y nombre del fichero de clasificación resultante ","", '*.shp')
        self.dlg.dir_clasif_result_unique_p2.setText(filename)
        
        if filename:
            self.dlg.dir_clasif_result_unique_p2.setStyleSheet("background-color: rgb(200, 255, 200)") 
        else:
            self.dlg.dir_clasif_result_unique_p2.setStyleSheet("background-color: rgb(255, 200, 200)") 

    #--------------------------------------------------------------------------
    def input_dir_multi_p2(self):  
        filename = QFileDialog.getExistingDirectory(
            self.dlg, "Seleccionar el directorio contenedor de los ficheros del proceso de de tratamiento de datos para todos los municipios","")
        self.dlg.dir_root_multi_p2.setText(filename)
        
        if exists(filename):
            self.dlg.dir_root_multi_p2.setStyleSheet("background-color: rgb(200, 255, 200)") 
        else:
            self.dlg.dir_root_multi_p2.setStyleSheet("background-color: rgb(255, 200, 200)") 

    #--------------------------------------------------------------------------
        
    def check_all_inputs(self):  
        ''' Function that checks if the input files are correct'''
        
        # using 3 colors for distinction between correct input file (green), 
        # correct format (yellow) or incorrect file, different municipality or
        # incorrect format (red)
        green = "background-color: rgb(200, 255, 200)"
        red = "background-color: rgb(255, 200, 200)"
        yellow = "background-color: rgb(255, 255, 175)"
        
        # evaluation is different if user selected unique municipality or 
        # multiple
        if self.dlg.option_multiple_mun_p1.isChecked() == True:
        
            # get all the filepaths
            text_cat_urb_multi = self.dlg.dir_cat_urb_multi.text()
            text_shp_urb_multi = self.dlg.dir_shp_urb_multi.text()
            text_cat_rust_multi = self.dlg.dir_cat_rust_multi.text()
            text_shp_rust_multi = self.dlg.dir_shp_rust_multi.text()
            
            # generate a list to store the municipality codes
            list_muns = []
            
            # append the basenames
            if text_cat_urb_multi not in ["", " "]:
                list_muns.append(basename(text_cat_urb_multi))
                
            if text_shp_urb_multi not in ["", " "]:
                list_muns.append(basename(text_shp_urb_multi))
                
            if text_cat_rust_multi not in ["", " "]:
                list_muns.append(basename(text_cat_rust_multi))
                
            if text_shp_rust_multi not in ["", " "]:
                list_muns.append(basename(text_shp_rust_multi))
                
            # get the code of the files of municipality introduced
            code_list = [file[:2] for file in list_muns]
            
            # declare control boolean variables
            input_correct_cat_urb = False
            input_correct_shp_urb = False
            input_correct_cat_rust = False
            input_correct_shp_rust = False
            
            # if all the files correspond to one municipality, proceed
            if len(set(code_list)) == 1:
                
                # check each file introduced
                if text_cat_urb_multi not in ["", " "]:
                    input_correct_cat_urb = input_check (text_cat_urb_multi, component = 'urban',
                                                         typology = "cat", form = 'multi')
                        
                if text_shp_urb_multi not in ["", " "]:
                    input_correct_shp_urb = input_check (text_shp_urb_multi, component = 'urban',
                                                         typology = "shp", form = 'multi')
                    
                if text_cat_rust_multi not in ["", " "]:
                    input_correct_cat_rust = input_check (text_cat_rust_multi, component = 'rustic',
                                                          typology = "cat", form = 'multi')
                    
                if text_cat_rust_multi not in ["", " "]:
                    input_correct_shp_rust = input_check (text_shp_rust_multi, component = 'rustic',
                                                          typology = "shp", form = 'multi')
                
                # if both, cadastral alphanumeric data, and spatial parcel data
                # are correct, set it to green and update the selected list
                if input_correct_cat_urb == True and input_correct_shp_urb == True:
                    
                    self.dlg.dir_cat_urb_multi.setStyleSheet(green)
                    self.dlg.dir_shp_urb_multi.setStyleSheet(green)
                    self.update_mun_list()
                    self.dlg.frame_seleccion_mun.setEnabled(True)
                
                # if onlye one is correct, in its format, set it to yellow and 
                # the other one to red. Delete and clear the selected muns
                elif input_correct_cat_urb == True and input_correct_shp_urb == False:
                    self.dlg.dir_cat_urb_multi.setStyleSheet(yellow)
                    self.dlg.dir_shp_urb_multi.setStyleSheet(red)
                    self.dlg.mun_list.clear()
                    self.dlg.mun_list_selected.clear()
                    self.dlg.frame_seleccion_mun.setEnabled(False)
                
                # same but viceversa
                elif input_correct_cat_urb == False and input_correct_shp_urb == True:
                    self.dlg.dir_cat_urb_multi.setStyleSheet(red)
                    self.dlg.dir_shp_urb_multi.setStyleSheet(yellow)
                    self.dlg.mun_list.clear()
                    self.dlg.mun_list_selected.clear()
                    self.dlg.frame_seleccion_mun.setEnabled(False)

                #otherwise set all to red
                else:
                    self.dlg.dir_cat_urb_multi.setStyleSheet(red)
                    self.dlg.dir_shp_urb_multi.setStyleSheet(red)
                    self.dlg.mun_list.clear()
                    self.dlg.mun_list_selected.clear()
                    self.dlg.frame_seleccion_mun.setEnabled(False)
                
                # check if rural option is checked
                if self.dlg.Option_include_rural.isChecked() == True:
                    
                    if input_correct_cat_urb == True and input_correct_shp_urb == True:

                        # same proceeding as before but with rural data
                        if input_correct_cat_rust == True and input_correct_shp_rust == True:
        
                            self.dlg.dir_cat_rust_multi.setStyleSheet(green)
                            self.dlg.dir_shp_rust_multi.setStyleSheet(green)
                            self.update_mun_list()
                            self.dlg.frame_seleccion_mun.setEnabled(True)
                            
                        elif input_correct_cat_rust == True and input_correct_shp_rust == False:
                            self.dlg.dir_cat_rust_multi.setStyleSheet(yellow)
                            self.dlg.dir_shp_rust_multi.setStyleSheet(red)
                            self.dlg.mun_list.clear()
                            self.dlg.mun_list_selected.clear()
                            self.dlg.frame_seleccion_mun.setEnabled(False)
    
                        elif input_correct_cat_rust == False and input_correct_shp_rust == True:
                            self.dlg.dir_cat_rust_multi.setStyleSheet(red)
                            self.dlg.dir_shp_rust_multi.setStyleSheet(yellow)
                            self.dlg.mun_list.clear()
                            self.dlg.mun_list_selected.clear()
                            self.dlg.frame_seleccion_mun.setEnabled(False)
                            
                        else:
                            self.dlg.dir_cat_rust_multi.setStyleSheet(red)
                            self.dlg.dir_shp_rust_multi.setStyleSheet(red)
                            self.dlg.mun_list.clear()
                            self.dlg.mun_list_selected.clear()
                            self.dlg.frame_seleccion_mun.setEnabled(False)
    
            # set all to red if there are more than 1 municipality code
            else:
    
                self.dlg.dir_cat_urb_multi.setStyleSheet(red)
                self.dlg.dir_cat_rust_multi.setStyleSheet(red)
                self.dlg.dir_shp_urb_multi.setStyleSheet(red)
                self.dlg.dir_shp_rust_multi.setStyleSheet(red)
                self.dlg.mun_list.clear()
                self.dlg.mun_list_selected.clear()
                self.dlg.frame_seleccion_mun.setEnabled(False)
                    
    #--------------------------------------------------------------------------
    
    def check_basic (self):
        ''' Function to perform a selection of all basic options'''
        
        self.dlg.Checkbox_Resi.setChecked(True)
        self.dlg.Checkbox_Indus.setChecked(True)
        self.dlg.Checkbox_Com_Ofi.setChecked(True)
        self.dlg.Checkbox_Equip.setChecked(True)
        self.dlg.Checkbox_Oc_Res.setChecked(True)
        self.dlg.Checkbox_Ed_Sing.setChecked(True)
    
    #--------------------------------------------------------------------------
    
    def uncheck_basic (self):
        ''' Function to perform a diselection of all basic option'''
        
        self.dlg.Checkbox_Resi.setChecked(False)
        self.dlg.Checkbox_Indus.setChecked(False)
        self.dlg.Checkbox_Com_Ofi.setChecked(False)
        self.dlg.Checkbox_Equip.setChecked(False)
        self.dlg.Checkbox_Oc_Res.setChecked(False)
        self.dlg.Checkbox_Ed_Sing.setChecked(False)
        
    #--------------------------------------------------------------------------
        
    def check_intermediate (self):
        ''' Function to select all intermediate options, including the optionals,
        which it make visible and checkable'''
        
        self.dlg.Checkbox_equip_otros.setChecked(True)
        self.dlg.Checkbox_hos_resta.setChecked(True)
        self.dlg.Checkbox_ofi.setChecked(True)
        self.dlg.Checkbox_equip_edu.setChecked(True)
        self.dlg.Checkbox_resi_uni_mix.setChecked(True)
        self.dlg.Checkbox_edif_sing.setChecked(True)
        self.dlg.Checkbox_resi_unif.setChecked(True)
        self.dlg.Checkbox_resi_plu.setChecked(True)
        self.dlg.Checkbox_comercio.setChecked(True)
        self.dlg.Checkbox_ocio_esp.setChecked(True)
        self.dlg.Checkbox_resi_plu_mix.setChecked(True)
        self.dlg.Checkbox_Indus_mix.setChecked(True)
        self.dlg.Checkbox_equip_sanit.setChecked(True)
        self.dlg.Checkbox_Indus_2.setChecked(True)
        
        self.dlg.Checkbox_resi_uni_mix.setEnabled(True)
        self.dlg.Checkbox_resi_plu_mix.setEnabled(True)
        self.dlg.Checkbox_Indus_mix.setEnabled(True)
        
        self.dlg.frame_threshold_uni_mx.setVisible(True)
        self.dlg.frame_threshold_plu_mx.setVisible(True)
        self.dlg.frame_threshold_ind_mx.setVisible(True)
        
    #--------------------------------------------------------------------------
     
    def uncheck_intermediate (self):
        ''' Function to diselect all intermediate options, including the optionals,
        which it make invisible and uncheckable'''       
        
        self.dlg.Checkbox_equip_otros.setChecked(False)
        self.dlg.Checkbox_hos_resta.setChecked(False)
        self.dlg.Checkbox_ofi.setChecked(False)
        self.dlg.Checkbox_equip_edu.setChecked(False)
        self.dlg.Checkbox_resi_uni_mix.setChecked(False)
        self.dlg.Checkbox_edif_sing.setChecked(False)
        self.dlg.Checkbox_resi_unif.setChecked(False)
        self.dlg.Checkbox_resi_plu.setChecked(False)
        self.dlg.Checkbox_comercio.setChecked(False)
        self.dlg.Checkbox_ocio_esp.setChecked(False)
        self.dlg.Checkbox_resi_plu_mix.setChecked(False)
        self.dlg.Checkbox_Indus_mix.setChecked(False)
        self.dlg.Checkbox_equip_sanit.setChecked(False)
        self.dlg.Checkbox_Indus_2.setChecked(False)
        
        self.dlg.Checkbox_resi_uni_mix.setDisabled(True)
        self.dlg.Checkbox_resi_plu_mix.setDisabled(True)
        self.dlg.Checkbox_Indus_mix.setDisabled(True)

        self.dlg.frame_threshold_uni_mx.setHidden(True)
        self.dlg.frame_threshold_plu_mx.setHidden(True)
        self.dlg.frame_threshold_ind_mx.setHidden(True)      
        
    #--------------------------------------------------------------------------
    
    def exclude_mx (self):
        ''' Function to exclude all the mixed options, making them hidden and
        unchecked'''
        
        if self.dlg.Checkbox_resi_unif.isChecked() == True:
            self.dlg.Checkbox_resi_uni_mix.setChecked(False)
            self.dlg.frame_threshold_uni_mx.setHidden(True)

        if self.dlg.Checkbox_resi_plu.isChecked() == True:
            self.dlg.Checkbox_resi_plu_mix.setChecked(False)
            self.dlg.frame_threshold_plu_mx.setHidden(True)
            
        if self.dlg.Checkbox_Indus_2.isChecked() == True:
            self.dlg.Checkbox_Indus_mix.setChecked(False)
            self.dlg.frame_threshold_ind_mx.setHidden(True)      

    #--------------------------------------------------------------------------
    
    def include_mx (self):
        ''' Function to include all the mixed options, making them visible and
        checked'''
        
        if self.dlg.Checkbox_resi_unif.isChecked() == True:
            self.dlg.Checkbox_resi_uni_mix.setChecked(True)
            self.dlg.frame_threshold_uni_mx.setVisible(True)

        if self.dlg.Checkbox_resi_plu.isChecked() == True:
            self.dlg.Checkbox_resi_plu_mix.setChecked(True)
            self.dlg.frame_threshold_plu_mx.setVisible(True)
            
        if self.dlg.Checkbox_Indus_2.isChecked() == True:
            self.dlg.Checkbox_Indus_mix.setChecked(True)
            self.dlg.frame_threshold_ind_mx.setVisible(True) 
        
    #--------------------------------------------------------------------------
    
    def update_double_value(self, value):
        ''' Function that ensure that the float values of sliders are correctly
        showed, of the slider of intermediate method'''
        
        if value == 0:
            self.dlg.value_value_threshold_not_built.setText("0 %")
            
        else:
            # method to update the value of the slider to the string box
            self.dlg.value_threshold_not_built.setText(str(int(value)) + " %")
    
    #--------------------------------------------------------------------------
    
    def update_double_value_2(self, value):
        ''' Function that ensure that the float values of sliders are correctly
        showed, of the slider of advanced method'''
        
        if value == 0:
            self.dlg.value_threshold_not_built_2.setText("0 %")
            
        else:
            self.dlg.value_threshold_not_built_2.setText(str(int(value)) + " %")
            
    #--------------------------------------------------------------------------
    
    def update_uni_mx_value(self, value):
        ''' Function that ensure correct visualization of slider value of the 
        single family mixed value'''
        
        self.dlg.value_threshold_uni.setText(str(value) + " %")
        
    #--------------------------------------------------------------------------
    
    def update_plu_mx_value(self, value):
        ''' Function that ensure correct visualization of slider value of the 
        multiple family mixed value'''
        
        self.dlg.value_threshold_plu.setText(str(value) + " %")
            
    #--------------------------------------------------------------------------
    
    def update_ind_mx_value(self, value):
        ''' Function that ensure correct visualization of slider value of the 
        industrial mixed value'''
        
        self.dlg.value_threshold_ind.setText(str(value) + " %")

    #--------------------------------------------------------------------------

    def add_row(self):
        
        ''' Function that perform the addition of a new row to the advanced 
        method table'''
        
        # get the number of categories 
        n_categories = self.dlg.table_advanced_option.rowCount()
        
        # if there is none, add row to 0 position
        if  n_categories == 0:
            self.dlg.table_advanced_option.insertRow(0)
            
        # in other case add the row to position after the last one (total rows
        # minus 1, cause it starts at 0)
        else:
            self.dlg.table_advanced_option.insertRow(n_categories)
            
        # create an appropriate item and set its state as checked
        item = QTableWidgetItem()
        item.setCheckState(Qt.CheckState.Checked)                               

        # Add the item to the table 
        self.dlg.table_advanced_option.setItem(n_categories, 0, item)
        
        # count new number of rows
        n_rows = self.dlg.table_advanced_option.rowCount()

        # vertical labels start at 0 and so on. They are not needed they will
        # be "removed". This is needed to do at the start and on every row
        # addition
        ls_vertical_labels = []
        
        # based on row number, add the label to a list
        for i in range (0, n_rows):
            ls_vertical_labels.append("")
            
        # set the empty label to each row
        self.dlg.table_advanced_option.setVerticalHeaderLabels(ls_vertical_labels)

    #--------------------------------------------------------------------------
    
    def delete_row(self):
        ''' Function to delete the last row of the advanced table'''
        
        # get the current cell item (user selects it)
        current_cell = self.dlg.table_advanced_option.currentItem()
        
        # Firsty, it's needed to delete the categories that the row has and
        # put them back in the selecion table. It's the same method as 
        # deselect_all_categories but changing the column index use, so that
        # it always take the 3rd (categories one) column
                
        # If he had selected one
        if current_cell:
            
            # get the row and column indexers
            current_row = current_cell.row()
            categories_column = 3
            
            current_cell = self.dlg.table_advanced_option.item(current_row,
                                                               categories_column)
            try:
                # get the strings (classes) that are saved to that category
                string_categories = current_cell.text()
            except:
                string_categories = None

            # if that category has any classes added (previously or by importing
            # the table)
            if string_categories:
            
                # make a list with the classes
                list_categories = list(string_categories.split(","))
                
                # iterate through all the classes of the table of selection
                for i in range(0, self.dlg.categories_list.count()):
                    
                    # get the item that relates to that class
                    item = self.dlg.categories_list.item(i)
                    
                    # iterate over the categories that are needed to delete
                    for category in list_categories:
                        
                        # if that class is part of the list of classes that are needed
                        # to delete
                        if category in item.text():
                            
                            # change the class characteristics, to de usable again
                            item.setFlags(item.flags() | Qt.ItemFlag.ItemIsSelectable |\
                                          Qt.ItemFlag.ItemIsEnabled |\
                                          Qt.ItemFlag.ItemIsDragEnabled |\
                                          Qt.ItemFlag.ItemIsUserCheckable)
                            
                            # delete its selection
                            item.setSelected(False)
               
                #--------------------------------------------------------------   
                # delete the classes that were included in the category
                current_cell.setText("")
                
                # enable the cell so it is again usable to add other classes
                current_cell.setFlags(item.flags() | Qt.ItemFlag.ItemIsEditable)
                
                # update the item to the table
                self.dlg.table_advanced_option.setItem(current_row, categories_column,
                                                       current_cell)
            
            # remove the selected row
            self.dlg.table_advanced_option.removeRow(self.dlg.table_advanced_option.currentRow())
    
    #--------------------------------------------------------------------------
    
    def set_column_widths(self):
        ''' Function to set the column widths of the advanced method table'''
        
        # get the header of the table and set the resize mode
        header = self.dlg.table_advanced_option.horizontalHeader()        
        header.setSectionResizeMode(2, QHeaderView.Stretch)
        header.setSectionResizeMode(3, QHeaderView.Stretch)
        
        # create a list with the width values
        widths = [1, 75, 0, 0, 15, 60]
        
        # count the number of columns and rows 
        n_cols = self.dlg.table_advanced_option.columnCount()
        n_rows = self.dlg.table_advanced_option.rowCount()
        
        # set the witdths of each column
        for i in range (0, n_cols):
            self.dlg.table_advanced_option.setColumnWidth(i, widths[i])
            
        # vertical labels start at 0 and so on. They are not needed they will
        # be "removed". This is needed to do at the start and on every row
        # addition
        ls_vertical_labels = []
        
        # based on row number, add the label to a list
        for i in range (0, n_rows):
            ls_vertical_labels.append("")
        
        # set the empty label to each row
        self.dlg.table_advanced_option.setVerticalHeaderLabels(ls_vertical_labels)

    #--------------------------------------------------------------------------

    def deselect_all_categories(self):
        ''' Function to perform the diselection of all categories that are 
        currently included in a row (category) to use on the classification'''
        
        # get the current cell item (user selects it )
        current_cell = self.dlg.table_advanced_option.currentItem()
        
        # If he had selected one
        if current_cell:
            
            # get the row and column indexers
            current_row = current_cell.row()
            categories_column = 3
            
            current_cell2 = self.dlg.table_advanced_option.item(current_row,
                                                               categories_column)
            try:
                # get the strings (classes) that are saved to that category
                string_categories = current_cell2.text()
            except:
                string_categories = None
            
            # if that category has any classes added (previously or by importing
            # the table)
            if string_categories:
            
                # make a list with the classes
                list_categories = list(string_categories.split(","))
                
                # iterate through all the classes of the table of selection
                for i in range(0, self.dlg.categories_list.count()):
                    
                    # get the item that relates to that class
                    item = self.dlg.categories_list.item(i)
                    
                    # iterate over the categories that are needed to delete
                    for category in list_categories:
                        
                        # if that class is part of the list of classes that are needed
                        # to delete
                        if category in item.text():
                            
                            # change the class characteristics, to de usable again
                            item.setFlags(item.flags() | Qt.ItemFlag.ItemIsSelectable |\
                                          Qt.ItemFlag.ItemIsEnabled |\
                                          Qt.ItemFlag.ItemIsDragEnabled |\
                                          Qt.ItemFlag.ItemIsUserCheckable)
                            
                            # delete its selection
                            item.setSelected(False)
               
                #--------------------------------------------------------------   
                # delete the classes that were included in the category
                current_cell2.setText("")
                
                # enable the cell so it is again usable to add other classes
                current_cell2.setFlags(item.flags() | Qt.ItemFlag.ItemIsEditable)
                
                # update the item to the table
                self.dlg.table_advanced_option.setItem(current_row, categories_column,
                                                       current_cell2)
 
    #-------------------------------------------------------------------------- 

    def add_categories(self):
        ''' Function to perform the addition of a class to the category that is 
        currently selected by the user'''
        
        # get the current cell that user selected
        current_cell = self.dlg.table_advanced_option.currentItem()
        
        # if there is a selected one
        if current_cell:
                        
            # get the row and column indexers
            current_row = current_cell.row()
            categories_column = 3
            
            current_cell2 = self.dlg.table_advanced_option.item(current_row,
                                                               categories_column)
            # create a list to store all the classes that user selected in the 
            # class table
            selected_items = []
            
            # get the selected thasses
            items_categorias = self.dlg.categories_list.selectedItems()
            
            # if there is any selected
            if items_categorias:
                
                # iterate over all selected
                for item in items_categorias:
                    
                    # remove its selection
                    item.setSelected(False)
                    
                    # delete all its flags (makes it disabled and unselectable)
                    item.setFlags(Qt.NoItemFlags)
                
                # remove all selections
                self.dlg.categories_list.setCurrentItem(None)

            # expand the list the the selected items
            selected_items.extend(items_categorias)

            # create a list to store the codification of each class
            categories_list = []
            
            # add the codification of each class
            for item in selected_items:
                categories_list.append(item.text()[1:5])
            
            # create a string that will store all the codes
            categories_string = ""
            
            # iterate over all the codes of classes
            for category in categories_list:
                
                # if its the last one, create an empty string, to avoid adding
                # a comma after last item
                if category == categories_list[len(categories_list) - 1]:
                    comma = ""
                    
                # any other case, use a comma
                else:
                    comma = ","
                
                # add the codes and commas to the string variable
                categories_string += category + comma
    
            try:
                # get the strings (classes) that are saved to that category
                current_text = current_cell2.text()
            except:
                current_text = None
            
            # if there is already a text (probably cause he added previously
            # other classes), add the new classes after a comma
            if current_text:
                current_cell2.setText(current_text + "," + categories_string)
                
            # if it was empty directly add the classes codes in string format
            else:
                current_cell2 = QTableWidgetItem()
                current_cell2.setText(categories_string)

            # Disable the option to modify the item, so user can't delete the
            # classes by mistake, and only do it with the other function
            current_cell2.setFlags(current_cell.flags() & ~Qt.ItemFlag.ItemIsEditable)
            
            # update the item to the table
            self.dlg.table_advanced_option.setItem(current_row, categories_column,
                                                   current_cell2)
        
    #--------------------------------------------------------------------------
    
    def save_clasif(self):
        ''' Function that saves the current clasification that user has on his
        advanced method table'''
        
        # open a dialog to select the ouput file name. By dafaults it will be 
        # saved on the directory on which the basic and intermediate are
        # saved
        filename, _filter = QFileDialog.getSaveFileName(self.dlg,
                                                        "Select output file ",
                                                        join(dirname(__file__), "clasif"),
                                                        '*.csv')
        # if a name was selected
        if filename:
            
            # create a dataframe to store the table values
            df_new_clasif_output = DataFrame()
            
            # get the number of rows and columns
            n_rows = self.dlg.table_advanced_option.rowCount()        
            n_cols = self.dlg.table_advanced_option.columnCount()
            
            # create a variable to store the identifier of each row
            identifier = 0
            
            # iterate over the rows and cols (cols start with 1 to avoid the
            # identifier used by qgis tables)
            for r in range (0, n_rows):
                for c in range (1, n_cols):
                    
                    # get the column name
                    col_name = self.dlg.table_advanced_option.horizontalHeaderItem(c).text()
                    
                    # get the item 
                    item = self.dlg.table_advanced_option.item(r,c)
                    
                    # if item has any text, save it to the same position but
                    # in the dataframe, using identifier and column name. If not
                    # just create an empty cell
                    if item:                
                        df_new_clasif_output.loc[int(identifier), col_name] = item.text()
                    else:
                        df_new_clasif_output.loc[int(identifier), col_name] = None
                
                # increase the identifier value
                identifier += 1
                
            # save the filled dataframe as a csv, with the name provided and 
            # no pandas indexers (redundant)
            df_new_clasif_output.to_csv(filename, index = False)
                    
    #--------------------------------------------------------------------------

    def import_clasif(self):
        ''' Function to open and import existing clasification to the table of 
        the advanced classfification method'''
        
        # open a dialog to select the file, in csv format, that contains the
        # classification parameters
        filename, _filter = QFileDialog.getOpenFileName(self.dlg,
                                                        "Select input file ",
                                                        join(dirname(__file__), "clasif"),
                                                        '*.csv')
        # if a file was selected
        if filename:
            
            # read the csv as a pandas dataframe
            df_imported_clasif = read_csv(filename)
            
            # get the count of the number of rows and columns of the current
            # table
            n_rows = self.dlg.table_advanced_option.rowCount()        
            n_cols = self.dlg.table_advanced_option.columnCount()
            
            # get the column names of the new imported dataframe
            col_names = list(df_imported_clasif.columns)
            
            # insert in the first position an empty string
            col_names.insert(0, "")
            
            # delete all rows currently in the table, starting from last to 
            # first (any other way wont properly work cause indexers update)
            for r in range (n_rows, -1, -1):
                self.dlg.table_advanced_option.removeRow(r)
                
            # delete all columns currently in the table, starting from last to 
            # first (any other way wont properly work cause indexers update)
            for c in range (n_cols, -1, -1):
                self.dlg.table_advanced_option.removeColumn(c)
            
            # add the new columns (including the first, empry string one)
            for i in range (0, len(col_names)):
                self.dlg.table_advanced_option.insertColumn(i)
                
            # give them the correct names
            self.dlg.table_advanced_option.setHorizontalHeaderLabels(col_names)
            
            # insert the number of rows
            for i in range (0, len(df_imported_clasif.index)):
                self.dlg.table_advanced_option.insertRow(i)
            
            # iterate over the new imported dataframe
            for index, row in df_imported_clasif.iterrows():
                
                # for each column
                for n_col in range (0, len(col_names)):
                    
                    # if the column is the first one, it will have a special
                    # item that is checkable (so that the user can check or
                    # uncheck it based on his willing to include or not that
                    # category in the advanced classification)
                    if n_col == 0:
                        
                        item = QTableWidgetItem()
                        item.setCheckState(Qt.CheckState.Checked)
                        
                    # for any other column, create a new item and give it
                    # the text of the import df value
                    else:
                
                        item = QTableWidgetItem()
                        item.setText(str(row[col_names[n_col]])) 
                    
                    # add the item to the advanced method table
                    self.dlg.table_advanced_option.setItem(index, n_col, item)
            
            # perform the column widths adjustment
            self.set_column_widths()
                               
    #--------------------------------------------------------------------------
    
    def setup_data(self, dlg):
        '''function that executes setup data task'''
                        
        task = setup_data_Task(self.dlg, self.d_codes, self.get_selected_mun())
        
        task.taskStarted.connect(self.disableButtons)
        task.taskEnded.connect(self.enableButtons)
        
        self.tm.addTask(task)

    #--------------------------------------------------------------------------
    
    def clasif(self, dlg):
        '''function that executes classification task'''

        task = clasif_Task(self.dlg)
        
        task.taskStarted.connect(self.disableButtons)
        task.taskEnded.connect(self.enableButtons)
        
        self.tm.addTask(task)
        
    #--------------------------------------------------------------------------
    
    def run(self):
        """ Initialization function """
        
        # set the initial sliders values
        self.dlg.value_threshold_uni.setText(str(self.dlg.slider_threshold_uni.value()) + " %")
        self.dlg.value_threshold_plu.setText(str(self.dlg.slider_threshold_plu.value()) + " %")
        self.dlg.value_threshold_ind.setText(str(self.dlg.slider_threshold_ind.value()) + " %")
        self.dlg.value_threshold_not_built.setText(str(int(self.dlg.slider_threshold_not_built.value())) + " %")
        self.dlg.value_threshold_not_built_2.setText(str(int(self.dlg.slider_threshold_not_built_2.value())) + " %")
        
        # show the interface
        self.dlg.show()