# -*- coding: utf-8 -*-

# This is a part of "ČHMÚ/CHMI – Meteorological Data Processing" a QGIS plugin
# that processes meteorological data published by CHMI.  
# Copyright (C) 2025 by Mikuláš Kadečka - mikulas.kad@seznam.cz  

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

# Partly generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QApplication
from qgis.core import QgsCoordinateReferenceSystem
from qgis.core import Qgis, QgsWkbTypes
from datetime import datetime
from qgis.gui import QgsFileWidget
from shutil import copytree
from qgis.utils import plugins

import time

import webbrowser
import os

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .data_chmu_dialog import DataCHMUDialog


import os.path
import sys

pluginDir = os.path.dirname(__file__)
externalModulesPath = os.path.join(pluginDir)
sys.path.append(externalModulesPath)

from .functions import *
from .tool import mainFunction

class DataCHMU:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor."""
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DataCHMU_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&ČHMÚ - data o počasí')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('DataCHMU', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/data_chmu/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Prostorová data o počasí'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&ČHMÚ - data o počasí'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = DataCHMUDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop

        ## Adding functionality to the user interface (UI)
        fontSize = self.dlg.categoryLabel.font().pointSize()
        self.dlg.helperText.setFontPointSize(fontSize)
        self.dlg.processingWarning.setFontPointSize(fontSize)
        self.dlg.spinBoxNN.hide()

        self.loadSourcePath()
        self.dlg.exportGroupBox.setStyleSheet("QGroupBox { border: none; }")
        self.dlg.saveGroupBox.setStyleSheet("QGroupBox { border: none; }")
        self.dlg.processingWarning.setStyleSheet("QTextEdit { background: transparent; border: none;}")
        self.dlg.processingWarning.setVisible(False)

        self.dlg.startDate.setDate(datetime(1961, 1, 1))
        self.dlg.endDate.setDate(datetime(2024, 1, 1))

        self.dlg.rasFileWidget.setDisabled(True)

        self.dlg.projectionWidget.setCrs(QgsCoordinateReferenceSystem(f'EPSG:{5514}'))
        self.dlg.interpolationMethodcomboBox.currentTextChanged.connect(self.updateInterpolationMethod)
        
        self.dlg.rasRadioButton01.setChecked(True)
        self.dlg.rasRadioButton01.clicked.connect(self.turnSavingOffRas)
        self.dlg.rasRadioButton02.clicked.connect(self.turnSavingOnRas)

        try:
            self.dlg.startButton.clicked.disconnect(self.runTool)
            self.dlg.helpPushButton.clicked.disconnect(self.openHelperSite)
            self.dlg.googleButton.clicked.disconnect(self.openGoogleDisk)
            self.dlg.gitButton.clicked.disconnect(self.openGitHub)
        except TypeError:
            # Raised if there are no previous connections, safe to ignore
            pass
        self.dlg.startButton.clicked.connect(self.runTool)
        self.dlg.helpPushButton.clicked.connect(self.openHelperSite)
        self.dlg.googleButton.clicked.connect(self.openGoogleDisk)
        self.dlg.gitButton.clicked.connect(self.openGitHub)

    def runTool(self):

        ## Performing tests for parameter consistency, printing a warning message
        ## if something is incorrect, and aborting the program if necessary
        if self.dlg.outputTabWidget.currentIndex() == 1:
            return 0

        if 'processing' not in plugins:
            message = self.tr("Chyba: Pro správné fungování nástroje zapněte 'Processing' plugin.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)

        dataSource = os.path.normpath(self.dlg.inputDataContainer.filePath())
        dataCheck = self.sourcePathCheck(dataSource)
        if dataCheck == 0:
            self.saveSourcePath(dataSource)
        elif dataCheck == 1:
            message = self.tr("Chyba: V sekci Nastavení zadejte platnou absolutní cestu vstupních dat.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif dataCheck == 2:
            message = self.tr("Chyba: V adresáři vstupních dat chybí jedna z kategorií.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif dataCheck == 3:
            message = self.tr("Chyba: V jedné z kategorií vstupních dat chybí bodová vrstva měřících stanic.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif dataCheck == 4:
            message = self.tr("Chyba: Zadaná adresářová cesta nesměřuje do složky 'spatial_data'.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)

        chosenCategory = self.dlg.categoryComboBox.currentText()

        intersectArea = self.dlg.extentComboBox.currentLayer()

        if intersectArea is None:
            message = self.tr("Chyba vstupních parametrů: Zadejte vrstvu pro rozsah výstupu.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif not intersectArea.isValid():
            message = self.tr("Chyba: Vámi zadanou vstupního vrstvu se nepodařilo načíst.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif not isinstance(intersectArea, QgsVectorLayer) or intersectArea.wkbType()==100:
            message = self.tr("Chyba vstupních parametrů: Pro volbu rozsahu zadejte vektorovou vrtvu.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        
        try:
            if intersectArea.wkbType()==100:
                message = self.tr("Chyba vstupních parametrů: Pro volbu rozsahu zadejte vektorovou vrtvu.")
                return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        except:
            pass

        outputPath = self.canCreateFile(self.dlg.rasFileWidget.filePath())

        interpolationMethod = self.dlg.interpolationMethodcomboBox.currentText()
        
        cellSize = self.dlg.cellSizeSpinBox.value()

        epsg = self.dlg.projectionWidget.crs()

        startDate = validation.getDate(self.dlg.startDate.date())
        endDate = validation.getDate(self.dlg.endDate.date())
        if endDate < startDate:
            message = self.tr("Chyba vstupních parametrů: Počáteční datum je až po koncovém datumu.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif startDate == endDate:
            message = self.tr("Chyba vstupních parametrů: Počáteční a koncové datum nemůže být stejné.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)

        try:
            dataPath = os.path.join(dataSource, str(user.observationCategoriesEN[chosenCategory]))
        except:
            dataPath = os.path.join(dataSource, str(user.observationCategoriesCZ[chosenCategory]))

        outputName = validation.getOutputName(chosenCategory, interpolationMethod, outputPath)

        tempDir = os.path.join(pluginDir, "temp")

        power = self.dlg.powerSpinBox.value()

        pointOutput = self.dlg.pointLayersExportButton.isChecked()

        parameters = {
            "dataPath" : dataPath,
            "outputPath" : outputPath,
            "outputName" : outputName,
            "intersectArea" : intersectArea,
            "startDate" : startDate,
            "endDate" : endDate,
            "interpolationMethod" : interpolationMethod,
            "cellSize" : cellSize,
            "epsg" : epsg,
            "tempDir" : tempDir,
            "power" : power,
            "category": chosenCategory,
            "pointOutput" : pointOutput
            }
        
        self.dlg.processingWarning.setVisible(True)
        QApplication.processEvents()

        ## Running the tool and returning a message based on its output
        output = mainFunction(parameters)
        self.dlg.processingWarning.hide()

        if output == 0:
            message = self.tr("Hotovo: Tedy doufám :)).")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Info)
        elif output == 1:
            message = self.tr("Chyba: Ve Vaší oblasti se nenachází žádné měřící stanice - oblast přesuňte či zvětšte.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif output == 2:
            message = self.tr("Chyba: Z Vámi zadaného časového období nelze vytvořit rastr.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif output == 3:
            message = self.tr("Chyba: Vzniklá bodová pole se nepovedlo nahrát.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        elif output == 4:
            message = self.tr("Chyba: Vámi zadanou vstupního vrstvu se nepodařilo načíst.")
            return self.iface.messageBar().pushMessage(message, level = Qgis.Warning)
        self.dlg.processingWarning.hide()

            
    def openHelperSite(self):
        webbrowser.open("https://github.com/mkik12/CHMU-weatherDataProcessing")

    def openGoogleDisk(self):
        webbrowser.open("https://drive.google.com/drive/folders/12qYemjNOktYcyJgaK6pPfBqbJCzDAlbV?usp=sharing")

    def openGitHub(self):
        webbrowser.open("https://github.com/mkik12/CHMU-weatherFilesScraper")

    def updateInterpolationMethod(self):
        if self.dlg.interpolationMethodcomboBox.currentText() == "IDW":
            self.dlg.interpolationStackedWidget.setCurrentIndex(0)
        elif self.dlg.interpolationMethodcomboBox.currentText() == "Nearest Neighbour":
            self.dlg.interpolationStackedWidget.setCurrentIndex(1)

    def canCreateFile(self, filePath):
        parentDir = os.path.dirname(filePath)
        if os.path.exists(filePath):
            return None
        elif os.access(parentDir, os.W_OK):
            root, ext = os.path.splitext(filePath)
            if ext:
                return filePath
            else:
                return filePath + ".gpkg"
        else:
            return None

    def turnSavingOffRas(self, name):
        self.dlg.rasFileWidget.setDisabled(True)
        self.dlg.rasFileWidget.setFilePath("")

    def turnSavingOnRas(self):
        self.dlg.rasFileWidget.setEnabled(True)
        self.dlg.rasFileWidget.setFilePath("")

    def sourcePathCheck(self, sourcePath):
        if os.path.exists(sourcePath) is False:
            return 1
        
        if os.path.basename(sourcePath) != "spatial_data":
            return 4
        
        folders = os.listdir(sourcePath)
        for i in range(12):
            if str(i) not in folders:
                return 2
            categoryPath = os.path.join(sourcePath, str(i))
            if "stations.gpkg" not in os.listdir(categoryPath):
                return 3
        return 0

    def loadSourcePath(self):
        with open(os.path.join(pluginDir, "data_path.txt"), "r") as file:
            if file.readlines != "":
                sourcePath = file.readline()
                self.dlg.inputDataContainer.setFilePath(sourcePath)
    
    def saveSourcePath(self, sourcePath):
        with open(os.path.join(pluginDir, "data_path.txt"), "w") as file:
            file.write(sourcePath)