######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################

from qgis.PyQt.QtWidgets import QDockWidget
from qgis.PyQt.QtCore import pyqtSignal
from qgis.core import QgsMapLayerProxyModel
from .ui.Ui_Preprocessing2DWidget import Ui_Preprocessing2DWidget
from qgis.core import (QgsVectorLayer,  QgsRasterLayer,QgsProject, QgsMapLayerProxyModel, QgsMessageLog, Qgis, QgsTask, QgsApplication,      QgsCoordinateReferenceSystem)
from ..core.PreProcessing2D import *

class Preprocessing2DWidget(QDockWidget, Ui_Preprocessing2DWidget):

    closingWidget = pyqtSignal(name="closingWidget")

    def __init__(self, iface, BV, param, parent = None):
        super(Preprocessing2DWidget, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        self.BV = BV
        self.param = param
        self.pushButton_compute.clicked.connect(self.compute)
        self.comboBox_rasterSelection.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.comboBox_rasterSelection.layerChanged.connect(self.loadDEM)

    def closeEvent(self, event):
        self.closingWidget.emit()
        event.accept()

    def ui_to_param(self):
        self.param['P']['findDikes'] = self.checkBox_findDikes.isChecked()
        self.param['P']['overtopping'] = int(self.lineEdit_overtopping.text())
        self.param['P']['averageSize'] = int(self.lineEdit_averageSize.text())
        self.param['P']['densityZone'] = self.checkBox_densityZone.isChecked()
        self.param['P']['minMeshSize'] = int(self.lineEdit_minMeshSize.text())
        self.param['P']['maxMeshSize'] = int(self.lineEdit_maxMeshSize.text())
        self.param['P']['findFlatZone'] = self.checkBox_findFlatZone.isChecked()
        self.param['P']['eps'] =  float(self.lineEdit_eps.text())
        
        
    def loadDEM(self):
        self.param.qgis_DEMlayer = self.comboBox_rasterSelection.currentLayer()

    def compute(self):
        self.ui_to_param()
        self.loadDEM
        self.task = BackgroundProcess('Pre Process', self.BV, self.param)
        QgsApplication.taskManager().addTask(self.task)


class BackgroundProcess(QgsTask):

    def __init__(self, description, BV, param):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param

    def run(self):
        QgsMessageLog.logMessage(message="Starting 2D Preprocessing", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        DEMfile= self.param.qgis_DEMlayer
        
        if self.param['P']['findDikes']:
            
            _, ncontour = find_dikes(self.BV,self.param,DEMfile)
            dikes_lines_filepath = path.join(self.param.work_path, 'detected_dikes.shp')
            n_layer = QgsVectorLayer(dikes_lines_filepath, name_bank)
            n_layer.setCrs(QgsCoordinateReferenceSystem(self.BV.crs.to_string()))
            QgsMessageLog.logMessage(message="number of detected dikes" +str(ncontour), tag="CEPHEE", level=Qgis.MessageLevel.Info)
        
        if self.param['P']['densityZone']:
            outname = density_zone(self.BV,self.param,DEMfile)
            density_layer = QgsRasterLayer(outname)  
            QgsProject.instance().addMapLayer(density_layer)
            QgsMessageLog.logMessage(message="density map created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
    
        if self.param['P']['findFlatZone']:
            outname = flat_zone(self.BV,self.param,DEMfile)
            water_layer = QgsRasterLayer(outname)  
            QgsProject.instance().addMapLayer(water_layer)
            QgsMessageLog.logMessage(message="water mask created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
            
      
        self.setProgress(100)
        return True

    def finished(self, result):
        if result:
            QgsMessageLog.logMessage(message="Process OK", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        else:
            QgsMessageLog.logMessage(message="Process Error", tag="CEPHEE", level=Qgis.MessageLevel.Critical)
