######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################

from qgis.PyQt.QtCore import pyqtSignal
from qgis.PyQt.QtWidgets import QDockWidget, QFileDialog
from qgis.core import (QgsVectorLayer, QgsRasterLayer,QgsProject, QgsMapLayerProxyModel, QgsMessageLog, Qgis, QgsTask,
                       QgsApplication, QgsCoordinateReferenceSystem)
from .ui.Ui_BathymetricWidget import Ui_BathymetricWidget
from ..core.Hydraulics import *
from ..core.Bathymetrics  import *
from os import path

class BathymetricWidget(QDockWidget, Ui_BathymetricWidget):

    closingWidget = pyqtSignal(name="closingWidget")

    def __init__(self, iface, BV, param, parent = None):
        super(BathymetricWidget, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        self.BV = BV
        self.param = param
        self.pushButton_compute.clicked.connect(self.compute)

    def closeEvent(self, event):
        self.closingWidget.emit()
        event.accept()

    def ui_to_param(self):
        self.param['B']['bathymetricSections'] = self.comboBox_bathymetricSections.currentText()
        self.param['B']['depth'] = float(self.lineEdit_reachInterStep.text())
        self.param['B']['interpolationMethod'] = self.comboBox_interpolationMethod.currentText()
        self.param['B']['useImportedSections'] = self.checkBox_useImportedSections.isChecked()

        if self.comboBox_bathymetricSections.currentText() == "rectangular":
            self.param['B']['bathymetricSections'] = "Rectangular"
        elif self.comboBox_bathymetricSections.currentText() == "parabolic":
            self.param['B']['bathymetricSections'] = "Parabolic"

        if self.comboBox_interpolationMethod.currentText() == "Normal":
            self.param['B']['interpolationMethod'] = "Normal"
        elif self.comboBox_interpolationMethod.currentText() == "1D":
            self.param['B']['interpolationMethod'] = "1D"
        elif self.comboBox_interpolationMethod.currentText() == "H imposed":
            self.param['B']['interpolationMethod'] = "Himposed"
        elif self.comboBox_interpolationMethod.currentText() == "Obs":
            self.param['B']['interpolationMethod'] = "Obs"

    def compute(self):
        self.ui_to_param()
        self.task = BackgroundProcess('process_bathymetric', self.BV, self.param)
        QgsApplication.taskManager().addTask(self.task)

class BackgroundProcess(QgsTask):

    def __init__(self, description, BV, param):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param

    def run(self):
        QgsMessageLog.logMessage(message="Starting bathymetric computation", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        if self.param['B']['depth'] ==0:
            self.param['B']['fromQ'] = True

        if  self.param['B']['interpolationMethod'] == "Obs":
            if os.path.exists(os.path.join(self.param.work_path, 'banks_lines.shp')):
                bank_filename = os.path.join(self.param.work_path, 'banks_lines.shp')
            else:
                QgsMessageLog.logMessage(message="banks_lines.shp must exist in workspace", tag="CEPHEE",
                                         level=Qgis.MessageLevel.Info)
        else:
            bank_filename =None

        drag_all_XS(self.BV, self.param, type_comp=self.param['B']['interpolationMethod'],
                            bank_filename=bank_filename)  # create the line of each section with the points of the banks

        if self.param['H']['mapping_method'] == 'interpolation':
            interpolate_MNTchannel(self.BV, self.param, self.param['H']['dxlat'])
        else:
            rasterize_MNTchannel(self.BV, self.param, self.param['H']['dxlat'])
        MergeRaster_channel_DEM(self.BV, self.param.work_path)
        QgsMessageLog.logMessage(message="New DEM created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        return True




