######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################

# global
import geopandas as gpd
from geopandas import GeoDataFrame
from shapely.geometry import Polygon
from scipy.spatial import cKDTree
import rasterio.mask
# local
from .Tools import *
from .Section import Section
from .Data import read_DEM
from shapely.ops import nearest_points

def importXSlines(BV,param, qgis_XS_layer=None, XS_filename=None):
    """ Import XS lines in a shp format. Associate them to the nearest reach

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    :param qgis_XS_layer: Shapefile in the QGIS open layer
    :type qgis_XS_layer: shapefile layer
    :param XS_filename: name of Shapefile with XS line
    :type qgis_XS_layer: str
    """
  
    if qgis_XS_layer or XS_filename:
        if qgis_XS_layer:
            XS_filename = qgis_XS_layer.dataProvider().dataSourceUri()

        XS_shp = gpd.read_file(XS_filename)
        Nsection = 0

        try:
            multi_linestring = Convert_sections(XS_shp)
        except (ValueError, TypeError) as e:
            raise Exception('Erreur lors de la conversion des sections: {}'.format(e))

        for reach in BV.reach:
            print(reach.name)
            list_of_points =[]
            list_of_lines = []
            reach.Xinterp = []
            line_reach = reach.geodata['geometry']
            new_coords = [(x, y) for x, y, z in line_reach.coords]
            line_reach_xy = LineString(new_coords)

            for indsection, line in enumerate(multi_linestring.geoms):
                if type(line) == LineString:
                    point_intersection = line.intersection(line_reach_xy)
                    if type(point_intersection) == Point:
                        list_of_points.append(point_intersection)
                        list_of_lines.append(line)
                else:
                    raise Exception('Cross section are not LineString')

            coords = list(line_reach.coords)
            projected_coords = []
            for point in list_of_points:
                # Calculer la position projetée le long de la ligne
                projected_distance = line_reach_xy.project(point)
                # Trouver les coordonnées réelles sur la ligne à cette distance projetée
                projected_point = line_reach_xy.interpolate(projected_distance)
                projected_coords.append((projected_point.x, projected_point.y))
            for projected_point in projected_coords:
                coords.append(projected_point)
            # Trier les points (coordonnées) dans l'ordre de la ligne en fonction de leur distance sur la ligne
            seen = set()  # Ensemble pour garder une trace des éléments déjà vus
            liste_sans_doublons = []  # Liste pour stocker les résultats sans doublons

            for item in coords:
                if item not in seen:
                    liste_sans_doublons.append(item)  # Ajouter l'élément s'il est unique
                    seen.add(item)
            coords=liste_sans_doublons
            sorted_coords = sorted(coords, key=lambda coord: line_reach_xy.project(Point(coord)))
            sorted_coords_xy = [(coord[0],coord[1]) for coord in sorted_coords]
            line_reach_tot = LineString(sorted_coords_xy)
            X_reach_tot =X_abscissa(line_reach_tot)


            reach.Xinterp=np.zeros(len(list_of_points))
            if len(sorted_coords)>1:
                for indpoint, point in enumerate(list_of_points):
                    point_xy =Point(point.x,point.y)
                    min_distance = float('inf')
                    indice= -1
                    for i, point_line in enumerate(sorted_coords_xy):
                        shapely_point = Point(point_line)  # Convertir chaque point en objet Shapely
                        dist = point_xy.distance(shapely_point)  # Utiliser la distance Shapely
                        if dist < min_distance:
                            min_distance = dist
                            indice = i

                    reach.Xinterp[indpoint] = X_reach_tot[indice]
                    line = list_of_lines[indpoint]

                    #creation section
                    S1 = Section(line)
                    if indice == 0:
                        S1.normal = [line_reach_tot.xy[0][indice] - line_reach_tot.xy[0][indice+1],
                                     line_reach_tot.xy[1][indice] - line_reach_tot.xy[1][indice+1]]
                    else:
                        S1.normal = [line_reach_tot.xy[0][indice-1] - line_reach_tot.xy[0][indice],
                                 line_reach_tot.xy[1][indice-1] - line_reach_tot.xy[1][indice]]

                    angle_river = np.arctan2(S1.normal[1], S1.normal[0])
                    angle_section = np.arctan2((line.coords[0][1] - point.y),
                                                (line.coords[0][0] - point.x))

                    #if (angle_river - angle_section) < np.pi :  # positionne la rive droite à droite
                        #line = LineString([coord for coord in line.coords[::-1]])

                    if not line.has_z:
                        line_int = LineString(
                            [line.interpolate(float(n) / param['XS']['numberOfPoints'], normalized=True) for n in range(param['XS']['numberOfPoints']+ 1)])
                        S1.line = line_int
                        zbed = [coord[2] for coord in line_int.coords]
                        S1.Zbed = np.min(zbed)

                    else:
                        zbed = [coord[2] for coord in line.coords]
                        S1.Zbed = np.min(zbed)
                        S1.line=line

                    S1.length = reach.Xinterp[indpoint]
                    S1.name = str('river_'+str(reach.geodata['River']) + '_Reach_'+str(reach.geodata['Reach'])+'_section_'+str(indpoint))
                    S1.centre = point # point du centre défini par la ligne centrale .shp
                    S1.distanceBord()
                    #if not line.has_z:
                    S1.originXSLocation('riverline', param['XS'][
                        'distSearchMin'])  # l'origine de la distance latérale est au point le plus bas

                    S1.manning = [10  for ii in range(len(S1.line.coords))]
                    reach.section.append(S1)

        reach.Xinterp = list(reach.Xinterp)
        reach.compute_slope()
        for ids, section in enumerate(reach.section):
            section.slope = reach.slope[ids]

    return Nsection

def createXSlines(BV, L, nx):
    """ Create all sections for all reaches

    :param L: half width of the section
    :type L : float
    :param n_x: number of point equally spaced regarding the curviline abscissa
    :type nx: int
    """
    # Cleaning previous sections
    for reach1 in BV.reach:
        reach1.section = []

    # Creating new sections
    n_sections = 0
    for reach1 in BV.reach:
        reach1.id_first_section = n_sections
        Xs = reach1.line_int.xy[0]
        Ys = reach1.line_int.xy[1]
        line2 = LineString([(x, y) for (x, y, z) in reach1.geodata['geometry'].coords])
        #recherche du point original le plus proche
        c0 = np.transpose(np.array([line2.xy[0],line2.xy[1]]))
        c1= np.transpose(np.array([Xs,Ys]))
        t0 = cKDTree(c0)
        distance, neighbours = t0.query(c1)

        for i in range(len(Xs)):
            angle, neighbours_down, neighbours_up = angle_from_line(reach1.line_int, line2, neighbours, i)
            Xstart= Xs[i]+L*np.cos(angle+np.pi/2)
            Ystart= Ys[i]+L*np.sin(angle+np.pi/2)
            Xend= Xs[i]+L*np.cos(angle-np.pi/2)
            Yend= Ys[i]+L*np.sin(angle-np.pi/2)
            
            line = LineString([(Xstart,Ystart), (Xend,Yend)])             
            if nx == 0:
                nx = 1 
            line_int = LineString([line.interpolate(float(n) / nx, normalized=True) for n in range(nx + 1)])

            S1 = Section(line_int)
            S1.length = reach1.Xinterp[i] #postion longitudinale de la section
            S1.name = str('river_'+str(reach1.geodata['River']) + '_Reach_'+str(reach1.geodata['Reach'])+'_section_'+str(i))
            S1.centre = Point(Xs[i],Ys[i])  #point du centre défini par la ligne centrale .shp
            S1.distanceBord()
            S1.normal = [line2.xy[0][neighbours_down]-line2.xy[0][neighbours_up],line2.xy[1][neighbours_down]-line2.xy[1][neighbours_up]]
            reach1.section.append(S1)
            Manning = [10 for _ in range(len(S1.line.xy[0]))]
            S1.manning=Manning
            S1.Zbed =0


        n_sections += len(reach1.section)

        reach1.compute_slope()
        for ids, section in enumerate(reach1.section):
            section.slope = reach1.slope[ids]

    return n_sections

def projectionXSline(BV, params):
    """ Provide an elevation for each point of the sections. Each reach contains a list of all the sections linked to it.

    The elevation is obtained by 2D interpolation with the method specified.
    The projection is made DEM file by DEM file. A section can cover 2 different DEM files.
    The raster mode is not operational yet

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    reach = BV.reach
    list_of_point_tot = []
    list_of_z = []
    list_of_id_tot = []
    list_of_dist_point_tot = []

    for idx, data_DEMi in enumerate(BV.DEM_stack['data_DEM']):
        print('Cross section projection on  DEM n°'+str(idx+1)+'/' + str(len(BV.DEM_stack['file_list'])))

        polygon = Polygon(data_DEMi['polygon_coords'])
        current_file = BV.DEM_stack['file_list'][idx]

        if params['XS']['typeProj'] == 'interpolation':
            _, _, DEMi = read_DEM(current_file, BV.crs)
        else: # raster
            DEMi = rasterio.open(current_file, 'r', crs=BV.crs)

        list_of_point =[]
        list_of_id = []
        list_of_dist_point =[]

        for current_reach in reach:
            for ids, section in enumerate(current_reach.section):
                points= [Point(coord[0],coord[1]) for coord in section.line.coords ]
                for ip,pts in enumerate(points):
                    if polygon.contains(pts):
                        list_of_point.append(pts)
                        Id = [current_reach.geodata['River'],current_reach.geodata['Reach'], ids]
                        list_of_dist_point.append(ip)
                        list_of_id.append(Id)

        X = [pt.x for pt in list_of_point]
        Y = [pt.y for pt in list_of_point]
        Z = projOnDEM(X, Y, DEMi, params['XS']['typeProj'], params['XS']['interpolationMethod'])
        list_of_z = list_of_z + Z
        list_of_point_tot = list_of_point_tot + list_of_point
        list_of_id_tot = list_of_id_tot + list_of_id
        list_of_dist_point_tot = list_of_dist_point_tot + list_of_dist_point

        #reattribution des points et des cotes aux bonnes sections
    for idr,reach1 in enumerate(reach):
        for ids,section in enumerate(reach1.section):

            id_selected_point=[i for i,d in enumerate(list_of_id_tot) if d==[reach1.geodata['River'],reach1.geodata['Reach'], ids]]
            list_point_selected = []
            list_of_dist_point_selected = []
            list_z_selected =[]
            if len(id_selected_point)>1:
                for i in range(len(id_selected_point)):
                    if not list_of_z[id_selected_point[i]] == data_DEMi['no_data'] or list_of_z[id_selected_point[i]]>-99:
                        list_point_selected.append(list_of_point_tot[id_selected_point[i]])
                        list_z_selected.append(list_of_z[id_selected_point[i]])
                        list_of_dist_point_selected.append(list_of_dist_point_tot[id_selected_point[i]])

                list_x=[pt.x for pt in list_point_selected]
                list_y=[pt.y for pt in list_point_selected]
                list_indpoint = [npoint for npoint in list_of_dist_point_selected]
                #trie des points dans le sens initial
                sort_list_x = [i for _, i in sorted(zip(list_indpoint, list_x))]
                sort_list_y = [i for _, i in sorted(zip(list_indpoint, list_y))]
                sort_list_z_selected = [i for _, i in sorted(zip(list_indpoint, list_z_selected))]

                section.line = LineString([(x,y,z) for x,y,z in zip(sort_list_x,sort_list_y,sort_list_z_selected)])

            section.distanceBord()
            section.originXSLocation(params['XS']['originSection'],params['XS']['distSearchMin'] ) #l'origine de la distance latérale est au point le plus bas
            Manning = [10 for _ in range(len(section.line.xy[0]))]
            section.manning=Manning
        
        reach1.compute_slope()
        for ids,section in enumerate(reach1.section):
            section.slope = reach1.slope[ids]
           
        


def interpolateAllXS(BV,param):
    """ Interpolate all cross-sections (calling Reach.interpolateXS function)

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    
    dz = param['H']['dz']
    dxlong= param['XS']['XSInterStep']
    method = param['XS']['methodChannel']
    method_levee = param['H']['levee']
    manning = param['XS']['defaultManning']
    reach=BV.reach
    for j,reach1 in enumerate(reach):           

        new_section=[]

        selected_points = []

        # Récupération des points de la boundary_line
        line = reach1.geodata['geometry']
        boundary_line = reach1.line_int

        list_of_subline = split_line_by_boundary(line, boundary_line)


        for i in range(len(reach1.section)-1): #boucle sur les portions entre sections

            subline = list_of_subline[i]

            #cote limite pour l'interpolation
            z_up= [reach1.section[i].line.coords[t][2] for t in range(len(reach1.section[i].line.coords))]
            Zmaxi=np.max(z_up)
            z_down =[reach1.section[i+1].line.coords[t][2] for t in range(len(reach1.section[i+1].line.coords))]
            Zmaxi1=np.max(z_down)
            maxZ = np.max([(Zmaxi-reach1.section[i].Zbed),(Zmaxi1-reach1.section[i+1].Zbed)])

            num_vert = int(round(subline.length / dxlong))
            if num_vert == 0:
                num_vert = 1
            
            line_int=LineString([subline.interpolate(float(n) / num_vert, normalized=True) for n in range(num_vert + 1)])
            dxlong_real=subline.length/num_vert

            c0 = np.transpose(np.array([subline.xy[0],subline.xy[1]]))  #point de la ligne originale entre section le plus proche du centre de la nouvelle section interpolée
            c1= np.transpose(np.array([line_int.xy[0],line_int.xy[1]]))
            t0 = cKDTree(c0)
            _, neighbours = t0.query(c1) 

            # on cherche la ligne définie au niveau de la section interp pour définir l'angle
            for ii in range(1,len(line_int.coords)-1): #boucle sur les points de la ligne entre 2 sections                

                X=[] #coordonées X des points définissant la nouvelle section
                Y=[]
                distance =[]
                Z=[]
                length_interp = ii * dxlong_real
               
                # #interpolation de la distance entre le point bas et la ligne centrale            
                dist_centre = np.interp(length_interp,[0,subline.length],[reach1.section[i].d0,reach1.section[i+1].d0])
                zbed = np.interp(length_interp,[0,subline.length],[reach1.section[i].Zbed,reach1.section[i+1].Zbed])
                angle, neighbours_down, neighbours_up = angle_from_line(line_int, subline, neighbours, ii)


                if method=='one channel':     #priorité à  l'interpolation du lit mineur jusqu'à cote  débordement

                    for z in np.arange(0.01,maxZ,dz):
                        # trouver l'intersection des points de
                        dbankup = reach1.section[i].findBankPoint(z+reach1.section[i].Zbed,True)
                        dbankdown = reach1.section[i+1].findBankPoint(z+reach1.section[i+1].Zbed,True) #prise en compte de la pente entre les 2 points bas              
                        distance_temp0= np.interp(length_interp,[0,subline.length],[dbankup[0],dbankdown[0]]) #interpolation rive gauche
                        distance_temp1= np.interp(length_interp,[0,subline.length],[dbankup[1],dbankdown[1]]) #interpolation rive droite

                        X.append(line_int.coords[ii][0]+distance_temp0*np.cos(angle+np.pi/2)) #rive droite
                        Y.append(line_int.coords[ii][1]+distance_temp0*np.sin(angle+np.pi/2))
                        Z.append(zbed + z)
                        X.append(line_int.coords[ii][0] + distance_temp1 * np.cos( angle + np.pi / 2))  # rive droite
                        Y.append(line_int.coords[ii][1] +  distance_temp1 * np.sin(angle + np.pi / 2))
                        Z.append(zbed+z)
                        distance.append(distance_temp0)#+dist_centre)
                        distance.append(distance_temp1)#+dist_centre)

                    sort_X =[x for _,x in sorted(zip(distance,X))]
                    sort_Y =[x for _,x in sorted(zip(distance,Y))]
                    sort_Z =[x for _,x in sorted(zip(distance,Z))]                     
                    distance.sort()
                 
                #--methode distance identique HEC RAS----
                elif method == 'multi channel':

                    distancetot = reach1.section[i].distance + reach1.section[i + 1].distance
                    distancetot = list(np.unique(distancetot))
                    distancetot.sort()
            
                    z_upinterp = np.interp(distancetot, reach1.section[i].distance, z_up)
                    z_downinterp = np.interp(distancetot, reach1.section[i + 1].distance, z_down)
                    z_interp = []

                    for jj in range(len(distancetot)):  # boucle sur les points de la section
                        z_interp.append(
                            np.interp(length_interp, [0, subline.length], [z_upinterp[jj], z_downinterp[jj]]))
                        X.append(line_int.coords[ii][0] + (dist_centre + distancetot[jj]) * np.cos(
                            angle - np.pi / 2))  # rive droite
                        Y.append(line_int.coords[ii][1] + (dist_centre + distancetot[jj]) * np.sin(angle - np.pi / 2))

                    sort_X = [x for _, x in sorted(zip(distancetot, X))]
                    sort_Y = [x for _, x in sorted(zip(distancetot, Y))]
                    sort_Z = [x for _, x in sorted(zip(distancetot, z_interp))]

                line_newSection = LineString([(sort_X[i], sort_Y[i], sort_Z[i]) for i in range(len(sort_X))])

                S1 = Section(line_newSection)
                S1.normal = [subline.xy[0][neighbours_down] - subline.xy[0][neighbours_up],
                             subline.xy[1][neighbours_down] - subline.xy[1][neighbours_up]]

                '''
                angle_river = np.arctan2(S1.normal[1], S1.normal[0])
                angle_section = np.arctan2((line_newSection.coords[0][1] - line_int.xy[1][ii]),
                                           (line_newSection.coords[0][0] - line_int.xy[0][ii]))

                
                if abs(angle_river - angle_section) > np.pi/2 : # positionne la rive droite à droite
                    line_newSection_inv = LineString(line_newSection.coords[::-1])
                    S1.line = line_newSection_inv
                '''

                S1.length = reach1.Xinterp[i] + length_interp  # postion longitudinale de la section
                S1.name = str('river_'+str(reach1.geodata['River']) + '_Reach_'+str(reach1.geodata['Reach'])+'_section_'+str(i))
                S1.centre = Point(line_int.coords[ii][0],
                                  line_int.coords[ii][1])  # point du centre défini par la ligne centrale .shp

                # l'origine de la distance latérale est au point le plus bas
                S1.distanceBord()
                S1.originXSLocation(param['XS']['originSection'], param['XS']['distSearchMin'])
                Manning = [manning for i in range(len(S1.line.xy[0]))]
                S1.manning = Manning
                zbed = [coord[2] for coord in S1.line.coords]
                #S1.Zbed = np.min(zbed)
                new_section.append(S1)
               
                
        for i in range(len(new_section)):
            reach1.section.append(new_section[i])
            
        #remet les sections de l'amont vers l'aval
        Xdist=[]
        for i in range(len(reach1.section)):
            Xdist.append(reach1.section[i].length)
        
        sort_section = [x for _,x in sorted(zip(Xdist,reach1.section))]
        Xdist.sort()
        reach1.section=sort_section
        reach1.Xinterp=Xdist
        reach1.compute_slope()
        for ids, section in enumerate(reach1.section):
            section.slope = reach1.slope[ids]

            
 
def optimizeXSlines(BV,method = "simple",resultType ='Normal'):
    """ Modify angle of XS to avoid intersections

   :param BV: Watershed
   :type BV: ModelCatchment
   :param method: method to divided XS
   :type method: str
   :param resultType: computation for banks position (2nd method only)
   :type resultType: str

    """

    for i,reach1 in enumerate(BV.reach):            
 
        angles =[]
        xs_lines_lob =[]
        xs_lines_rob = []
        xs_lines_channel = []
        length_lob =[]
        length_rob =[]
        length_channel = []
        point_inter1=[]
        point_inter2=[]

        if method =="simple":

            nx = len(reach1.section[0].line.coords)
            for i in range(len(reach1.section)):  # boucle sur les portions entre sections
                S1 = reach1.section[i]
                angles.append(np.arctan2(S1.normal[1], S1.normal[0]) - np.pi / 2) # * 180.0 / np.pi)
                xs_lines_channel.append(LineString([(S1.start.x, S1.start.y), (S1.end.x, S1.end.y)]))
                length_channel.append(S1.line.length)
                point_inter1.append(S1.centre)

            angles_channel= angles.copy()
            # Flag intersecting lines
            intersect_flag= np.zeros(len(xs_lines_channel), dtype=int)

            for ix1 in range(0, len(xs_lines_channel)):
                for ix2 in range(ix1 + 1, len(xs_lines_channel)):
                    if xs_lines_channel[ix1].intersects(xs_lines_channel[ix2]):
                        intersect_flag[ix1] += 1

            nintersect = np.sum(intersect_flag)
            print("Number of intersections:", nintersect)
            if nintersect > 0:
                # traitement du lit majeur rive droite
                # --------------------------------------
                xs_lines_channel, angles_channel= remove_intersection(xs_lines_channel, angles_channel, reach1.Xinterp, 'channel',
                                                               length_channel,
                                                               point_inter1)

            for i in range(len(reach1.section)):  # boucle sur les portions entre sections
                S1 = reach1.section[i]
                line = LineString([xs_lines_channel[i].coords[-1], xs_lines_channel[i].coords[0]])
                if nx == 0:
                    nx = 1
                line_int = LineString([line.interpolate(float(n) / nx, normalized=True) for n in range(nx + 1)])

                S1.name = str('river_'+str(reach1.geodata['River']) + '_Reach_'+str(reach1.geodata['Reach'])+'_section_'+str(i))
                S1.line = line_int
                S1.distanceBord()

        elif method == "from bank" :

            if resultType == 'Normal':
                df = reach1.resNormal
            elif resultType == '1D':
                df = reach1.res1D
            if resultType == 'Himposed':
                df= reach1.resHimposed


            for i in range(len(reach1.section)): #boucle sur les portions entre sections
                S1 = reach1.section[i]
                # Compute normals angles
                angles.append(np.arctan2(S1.normal[0], S1.normal[1]))  # * 180.0 / np.pi)

                df_filter = df[df['idSection'] == i]

                xbank1 = df_filter['bank'].iloc[0][1][0].x
                xbank2 = df_filter['bank'].iloc[0][1][1].x
                ybank1 = df_filter['bank'].iloc[0][1][0].y
                ybank2 = df_filter['bank'].iloc[0][1][1].y

                inter1 = Point(xbank1, ybank1)
                inter2 = Point(xbank2, ybank2)

                dist1 = inter1.distance(S1.start)
                dist2 = inter2.distance(S1.end)

                xs_lines_rob.append(LineString([(S1.start.x, S1.start.y), (inter1.x , inter1.y)]))
                length_rob.append(S1.start.distance(inter1))
                point_inter1.append(inter1)
                xs_lines_lob.append(LineString([(inter2.x, inter2.y), (S1.end.x, S1.end.y)]))
                length_lob.append(S1.end.distance(inter2))
                point_inter2.append(inter2)

            angles_lob = angles.copy()
            angles_rob = angles.copy()

            # Flag intersecting lines
            intersect_flag_lob = np.zeros(len(xs_lines_lob), dtype=int)
            intersect_flag_rob = np.zeros(len(xs_lines_rob), dtype=int)
            for ix1 in range(0, len(xs_lines_lob)):
                for ix2 in range(ix1 + 1, len(xs_lines_lob)):
                    if xs_lines_lob[ix1].intersects(xs_lines_lob[ix2]):
                        intersect_flag_lob[ix1] += 1
                    if xs_lines_rob[ix1].intersects(xs_lines_rob[ix2]):
                        intersect_flag_rob[ix2] += 1

            nintersect = np.sum(intersect_flag_lob)+np.sum(intersect_flag_rob)
            print("Number of intersections:", nintersect)

            if nintersect > 0:
                # traitement du lit majeur rive droite
                # --------------------------------------
                xs_lines_rob, angles_rob = remove_intersection(xs_lines_rob, angles_rob, reach1.Xinterp, 'right', length_rob,
                                                               point_inter1)
                #traitement du lit majeur rive gauche
                #--------------------------------------
                xs_lines_lob,angle_lob = remove_intersection(xs_lines_lob, angles_lob, reach1.Xinterp, 'left', length_lob, point_inter2)

                # Recompute normals
            for i in range(len(reach1.section)):  # boucle sur les portions entre sections
                S1 = reach1.section[i]
                line = LineString([xs_lines_rob[i].coords[0],xs_lines_rob[i].coords[1],
                                    xs_lines_lob[i].coords[0], xs_lines_lob[i].coords[1],
                                    ])
                S1.line = line
                S1.distanceBord()
                S1.name = str('river_'+str(reach1.geodata['River']) + '_Reach_'+str(reach1.geodata['Reach'])+'_section_'+str(i))           


def saveXSlines(BV, filepath):
    """ save XS in a shapefile format

     :param BV: Watershed
     :type BV: ModelCatchment
     :param filepath: folder in which save XS
     :type method: str

      """
    gdf_tosave = GeoDataFrame(columns = ['geometry'],crs =BV.crs)
    count =0
    for i in range(len(BV.reach)):
        for j in range(0, len(BV.reach[i].section)):
            gdf_tosave.loc[count,'geometry'] = BV.reach[i].section[j].line
            count +=1
    gdf_tosave.to_file(filepath)

def split_line_by_boundary(line, boundary_line):
    """
    Divise `line` en plusieurs LineStrings basées sur les projections des segments de `boundary_line`.

    :param line: Une LineString cible à diviser.
    :param boundary_line: Une LineString définissant les segments pour la projection.
    :return: Une liste de LineStrings correspondant aux intervalles.
    """
    if not isinstance(line, LineString) or not isinstance(boundary_line, LineString):
        raise TypeError("Les deux entrées doivent être des objets de type LineString.")

    # Récupérer les projections des points de `boundary_line` sur `line`
    projected_points = []
    for boundary_point in boundary_line.coords:
        point = Point(boundary_point)
        projected_point = line.interpolate(line.project(point))
        projected_points.append(projected_point)

    # Ordonner les projections le long de la `line`
    projected_points = sorted(projected_points, key=lambda p: line.project(p))

    # Construire des LineStrings pour chaque intervalle
    intervals = []
    for i in range(len(projected_points) - 1):
        start_projection = projected_points[i]
        end_projection = projected_points[i + 1]

        # Extraire les points de la `line` entre les projections
        segment_points = []
        for coord in line.coords:
            point = Point(coord)
            if line.project(start_projection) <= line.project(point) <= line.project(end_projection):
                segment_points.append(coord)

        # Ajouter les projections explicitement pour garantir qu'elles sont incluses
        if tuple(start_projection.coords[0]) not in segment_points:
            segment_points.insert(0, tuple(start_projection.coords[0]))
        if tuple(end_projection.coords[0]) not in segment_points:
            segment_points.append(tuple(end_projection.coords[0]))

        # Créer une LineString pour cet intervalle
        intervals.append(LineString(segment_points))

    return intervals

