######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################
import os.path

# global
import geopandas as gpd
import matplotlib.pyplot as plt
from geopandas import GeoDataFrame
# local
from .Tools import *
from shapely.geometry import Point,polygon, linestring, MultiPoint, MultiPolygon, mapping

def addQtoSection(BV, param):
    """ Add discharge at the network linked to the chosen outlet. Discharge is computed considering drained area ratio.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    #select river
    reach = BV.reach
    Q = param['H']['outletDischarge']
    Q_outlet = Q
    method = param['H']['dischargeMethod']
    crs = param['C']['DEM_CRS_ID']#
    window_size = param['C']['window_size']

    ind_river = BV.list_of_outlet[BV.id_outlet][1] #indice des rivières du BV
    print(ind_river)
    ind1 = [BV.list_of_outlet[BV.id_outlet][2]] #tronçon aval [river,reach]
    ind_river_reach = []

    for i in range(len(reach)):

        ind_river_reach.append([reach[i].geodata['River'],reach[i].geodata['Reach']])

    for i in range(len(reach)):
        if reach[i].geodata['River'] in ind_river:
            for j in range(len(reach[i].section)):
                section = reach[i].section[j]
                if param['C']['computeGlobal']:
                    neighborhood = find_pixels_in_neighborhood(BV.Map['acc'], BV.Map['X'], BV.Map['Y'],
                                                            section.centre.coords[0][0],
                                                            section.centre.coords[0][1],
                                                            window_size)
                    Area = np.max(neighborhood)
                else:
                    Area = 0
                section.acc =Area

    if method == 'Per reach':

        ind_reach = ind_river_reach.index(ind1[0])
        reach[ind_reach].Q = Q  #intialisation du bief aval à la valeur de Q
        neighborhood = find_pixels_in_neighborhood(BV.Map['acc'], BV.Map['X'], BV.Map['Y'],
                                                   reach[ind_reach].section[-2].centre.coords[0][0],
                                                   reach[ind_reach].section[-2].centre.coords[0][1],
                                                   window_size)

        reach[ind_reach].area = np.max(neighborhood)
        n_reach_withQ =1

        while n_reach_withQ < len(reach):

            for j in range(len(BV.junction)):
                if BV.junction.loc[j, 'River1'] in ind_river:

                    ind_new_reach = ind_river_reach.index( [BV.junction.loc[j,'River1'],BV.junction.loc[j,'Reach1']])
                    if reach[ind_new_reach].Q != 0:
                        ind_reach_upstream1 = ind_river_reach.index([BV.junction.loc[j,'River2'],BV.junction.loc[j,'Reach2']])
                        ind_reach_upstream2 = ind_river_reach.index([BV.junction.loc[j, 'River3'], BV.junction.loc[j,'Reach3']])

                        #accumalative area at upstream
                        neighborhood = find_pixels_in_neighborhood(BV.Map['acc'], BV.Map['X'], BV.Map['Y'],
                                                                   reach[ind_reach_upstream1].section[-2].centre.coords[0][0],
                                                                   reach[ind_reach_upstream1].section[-2].centre.coords[0][1],
                                                                   window_size)

                        reach[ind_reach_upstream1].area = np.max(neighborhood)
                        reach[ind_reach_upstream1].Q = reach[ind_reach_upstream1].area/reach[ind_new_reach].area*reach[ind_new_reach ].Q
                        n_reach_withQ +=1
                        # accumalative area at upstream
                        neighborhood = find_pixels_in_neighborhood(BV.Map['acc'], BV.Map['X'], BV.Map['Y'],
                                                                   reach[ind_reach_upstream2].section[-2].centre.coords[0][0],
                                                                   reach[ind_reach_upstream2].section[-2].centre.coords[0][1],
                                                                   window_size)

                        reach[ind_reach_upstream2].area = np.max(neighborhood)
                        reach[ind_reach_upstream2].Q = reach[ind_reach_upstream2].area / reach[ind_new_reach].area * reach[
                            ind_new_reach].Q
                        n_reach_withQ += 1


        for i in range(len(BV.reach)):
            for j in range(len(reach[i].section)):
                reach[i].section[j].Q = reach[i].Q

    elif method == 'Accumulated area':
        for i in range(len(reach)):
            reach[i].Q = Q
            #if reach[i].geodata['River'] in ind_river:
            for j in range(len(reach[i].section)):
                section = reach[i].section[j]
                neighborhood = find_pixels_in_neighborhood(BV.Map['acc'], BV.Map['X'], BV.Map['Y'],
                                                           section.centre.coords[0][
                                                               0],
                                                           section.centre.coords[0][
                                                               1],
                                                           window_size)
                section.acc =  np.max(neighborhood)

                section.Q = section.acc/BV.list_of_outlet[BV.id_outlet][3]*Q

    elif method == 'Constant':
        for reach in BV.reach:
            reach.Q = Q

            for j in range(len(reach.section)):
                section = reach.section[j]
                section.Q = Q

    if method == 'Per reach':

        print('discharge constant per reach, discharge file created and/or used')

        if os.path.exists(os.path.join(param.work_path,'discharge.shp')):
            discharge_df = gpd.read_file(os.path.join(param.work_path, 'discharge.shp'))
            if len(discharge_df) == len(reach):
                    for index, row in discharge_df.iterrows():
                        row['Q computed'] = reach[index].Q

                        if float(row['Q imposed'])>0:
                            if index == ind_reach:
                                Q_outlet = float(row['Q imposed'])

                            reach[index].Q = float(row['Q imposed'])
                            for j in range(len(reach[index].section)):
                                reach[index].section[j].Q = reach[index].Q

            else:
                print('Discharge data are not consistent with hydro network, remove former discharge file')

        else:
            BV.discharge = GeoDataFrame(columns=['idReach', 'River', 'Reach', 'Q computed', 'Q imposed', 'geometry'],
                                        crs=crs)
            count = 0
            for reach1 in reach:
                BV.discharge.loc[count, 'idReach'] = count
                BV.discharge.loc[count, 'River'] = reach1.geodata['River']
                BV.discharge.loc[count, 'Reach'] = reach1.geodata['Reach']
                BV.discharge.loc[count, 'Q computed'] = reach1.Q
                BV.discharge.loc[count, 'Q imposed'] = -1
                BV.discharge.loc[count, 'geometry'] = reach1.geodata['geometry']
                count +=1

            BV.discharge.to_file(os.path.join(param.work_path,'discharge.shp'))

    return Q_outlet

def addWidthtoSection(BV,param):
    """ Compute for all sections the curve of wetted area, a width as a function of waterdepth.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    reach = BV.reach
    ind_river = BV.list_of_outlet[BV.id_outlet][1]
    dz = param['H']['dz']
    method = param['H']['levee'] #méthode des digues ou non
    dxlat = param['H']['dxlat']
    for i in range(len(reach)):
            if reach[i].geodata['River'] in ind_river:
                for j in range(len(reach[i].section)):
                    
                    section = reach[i].section[j]
                    section.hydro = {'depth': [], 'A': [], 'W': [], 'Q': [], 'equivalent': {'w1': 0, 'd1': 0, 'i2': 0}}
                    hmax = param['H']['hmax']#np.max(z_up)

                    for z in np.arange(0.01,hmax,dz):

                        dbank = section.findBankPoint(section.Zbed+z,method)
                        if len(dbank)>1:
                            averagedValue, bank, hydro_distr, manning_distr = section.computeHydraulicGeometry(section.Zbed+z, dxlat,method)
                            section.hydro['A'].append(averagedValue['A'])
                            section.hydro['depth'].append(z)
                            section.hydro['W'].append(dbank[-1]-dbank[0])

                            dist_distr = hydro_distr[:, 0]
                            V = np.zeros((len(dist_distr),))
                            for idx, Rh in enumerate(hydro_distr[:, -1]):
                                V[idx] = 1 / manning_distr[idx] * Rh ** (2 / 3) * section.slope ** 0.5
                            Qcomputed = np.sum(V*hydro_distr[:, 2])
                            section.hydro['Q'].append(Qcomputed )
                        else:
                            print('No width at section ' +str(j)+ 'of reach ' + reach[i].name)


    try :
        with pd.ExcelWriter(os.path.join(param.work_path,"hydraulic_curve.xlsx")) as writer:
            # save result as Excel file
            for reach1 in BV.reach:
                for section in reach1.section:
                    df = pd.DataFrame(columns=['depth','Area','Width','Discharge'])
                    for i in range(len(section.hydro['depth'])):
                        df.loc[i, 'depth'] = section.hydro['depth'][i]
                        df.loc[i, 'Area'] = section.hydro['A'][i]
                        df.loc[i, 'Width'] = section.hydro['W'][i]
                        df.loc[i, 'Discharge'] = section.hydro['Q'][i]
                    # use to_excel function and specify the sheet_name and index
                    # to store the dataframe in specified sheet
                    df.to_excel(writer, sheet_name=section.name, index=False)
    except :
        df = pd.DataFrame(columns=['reach','section','depth', 'Area', 'Width','Discharge'])
        count=0
        for reach1 in BV.reach:
            for ids,section in enumerate(reach1.section):
                for i in range(len(section.hydro['depth'])):
                    df.loc[count, 'reach'] = reach1.name
                    df.loc[count, 'section'] = 'XS_' + str(reach1.id_first_section + ids)
                    df.loc[count, 'depth'] = section.hydro['depth'][i]
                    df.loc[count, 'Area'] = section.hydro['A'][i]
                    df.loc[count, 'Width'] = section.hydro['W'][i]
                    df.loc[count, 'Discharge'] = section.hydro['Q'][i]
                    count +=1
                # use to_excel function and specify the sheet_name and index
                # to store the dataframe in specified sheet
        df.to_csv(os.path.join(param.work_path,"hydraulic_curve.csv"))



def computeNormalAndCriticalDepth(BV, param):
    """Compute the normal and critical depth at each section as a function of the variable Q of each section.
     Only one channel is possible presently.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """

    for reach in BV.reach:
        #if reach[i].geodata['River'] in ind_river:
        header = ['ID', 'idSection', 'Q', 'WSE', 'CritDepth', 'bank', 'distance', 'h', 'A', 'P', 'Rh', 'V', 'Sf']
        reach.resNormal = pd.DataFrame(columns=header)
        reach.reachNormalAndCriticalDepth(param['H'])


def backWaterProfile(BV,param):
    """Computation of the backwater profile. If regime is supercritical, the water deth is milited to the crotical one.
    Avaliable only for one reach presently.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """

    reach = BV.reach
    ind1 = [BV.list_of_outlet[BV.id_outlet][2]]  # tronçon aval [river,reach]
    ind_river = [BV.list_of_outlet[BV.id_outlet][1]]  # tronçon aval [river,reach]
    ind_river_reach = []


    #find inline_structure data
    if os.path.exists(os.path.join(param.work_path,'inline_structure.shp')):
        inline_df = gpd.read_file(os.path.join(param.work_path,'inline_structure.shp'))
        list_of_centre =[]
        list_ind_reach =[]
        list_ind_section = []
        for idr,reach1 in enumerate(BV.reach):
            for ids,section in enumerate(reach1.section):
                list_of_centre.append(section.centre)
                list_ind_reach.append(idr)
                list_ind_section.append(ids)

        for index, row in inline_df.iterrows():
            point_inline = row['geometry']
            ids = min(range(len(list_of_centre)), key=lambda i: list_of_centre[i].distance(point_inline))
            BV.reach[list_ind_reach[ids]].section[list_ind_section[ids]].type = 'inline_structure'
            BV.reach[list_ind_reach[ids]].section[list_ind_section[ids]].inlinedata = [float(row['Cd']),
                                                                                       float(row['L']),
                                                                                       float(row['Zs']),
                                                                                       row['structure']]



    header = ['ID', 'idSection', 'Q', 'WSE', 'CritDepth', 'bank', 'distance', 'h', 'A', 'P', 'Rh', 'V', 'Sf']
    for i in range(len(reach)):
        ind_river_reach.append([reach[i].geodata['River'], reach[i].geodata['Reach']])

    ind_reach_down = ind_river_reach.index(ind1[0])
    print('1D computation for reach' + str(ind1[0]))


    reach[ind_reach_down].res1D = pd.DataFrame(columns=header)
    reach[ind_reach_down].reachBackWater(param['H']['hWaterOutlet'], param['H'])
    reach[ind_reach_down].flag1D=1
    n_reach_with_comp = 1

    while n_reach_with_comp < len(reach):
        for j in range(len(BV.junction)):
            if BV.junction.loc[j, 'River1'] in ind_river and BV.junction.loc[j, 'River2'] in ind_river\
                    and BV.junction.loc[j, 'River3'] in ind_river:
                ind_new_reach = ind_river_reach.index([BV.junction.loc[j, 'River1'], BV.junction.loc[j, 'Reach1']])
                if reach[ind_new_reach].flag1D != 0:
                    df_filter = reach[ind_new_reach].res1D[reach[ind_new_reach].res1D['idSection'] == 0]
                    h_junction = df_filter.loc[df_filter.index[0], 'WSE']
                    ind_reach_upstream1 = ind_river_reach.index(
                        [BV.junction.loc[j, 'River2'], BV.junction.loc[j, 'Reach2']])
                    ind_reach_upstream2 = ind_river_reach.index(
                        [BV.junction.loc[j, 'River3'], BV.junction.loc[j, 'Reach3']])
                    if reach[ind_reach_upstream1].flag1D == 0:
                        print('1D computation for reach ' + str(BV.junction.loc[j, 'River2']) + ', ' + str(
                            BV.junction.loc[j, 'Reach2']))
                        reach[ind_reach_upstream1].res1D = pd.DataFrame(columns=header)
                        reach[ind_reach_upstream1].reachBackWater(h_junction, param['H'])
                        reach[ind_reach_upstream1].flag1D = 1
                        n_reach_with_comp += 1

                    if reach[ind_reach_upstream2].flag1D == 0:
                        reach[ind_reach_upstream2].res1D = pd.DataFrame(columns=header)
                        reach[ind_reach_upstream2].reachBackWater(h_junction, param['H'])
                        print('1D computation for reach ' + str(BV.junction.loc[j, 'River3']) +
                          ', ' + str(BV.junction.loc[j, 'Reach3']))
                        reach[ind_reach_upstream2].flag1D = 1
                        n_reach_with_comp += 1


def impose_water_depth(BV,param):
    """Computation of hydaulic parameter for an imosed wtare depth at each section. water balance is not verified
    Avaliable only for one reach presently.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    
    reach = BV.reach
    ind_river = BV.list_of_outlet[BV.id_outlet][1]
    header = ['ID', 'idSection', 'Q', 'WSE', 'CritDepth', 'bank', 'distance', 'h', 'A', 'P', 'Rh', 'V', 'Sf']

    #if type(ind_river) == int:
     #   ind_river = [ind_river] # une seule rivière dans le réseau
    for reach in BV.reach:

        #if reach[i].geodata['River'] in ind_river:
        reach.resHimposed = pd.DataFrame(columns=header)
        reach.reachImposedWaterDepth(param['H'])



def projOnFrictionMap(BV,poly_filename):
    """Provide the friction coefficient to each point af all sections according a map value.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param poly_filename: name of the shapefile containing the friction area (multipolygone type)
    :type poly_filename: str
    """

    poly_shp = gpd.read_file(poly_filename)
    list_header = poly_shp.columns.to_list()
    ManningName = list_header[1]
    for i, row in poly_shp.iterrows():
        for reach1 in BV.reach:
            for section in reach1.section:
                list_of_point = [Point(coord) for coord in section.line.coords]
                for ip,pt in enumerate(list_of_point):
                    if row['geometry'].contains(pt):
                        manning_selected =row[ManningName]
                        if section.manning[ip]>manning_selected:
                            section.manning[ip] = manning_selected


def setConstantFriction(BV,manning_value):
    """Provide the friction coefficient to each point af all sections according a cosntant  value.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param manning_value: constant friciton value
    :type manning_value: float
    """


    for reach1 in BV.reach:
        for section in reach1.section:
            for ip, coord in enumerate(section.line.coords):
                section.manning[ip] = manning_value

def createAllBankLines(BV, resultType = 'Normal'):
    """Create a geopanda dataframe for all reaches with banks as geometry.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param resultType: type of the computation for which the bank are calculated
    :type resultType: str
    """
    for reach1 in BV.reach:
        reach1.createBankLines(resultType=resultType)


def saveBankslines(BV, filepath):
    """save the geopanda dataframe for all reaches with banks as geometry.

    :param BV: Watershed
    :type BV: ModelCatchment
    :param filepath: path of the repertory for the shapefile of all bank lines
    :type filepath: str
    """
    gdf_tosave = GeoDataFrame(columns=['geometry'],crs=BV.crs)
    count = 0
    for reach1 in BV.reach:
        idRiver, idReach = reach1.geodata['River'], reach1.geodata['Reach']
        gdf_tosave.loc[count,'geometry'] = reach1.leftLine
        gdf_tosave.loc[count,'Name'] = 'LeftBank_river' +str(idRiver) +'_reach' + str(idReach)
        count +=1
        gdf_tosave.loc[count,'geometry'] = reach1.rightLine
        gdf_tosave.loc[count,'Name'] = 'RightBank_river' +str(idRiver) +'_reach' + str(idReach)
        count +=1
    gdf_tosave.to_file(os.path.join(filepath,'banks_lines.shp'))



def dectect_inline_structure(BV,param):
    """find section with a slope superior to a threshold (inline structure)

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    crs = BV.crs.to_string()
    BV.inline_structure = GeoDataFrame(columns=['Cd','L','Zs','structure','geometry'],crs = crs)
    count =0
    if param['H']['createBanksMethods'] == 'Himposed':

        for reach in BV.reach:
            df = reach.resHimposed

            for ids,section_u in enumerate(reach.section[:-1]):
                section_u = reach.section[ids-1]
                WS_d =  df['WSE'].iloc[ids+1]
                WS_u = df['WSE'].iloc[ids]
                dL = reach.Xinterp[ids+1]-reach.Xinterp[ids]
                slope = abs((WS_u-WS_d)/dL)
                if slope >  param['H']['slope_structure']:
                    BV.inline_structure.loc[count,'Cd']=0.4
                    width =  abs(df['bank'].iloc[ids][0][0] - df['bank'].iloc[ids][0][1])
                    BV.inline_structure.loc[count, 'L'] = width
                    BV.inline_structure.loc[count, 'Zs'] =WS_u
                    BV.inline_structure.loc[count, 'structure'] = 'weir'
                    BV.inline_structure.loc[count, 'geometry'] = section_u.centre
                    section_u.type = 'inline'
                    count +=1

        BV.inline_structure.to_file(os.path.join(param.work_path,'inline_structure.shp'))


def find_bank_from_poly(BV,dxlat,file_multishape):
    """find position of bank by intersection between XS and a bank line.
    Compute hydraulic geometry of the section considering intersection as observatedwater surface elevation

    :param BV: Watershed
    :type BV: ModelCatchment
    :param dxlat: lateral distance space between 2 vertical lines (sub domain computation)
    :type dxlat: float
    :param file_multishape: bank line obtained from computation or user defined
    :type file_multishape: str or shapely geometry

    """

    if type(file_multishape) == str:
        multishape =gpd.read_file(file_multishape)
    elif type(file_multishape) ==MultiLineString:

        multishape = gpd.GeoDataFrame(geometry=[],crs = BV.crs)
        for idl, line in enumerate(file_multishape.geoms):
            multishape.loc[idl,'geometry'] = line



    for reach in BV.reach:
        for ids,section in enumerate(reach.section):
            list_inter =[]
            Z = [section.line.coords[i][2] for i in range(len(section.line.coords))]


            for i, shape in multishape.iterrows():
                if type(shape['geometry']) == linestring.LineString:
                   line_to_intersect = shape['geometry']
                elif type(shape['geometry']) == polygon.Polygon:
                    line_to_intersect = LineString(shape['geometry'].exterior)

                inter = line_to_intersect.intersection(section.line)

                if type(inter) == MultiPoint:
                    list_inter += [geom for geom in inter.geoms]
                elif type(inter) == Point:
                    list_inter += [inter]

            dist_centre_start = section.start.distance(section.centre)
            dist_centre_end = section.end.distance(section.centre)
            dist_point_to_start =[]
            dbank1 =[-dist_centre_start]
            dbank2 = [dist_centre_end]
            list_point1 =[section.start]
            list_point2 = [section.end]
            for point in list_inter:
                dist_point_to_centre= point.distance(section.centre)
                dist_point_to_start.append(point.distance(section.start))
                if point.distance(section.start)<dist_centre_start :#rive gauche
                    dbank1.append(-1*dist_point_to_centre)
                    list_point1.append(point)
                else:
                    dbank2.append(dist_point_to_centre)
                    list_point2.append(point)

            ind1 = np.argmax(dbank1)
            ind2 = np.argmin(dbank2)
            point_bank1=list_point1[ind1]
            point_bank2=list_point2[ind2]

            Z1 =np.interp(dbank1[ind1],section.distance,Z)
            Z2 = np.interp(dbank2[ind2], section.distance, Z)
            WS = np.min([Z1,Z2])
            section.Zbed = WS
            point_bank1 =Point(point_bank1.x,point_bank1.y,WS)
            point_bank2 = Point(point_bank2.x, point_bank2.y, WS)

            bank =[[dbank1[ind1],dbank2[ind2]],  [point_bank1 ,point_bank2]]
            ID = [reach.geodata['River'], reach.geodata['Reach']]
            #['ID', 'idSection', 'Q', 'WSE', 'CritDepth', 'bank', 'distance', 'h', 'A', 'P', 'Rh', 'V', 'Sf']
            dist_init = np.arange(dbank1[ind1],dbank2[ind2],dxlat)
            init_vector = np.zeros(len(dist_init))
            reach.resObs.loc[ids] = [ID, ids, section.Q, WS, 0, bank, dist_init, init_vector,init_vector,init_vector,init_vector,init_vector, []]


def mapForMarine(BV, param, shape):
    """create the map of drainage network parameter from DEM for hydrological computation in MARINE softaware

    :param BV: Watershed
    :type BV: ModelCatchment
    :param param: Parameters requires for computation
    :type param: Parameters
    """
    list_w1, list_d1,list_i2, x_pos_obj, y_pos_obj  = [], [], [], [], []
    for i in range(len(BV.reach)):
        for j in range(0,len(BV.reach[i].section)):
            [w1,d1,i2] = BV.reach[i].section[j].equivalentChannel(shape)
            list_w1.append(w1)
            list_d1.append(d1)
            list_i2.append(i2)
            x_pos_obj.append(BV.reach[i].section[j].centre.x)
            y_pos_obj.append(BV.reach[i].section[j].centre.y)

    x = BV.Map['X'][0,:]
    y = BV.Map['Y'][:,0]
    resolution = param['C']['resolution']
    W1 = np.zeros(BV.globalDEM.shape)
    D1 = np.zeros(BV.globalDEM.shape)
    I2 = np.zeros(BV.globalDEM.shape)

    for idx,x_pos in enumerate(x):
        for idy,y_pos in enumerate(y):
            index = np.where((abs(x_pos_obj - x_pos) <= resolution/2) & (abs(y_pos_obj - y_pos) <= resolution/2))
            W1[idy,idx] = np.nanmean(np.array(list_w1)[index])
            D1[idy,idx] = np.nanmean(np.array(list_d1)[index])
            I2[idy,idx] = np.nanmean(np.array(list_i2)[index])

    BV.Map['w1'] = W1
    BV.Map['d1'] = D1
    BV.Map['i2'] = I2


def Limerinos(manning_distr, waterdepth_distr):
    """Limerinos formula for bed friction

    :param manning_distr: value of friciton coefficicent (m) for each subdomain
    :type BV: list
    :param waterdepth_distr: value of wayterdepth (m) for each subdomain
    :type waterdepth_distr: list
    :return:
        - friction coefficient for section (list)


    """

    new_manning_distr = []
    for ix, ks in enumerate(manning_distr):
        new_manning_distr.append(waterdepth_distr[ix]**(1/6) *0.0926 /(1.16 + 2* np.log10(waterdepth_distr[ix]/ks)))
    return  new_manning_distr


def find_WaterdepthFromQ(Zbed, section, paramH):
    """ Solve manning equation for the input section

    :param Zbed: lower point of the section used in reach bed
    :type Zbed: float
    :param section: considered section
    :type section: Section
    :param paramH: Parameters requires for hydraulics computation
    :type paramH: dict
    :return:
        - water surface elevation (float)

    """

    WSsup = Zbed + paramH['hsup']
    WSinf = Zbed + paramH['hinf']
    WS = (WSsup + WSinf) / 2
    ecart = WSsup - WSinf
    iter_i =0

    while ecart > paramH['eps'] and iter_i < paramH['MaxIter']:
        iter_i += 1
        # calcul du milieu
        WS = (WSsup + WSinf) / 2
        _, _, hydro_distr, manning_distr = section.computeHydraulicGeometry(
            WS, paramH['dxlat'], paramH['levee'],paramH['frictionLaw'])
        dist_distr = hydro_distr[:, 0]
        A_distr = hydro_distr[:, 2]
        Rh_distr = hydro_distr[:, 4]
        V = np.zeros((len(dist_distr),))

        for idx, Rh in enumerate(Rh_distr):
            V[idx] = 1 / manning_distr[idx] * Rh ** (2 / 3) * section.slope ** 0.5
        Q1 = np.sum(V * A_distr)

        if Q1 > section.Q:
            # la solution est inférieure à m
            WSsup = WS
        else:
            # la solution est supérieure à m
            WSinf = WS
        ecart = WSsup - WSinf

    return WS


def find_CriticaldepthFromQ(Zbed, section, paramH):
    """ Solve Fr=1

    :param Zbed: lower point of the section used in reach bed
    :type Zbed: float
    :param section: considered section
    :type section: Section
    :param paramH: Parameters requires for hydraulics computation
    :type paramH: dict
    :return:
        - critical elevation (float)
    """
    hsup = Zbed + paramH['hsup']
    hinf = Zbed + paramH['hinf']
    h = (hsup + hinf) / 2
    iter_i = 0
    ecart = hsup -hinf
    Fr=0

    while ecart > paramH['eps'] and iter_i < paramH['MaxIter']:
        iter_i += 1
        # calcul du milieu
        h = (hsup + hinf) / 2
        averagedValue, bank, _, _ = section.computeHydraulicGeometry(h, 0, paramH['levee'])
        if averagedValue['A']>0:
            Fr = section.Q ** 2 * (bank[0][1] - bank[0][0]) / 9.81 / averagedValue['A'] ** 3

        if Fr < 1:
            # la solution est inférieure à m
            hsup = h
        else:
            # la solution est supérieure à m
            hinf = h
        ecart = hsup - hinf
    return h


def charge(Q, A, WS):
    """ compute head (m)

    :param Q: objective discharge
    :param A: wetted area
    :param WS: water elevation
    :return:
        - head (float)
    """
    H = WS + Q ** 2 / 2 / 9.81 / A ** 2
    return H


def find_hsup(Zbed, section1, WS2, section2, dx, paramH):
    """solve head loss equation between 2 sections , integral formulation

    :WS1: water elevation at upstream section (m)
    :A2,Rh2,WS2,: hydraulic parameter at downstream section (m2,m,m)
    :line: upstream crossecion line
    :centre : upstream crossection centre location
    :Q: discharge at the upstream section (m3/s)
    :dx: distance between the 2 sections (m)
    :return:
        - water elevation (float)
        - friction slope  (float)

    """
    WSsup = Zbed + paramH['hsup']
    WSinf = Zbed + paramH['hinf']
    WS = (WSsup + WSinf) / 2
    Sf = 0
    ecart = WSsup - WSinf
    iter_i = 0

    while ecart > paramH['eps'] and iter_i < paramH['MaxIter']:
        iter_i += 1
        # calcul du milieu
        WS = (WSsup + WSinf) / 2

        #downstream section
        averagedValue, bank, hydro_distr, manning_distr = section2.computeHydraulicGeometry(
            WS2, paramH['dxlat'], paramH['levee'],paramH['frictionLaw'])
        Pm_distr = [p for p in hydro_distr[:, 3]]
        Rh_distr = [p for p in hydro_distr[:, 4]]

        tau2 = [Pm_distr[i]*Rh_distr[i]**(5/3)/manning_distr[i] for i in range(len(manning_distr))]
        neq2 = averagedValue['P']*averagedValue['Rh']**(5/3)/ np.sum(tau2)
        Sf2 = (section2.Q ** 2 / averagedValue['A'] ** 2) * neq2 ** 2 / (averagedValue['Rh']) ** (
                    4 / 3)
        A2 = averagedValue['A']

        #upstream section
        averagedValue, bank, hydro_distr, manning_distr = section1.computeHydraulicGeometry(
            WS, paramH['dxlat'], paramH['levee'],paramH['frictionLaw'])
        Pm_distr = [p for p in hydro_distr[:, 3]]
        Rh_distr = [p for p in hydro_distr[:, 4]]

        tau1 = [Pm_distr[i]*Rh_distr[i]**(5/3)/manning_distr[i] for i in range(len(manning_distr))]
        neq1 = averagedValue['P']*averagedValue['Rh']**(5/3)/ np.sum(tau1)
        Sf1 = (section1.Q ** 2 / averagedValue['A'] ** 2) * neq1 ** 2 / (averagedValue['Rh']) ** (
                    4 / 3)

        dh = (Sf1 + Sf2) / 2 * dx
        dh2 = charge(section1.Q, averagedValue['A'], WS) - charge(section2.Q, A2, WS2)

        if dh < dh2:
            # la solution est inférieure à m
            WSsup = WS
        else:
            # la solution est supérieure à m
            WSinf = WS

        ecart = WSsup - WSinf
        Sf = dh/dx

    return WS, Sf


def read_obs(BV, paramH, in_obs, time, var='Z', type='EnsObs'):
    """read in situ observatino of water elevation

    :param BV: Watershed
    :type BV: ModelCatchment
    :param paramH: Parameters requires for computation
    :type param: dict
    :param in_obs: observation data
    :type param: csv file/ pickle
    :param time: considered time of observation
    :type time: datetime
    :param var: considered variable of observation
    :type param: str
    :param  type: data observation source
    :type type: str

    """

    time_list = in_obs.times_list(var)

    indtime = np.argmin([abs(time - t) for t in time_list])
    near_time = int(time_list[indtime])

    if type == 'EnsObs':

        for reach in BV.reach:
            X_posObs = []
            var_posObs = []
            distance_reach = X_abscissa(reach.geodata['geometry'])
            idRiver, idReach = reach.geodata['River'], reach.geodata['Reach']
            for ip, p in enumerate(in_obs[var][near_time]['pos']):  # abscisse curviligne de stations de mesure
                if reach.geodata['geometry'].has_z:
                    centerline = LineString([(coord[0], coord[1]) for coord in reach.geodata['geometry'].coords])
                else:
                    centerline = reach.geodata['geometry']

                p_on_centerline = nearest_points(centerline, Point(p))[0]
                count_pts = 1
                while (distance_reach[count_pts] - distance_reach[count_pts - 1] < Point(
                        centerline.coords[count_pts]).distance(p_on_centerline)) \
                        and (count_pts < len(distance_reach) - 1):
                    count_pts += 1

                list_of_points = [coord for coord in
                                  centerline.coords[:count_pts - 1]] + [(p_on_centerline.x, p_on_centerline.y)]

                line_temp = LineString(list_of_points)
                X_posObs.append(line_temp.length)
                var_posObs.append(in_obs[var][near_time]['val'][ip])

            Z_posObs = var_posObs

            if var == 'H':
                point_of_centerline = [s.centre for s in reach.section]
                z_of_centerline = [s.Zbed for s in reach.section]
                for p in range(len(var_posObs)):
                    point_obs = Point(in_obs[var][near_time]['pos'][p][0], in_obs[var][near_time]['pos'][p][1])
                    # Z fond pris sur le MNT
                    Zneighborhood = np.min(
                        find_pixels_in_neighborhood(BV.globalDEM, BV.Map['X'], BV.Map['Y'], point_obs.x, point_obs.y,
                                                    1))
                    Z_posObs[p] = Z_posObs[p] + Zneighborhood
                    # Z fond pris sur les sections
                    # ind_section = np.argmin([point_obs.distance(pc) for pc in point_of_centerline])
                    # Z_posObs[p] =Z_posObs[p]+z_of_centerline[ind_section]

            # interpolation des cotes sur les sections
            Z_section = np.interp(reach.Xinterp, X_posObs, Z_posObs)

            slope = []
            for i in range(len(Z_section) - 1):
                slope.append(abs((Z_section[i + 1] - Z_section[i]) / (reach.Xinterp[i + 1] - reach.Xinterp[i])))

            slope.append(0)

            # calcul des grandeurs hydrauliques pour chaque section

            for j, section in enumerate(reach.section):

                averagedValue, bank, hydro_distr, manning_distr = section.computeHydraulicGeometry(Z_section[j],
                                                                                                   paramH['dxlat'],
                                                                                                   paramH['levee'])

                dist_distr = hydro_distr[:, 0]
                V = np.zeros((len(dist_distr),))

                for idx, Rh in enumerate(hydro_distr[:, -1]):
                    V[idx] = 1 / manning_distr[idx] * Rh ** (2 / 3) * slope[j] ** 0.5

                # Store data in dataframe
                ID = [idRiver, idReach]

                reach.resObs.loc[len(reach.resObs)] = [ID, j, section.Q, Z_section[j], 0, bank,
                                                       hydro_distr[:, 0], hydro_distr[:, 1],
                                                       hydro_distr[:, 2], hydro_distr[:, 3], hydro_distr[:, 4],
                                                       V, slope[j]]

