######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################

from qgis.PyQt.QtWidgets import QDockWidget, QFileDialog
from qgis.PyQt.QtCore import pyqtSignal
from qgis.core import   (QgsMessageLog, Qgis, QgsTask, QgsApplication, QgsVectorLayer, QgsProject,
                         QgsCoordinateReferenceSystem)
from .ui.Ui_CrossSectionWidget import Ui_CrossSectionWidget
from os import path
from ..core.CrossSection import *


class CrossSectionWidget(QDockWidget, Ui_CrossSectionWidget):

    closingWidget = pyqtSignal(name="closingWidget")

    def __init__(self, iface, BV, param, parent = None):
        super(CrossSectionWidget, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        self.BV = BV
        self.param = param
        self.flag_import = False
        self.task = None
        self.checkBox_interpolateCrossSections.stateChanged.connect(self.updatePanel)
        self.radioButton_interpolation.toggled.connect(self.updatePanel)
        self.pushButton_generate.clicked.connect(self.generateXS)
        self.pushButton_selectFile.clicked.connect(self.select_file_path)
        self.pushButton_import.clicked.connect(self.importXS)
        self.pushButton_projection.clicked.connect(self.projectAndInterpolateXS)
        self.updatePanel()

    def closeEvent(self, event):
        self.closingWidget.emit()
        event.accept()

    def updatePanel(self):
        if self.radioButton_interpolation.isChecked():
            self.comboBox_interpolationMethod.setEnabled(True)
        else:
            self.comboBox_interpolationMethod.setDisabled(True)
        if self.checkBox_interpolateCrossSections.isChecked():
            self.lineEdit_XSinterstep.setEnabled(True)
        else:
            self.lineEdit_XSinterstep.setDisabled(True)

    def ui_to_param(self):
        self.param['XS']['reachInterStep'] = float(self.lineEdit_reachInterStep.text())
        self.param['XS']['width'] = float(self.lineEdit_width.text())
        self.param['XS']['numberOfPoints'] = int(self.lineEdit_numberOfPoints.text())
        self.param['XS']['originSection'] = 'riverline'
        self.param['XS']['distSearchMin'] = float(self.lineEdit_distSearchMin.text())
        self.param['XS']['XSInterStep'] = float(self.lineEdit_XSinterstep.text())
        self.param['XS']['methodChannel'] = 'multi channel'
        if self.radioButton_interpolation.isChecked():
            self.param['XS']['typeProj'] = 'interpolation'
            self.param['XS']['interpolationMethod'] = self.comboBox_interpolationMethod.currentText()
        else:
            self.param['XS']['typeProj'] = 'raster'
            self.param['XS']['interpolationMethod'] = None
        self.param['XS']['interpXS'] = self.checkBox_interpolateCrossSections.isChecked()
        self.param['XS']['optimizeXS'] = self.checkBox_optimizeCrossSections.isChecked()
        

    def generateXS(self):
        self.ui_to_param()
        self.flag_import = False
        self.task = GenerateXS('GenerateXS', self.BV, self.param)
        QgsApplication.taskManager().addTask(self.task)

    def select_file_path(self):
        filename, filetype = QFileDialog.getOpenFileName(parent = None, caption = "Open File",
                                                         directory = ".")
        if filename:
            self.lineEdit_filePath.setText(filename)
            self.param['XS']['XS_filename'] = filename

    def importXS(self):
        self.ui_to_param()
        self.flag_import =True
        self.task = ImportXS('GenerateXS', self.BV, self.param)
        QgsApplication.taskManager().addTask(self.task)
        

    def projectAndInterpolateXS(self):
        self.ui_to_param()
        self.task = ProjectAndInterpolateXS('ProjectAndInterpolateXS', self.BV, self.param,self.flag_import)
        QgsApplication.taskManager().addTask(self.task)

class GenerateXS(QgsTask):

    def __init__(self, description, BV, param):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param

    def run(self):
        QgsMessageLog.logMessage(message="Starting Cross Section Process", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(0)
        n_reaches = self.BV.createReach()
        self.BV.interpolateReach(self.param['XS']['reachInterStep'])
        QgsMessageLog.logMessage(message="Reaches created and interpolated : "+str(n_reaches),
                                 tag="CEPHEE", level=Qgis.MessageLevel.Info)

        n_sections = createXSlines(self.BV,self.param['XS']['width'], self.param['XS']['numberOfPoints'])
  
        QgsMessageLog.logMessage(message="Number of sections " + str(n_sections), tag="CEPHEE",
                                 level=Qgis.MessageLevel.Info)
        XS_lines_filepath = path.join(self.param.work_path, 'XS_lines.shp')
        saveXSlines(self.BV,XS_lines_filepath)
        n_layer = QgsVectorLayer(XS_lines_filepath, 'XS lines')
        n_layer.setCrs(QgsCoordinateReferenceSystem(self.BV.crs.to_string()))
        QgsProject.instance().addMapLayer(n_layer)
        QgsMessageLog.logMessage(message="Cross sections created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(100)
        return True

    def finished(self, result):
        if result:
            QgsMessageLog.logMessage(message="Process OK", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        else:
            QgsMessageLog.logMessage(message="Process Error", tag="CEPHEE", level=Qgis.MessageLevel.Critical)


class ImportXS(QgsTask):

    def __init__(self, description, BV, param):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param

    def run(self):
        QgsMessageLog.logMessage(message="Starting Cross Section Process", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(0)
        n_reaches = self.BV.createReach()
        self.BV.interpolateReach(self.param['XS']['reachInterStep'])
        QgsMessageLog.logMessage(message="Reaches created and interpolated : "+str(n_reaches),
                                 tag="CEPHEE", level=Qgis.MessageLevel.Info)

        n_sections = importXSlines(self.BV,self.param,XS_filename=self.param['XS']['XS_filename'] )
            
        QgsMessageLog.logMessage(message="Number of sections " + str(n_sections), tag="CEPHEE",
                                 level=Qgis.MessageLevel.Info)
        XS_lines_filepath = path.join(self.param.work_path, 'XS_lines.shp')
        saveXSlines(self.BV,XS_lines_filepath)
        n_layer = QgsVectorLayer(XS_lines_filepath, 'XS lines')
        n_layer.setCrs(QgsCoordinateReferenceSystem(self.BV.crs.to_string()))
        QgsProject.instance().addMapLayer(n_layer)
        QgsMessageLog.logMessage(message="Cross sections created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(100)
        return True

    def finished(self, result):
        if result:
            QgsMessageLog.logMessage(message="Process OK", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        else:
            QgsMessageLog.logMessage(message="Process Error", tag="CEPHEE", level=Qgis.MessageLevel.Critical)



class ProjectAndInterpolateXS(QgsTask):

    def __init__(self, description, BV, param,flag_import):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param
        self.flag_import = flag_import

    def run(self):
        QgsMessageLog.logMessage(message="Starting Cross Section Process", tag="CEPHEE", level=Qgis.MessageLevel.Info)
    
        if self.param['XS']['optimizeXS'] and not self.param['XS']['interpXS']:
            QgsMessageLog.logMessage(message="Optimize cross sections", tag="CEPHEE", level=Qgis.MessageLevel.Info)
            optimizeXSlines(self.BV)

        if self.flag_import:
            if not self.BV.reach[0].section[0].line.has_z:
                projectionXSline(self.BV,self.param)
        else:
            projectionXSline(self.BV, self.param)
        QgsMessageLog.logMessage(message="Cross sections projeted", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        
        if self.param['XS']['interpXS']:
            QgsMessageLog.logMessage(message="Interpolate cross sections", tag="CEPHEE", level=Qgis.MessageLevel.Info)
            interpolateAllXS(self.BV,self.param)  
            QgsMessageLog.logMessage(message="Cross sections interpolated", tag="CEPHEE", level=Qgis.MessageLevel.Info)
            if self.param['XS']['optimizeXS']:
                QgsMessageLog.logMessage(message="Optimize cross sections", tag="CEPHEE", level=Qgis.MessageLevel.Info)
                optimizeXSlines(self.BV)                
                projectionXSline(self.BV,self.param)

        self.setProgress(50)
        XS_lines_filepath = path.join(self.param.work_path, 'XS_lines_projeted.shp')
        saveXSlines(self.BV,XS_lines_filepath)
        n_layer = QgsVectorLayer(XS_lines_filepath, 'XS lines projeted')
        n_layer.setCrs(QgsCoordinateReferenceSystem(self.BV.crs.to_string()))
        QgsProject.instance().addMapLayer(n_layer)
        self.setProgress(80)
        export_as_csv_for_HECRASgeometry(self.BV, self.param.work_path)
        QgsMessageLog.logMessage(message="Cross sections exported for HECRAS", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        
        self.setProgress(100)
        return True

    def finished(self, result):
        if result:
            QgsMessageLog.logMessage(message="Process OK", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        else:
            QgsMessageLog.logMessage(message="Process Error", tag="CEPHEE", level=Qgis.MessageLevel.Critical)
