######################################################################
#             / ____|  ____|  __ \| |  | |  ____|  ____|             #
#            | |    | |__  | |__) | |__| | |__  | |__                #
#            | |    |  __| |  ___/|  __  |  __| |  __|               #
#            | |____| |____| |    | |  | | |____| |____              #
#             \_____|______|_|    |_|  |_|______|______|             #
######################################################################
#
# CEPHEE
# Copyright (C) 2024 Toulouse INP
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details :
# <http://www.gnu.org/licenses/>.
#
######################################################################

from qgis.PyQt.QtWidgets import QDockWidget, QFileDialog
from qgis.PyQt.QtCore import pyqtSignal
from qgis.core import (QgsVectorLayer, QgsRasterLayer,QgsProject, QgsMapLayerProxyModel, QgsMessageLog, Qgis, QgsTask,
                       QgsApplication, QgsCoordinateReferenceSystem)
from .ui.Ui_HydraulicWidget import Ui_HydraulicWidget
from os import path
from ..core.postCEPHEE import *
from ..core.Hydraulics import *
from ..core.Tools import export_result_as_csv
class HydraulicWidget(QDockWidget, Ui_HydraulicWidget):

    closingWidget = pyqtSignal(name="closingWidget")

    def __init__(self, iface, BV, param, parent = None):
        super(HydraulicWidget, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        self.BV = BV
        self.param = param
        self.task = None
        self.pushButton_compute.clicked.connect(self.compute)
        self.checkBox_useFrictionMap.stateChanged.connect(self.updatePanel)
        self.mMapLayerComboBox.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.checkBox_hydraulicCurve.stateChanged.connect(self.updatePanel)
        self.checkBox_createBanks.stateChanged.connect(self.updatePanel)
        self.comboBox_createBanks.currentIndexChanged.connect(self.updatePanel)
        self.updatePanel()

    def closeEvent(self, event):
        self.closingWidget.emit()
        event.accept()

    def updatePanel(self):
        if self.checkBox_useFrictionMap.isChecked():
            self.mMapLayerComboBox.setEnabled(True)
            #self.comboBox_frictionLaw.setDisabled(True)
            self.lineEdit_frictionValue.setDisabled(True)
        else:
            self.mMapLayerComboBox.setDisabled(True)
            self.comboBox_frictionLaw.setEnabled(True)
            self.lineEdit_frictionValue.setEnabled(True)
        if self.checkBox_hydraulicCurve.isChecked():
            self.lineEdit_dz.setEnabled(True)
            self.lineEdit_hmax.setEnabled(True)
        else:
            self.lineEdit_dz.setDisabled(True)
            self.lineEdit_hmax.setDisabled(True)
        self.lineEdit_hWaterOutlet.setDisabled(True)
        if self.checkBox_createBanks.isChecked():
            self.comboBox_createBanks.setEnabled(True)
            if self.comboBox_createBanks.currentText() == "1D backwater":
                self.lineEdit_hWaterOutlet.setEnabled(True)
            if self.comboBox_createBanks.currentText() == "Constant depth":
                self.lineEdit_hWaterOutlet.setEnabled(True)
        else:
            self.comboBox_createBanks.setDisabled(True)

    def ui_to_param(self):
        self.param['H']['outletDischarge'] = float(self.lineEdit_outletDischarge.text())
        self.param['H']['dischargeMethod'] = self.comboBox_dischargeMethod.currentText()
        self.param['H']['frictionLaw'] = self.comboBox_frictionLaw.currentText()
        self.param['H']['frictionValue'] = float(self.lineEdit_frictionValue.text())

        self.param['H']['dxlat'] = float(self.lineEdit_lateralStep.text())

        if self.checkBox_useFrictionMap.isChecked():
            self.param['H']['frictionMap'] = True
            map_name = self.mMapLayerComboBox.currentLayer()
            self.param['H']['friction_filename'] = map_name.dataProvider().dataSourceUri()

        else:
            self.param['H']['frictionMap'] = False


        if self.checkBox_levee.isChecked():
            self.param['H']['levee'] = True
        else:
            self.param['H']['levee'] = False
           
        if self.checkBox_hydraulicCurve.isChecked():
            self.param['H']['hydraulicCurve'] = True
            self.param['H']['dz'] = float(self.lineEdit_dz.text())
            self.param['H']['hmax'] = float(self.lineEdit_hmax.text())
        else:
            self.param['H']['hydraulicCurve'] = False
            
        if self.comboBox_createBanks.currentText() == "Normal depth":
            self.param['H']['createBanksMethods'] = "Normal"
        elif self.comboBox_createBanks.currentText() == "Critical depth":
            self.param['H']['createBanksMethods'] = "Normal"
        elif self.comboBox_createBanks.currentText() == "1D backwater":
            self.param['H']['createBanksMethods'] = "1D"
        elif self.comboBox_createBanks.currentText() == "Constant depth":
            self.param['H']['createBanksMethods'] = "Himposed"

        self.param['H']['hWaterOutlet'] = float(self.lineEdit_hWaterOutlet.text())
        self.param['H']['himposed'] = float(self.lineEdit_hWaterOutlet.text())

        if self.checkBox_createBanks.isChecked():
            self.param['H']['createBanks'] = True
        else:
            self.param['H']['createBanks'] = False


    def compute(self):
        self.ui_to_param()
        self.task = BackgroundProcess('process_hydraulicComputation', self.BV, self.param)
        QgsApplication.taskManager().addTask(self.task)

class BackgroundProcess(QgsTask):

    def __init__(self, description, BV, param):
        super().__init__(description, QgsTask.CanCancel)
        self.BV = BV
        self.param = param

    def run(self):
        QgsMessageLog.logMessage(message="Starting Hydraulic computation", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        Q_outlet = addQtoSection(self.BV,self.param)
        QgsMessageLog.logMessage(message="Discharge associated to sections", tag="CEPHEE", level=Qgis.MessageLevel.Info)

        setConstantFriction(self.BV, self.param['H']['frictionValue'])
        if self.param['H']['frictionMap']:
            projOnFrictionMap(self.BV, self.param['H']['friction_filename'])
        QgsMessageLog.logMessage(message="friction coefficient associated to sections", tag="CEPHEE", level=Qgis.MessageLevel.Info)

        if self.param['H']['createBanksMethods'] == "Normal":
            computeNormalAndCriticalDepth(self.BV,self.param)
            QgsMessageLog.logMessage(message="normal depths calculation finished", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        elif self.param['H']['createBanksMethods'] == "1D":
            backWaterProfile(self.BV,self.param) 
            QgsMessageLog.logMessage(message="back water calculation finished", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        elif self.param['H']['createBanksMethods'] == "Himposed":
            impose_water_depth(self.BV,self.param)
            QgsMessageLog.logMessage(message="constant water depth calculation finished", tag="CEPHEE", level=Qgis.MessageLevel.Info)

        config_cal = self.param['H']['createBanksMethods']
        export_result_as_csv(self.BV, self.param)

        if self.param['H']['createBanksMethods'] == 'Normal':
            config_cal = config_cal + '_Q' + str(self.param['H']['outletDischarge'])
        elif self.param['H']['createBanksMethods'] == '1D':
            config_cal = config_cal + '_Q' + str(self.param['H']['outletDischarge']) + '_h' + str(self.param['H']['hWaterOutlet'])
        elif self.param['H']['createBanksMethods'] == 'Himposed':
            self.config_cal = config_cal + '_h' + str(self.param['H']['himposed'])


        if self.param['H']['createBanks']:
            createAllBankLines(self.BV,resultType =self.param['H']['createBanksMethods'])
            banks_lines_filepath = path.join(self.param.work_path, 'banks_lines.shp')
            saveBankslines(self.BV,self.param.work_path)
            name_bank ='banks_lines_' + config_cal
            n_layer = QgsVectorLayer(banks_lines_filepath, name_bank)
            n_layer.setCrs(QgsCoordinateReferenceSystem(self.BV.crs.to_string()))
            QgsProject.instance().addMapLayer(n_layer)
            QgsMessageLog.logMessage(message="banks lines created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(25)

        if self.param['H']['hydraulicCurve']:
            addWidthtoSection(self.BV,self.param)
            QgsMessageLog.logMessage(message="hydraulic curves created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
            self.setProgress(50)

        pixel_size =self.param['H']['dxlat']#self.param['C']['resolution']#self.BV.resmin

        data_poly =create_poly_CEPHEE(self.BV,
                                      nx_trans = 0, 
                                      pixel_size=pixel_size,
                                      resultType =self.param['H']['createBanksMethods'] )
        QgsMessageLog.logMessage(message="polygones of results created", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        self.setProgress(75)

        if self.param['H']['mapping_method'] == 'interpolation':
            raster_result_path = interpolate_result_CEPHEE(data_poly, self.param.work_path, config_cal, [pixel_size, pixel_size],
                                                             self.param['C']['DEM_CRS_ID'])
        else:
            raster_result_path = rasterize_poly_CEPHEE(data_poly,
                                                   self.param.work_path,config_cal, 
                                                   [pixel_size,pixel_size],
                                                   self.BV.global_DEM_path)

        depth_layer = QgsRasterLayer(raster_result_path['depth'],path.splitext(path.basename(raster_result_path['depth']))[0])   
        vel_layer = QgsRasterLayer(raster_result_path['velocity'],path.splitext(path.basename(raster_result_path['velocity']))[0])
        WSE_layer = QgsRasterLayer(raster_result_path['WSE'],path.splitext(path.basename(raster_result_path['WSE']))[0])
        QgsProject.instance().addMapLayer(depth_layer)
        QgsProject.instance().addMapLayer(vel_layer)
        QgsProject.instance().addMapLayer(WSE_layer)
        QgsMessageLog.logMessage(message="rasters of results created", tag="CEPHEE", level=Qgis.MessageLevel.Info)

        if os.path.exists(os.path.join(self.param.work_path, 'inline_structure.shp')):
            inline_layer = QgsVectorLayer(path.join(self.param.work_path, 'inline_structure.shp'), 'inline_structure')
            QgsMessageLog.logMessage(message="inline structure detected", tag="CEPHEE",
                                     level=Qgis.MessageLevel.Info)
            QgsProject.instance().addMapLayer(inline_layer)

        if self.param['H']['dischargeMethod'] == 'Per reach':
            if os.path.exists(os.path.join(self.param.work_path, 'discharge.shp')):
                discharge_layer = QgsVectorLayer(path.join(self.param.work_path, 'discharge.shp'), 'discharge')
                QgsMessageLog.logMessage(message="discharge repartition computed", tag="CEPHEE",
                                         level=Qgis.MessageLevel.Info)
                QgsProject.instance().addMapLayer(discharge_layer)
        self.setProgress(100)
        return True

    def finished(self, result):
        if result:
            QgsMessageLog.logMessage(message="Process OK", tag="CEPHEE", level=Qgis.MessageLevel.Info)
        else:
            QgsMessageLog.logMessage(message="Process Error", tag="CEPHEE", level=Qgis.MessageLevel.Critical)
