# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BioDispersal
                                 A QGIS plugin
 Computes ecological continuities based on environments permeability
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2018-04-12
        git sha              : $Format:%H$
        copyright            : (C) 2018 by IRSTEA
        email                : mathieu.chailloux@irstea.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
    
from PyQt5.QtCore import QCoreApplication, QVariant
from PyQt5.QtGui import QIcon
from qgis.core import (Qgis,
                       QgsFields,
                       QgsField,
                       QgsFeature,
                       QgsFeatureRequest,
                       QgsGraduatedSymbolRenderer,
                       QgsProcessing,
                       QgsProcessingUtils,
                       QgsProcessingAlgorithm,
                       QgsProcessingException,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingParameterExtent,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterMatrix,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterExpression,
                       QgsProcessingParameterString,
                       QgsProcessingParameterField,
                       QgsProcessingParameterRange,
                       QgsProcessingParameterVectorDestination,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFile,
                       QgsFeatureSink)

import processing
from processing.algs.gdal.rasterize import rasterize

from ..qgis_lib_mc import utils, qgsUtils, qgsTreatments, feedbacks, styles
from .patch_algs import ExtractPatchesR

class QualifAlgorithm(qgsUtils.BaseProcessingAlgorithm):

    def group(self):
        return self.tr("Patch qualification (vector)")
    def groupId(self):
        return "qualif"
        
class ExtractPatchesV(QualifAlgorithm):

    ALG_NAME = 'extractPatchesV'

    FIELD = 'FIELD'
    VALUES = 'VALUES'
    SURFACE = 'SURFACE'
    DILAT = 'DILAT'
        
    def displayName(self):
        return self.tr('Extract patches (Vector to Vector)')
        
    def shortHelpString(self):
        s = "Extract patches from land use vector layer according to"
        s += " specified land use types and minimum surface."
        s += "\nLand use values are integer separated by semicolons (';')."
        return self.tr(s)
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                description=self.tr('Input layer')))
        self.addParameter(
            QgsProcessingParameterField(
                self.FIELD,
                description=self.tr('Field'),
                defaultValue=None,
                type=QgsProcessingParameterField.Numeric,
                parentLayerParameterName=self.INPUT,
                optional=True))
        self.addParameter(
            QgsProcessingParameterString (
                self.VALUES,
                description=self.tr('Field values (separated by \';\')'),
                optional=True))
        self.addParameter(
            QgsProcessingParameterNumber (
                self.SURFACE,
                description=self.tr('Patch minimum surface (square meters)'),
                type=QgsProcessingParameterNumber.Double,
                optional=True))
        self.addParameter(
            QgsProcessingParameterNumber (
                self.DILAT,
                description=self.tr('Expansion erosion buffer value'),
                type=QgsProcessingParameterNumber.Integer,
                optional=True))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parse params
        input_source = self.parameterAsSource(parameters,self.INPUT,context)
        input = input_source.materialize(QgsFeatureRequest(),feedback=feedback)
        if input is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
        values = self.parameterAsInts(parameters,self.VALUES,context)
        surface = self.parameterAsDouble(parameters,self.SURFACE,context)
        dilat = self.parameterAsDouble(parameters,self.DILAT,context)
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        mf = QgsProcessingMultiStepFeedback(6,feedback)
        # Extract values
        fieldname = self.parameterAsString(parameters,self.FIELD,context)
        if fieldname:
            values = self.parameterAsInts(parameters,self.VALUES,context)
            values_str = [ str(v) for v in values ]
            values_expr = ",".join(values_str)
            extract_expr = "{} in ({})".format(fieldname,values_expr)
            extracted = self.mkTmpPath("extracted.gpkg")
            qgsTreatments.extractByExpression(input,extract_expr,extracted,feedback=mf)
        else:
            extracted = input
        mf.setCurrentStep(1)
        # Expansion
        if dilat:
            expanded = self.mkTmpPath("expanded.gpkg")
            qgsTreatments.applyBufferFromExpr(extracted,dilat,expanded,feedback=mf)
        else:
            expanded = extracted
        mf.setCurrentStep(2)
        # Dissolve
        dissolved = self.mkTmpPath("dissolved.gpkg")
        qgsTreatments.dissolveLayer(expanded,dissolved,feedback=mf)
        mf.setCurrentStep(3)
        # dilatation
        if dilat:
            eroded = self.mkTmpPath("eroded.gpkg")
            qgsTreatments.applyBufferFromExpr(dissolved,-dilat,eroded,feedback=mf)
        else:
            eroded = dissolved
        mf.setCurrentStep(4)
        # Mutli to single geom
        singleGeom = self.mkTmpPath("singleGeom.gpkg")
        qgsTreatments.multiToSingleGeom(eroded,singleGeom,feedback=mf)
        mf.setCurrentStep(5)
        # Filter surface
        surface_expr = "$area > {}".format(surface)
        qgsTreatments.extractByExpression(singleGeom,surface_expr,output,feedback=mf)
        mf.setCurrentStep(6)
        return { 'OUTPUT' : output }
        
class ExtractPatchesRV(QualifAlgorithm):

    ALG_NAME = 'extractPatchesRV'

    VALUES = 'VALUES'
    SURFACE = 'SURFACE'
        
    def displayName(self):
        return self.tr('Extract patches (Raster to Vector)')
        
    def shortHelpString(self):
        s = "Extract patches from land use raster layer according to"
        s += " specified land use types and minimum surface."
        s += "\nLand use values are integer separated by semicolons (';')."
        return self.tr(s)
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                description=self.tr('Input layer')))
        self.addParameter(
            QgsProcessingParameterString (
                self.VALUES,
                description=self.tr('Land use values (separated by \';\')')))
        self.addParameter(
            QgsProcessingParameterNumber (
                self.SURFACE,
                description=self.tr('Patch minimum surface (square meters)'),
                type=QgsProcessingParameterNumber.Double,
                optional=True))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parse params
        surface = self.parameterAsDouble(parameters,self.SURFACE,context)
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        mf = QgsProcessingMultiStepFeedback(4,feedback)
        # Extract patches R
        extract_params = dict(parameters)
        patchR_path = self.mkTmpPath("patchesR.tif")
        extract_params[self.SURFACE] = None
        extract_params[self.OUTPUT] = patchR_path
        processing.run("BioDispersal:" + ExtractPatchesR.ALG_NAME,
            extract_params,context=context,feedback=mf)
        mf.setCurrentStep(1)
        # Polygonize
        patchV_path = self.mkTmpPath("patchesV.gpkg")
        qgsTreatments.applyPolygonizeFieldOpt(patchR_path,patchV_path,feedback=mf)
        mf.setCurrentStep(2)
        # Filter by surface
        expr = "$area >= " + str(surface)
        area_path = self.mkTmpPath("area.gpkg")
        qgsTreatments.extractByExpression(patchV_path,expr,area_path,feedback=mf)
        mf.setCurrentStep(3)
        # Filter by surface
        fieldname = "join_index"
        qgsTreatments.addIdxField(area_path,fieldname,output,feedback=mf)
        mf.setCurrentStep(4)
        # return
        return { 'OUTPUT' : output }


class QualifAlgClassif(QualifAlgorithm):
    # Apply Jenks classif on self.output self.fieldname
    def postProcessAlgorithm(self,context,feedback):
        out_layer = QgsProcessingUtils.mapLayerFromString(self.output,context)
        if not out_layer:
            raise QgsProcessingException("No layer found for " + str(self.output))
        styles.setRdYlGnGraduatedStyle(out_layer,self.fieldname)
        return {self.OUTPUT: self.output }
        
class QualifAlg1Layer(QualifAlgClassif):
    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                description=self.tr('Patch layer')))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        input = self.parameterAsVectorLayer(parameters,self.INPUT,context)
        if input is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        # Processing
        

class QualifAlg2Layers(QualifAlgClassif):
    LAYER_A = 'LAYER_A'
    LAYER_B = 'LAYER_B'
    VALUES = 'VALUES'
    OUT_FIELDNAME = 'OUT_FIELDNAME'
    
    def initLayerAV(self):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.LAYER_A,
                description=self.tr('Patch layer')))
    def initLayerBV(self):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.LAYER_B,
                description=self.tr('Layer B (relative surface layer)')))
    def initLayerBR(self):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.LAYER_B,
                description=self.tr('Layer B (relative surface layer)')))
    def initOutLayerV(self):
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
    def initOutLayerR(self):
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.tr("Output layer")))
    def initValues(self):
        self.addParameter(
            QgsProcessingParameterString (
                self.VALUES,
                optional=True,
                description=self.tr('Values (separated by \';\', all values by default)')))
    def initOutFieldname(self):
        self.addParameter(
            QgsProcessingParameterString (
                self.OUT_FIELDNAME,
                optional=True,
                description=self.tr('Output fieldname')))
        

class QualifAlgVV(QualifAlg2Layers):
    def initAlgorithm(self, config=None):
        self.initLayerAV()
        self.initLayerBV()
        self.initOutLayerV()
class QualifAlgVR(QualifAlg2Layers):
    def initAlgorithm(self, config=None):
        self.initLayerAV()
        self.initLayerBR()
        self.initValues()
        self.initOutLayerV()
        self.initOutFieldname()
                  
# Relative surface with patch vector and landuse raster
class RelativeSurfaceVR(QualifAlgVR):

    ALG_NAME = 'relativeSurfaceVR'
    
    NODATA_FLAG = 'NODATA_FLAG'

    def displayName(self):
        return self.tr("Relative surface")
    def shortHelpString(self):
        return self.tr("Relative surface (percentage of B surface in each patch of layer A)")
        
    def initAlgorithm(self, config=None):
        super().initAlgorithm()
        self.addParameter(
            QgsProcessingParameterBoolean(
            self.NODATA_FLAG,
            description = self.tr('Include NoData pixels in pixel count'),
            defaultValue=False))
        
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        layerA = self.parameterAsVectorLayer(parameters,self.LAYER_A,context)
        if layerA is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        layerB = self.parameterAsRasterLayer(parameters,self.LAYER_B,context)
        if layerB is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_B))
        values = self.parameterAsString(parameters,self.VALUES,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        nodataFlag = self.parameterAsBool(parameters,self.NODATA_FLAG,context)
        out_fieldname = self.parameterAsString(parameters,self.OUT_FIELDNAME,context)
        mf = QgsProcessingMultiStepFeedback(4,feedback)
        # Values extraction
        if values:
            values = self.parameterAsInts(parameters,self.VALUES,context)
            extract_path = self.mkTmpPath("extract.tif")
            extract_params = { ExtractPatchesR.INPUT : parameters[self.LAYER_B],
                ExtractPatchesR.VALUES : parameters[self.VALUES],
                ExtractPatchesR.OUTPUT : extract_path }
            processing.run("BioDispersal:" + ExtractPatchesR.ALG_NAME,
                extract_params,context=context,feedback=mf)
            # layerB = qgsUtils.loadRasterLayer(extract_path) 
        else:
            extract_path = layerB
        mf.setCurrentStep(1)
        # Prepare nodata pixels if needed
        if nodataFlag:
            nodata_path = qgsUtils.mkTmpPath("nodata.tif")
            #nodata_val = layerB.dataProvider().sourceNoDataValue(1)
            qgsTreatments.applyRNull(layerB,0,nodata_path,context=context,feedback=mf)
        else:
            nodata_path = layerB
        # Zonal stats (total count)
        # Stats 0 = count
        prefixC = "count_"
        count_path = qgsUtils.mkTmpPath("count.gpkg")
        qgsTreatments.rasterZonalStats(layerA,nodata_path,count_path,
            prefix=prefixC,stats=[0],feedback=mf) 
        mf.setCurrentStep(2)
        # Zonal stats
        # Stats 0 = count
        values_str = "".join(str(v) for v in values)
        prefixV = values_str + "_"
        stats_path = qgsUtils.mkTmpPath("stats.gpkg")
        qgsTreatments.rasterZonalStats(count_path,extract_path,stats_path,
            prefix=prefixV,stats=[0],feedback=mf) 
        mf.setCurrentStep(3)
        # Compute relative surface
        # x_res = layerB.rasterUnitsPerPixelX()
        # y_res = layerB.rasterUnitsPerPixelY()
        # pixel_surf = x_res * y_res
        # feedback.pushDebugInfo("pixel_surf = " + str(pixel_surf))
        expr = '"%scount" / "%scount"' % (prefixV,prefixC)
        mf.pushDebugInfo("expr = " + expr)
        self.fieldname = out_fieldname if out_fieldname else "SurfRel_" + values_str
        # surf_path = qgsUtils.mkTmpPath("surface_relative.gpkg")
        qgsTreatments.fieldCalculator(stats_path,self.fieldname,expr,self.output,
            precision=4,feedback=mf)
        mf.setCurrentStep(4)
        # Apply classif
        return { self.OUTPUT : self.output }
        
    def processAlgorithmOld(self,parameters,context,feedback):
        # Parameters
        layerA = self.parameterAsVectorLayer(parameters,self.LAYER_A,context)
        if layerA is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        layerB = self.parameterAsRasterLayer(parameters,self.LAYER_B,context)
        if layerB is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_B))
        values = self.parameterAsString(parameters,self.VALUES,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        # Values extraction
        if values:
            values = self.parameterAsInts(parameters,self.VALUES,context)
            extract_path = self.mkTmpPath("extract.tif")
            extract_params = { ExtractPatchesR.INPUT : parameters[self.LAYER_B],
                ExtractPatchesR.VALUES : parameters[self.VALUES],
                ExtractPatchesR.OUTPUT : extract_path }
            processing.run("BioDispersal:" + ExtractPatchesR.ALG_NAME,
                extract_params,context=context,feedback=feedback)
            # layerB = qgsUtils.loadRasterLayer(extract_path) 
        else:
            extract_path = layerB
        # Zonal stats
        values_str = "".join(str(v) for v in values)
        prefix = values_str + "_"
        stats_path = qgsUtils.mkTmpPath("stats.gpkg")
        # Stats = 0 <=> count
        qgsTreatments.rasterZonalStats(layerA,extract_path,stats_path,
            prefix=prefix,stats=[0],feedback=feedback) 
        # Compute relative surface
        x_res = layerB.rasterUnitsPerPixelX()
        y_res = layerB.rasterUnitsPerPixelY()
        pixel_surf = x_res * y_res
        feedback.pushDebugInfo("pixel_surf = " + str(pixel_surf))
        expr = '("%scount" * %d) / $area' % (prefix,pixel_surf)
        feedback.pushDebugInfo("expr = " + expr)
        self.fieldname = "SurfRel_" + values_str
        # surf_path = qgsUtils.mkTmpPath("surface_relative.gpkg")
        qgsTreatments.fieldCalculator(stats_path,self.fieldname,expr,self.output,
            precision=4,feedback=feedback)
        # Apply classif
        return { self.OUTPUT : self.output }
        
    def postProcessAlgorithm(self,context,feedback):
        out_layer = QgsProcessingUtils.mapLayerFromString(self.output,context)
        if not out_layer:
            raise QgsProcessingException("No layer found for " + str(self.output))
        styles.setRdYlGnGraduatedStyle(out_layer,self.fieldname)
        return {self.OUTPUT: self.output }
          
          
# Relative surface with patch vector and landuse raster
class ReachableSurface(QualifAlgVR):

    ALG_NAME = 'reachableSurface'
    
    BUFFER = 'BUFFER'
    NODATA_FLAG = 'NODATA_FLAG'
    JOIN_FIELDNAME = 'JOIN_FIELDNAME'

    def displayName(self):
        return self.tr("Reachable surface")
    def shortHelpString(self):
        return self.tr("Reachable surface")
        
    def initAlgorithm(self, config=None):
        super().initAlgorithm()
        self.addParameter(
            QgsProcessingParameterNumber (
                self.BUFFER,
                description=self.tr('Euffer value'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=100))
        self.addParameter(
            QgsProcessingParameterBoolean(
            self.NODATA_FLAG,
            description = self.tr('Include NoData pixels in pixel count'),
            defaultValue=False))
        self.addParameter(
            QgsProcessingParameterField(
                self.JOIN_FIELDNAME,
                description=self.tr('Fieldname used to join layers (join by location otherwise)'),
                parentLayerParameterName=self.LAYER_A,
                optional=True))
        
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        layerA = self.parameterAsVectorLayer(parameters,self.LAYER_A,context)
        if layerA is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        layerB = self.parameterAsRasterLayer(parameters,self.LAYER_B,context)
        if layerB is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_B))
        values = self.parameterAsString(parameters,self.VALUES,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        buffer = self.parameterAsDouble(parameters,self.BUFFER,context)
        nodataFlag = self.parameterAsBool(parameters,self.NODATA_FLAG,context)
        join_fieldname = self.parameterAsString(parameters,self.JOIN_FIELDNAME,context)
        out_fieldname = self.parameterAsString(parameters,self.OUT_FIELDNAME,context)
        mf = QgsProcessingMultiStepFeedback(8,feedback)
        # Values extraction
        if values:
            values = self.parameterAsInts(parameters,self.VALUES,context)
            extract_path = self.mkTmpPath("extract.tif")
            extract_params = { ExtractPatchesR.INPUT : parameters[self.LAYER_B],
                ExtractPatchesR.VALUES : parameters[self.VALUES],
                ExtractPatchesR.OUTPUT : extract_path }
            processing.run("BioDispersal:" + ExtractPatchesR.ALG_NAME,
                extract_params,context=context,feedback=mf)
        else:
            extract_path = layerB
        mf.setCurrentStep(1)
        # Prepare nodata pixels if needed
        if nodataFlag:
            nodata_path = qgsUtils.mkTmpPath("nodata.tif")
            qgsTreatments.applyRNull(layerB,0,nodata_path,context=context,feedback=mf)
        else:
            nodata_path = layerB
        mf.setCurrentStep(2)
        # Zonal stats (init count) - Stats 0 = count
        prefixCountInit = "init_"
        countInitField = prefixCountInit + "count"
        count_init_path = qgsUtils.mkTmpPath("countInit.gpkg")
        if qgsUtils.hasPolygonGeometry(layerA):
            qgsTreatments.rasterZonalStats(layerA,layerB,count_init_path,
                prefix=prefixCountInit,stats=[0],feedback=mf)
        else:
            qgsTreatments.fieldCalculator(layerA,countInitField,"0",count_init_path,
                precision=4,feedback=mf)
        mf.setCurrentStep(3)
        # Buffer
        expanded = self.mkTmpPath("expanded.gpkg")
        qgsTreatments.applyBufferFromExpr(count_init_path,buffer,expanded,feedback=mf)
        mf.setCurrentStep(4)
        # Zonal stats (total count) - Stats 0 = count
        prefixCountNew = "new_"
        countNewField = prefixCountNew + "count"
        countNew_path = qgsUtils.mkTmpPath("countNew.gpkg")
        qgsTreatments.rasterZonalStats(expanded,nodata_path,countNew_path,
            prefix=prefixCountNew,stats=[0],feedback=mf) 
        mf.setCurrentStep(5)
        # Zonal stats (values count) - Stats 0 = count
        prefixCountValues = "values_"
        countValField = prefixCountValues + "count"
        countVal_path = qgsUtils.mkTmpPath("countVal.gpkg")
        qgsTreatments.rasterZonalStats(countNew_path,extract_path,countVal_path,
            prefix=prefixCountValues,stats=[0],feedback=mf) 
        mf.setCurrentStep(6)
        # Compute results
        # expr = '"{}" / ("{}" - "{}")'.format(countNewField,countValField,countInitField)
        expr = '"{}" / ("{}" - "{}")'.format(countValField,countNewField,countInitField)
        mf.pushDebugInfo("expr = " + expr)
        if out_fieldname:
            self.fieldname = out_fieldname
        else:
            values_str = "".join(str(v) for v in values)
            prefixV = values_str + "_"
            self.fieldname = "ReachSurf_" + values_str
        surfPath = qgsUtils.mkTmpPath("surf.gpkg")
        qgsTreatments.fieldCalculator(countVal_path,self.fieldname,expr,surfPath,
            precision=4,feedback=mf)
        mf.setCurrentStep(7)
        # Join on original geom
        fields = [countInitField,countNewField,countValField,self.fieldname]
        if join_fieldname:
            qgsTreatments.joinByAttribute(layerA,join_fieldname,surfPath,
                join_fieldname,self.output,copy_fields=fields,feedback=mf)
        else:
            qgsTreatments.joinByLoc(layerA,surfPath,out_path=self.output,
                method=1,predicates=[5],fields=fields,feedback=mf)
        mf.setCurrentStep(8)
        return { self.OUTPUT : self.output }
        
    def postProcessAlgorithm(self,context,feedback):
        out_layer = QgsProcessingUtils.mapLayerFromString(self.output,context)
        if not out_layer:
            raise QgsProcessingException("No layer found for " + str(self.output))
        styles.setRdYlGnGraduatedStyle(out_layer,self.fieldname)
        return {self.OUTPUT: self.output }
    
class CompactnessAlg(QualifAlg1Layer):

    ALG_NAME = 'compactness'   
    def displayName(self):
        return self.tr("Compactness")
    def shortHelpString(self):
        return self.tr("Computes compactness index (area / perimeter) for each feature of input layer")
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        input = self.parameterAsVectorLayer(parameters,self.INPUT,context)
        if input is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        # Processing
        # Compactness could be based on minimum enclosing circles 
        expr = "$area / $perimeter"
        expr = "$area / ($perimeter * $perimeter)"
        expr = "$area / ($perimeter * $perimeter / (4*pi()))"
        self.fieldname = "compactness"
        qgsTreatments.fieldCalculator(input,self.fieldname,expr,self.output,
            context=context,feedback=feedback)
        return { self.OUTPUT : self.output }
    

class DistanceAlg(QualifAlgVR):

    ALG_NAME = 'distanceAlgVR'

    def displayName(self):
        return self.tr("Distance to layer")
    def shortHelpString(self):
        return self.tr("Computes minimal distance to layer B pixel for each feature of layer A")
        
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        layerA = self.parameterAsVectorLayer(parameters,self.LAYER_A,context)
        if layerA is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_A))
        layerB = self.parameterAsRasterLayer(parameters,self.LAYER_B,context)
        if layerB is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LAYER_B))
        values = self.parameterAsString(parameters,self.VALUES,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        out_fieldname = self.parameterAsString(parameters,self.OUT_FIELDNAME,context)
        mf = QgsProcessingMultiStepFeedback(3,feedback)
        # Values extraction
        if values:
            values = self.parameterAsInts(parameters,self.VALUES,context)
            extract_path = self.mkTmpPath("extract.tif")
            extract_params = { ExtractPatchesR.INPUT : parameters[self.LAYER_B],
                ExtractPatchesR.VALUES : parameters[self.VALUES],
                ExtractPatchesR.OUTPUT : extract_path }
            processing.run("BioDispersal:" + ExtractPatchesR.ALG_NAME,
                extract_params,context=context,feedback=mf)
            # layerB = qgsUtils.loadRasterLayer(extract_path) 
        else:
            extract_path = layerB
        mf.setCurrentStep(1)
        # Distance
        distance_path = qgsUtils.mkTmpPath("distance.tif")
        qgsTreatments.applyProximity(extract_path,distance_path,feedback=mf)
        mf.setCurrentStep(2)
        # Zonal stats
        if out_fieldname:
            prefix = out_fieldname + "_"
        else:
            values_str = "".join(str(v) for v in values)
            prefix = values_str + "_dist_"
        self.fieldname = prefix + "min"
        # Stats = 5 <=> minimum
        qgsTreatments.rasterZonalStats(layerA,distance_path,self.output,
            prefix=prefix,stats=[5],feedback=mf) 
        mf.setCurrentStep(3)
        return { self.OUTPUT : self.output }
        
    def postProcessAlgorithm(self,context,feedback):
        out_layer = QgsProcessingUtils.mapLayerFromString(self.output,context)
        if not out_layer:
            raise QgsProcessingException("No layer found for " + str(self.output))
        styles.setRdYlGnGraduatedStyle(out_layer,self.fieldname,
            invert_ramp=True,invert_ranges=True)
        return {self.OUTPUT: self.output }

# Computes Shannon Diversity Index
class ShannonDiversityIndex(QualifAlgClassif):

    ALG_NAME = 'shannonIndex'

    PATCHES = 'PATCHES'
    LANDUSE = 'LANDUSE'
    
    def displayName(self):
        return self.tr("Shannon index")
    def shortHelpString(self):
        return self.tr("Creates field with classification value from active symbology")
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PATCHES,
                description=self.tr('Patch vector layer')))
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.LANDUSE,
                description=self.tr('Landuse raster layer')))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        patches = self.parameterAsVectorLayer(parameters,self.PATCHES,context)
        if patches is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.PATCHES))
        landuse = self.parameterAsRasterLayer(parameters,self.LANDUSE,context)
        if landuse is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.LANDUSE))
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        self.fieldname = "shannon"
        mf = QgsProcessingMultiStepFeedback(2,feedback)
        # Computes zonal histo
        hist_path = qgsUtils.mkTmpPath("zonal_histo.gpkg")
        qgsTreatments.zonalHisto(patches,landuse,hist_path,feedback=mf)
        mf.setCurrentStep(1)
        hist_layer = qgsUtils.loadVectorLayer(hist_path)
        # Computes index
        prefix = "HISTO_"
        fields_histo = [f.name() for f in hist_layer.fields() if f.name().startswith(prefix)]
        tot_expr = " + ".join(fields_histo)
        fields_expr = ['if ({0} > 0, {0} / ({1}) * ln({0} / ({1})) * -1, 0)'.format(fh,tot_expr) for fh in fields_histo]
        expr = " + ".join(fields_expr)
        qgsTreatments.fieldCalculator(hist_path,self.fieldname,expr,self.output,feedback=mf)
        mf.setCurrentStep(2)
        return { self.OUTPUT: self.output }

class PatchAreaV(QualifAlgClassif):

    ALG_NAME = 'patchAreaV'
    PATCHES = 'PATCHES'
    
    def displayName(self):
        return self.tr("Patch area")
    def shortHelpString(self):
        return self.tr("Computes patch area")
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PATCHES,
                description=self.tr('Patch vector layer')))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))

    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        patches = self.parameterAsVectorLayer(parameters,self.PATCHES,context)
        if patches is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.PATCHES))
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        self.fieldname = "area"
        # Computes area
        expr = "$area"
        qgsTreatments.fieldCalculator(patches,self.fieldname,expr,self.output,feedback=feedback)
        return { self.OUTPUT: self.output }
        
  
class ClassifySymbology(QualifAlgClassif):

    ALG_NAME = 'classifySymbology'
    
    PREFIX = 'PREFIX'
    FIELD_PREFIX = 'classif_'

    def displayName(self):
        return self.tr("Classify from symbology")
    def shortHelpString(self):
        return self.tr("Creates field with classification value from active symbology")
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                description=self.tr('Input layer')))
        self.addParameter(
            QgsProcessingParameterString(
                self.PREFIX,
                description=self.tr('Prefix for output fieldname'),
                defaultValue=self.FIELD_PREFIX))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
    
    def classifyFromSymbology(self,layer,prefix,output,feedback):
        renderer = layer.renderer()
        feedback.pushDebugInfo("Rendered class = " + str(renderer.__class__.__name__))
        if isinstance(renderer,QgsGraduatedSymbolRenderer):
            ranges = renderer.ranges()
            field_r = renderer.classAttribute()
            self.fieldname = prefix + field_r
            copy_fields = [field_r,self.fieldname]
            feedback.pushDebugInfo("out_fname = " + str(self.fieldname))
            # Computes formula
            formula = ''
            for idx, r in enumerate(ranges):
                feedback.pushDebugInfo("formula = " + str(formula))
                lowerVal = r.lowerValue()
                upperVal = r.upperValue()
                formula += 'if (({1} <= "{0}") and ({2} > "{0}"), {3}, '.format(field_r,lowerVal,upperVal,idx)
            formula += ' ' + str(len(ranges) - 1) + (')' * len(ranges))
            feedback.pushDebugInfo("formula2 = " + str(formula))
            # Apply classification
            qgsTreatments.fieldCalculator(layer,self.fieldname,formula,output,feedback=feedback)
            return copy_fields
        else:
            raise QgsProcessingException(self.tr("Input layer renderer is not graduated"))
        
    
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        layer = self.parameterAsLayer(parameters,self.INPUT,context)
        # source = self.parameterAsSource(parameters,self.INPUT,context)
        if layer is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))
        out_prefix = self.parameterAsString(parameters,self.PREFIX,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        # Retrieve symbology classes
        self.classifyFromSymbology(layer,out_prefix,self.output,feedback)
        return { self.OUTPUT: self.output }
        
        
class AgregateCriterias(ClassifySymbology):

    ALG_NAME = 'agregateQualif'
    
    INPUTS = 'INPUTS'
    JOIN_FIELDNAME = 'JOIN_FIELDNAME'
    AGR_FIELDNAME = 'AGR_FIELDNAME'
    #AREA_FIELDNAME = 'AREA_FIELDNAME'

    def displayName(self):
        return self.tr("Agregate criterias")
    def shortHelpString(self):
        str = self.tr("Agregate qualification layers according to current renderers.")
        str += self.tr("Each rendering class is replaced by ")
        return str
        
    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.INPUTS,
                description=self.tr('Input layers'),
                layerType=QgsProcessing.TypeVector))
        self.addParameter(
            QgsProcessingParameterString(
                self.PREFIX,
                description=self.tr('Prefix for output classification fieldnames'),
                defaultValue=self.FIELD_PREFIX))
        self.addParameter(
            QgsProcessingParameterString(
                self.JOIN_FIELDNAME,
                description=self.tr('Fieldname used to join layers (join by location otherwise)'),
                optional=True))
        self.addParameter(
            QgsProcessingParameterString(
                self.AGR_FIELDNAME,
                description=self.tr('Output qualification fieldname'),
                defaultValue='QUALIF'))
        # self.addParameter(
        #     QgsProcessingParameterString(
        #         self.AREA_FIELDNAME,
        #         description=self.tr('Area fieldname'),
        #         optional=True))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Output layer")))
    
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        input_layers = self.parameterAsLayerList(parameters,self.INPUTS,context)
        if not input_layers:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUTS))
        join_fieldname = self.parameterAsString(parameters,self.JOIN_FIELDNAME,context)
        agr_fieldname = self.parameterAsString(parameters,self.AGR_FIELDNAME,context)
        #area_fieldname = self.parameterAsString(parameters,self.AREA_FIELDNAME,context)
        out_prefix = self.parameterAsString(parameters,self.PREFIX,context)
        self.output = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)
        # Feedback
        nb_layers = len(input_layers)
        nb_steps = nb_layers * 2 + 1#(2 if area_fieldname else 1) 
        mf = QgsProcessingMultiStepFeedback(nb_steps,feedback)
        # Classification loop on each layer
        formula = ""
        for cpt, layer in enumerate(input_layers):
            mf.pushDebugInfo("cpt = {}".format(cpt))
            classified = qgsUtils.mkTmpPath("classified{}.gpkg".format(cpt))
            copy_fields = self.classifyFromSymbology(layer,out_prefix,classified,mf)
            mf.setCurrentStep(cpt * 2 + 1)
            if cpt > 0:
                # Join classif with previous loop output
                joined = qgsUtils.mkTmpPath("joined{}.gpkg".format(cpt))
                if join_fieldname:
                    qgsTreatments.joinByAttribute(previous,join_fieldname,classified,
                        join_fieldname,joined,copy_fields=copy_fields,feedback=mf)
                else:
                    qgsTreatments.joinByLoc(previous,classified,out_path=joined,
                        fields=copy_fields,feedback=mf)
                previous = joined
            else:
                previous = classified
            mf.setCurrentStep(cpt * 2 + 2)
            # Build formula
            formula += '\"' + self.fieldname + '\"'
            if cpt < nb_layers - 1:
                formula += " + "
        # Computes area
        # if area_fieldname:
        #     area_path = qgsUtils.mkTmpPath("area.gpkg")
        #     qgsTreatments.fieldCalculator(joined,area_fieldname,"$area",area_path,feedback=mf)
        # else:
        #     area_path = joined
        # mf.setCurrentStep(nb_layers * 2 + 1)
        # Build output layer
        mf.pushDebugInfo("formula = {}".format(formula))
        #qgsTreatments.fieldCalculator(area_path,agr_fieldname,formula,self.output,feedback=mf)
        qgsTreatments.fieldCalculator(joined,agr_fieldname,formula,self.output,feedback=mf)
        self.fieldname = agr_fieldname
        mf.setCurrentStep(nb_layers * 2 + 1)#2)
        return { self.OUTPUT: self.output }
        
        
    # Apply unique values renderer
    def postProcessAlgorithm(self,context,feedback):
        out_layer = QgsProcessingUtils.mapLayerFromString(self.output,context)
        if not out_layer:
            raise QgsProcessingException("No layer found for " + str(self.output))
        styles.setRendererUniqueValues(out_layer,self.fieldname)
        return {self.OUTPUT: self.output }
