# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BioDispersal
                                 A QGIS plugin
 Computes ecological continuities based on environments permeability
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2018-04-12
        git sha              : $Format:%H$
        copyright            : (C) 2018 by IRSTEA
        email                : mathieu.chailloux@irstea.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import numpy as np
    
from PyQt5.QtCore import QCoreApplication, QVariant
from PyQt5.QtGui import QIcon
from qgis.core import (Qgis,
                       QgsField,
                       QgsProcessing,
                       QgsProcessingUtils,
                       QgsProcessingAlgorithm,
                       QgsProcessingException,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingParameterExtent,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterMatrix,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterExpression,
                       QgsProcessingParameterString,
                       QgsProcessingParameterField,
                       QgsProcessingParameterRange,
                       QgsProcessingParameterVectorDestination,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFile)

import processing
from processing.algs.gdal.rasterize import rasterize

from ..qgis_lib_mc import utils, qgsUtils, qgsTreatments, feedbacks, styles

class CircuitscapeAlgorithm(qgsUtils.BaseProcessingAlgorithm):
    def tr(self, string):
       return QCoreApplication.translate(self.__class__.__name__, string)
    def group(self):
        return self.tr("Circuitscape")
    def groupId(self):
        return 'circuitscape'
   
    
class ASCIIOutput(QgsProcessingParameterRasterDestination):
    def __init__(self, name, description):
        super().__init__(name, description)
    def defaultFileExtension(self):
        return 'asc'
    def supportedOutputRasterLayerExtensions(self):
        return ['asc']
    
    
class ExportPatchesToCircuitscape(CircuitscapeAlgorithm):

    ALG_NAME = 'exportpatchestocircuitscape'
    
    CLASS = 'CLASS'
        
    def displayName(self):
        return self.tr('Export to Circuitscape (start points)')
        
    def shortHelpString(self):
        return self.tr('Export patch layer (focal nodes, biodiversity reservois, ...) to Circuitscape')

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                description=self.tr('Input patch layer')))
        self.addParameter(QgsProcessingParameterNumber(
            self.CLASS, "Choose Landscape Class", type=QgsProcessingParameterNumber.Integer,
            defaultValue=None,optional=True))
        self.addParameter(
            ASCIIOutput(
                self.OUTPUT,
                self.tr("Exported layer (start points)")))
                
    def processAlgorithm(self,parameters,context,feedback):
        if not utils.scipyIsInstalled():
            msg = "Scipy (python library) import failed. You can install it through OSGEO installer"
            raise QgsProcessingException(msg)
        import scipy
        input = self.parameterAsRasterLayer(parameters,self.INPUT,context)
        if input is None:
            raise QgsProcessingException(self.invalidRasterError(parameters, self.INPUT))
        cl = self.parameterAsInt(parameters, self.CLASS, context)
        cl_param = parameters[self.CLASS]
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
                
        input_filename = input.source()
        input_nodata_val = input.dataProvider().sourceNoDataValue(1)
        feedback.pushDebugInfo("Input NoData value = " + str(input_nodata_val))
        if input_nodata_val == 1:
            raise QgsProcessingException("Input NoData value cannot be equal to 1.")
        input_type = input.dataProvider().dataType(1)
        feedback.pushDebugInfo("Input type = " + str(input_type))
        # if 0 in input_vals:
            # raise QgsProcessingException("Input layer contains 0 value")
        feedback.pushDebugInfo("output = " + str(output))
        output_basename, ext = os.path.splitext(output)
        output_tif = output_basename + ".tif"
                
        classes, array = qgsUtils.getRasterValsAndArray(str(input_filename))
        new_array = np.copy(array)
        feedback.pushDebugInfo("class = " + str(cl) + ", " + str(parameters[self.CLASS]))
        feedback.pushDebugInfo("array = " + str(array[0][0]))
        feedback.pushDebugInfo("new_array = " + str(new_array[0][0]))
        if cl_param is None:
            new_array[array!=input_nodata_val] = 1
            new_array[array==input_nodata_val] = 0
            new_array[array!=array] = 0
        elif cl not in classes:
            raise QgsProcessingException("No pixel found with class value " + str(cl))
        else:
            new_array[new_array!=cl] = 0
            new_array[array==cl] = 1
        struct = scipy.ndimage.generate_binary_structure(2,2)
        labeled_array, nb_patches = scipy.ndimage.label(new_array,struct)
        labeled_array[labeled_array==0] = input_nodata_val
        
        qgsUtils.exportRaster(labeled_array,input_filename,output_tif,
            nodata=input_nodata_val,type=input_type)
        out = qgsTreatments.applyTranslate(output_tif,output,data_type=input_type,
            nodata_val=input_nodata_val,context=context,feedback=feedback)
        # Data type coule be problematic if input layer has small type 
        # (suchs as Byte but BioDispersal exports Float32) and lot of patches.
        return {'OUTPUT' : out }
      
      
class ExportFrictionToCircuitscape(CircuitscapeAlgorithm):

    ALG_NAME = 'exportfrictiontocircuitscape'
        
    def displayName(self):
        return self.tr('Export to Circuitscape (friction layer)')
        
    def shortHelpString(self):
        return self.tr('Export friction (resistance) layer to Circuitscape, converting existing layer to ASCII format.')

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                description=self.tr('Input resistance layer')))
        self.addParameter(
            ASCIIOutput(
                self.OUTPUT,
                self.tr("Exported layer (resistance)")))
                
    def processAlgorithm(self,parameters,context,feedback):    
        input = self.parameterAsRasterLayer(parameters,self.INPUT,context)
        if input is None:
            raise QgsProcessingException(self.invalidRasterError(parameters, self.INPUT))
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)        
                
        if output is None:
            raise QgsProcessingException("Empty output layer")
        base, ext = os.path.splitext(output)
        if ext != '.asc':
            raise QgsProcessingException("Invalid extension for output ASCII file : " + str(output))
        input_filename = input.source()
        input_nodata_val = input.dataProvider().sourceNoDataValue(1)
        feedback.pushDebugInfo("Input NoData value = " + str(input_nodata_val))
        if input_nodata_val == 1:
            raise QgsProcessingException("Input NoData value cannot be equal to 1.")
            
        input_type = input.dataProvider().dataType(1)
        out = qgsTreatments.applyTranslate(input_filename,output,data_type=input_type,
            nodata_val=input_nodata_val,context=context,feedback=feedback)
        return {'OUTPUT' : output }
        
        
class RandomStartPointsCircuitscape(CircuitscapeAlgorithm):

    ALG_NAME = 'randomStartPointsCircuitscape'
    
    def displayName(self):
        return self.tr('Random start points (Circuitscape)')
        
    def shortHelpString(self):
        return self.tr('Generates random start points layer for circuitscape input')
        
    PATCH_LAYER = 'PATCH_LAYER'
    NB_POINTS = 'NB_POINTS'
    NB_LAUNCHES = 'NB_LAUNCHES'
    DEFAULT_NB_LAUNCHES = 1
    RESISTANCE_LAYER = 'RESISTANCE_LAYER'
    # OUTPUT_DIR = 'OUTPUT_DIR'
    OUTPUT = 'OUTPUT'

    POINTS_VECTOR = 'POINTS_VECTOR'

    def initAlgorithm(self,config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PATCH_LAYER,
                description=self.tr('Patch layer')))
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.RESISTANCE_LAYER,
                description=self.tr('Resistance layer')))
        self.addParameter(
            QgsProcessingParameterNumber (
                self.NB_POINTS,
                description=self.tr('Number of points'),
                type=QgsProcessingParameterNumber.Integer))
        self.addParameter(
            QgsProcessingParameterNumber (
                self.NB_LAUNCHES,
                description=self.tr('Number of launches'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=self.DEFAULT_NB_LAUNCHES))
        self.addParameter(
            QgsProcessingParameterFolderDestination (
                self.OUTPUT,
                description=self.tr('Output directory')))
        
        
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        patch_source, patch_layer = qgsTreatments.parameterAsSourceLayer(
            self,parameters,self.PATCH_LAYER,context,feedback=feedback)
        resistance_layer = self.parameterAsRasterLayer(parameters,self.RESISTANCE_LAYER,context)
        nb_points = self.parameterAsInt(parameters, self.NB_POINTS, context)
        nb_launches = self.parameterAsInt(parameters, self.NB_LAUNCHES, context)
        out_dir = self.parameterAsFileOutput(parameters,self.OUTPUT,context)
        resistance_path = qgsUtils.pathOfLayer(resistance_layer)
        # Init input / output
        utils.mkDir(out_dir)
        dirname = os.path.dirname(__file__)
        base_config_file = os.path.join(dirname,"baseConfig.ini")
        fin = open(base_config_file,"rt")
        config_content = fin.read()
        fin.close()
        # Resistance translation
        EPT = ExportFrictionToCircuitscape
        out_resistance = os.path.join(out_dir,'resistance.asc')
        parameters = { EPT.INPUT : resistance_layer , EPT.OUTPUT : out_resistance } 
        qgsTreatments.applyProcessingAlg('BioDispersal',EPT.ALG_NAME,
            parameters,context=context,feedback=feedback)
        # Loop
        step_feedback = feedbacks.ProgressMultiStepFeedback(nb_launches*4,feedback)
        outfiles = []
        for i in range(nb_launches):
            bname = "launch" + str(i)
            basepath = os.path.join(out_dir,bname)
            # Sort random points from patch layer
            points_vector = basepath + "_start.gpkg"
            qgsTreatments.applyVRandom(patch_layer,nb_points,points_vector,
                context=context,feedback=step_feedback)
            #qgsUtils.loadVectorLayer(points_vector,loadProject=True)
            step_feedback.setCurrentStep(4*i+1)
            # Rasterize points layer
            extent = resistance_layer.extent()
            resolution = resistance_layer.rasterUnitsPerPixelX()
            #points_tif = QgsProcessingUtils.generateTempFilename('points_raster.tif')
            points_tif = basepath + "_start.tif"
            qgsUtils.removeRaster(points_tif)
            qgsTreatments.applyRasterization(points_vector,points_tif,extent,resolution,
                field='cat',out_type=Qgis.Int16,nodata_val=-9999,
                context=context,feedback=step_feedback)
            #qgsUtils.loadRasterLayer(points_tif,loadProject=True)
            step_feedback.setCurrentStep(4*i+2)
            # Convert to asc
            #points_asc = QgsProcessingUtils.generateTempFilename('points_raster.asc')
            points_asc = basepath + "_start.asc"
            qgsUtils.removeRaster(points_asc)
            qgsTreatments.applyTranslate(points_tif,points_asc,
                nodata_val=0,context=context,feedback=step_feedback)
            #qgsUtils.loadRasterLayer(points_asc,loadProject=True)
            step_feedback.setCurrentStep(4*i+3)
            # Out config file
            out_file = basepath + ".out"
            out_log = basepath + ".log"
            out_content = config_content.replace('OUT_FILE',out_file)
            out_content = out_content.replace('LOG_FILE',out_log)
            out_content = out_content.replace('START_LAYER',points_asc)
            out_content = out_content.replace('RESISTANCE_LAYER',out_resistance)
            outfile = basepath + "_config.ini"
            with open(outfile,"wt") as fout:
                fout.write(out_content)
            outfiles.append(outfile)
            step_feedback.setCurrentStep(4*i+4)
        # Output Julia script
        outscript = os.path.join(out_dir,"launchCircuitscape.jl")
        outscript = utils.normPath(outscript)
        script_text = "#include(\"" + outscript +"\")\n"
        script_text += "using Circuitscape\n"
        for outfile in outfiles:
            # f_posix = str(Path(outfile))
            # f_posix = f_posix.replace("\\","/")
            f_posix = utils.normPath(outfile)
            script_text += "compute(\"" + f_posix + "\")\n"
        utils.removeFile(outscript)
        with open(outscript,"w+") as fout:
            fout.write(script_text)
        os.chmod(outscript,0o7242)
        #os.chmod(outscript,stat.S_IXUSR)
        return { self.OUTPUT : out_dir }
        
        
class AggregateCirctuitscapeCurrentMaps(CircuitscapeAlgorithm):

    ALG_NAME = 'aggrCurrMapsLayers'
    
    def displayName(self):
        return self.tr('Aggregates current maps')
        
    def shortHelpString(self):
        return self.tr('Aggregates current maps (Circuitscape output) into one cumulative current map')
        
    INPUT_LAYERS = 'INPUT_LAYERS'

    def initAlgorithm(self,config=None):
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.INPUT_LAYERS,
                description=self.tr('Input layers'),
                layerType=QgsProcessing.TypeRaster))
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.tr("Output layer")))
                
    # def createFileFilter(self):
        # return "*.asc"
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        input_layers = self.parameterAsLayerList(parameters,self.INPUT_LAYERS,context)
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        # Serie
        aggregate = QgsProcessingUtils.generateTempFilename('aggregate.tif')
        qgsTreatments.applyRSeries(input_layers,aggr_func=0,output=aggregate,
            context=context,feedback=feedback)
        # Set null
        qgsTreatments.applyRSetNull(aggregate,0,output,context=context,feedback=feedback)
        # Set style
        # self.output = output
        # self.out_layer = qgsUtils.loadRasterLayer(output)
        return { self.OUTPUT : output }

    # def postProcessAlgorithm(self, context, feedback):
        # out_layer = self.out_layer#qgsUtils.loadRasterLayer(output)
        # color_ramp = styles.mkColorRamp('Plasma')
        # shader = styles.mkRasterShader(out_layer,color_ramp)
        # styles.setSBPCRasterRenderer(out_layer,shader)
        # out_layer.triggerRepaint()
        # return { self.OUTPUT : self.out_layer }
        
        
class AggregateCirctuitscapeResults(CircuitscapeAlgorithm):

    ALG_NAME = 'aggrCircuitscapeResults'
    
    def displayName(self):
        return self.tr('Aggregates Circuitscape results')
        
    def shortHelpString(self):
        return self.tr('Aggregates Circuitscape results (start points and current maps) from directory')
        
    RESULTS_DIR = 'RESULTS_DIR'
    AGGR_CURR = 'AGGR_CURR'
    AGGR_START = 'AGGR_START'

    def initAlgorithm(self,config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.RESULTS_DIR,
                description=self.tr('Results directory'),
                behavior=QgsProcessingParameterFile.Folder))
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.AGGR_CURR,
                self.tr("Aggregate current map")))
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.AGGR_START,
                self.tr("Aggregate start points")))
                
    def processAlgorithm(self,parameters,context,feedback):
        # Parameters
        input_dir = self.parameterAsFile(parameters,self.RESULTS_DIR,context)
        aggr_curr = self.parameterAsOutputLayer(parameters, self.AGGR_CURR, context)
        aggr_start = self.parameterAsOutputLayer(parameters, self.AGGR_START, context)
        # Retrieve files
        files = os.listdir(input_dir)
        curr_maps = [os.path.join(input_dir,f) for f in files if f.endswith("_cum_curmap.asc")]
        start_layers = [os.path.join(input_dir,f) for f in files if f.endswith("_start.gpkg")]
        nb_curr_maps, nb_start_layers = len(curr_maps), len(start_layers)
        if nb_curr_maps != nb_start_layers:
            raise QgsProcessingException("Inconsistent number of maps : "
                + str(nb_curr_maps) + " current maps vs "
                + str(nb_start_layers) + " start points layers")
        if nb_curr_maps == 0:
            raise QgsProcessingException("No layers found in " + str(input_dir))
        # Aggregate current
        ACC = AggregateCirctuitscapeCurrentMaps
        parameters = { ACC.INPUT_LAYERS : curr_maps, ACC.OUTPUT : aggr_curr} 
        qgsTreatments.applyProcessingAlg('BioDispersal',ACC.ALG_NAME,parameters,
            context=context,feedback=feedback)
        # Aggregate start points
        layer = qgsUtils.loadVectorLayer(start_layers[0])
        crs = layer.crs()
        qgsTreatments.mergeVectorLayers(start_layers,crs,aggr_start,
            context=context,feedback=feedback)
        # Return
        return { self.AGGR_CURR : aggr_curr, self.AGGR_START : aggr_start }
