# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BGA
                                 A QGIS plugin
 This is the plugin used to analyse the Urban Areas builtUp.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-03
        git sha              : $Format:%H$
        copyright            : (C) 2025 by NRSC
        email                : konkimallamanvitha@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""


# Import all the necessary modules
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.gui import QgsFileWidget
from qgis.PyQt.QtWidgets import QHeaderView , QAction, QFileDialog, QMessageBox, QColorDialog, QTableWidgetItem, QDialogButtonBox
from qgis.PyQt import QtWidgets
from qgis.core import (QgsProject, Qgis, QgsWkbTypes, QgsMapLayerType, QgsMapLayerProxyModel, QgsPointXY, QgsGeometry, 
    QgsVectorLayer, QgsRasterLayer, QgsCoordinateReferenceSystem, QgsVectorFileWriter, QgsProcessingFeatureSourceDefinition
 )
from PyQt5.QtCore import Qt, QEvent
from PyQt5.QtGui import QColor
from qgis.utils import reloadPlugin
from PyQt5 import uic
from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform
import processing

import tempfile
import os
from datetime import datetime
import pandas as pd

# Importing core project functionality
from .backend.CityRasterProcessor import CityRasterProcessor

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .BGA_dialog import BGADialog

from PyQt5.QtWidgets import QDialog

from PyQt5.QtCore import QObject, QEvent

class DialogCloseFilter(QObject):
    def __init__(self, plugin):
        super().__init__()
        self.plugin = plugin

    def eventFilter(self, obj, event):
        if event.type() == QEvent.Close:
            self.plugin.clear_inputs_on_cancel()
            return False  # Allow the dialog to close
        return False  # Let other events pass through


class BGA:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
		
		# customized init variables
        self.dlg = BGADialog()
        self.all_raster_layers = {} # Have all the raster layers present in the project
        self.selected_raster_layers = {}      # names along with raster paths
        self.vector_path = ''       # Get selected vector path
        self.colors = []
		
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'BGA_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&BGA')


        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('BGA', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action
    
    
    '''
    Get all the raster layers present in the QgsProject into QgsCheckableComboBox and 
    also stores the URI of all the layers in a dicttionary.
    '''
    def add_layers_to_combo(self):
        self.dlg.rasterComboBox.clear() 
        
        # Get all layers currently loaded in the QGIS project
        layers = QgsProject.instance().mapLayers().values()
        
        for layer in layers:
            # Add if the layer is a raster layer into the comboBox
            if layer.type() == QgsMapLayerType.RasterLayer:
                # Add layer object to combo box while Displaying layer name
                self.dlg.rasterComboBox.addItem(layer.name(), layer)
                # Store the data source URI (path) of the raster layer in the dictionary
                self.all_raster_layers[layer.name()] = layer.dataProvider().dataSourceUri()

    def validate_frame_input(self):
        # Analysis allowed only for maximum of 4 and minimum of 1 raster layers
        if len(self.dlg.rasterComboBox.checkedItems()) > 4:
            self.dlg.rasterLayerWarning.setVisible(True)    # Shows the warning
            self.dlg.rasterLayerWarning.setText("⚠ Please select at most 6 layer.")
        elif len(self.dlg.rasterComboBox.checkedItems()) < 1:
            self.dlg.rasterLayerWarning.setVisible(True)
            self.dlg.rasterLayerWarning.setText("⚠ Please select at least 1 layer.")
        else:
            self.dlg.rasterLayerWarning.setVisible(False)
            # Check if a vector layer is selected
            if self.dlg.vectorComboBox.currentLayer() is None: 
                self.dlg.vectorLayerWarning.setVisible(True)
            else:
                self.dlg.vectorLayerWarning.setVisible(False)
                # Check if city/region input is not empty
                if self.dlg.regionLineEdit.displayText() == "":
                    self.dlg.regionWarning.setVisible(True)
                else:
                    self.dlg.regionWarning.setVisible(False)
                    return True     # All validations passed
        return False
        
    '''
    Checks whether all the raster layers are of same region and 
    AOI also belong to that region
    '''
    def check_geometry(self):
        raster_layers = self.dlg.rasterComboBox.checkedItems()
        def check_raster(init_name, final_name):
            init_raster = QgsProject.instance().mapLayersByName(init_name)[0]
            final_raster = QgsProject.instance().mapLayersByName(final_name)[0]

            # Check if the extents (bounding boxes) of the rasters are exactly the same
            same_extent = init_raster.extent() == final_raster.extent()

            # Check if the raster pixel dimensions (width & height) are equal
            same_size = (init_raster.width() == final_raster.width() and
                         init_raster.height() == final_raster.height())

            # Check if the Coordinate Reference Systems (CRS) of the rasters are the same
            same_crs = init_raster.crs() == final_raster.crs()

            # Check if the spatial resolution (pixel size in map units) is the same
            same_res = (init_raster.rasterUnitsPerPixelX() == final_raster.rasterUnitsPerPixelX() and
                        init_raster.rasterUnitsPerPixelY() == final_raster.rasterUnitsPerPixelY())
            
            
            # If any of the above checks fail, show a warning and return False
            if not (same_extent and same_size and same_crs and same_res):
                QMessageBox.warning(
                    self.dlg,  # Parent widget for the message box
                    self.tr("Geometry Mismatch"),
                    self.tr("Selected rasters differ in extent, resolution, or CRS."),
                )
                return False

            return True

        def check_AOI_Coverage():
            # Get the vector (AOI) and raster layers
            aoi_layer = self.dlg.vectorComboBox.currentLayer()
            raster_layer = QgsProject.instance().mapLayersByName(raster_layers[0])[0]

            # Get the raster extent and transform to AOI CRS if needed
            raster_extent = raster_layer.extent()
            if raster_layer.crs().authid() != 'EPSG:4326' or aoi_layer.crs().authid() != 'EPSG:4326':
                print(raster_layer.crs(),'-----',aoi_layer.crs() )
                # transform = QgsCoordinateTransform(raster_layer.crs(), aoi_layer.crs(), QgsProject.instance())
                # raster_extent = transform.transformBoundingBox(raster_extent)
                QMessageBox.warning(
                    self.dlg,  # Parent widget for the message box
                    self.tr("CRS Mismatch between raster layer CRS and AOI CRS"),
                    self.tr("Selected rasters and AOI differ in CRS or not in 'EPSG:4326'."),
                )
                return False


            # Check if AOI and raster intersect
            for feature in aoi_layer.getFeatures():
                if feature.geometry().intersects(raster_extent):
                    return True  # AOI overlaps raster

            # If no overlap found
            QMessageBox.warning(
                self.dlg,
                self.tr("AOI Outside Raster"),
                self.tr("The selected AOI does not overlap with the raster extent."),
            )
            return False


        for i in range(len(raster_layers)-1):
            bool_check = check_raster(raster_layers[i], raster_layers[i+1])
            if bool_check == False:
                return False
        
        return check_AOI_Coverage()
                
                
    '''
    After validating all items in a frame, Store all selected raster and vector layers 
    and call the function to add these layers into table.
    '''
    def collect_and_load_table_data(self):
        # validate frame 1(Raster layers, vector layer, region) data before loading into the table
        if self.validate_frame_input():
            # Checks all layers are of same region
            if self.check_geometry():
                # Get all selected raster layer names from the raster checkable combo box
                raster_layer_names = self.dlg.rasterComboBox.checkedItems()
                
                for raster_layer in raster_layer_names:
                    # self.reproject_raster_layer(raster_layer)
                    layer = {}
                    layer[raster_layer] =  self.all_raster_layers[raster_layer]
                    self.selected_raster_layers.update(layer)
                    

                # Get the selected vector layer from the vector combo box
                vector_layer = self.dlg.vectorComboBox.currentLayer()
                # self.reproject_vector_layer(vector_layer)
                # Store the vector layer path
                self.vector_path = vector_layer.dataProvider().dataSourceUri()
                
                # Load the data into the table
                self.load_data_into_table()
            
    '''
    Convert degree, minutes, seconds to decimal degrees using formula.
    '''
    def dms_to_dd(self, dms_str):
        # Replace apostrophes or similar symbols with "." and split using it
        dms_str = dms_str.replace("'", ".").replace("’", ".").replace("‘", ".")  # Replace apostrophes if present
        parts = [float(p.strip()) for p in dms_str.strip().split(".") if p.strip()]
        
        if len(parts) == 3:
            degrees, minutes, seconds = parts
        elif len(parts) == 2:
            degrees, minutes = parts
            seconds = 0
        elif len(parts) == 1:
            return parts[0]
        else:
            raise ValueError("Invalid DMS format.")
        
        # Determine sign (for negative coordinates) and convert to decimal degrees
        sign = -1 if degrees < 0 else 1
        return sign * (abs(degrees) + minutes / 60 + seconds / 3600)
    
    
    
    def validate_centroid(self):
        # Get the text from the centroid input field
        self.dlg.centroidWarning.setText("⚠ Please enter valid centroid!")
        centroid_text = self.dlg.centroidLineEdit.displayText()
        if centroid_text == "":
            self.dlg.centroidWarning.setVisible(False)
            return None, True
        else:    
            point = centroid_text.split(",")
            if len(point) != 2:
                self.dlg.centroidWarning.setVisible(True)
                return None, False
            else:
                self.dlg.centroidWarning.setVisible(False)
                x_str, y_str = point
                try:
                    float(x_str)
                    float(y_str)
                    point = (x_str, y_str)
                    self.dlg.centroidWarning.setVisible(False)
                    return point, True
                except ValueError:
                    self.dlg.centroidWarning.setVisible(True)
                    return None, False
    
    
    '''
    Gets the user-entered centroid, 
    converts it if needed (from DMS to decimal), 
    and returns it as a QgsPointXY.
    '''
    def get_centroid_point(self, centroid_type, point):
        x_str, y_str = point
        # If input is in Decimal Degrees (DD), convert directly to QgsPointXY
        if centroid_type == 'Decimal Degrees (DD)':
            centroid_point = QgsPointXY(float(x_str), float(y_str))
            return centroid_point
        # If input is in Degrees, Minutes, Seconds (DMS), convert each to decimal degrees
        elif centroid_type == 'Degrees, Minutes, Seconds (DMS)':
            x_str = self.dms_to_dd(x_str)
            y_str = self.dms_to_dd(y_str)
            centroid_point = QgsPointXY(float(x_str), float(y_str))
            return centroid_point
            
    '''
    Checks whether the given centroid point lies within the AOI (Area of Interest) polygon layer.
    '''
    def is_centroid_inside_aoi(self, centroid_point):
        self.dlg.centroidWarning.setText("⚠ Invalid Centroid: Point lies outside the selected region's AOI")
        # Convert the centroid point to a QgsGeometry object
        centroid_geom = QgsGeometry.fromPointXY(centroid_point)
        
        # Load the AOI vector layer from the stored path
        aoi_layer = QgsVectorLayer(self.vector_path, "AOI", "ogr")
        for feature in aoi_layer.getFeatures():
            # Check if the centroid lies within the feature geometry
            if feature.geometry().contains(centroid_geom):
                self.dlg.centroidWarning.setVisible(False)
                return True
                
        self.dlg.centroidWarning.setVisible(True)
        return False
    
    '''
    Validates that each row in the table has a valid year 
    (numeric, not empty, and between 1900 and the current year).
    '''
    def validate_year_in_table(self):
        row_count = self.dlg.tableWidget.rowCount()
        for i in range(row_count):
            item = self.dlg.tableWidget.item(i, 1)
            
            # Check if the cell is empty
            if item is None:
                self.dlg.tableWarnings.setText("⚠ Please enter valid year!")
                self.dlg.tableWarnings.setVisible(True)
                return False
            year_text = item.text().strip()
            
            # Check if the text is a digit
            if not year_text.isdigit():
                self.dlg.tableWarnings.setVisible(True)
                return False

            year = int(year_text)
            current_year = datetime.now().year
            
            # Check if the year is within a valid range
            if year < 1900 or year > current_year:
                self.dlg.tableWarnings.setVisible(True)
                return False
            else:
                self.dlg.tableWarnings.setVisible(False)
        return True
    
    
    '''
    Validates that each color entry in the table is in the correct RGBA format 
    (4 integers between 0 and 255), and ensures all colors are distinct.
    '''
    def validate_color_in_table(self):
        self.dlg.tableWarnings.setText("⚠ Please enter valid color!")
        row_count = self.dlg.tableWidget.rowCount()
        for i in range(row_count):
            item = self.dlg.tableWidget.item(i, 2)
            # If the cell is empty 
            if item == "" or not item:
                continue    # iteration skip as default colors will be assigned later.
            color_text = item.text().strip().split(",")
            if color_text[0] == '':
                continue 
            print(color_text)
            
            
            if len(color_text) != 4:            # R,G,B, alpha
                
                self.dlg.tableWarnings.setVisible(True)
                return False
                
            for color in color_text:
                color = color.strip()
                if not color.isdigit():         # Must be numeric
                    self.dlg.tableWarnings.setVisible(True)
                    return False
                    
                color = int(color)
                if 0 > color or color > 255:    # Must be in 0–255 range
                    self.dlg.tableWarnings.setVisible(True)
                    return False
        self.dlg.tableWarnings.setVisible(False)
        
        # Check for color uniqueness using get_colors
        if not self.get_colors():
            self.dlg.tableWarnings.setText("⚠ Duplicate colors found!")
            self.dlg.tableWarnings.setVisible(True)
            return False
        else:
            self.dlg.tableWarnings.setVisible(False)
        return True      # All validations passed
            
    '''
    Opens a color picker when the user clicks the color column in the table, 
    then sets the selected RGBA color in that cell.
    '''
    def choose_color(self, row, column):
        # Trigger only if the clicked column is the color column (index 2)
        if column == 2: 
            color = QColorDialog.getColor()     # Open color picker dialog
            if color.isValid():
                r = color.red()
                g = color.green()
                b = color.blue()
                a = color.alpha()  # Usually 255 unless transparency selected

                rgba_text = f"{r},{g},{b},{a}"

                item = QTableWidgetItem(rgba_text)
                item.setBackground(color)
                
                # Make item Editable, Selectable
                item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsSelectable |  Qt.ItemIsEditable)
                # Adding the rgba values into table
                self.dlg.tableWidget.setItem(row, column, item)

    '''
    Collects valid raster layer names and years from the table, 
    sorts them by year, and returns the corresponding file paths and year labels.
    '''
    def get_year_labels(self):
        data = []
        row_count = self.dlg.tableWidget.rowCount()

        for i in range(row_count):
            value_item = self.dlg.tableWidget.item(i, 1)
            name_item = self.dlg.tableWidget.item(i, 0)
            
            year = value_item.text().strip()
            name = name_item.text().strip()
            
            # Get the raster path using the layer name from dictionary
            path = self.selected_raster_layers.get(name)
            data.append({'layerName': name, 'path': path, 'Year': int(year)})

        df = pd.DataFrame(data)
        df = df.sort_values(by='Year')      # Sort the paths by year

        sorted_paths = df['path'].tolist()
        sorted_years = df['Year'].tolist()
        return sorted_paths, sorted_years


    '''
    Ensures each row in the color column has a unique RGBA color string by
    returning False if duplicate colors are found; otherwise updates colors and returns True.
    If missing, assigns a default unused color. 
    '''
    def get_colors(self):
        row_count = self.dlg.tableWidget.rowCount()
        self.colors = [None] * row_count

        default_colors = [
            (0, 0, 255, 255),      # Blue
            (255, 170, 0, 255),    # Mustard
            (0, 255, 0, 255),      # Green
            (255, 0, 0, 255),      # Red
            (255, 165, 0, 255),    # Orange
            (255, 255, 0, 255),    # Yellow
            (255, 0, 255, 255),    # Magenta
            (128, 0, 128, 255),    # Purple
            (0, 128, 128, 255),    # Teal
            (128, 128, 0, 255)     # Olive
        ]

        used_colors = set()

        # Step 1: Gather year and row index
        year_row_pairs = []
        for i in range(row_count):
            year_item = self.dlg.tableWidget.item(i, 1)
            if year_item:
                try:
                    year = int(year_item.text())
                    year_row_pairs.append((year, i))
                except ValueError:
                    pass  # skip non-integer years

        # Step 2: Sort by year (ascending)
        year_row_pairs.sort()

        # Step 3: Assign user-provided colors or detect duplicates
        for year, row in year_row_pairs:
            item = self.dlg.tableWidget.item(row, 2)
            if item.text() != "" and item and item.text().strip():
                print("in step3 if")
                rgba_str = ",".join(part.strip() for part in item.text().strip().split(","))
                if rgba_str not in used_colors:
                    self.colors[row] = rgba_str
                    used_colors.add(rgba_str)
                else:
                    return False  # Duplicate color found
            else:
                print("in step3 else")
                continue

        # Step 4: Assign default colors for missing ones
        for year, row in year_row_pairs:
            item = self.dlg.tableWidget.item(row, 2)
            if not item or item.text() == "":
                print("in if not in step4 if")
                next_color = None
                for r, g, b, a in default_colors:
                    rgba_str = f"{r},{g},{b},{a}"
                    if rgba_str not in used_colors:
                        next_color = rgba_str
                        used_colors.add(rgba_str)
                        break

                if next_color:
                    color = QColor(*map(int, next_color.split(',')))
                    item = QTableWidgetItem(next_color)
                    item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsSelectable)
                    self.dlg.tableWidget.setItem(row, 2, item)
                    self.colors[row] = next_color

        # Step 5: Sort self.colors by year
        sorted_colors = [None] * row_count
        for idx, (year, row) in enumerate(year_row_pairs):
            sorted_colors[idx] = self.colors[row]
        self.colors = sorted_colors


        self.colors = self.colors[::-1]
        return True

    '''
    Check if the output directory path is set; show warning if empty
    '''
    def validate_output_dir(self):
        if self.dlg.outputFileWidget.filePath() == "":
            self.dlg.outputDirWarning.setVisible(True)
        else:
            self.dlg.outputDirWarning.setVisible(False)
            return True
        return False
    
    
    '''
    Sorts the table rows by the year column (index 1) after validating the year data,
    then updates the table with the sorted entries.
    '''
    def sort_table_entries(self, index):
        if self.validate_year_in_table():
            if index == 1:          # Only sort if the triggered column index is 1 (Year column)
                row_count = self.dlg.tableWidget.rowCount()
                
                layers = []
                # Safely extract all data (layers, years)
                for i in range(row_count):
                    row_data = []
                    for j in range(3):
                        item = self.dlg.tableWidget.item(i, j)
                        row_data.append(item.text() if item else "")
                    layers.append(row_data)
                
                # Sort by the second column (year)
                layers = sorted(layers, key=lambda x: int(x[1]))

                # Clear the table
                self.dlg.tableWidget.setRowCount(0)

                # Repopulate using sorted data
                for row_data in layers:
                    row_pos = self.dlg.tableWidget.rowCount()
                    self.dlg.tableWidget.insertRow(row_pos)
                    for col_index, value in enumerate(row_data):
                        self.dlg.tableWidget.setItem(row_pos, col_index, QTableWidgetItem(value))
                        
                        
    def clear_inputs_on_cancel(self):
        self.dlg.reject()
        reloadPlugin('BGA')
    
    
    '''
    Validates user inputs including centroid location, year and color data, and output directory. 
    If all validations pass and the centroid lies within the AOI (if provided), 
    it collects parameters, initializes the processing class CityRasterProcessor, 
    updates the progress bar, and displays a success message.
    '''
    def clickOk(self):
        # Check the entries of Frame input data before clicking ok
        if self.validate_frame_input():
            if self.dlg.tableWidget.rowCount() == 0:
                self.dlg.tableWarnings.setText("⚠ Table is empty! Click Upload Data Button.")
                self.dlg.tableWarnings.setVisible(True)
            else:
                self.dlg.tableWarnings.setVisible(False)
                # Validate year entries in the table
                if self.validate_year_in_table(): 
                    # Validate color entries in the table
                    if self.validate_color_in_table():
                        # Proceed only if centroid check passed or no centroid provided (dynamically takes calculated centroid)
                        point, is_centroid_valid = self.validate_centroid()
                        if is_centroid_valid:
                            if point is None:
                                centroid_point = None
                            else:
                                # Get the selected centroid coordinate format from the combo box
                                centroidType = self.dlg.centroidComboBox.currentText()
                                # Get the centroid point based on user input and coordinate format
                                centroid_point = self.get_centroid_point(centroidType, point)
                                # If centroid is provided, check if it lies inside the AOI polygon
                                if not self.is_centroid_inside_aoi(centroid_point):
                                    is_centroid_valid = False
                            if is_centroid_valid:
                                # Validate that output directory is selected
                                if self.validate_output_dir():
                                    # Get city input from user
                                    region = self.dlg.regionLineEdit.displayText()
                                    
                                    # Get number of sectors from the combo box (as int)
                                    no_of_sectors = int(self.dlg.sectorComboBox.currentText())
                                    # Get output folder path
                                    output_path = self.dlg.outputFileWidget.filePath()
                                    
                                    # Get sorted raster layer paths and year labels from the table
                                    sorted_paths, years = self.get_year_labels()

                                    if not sorted_paths or not years:
                                        QMessageBox.warning(self.dlg, "Missing Data", "Ensure that raster layers and valid years are provided.")
                                        return
                                        
                                    # Initialize progress bar
                                    self.dlg.progressBar.setValue(0)
                                    self.dlg.progressBar.setValue(1)
                                    
                                    # Disable the OK button to prevent re-submission
                                    ok_button = self.dlg.buttonBox.button(QDialogButtonBox.Ok)
                                    ok_button.setEnabled(False)
                                    
                                    # Call processing class with or without centroid point as argument
                                    CityRasterProcessor(output_path, self.dlg, self.iface, region, sorted_paths, self.vector_path, years, no_of_sectors, self.colors, centroid_point)
                                    
                                    # Show success message on QGIS message bar
                                    self.iface.messageBar().pushMessage("Success", "Output Folder saved at " + output_path , level=Qgis.Success)
                                    self.dlg.progressBar.setValue(100)                                  
                            
                    
    
    '''
    Loads raster layer names into the first column of the table widget.
    '''
    def load_data_into_table(self):
        row = 0
        # Set table row count to the number of raster paths available
        self.dlg.tableWidget.setRowCount(len(self.selected_raster_layers))
        for layer_name in self.selected_raster_layers.keys():
            self.dlg.tableWidget.setItem(row, 0, QtWidgets.QTableWidgetItem(layer_name))
            self.dlg.tableWidget.setItem(row, 2, QTableWidgetItem(""))
            row = row + 1

    '''
    Loads selected raster files from user data into the QGIS project, 
    adds them to a combo box, and stores their paths.
    '''
    def click_add_raster(self):
        # Retrieve the combined file path string
        raster_path_str = self.dlg.rasterFileWidget.filePath()
        
        # Split the string into a list of individual file paths 
        raster_paths = QgsFileWidget.splitFilePaths(raster_path_str)
        
        for raster_path in raster_paths:
            if raster_path:
                # Extract the file name from the full path
                layer_name = os.path.basename(raster_path)
                # Create a QgsRasterLayer from the file path and name
                layer = QgsRasterLayer(raster_path, layer_name)
                if layer.isValid():
                    # Add the raster layer to the current QGIS project
                    QgsProject.instance().addMapLayer(layer)
                    # Add the layer name to the combo box with layer 
                    self.dlg.rasterComboBox.addItem(layer.name(), layer)
                    # Store the data source URI for the layer in a dictionary
                    self.all_raster_layers[layer.name()] = raster_path
    

    '''
    Loads a selected vector file from user data into the QGIS project
    if it's valid and clears the file input widget.
    '''
    def click_add_vector(self):
        # Get the file path from the vector file widget
        vector_path = self.dlg.vectorFileWidget.filePath()
        if vector_path != "":
            # Extract the file name from the full path
            layer_name = os.path.basename(vector_path)
            # Create a vector layer using the path, name, and 'ogr' provider
            layer = QgsVectorLayer(vector_path, layer_name, "ogr")
            if layer.isValid():
                QgsProject.instance().addMapLayer(layer)    # Add it to the QGIS project


    def eventFilter(self, obj, event):
        if obj == self.dlg and event.type() == QEvent.Close:
            self.clear_inputs_on_cancel()
            return False  # Allow the dialog to close
        return super().eventFilter(obj, event)


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path =  os.path.join(os.path.dirname(__file__), 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'BuiltUp Growth Analysis'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # init all variables from pyqt
        self.dlg.rasterLayerWarning.setVisible(False)
        self.dlg.regionWarning.setVisible(False)
        self.dlg.tableWarnings.setVisible(False)
        self.dlg.outputDirWarning.setVisible(False)
        self.dlg.centroidWarning.setVisible(False)
        self.dlg.vectorLayerWarning.setVisible(False)
        
        self.dlg.sectorComboBox.setCurrentIndex(2)
        
        # Set Hovers (ToolTip) or place holder item
        self.dlg.rasterComboBox.setToolTip("Select Raster Images")
        self.dlg.vectorComboBox.setToolTip("Select Region Area Of Interest")
        self.dlg.uploadDataPushButton.setToolTip("Add to Table")
        self.dlg.centroidComboBox.setToolTip("Select Units")
        self.dlg.outputLabel.setToolTip("Save to location")
        self.dlg.outputFileWidget.setToolTip("Save to region")
        self.dlg.sectorComboBox.setToolTip("Select No of Sectors for Display")
        self.dlg.tableWidget.horizontalHeaderItem(1).setToolTip("Sort by Year")
        self.dlg.tableWidget.horizontalHeaderItem(2).setToolTip("Choose Color (Optional)")
        self.dlg.regionLineEdit.setPlaceholderText("e.g. Bhopal")
        self.dlg.centroidLineEdit.setPlaceholderText("(Optional) e.g. 77.40, 23.23  (lon, lat)")
        
        # Restricting the file selection to specific formats and types
        self.dlg.rasterFileWidget.setFilter("GeoTIFF (*.tif *.tiff);;ERDAS IMG (*.img)")
        self.dlg.vectorFileWidget.setFilter("ESRI Shapefile (*.shp)")
        self.dlg.vectorComboBox.setFilters(QgsMapLayerProxyModel.PolygonLayer)   # Limit the vector layer combo box to show only polygon layers 
        self.dlg.rasterFileWidget.setStorageMode(QgsFileWidget.GetMultipleFiles) # Enables multi-file selection
        
        # Get the files from user data
        self.dlg.rasterFileWidget.fileChanged.connect(self.click_add_raster)
        self.dlg.vectorFileWidget.fileChanged.connect(self.click_add_vector)

        # Sets table size dynamically
        header = self.dlg.tableWidget.horizontalHeader()
        header.setSectionResizeMode(0, QHeaderView.Stretch)         # Column 0 stretches
        header.setSectionResizeMode(1, QHeaderView.Interactive)     # Column 1 fixed or user-resizable
        header.setSectionResizeMode(2, QHeaderView.Interactive)     # Column 2 fixed or user-resizable
        self.dlg.tableWidget.setColumnWidth(1, 100)
        self.dlg.tableWidget.setColumnWidth(2, 200)
        
        self.close_event_filter = DialogCloseFilter(self)
        self.dlg.installEventFilter(self.close_event_filter)

        
        # Connect UI buttons and table events to their respective handler functions
        self.dlg.uploadDataPushButton.clicked.connect(self.collect_and_load_table_data)
        self.dlg.tableWidget.horizontalHeader().sectionClicked.connect(self.sort_table_entries)
        self.dlg.tableWidget.cellClicked.connect(self.choose_color)
        
        self.dlg.buttonBox.accepted.connect(self.clickOk)
        self.dlg.buttonBox.rejected.connect(self.clear_inputs_on_cancel)

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&BGA'),
                action)
            self.iface.removeToolBarIcon(action)
            
            
        if self.dlg:
            self.dlg.deleteLater()
            self.dlg = None


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False

        # Populate raster layers into the combo box
        self.add_layers_to_combo()

        # Show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            # Call the main logic or processing here
            # Example: self.processData() or handle the OK button logic
            pass
        

