from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QPushButton,
    QTableWidget, QTableWidgetItem, QFileDialog, QMessageBox, QAction, QToolBar,
    QHeaderView, QDesktopWidget, QInputDialog, QGridLayout, QSizePolicy,
    QComboBox, QSpinBox, QDialogButtonBox
)
from qgis.PyQt.QtCore import Qt, QVariant, QCoreApplication, QTranslator, QSettings, QLocale, QItemSelectionModel
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsFeature, QgsGeometry, QgsPointXY, QgsField, QgsFields, QgsWkbTypes,
    QgsVectorFileWriter, QgsProject, QgsVectorLayer, QgsCoordinateTransform,
    QgsSnappingConfig, QgsTolerance, QgsApplication
)
from qgis.gui import QgsMapToolEmitPoint, QgsRubberBand
from qgis.utils import iface
import math
import os
import tempfile

class SnappingPointMapTool(QgsMapToolEmitPoint):
    def __init__(self, canvas):
        super().__init__(canvas)
        self.canvas = canvas
        self.rubberBand = QgsRubberBand(self.canvas, QgsWkbTypes.PointGeometry)
        self.rubberBand.setColor(Qt.red)
        self.rubberBand.setWidth(2)
        self.rubberBand.setIcon(QgsRubberBand.ICON_CIRCLE)
        self.rubberBand.setIconSize(10)
        self.snappingUtils = iface.mapCanvas().snappingUtils()

    def canvasMoveEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        match = self.snappingUtils.snapToMap(point)
        if match.isValid():
            snap_point = match.point()
            self.rubberBand.setToGeometry(QgsGeometry.fromPointXY(snap_point), None)
            self.rubberBand.show()
        else:
            self.rubberBand.hide()

    def canvasPressEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        match = self.snappingUtils.snapToMap(point)
        if match.isValid():
            snap_point = match.point()
        else:
            snap_point = point
        self.rubberBand.hide()
        self.canvasClicked.emit(snap_point, Qt.LeftButton)

    def deactivate(self):
        self.rubberBand.hide()
        super().deactivate()

class ImportOptionsDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle(QCoreApplication.translate('ImportOptionsDialog', 'Import Options'))
        self.layout = QVBoxLayout()
        self.angle_type_label = QLabel(QCoreApplication.translate('ImportOptionsDialog', 'Select angle type:'))
        self.angle_type_combo = QComboBox()
        self.angle_type_combo.addItems([
            QCoreApplication.translate('ImportOptionsDialog', 'Azimuth'),
            QCoreApplication.translate('ImportOptionsDialog', 'Bearing')
        ])
        self.layout.addWidget(self.angle_type_label)
        self.layout.addWidget(self.angle_type_combo)
        self.decimal_separator_label = QLabel(QCoreApplication.translate('ImportOptionsDialog', 'Select decimal separator:'))
        self.decimal_separator_combo = QComboBox()
        self.decimal_separator_combo.addItems([
            QCoreApplication.translate('ImportOptionsDialog', 'Comma'),
            QCoreApplication.translate('ImportOptionsDialog', 'Dot')
        ])
        self.layout.addWidget(self.decimal_separator_label)
        self.layout.addWidget(self.decimal_separator_combo)
        self.angle_precision_label = QLabel(QCoreApplication.translate('ImportOptionsDialog', 'Select angle precision (0-10):'))
        self.angle_precision_spin = QSpinBox()
        self.angle_precision_spin.setRange(0, 10)
        self.angle_precision_spin.setValue(3)
        self.layout.addWidget(self.angle_precision_label)
        self.layout.addWidget(self.angle_precision_spin)
        self.distance_precision_label = QLabel(QCoreApplication.translate('ImportOptionsDialog', 'Select distance precision (0-10):'))
        self.distance_precision_spin = QSpinBox()
        self.distance_precision_spin.setRange(0, 10)
        self.distance_precision_spin.setValue(3)
        self.layout.addWidget(self.distance_precision_label)
        self.layout.addWidget(self.distance_precision_spin)
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        self.layout.addWidget(self.button_box)
        self.setLayout(self.layout)

    def get_options(self):
        return {
            'angle_type': self.angle_type_combo.currentText(),
            'decimal_separator': 'Comma' if self.decimal_separator_combo.currentIndex() == 0 else 'Dot',
            'angle_precision': self.angle_precision_spin.value(),
            'distance_precision': self.distance_precision_spin.value()
        }

class AzimuthToolDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle(self.tr('Azimuth Tool'))
        self.setGeometry(300, 300, 600, 400)
        self.setWindowFlag(Qt.WindowStaysOnTopHint)
        self.layout = QVBoxLayout()
        self.setup_ui()
        self.mapTool = None
        self.center_window()

    def tr(self, message):
        return QCoreApplication.translate('AzimuthToolDialog', message)

    def center_window(self):
        qr = self.frameGeometry()
        cp = QDesktopWidget().availableGeometry().center()
        qr.moveCenter(cp)
        self.move(qr.topLeft())

    def setup_ui(self):
        self.setup_output_shapefile_field()
        self.setup_initial_point_field()
        self.setup_distance_azimuth_table()
        self.setup_buttons()
        self.setup_process_button()
        self.setLayout(self.layout)

    def setup_output_shapefile_field(self):
        self.output_shapefile_label = QLabel(self.tr('Output Layer:'))
        self.output_shapefile_edit = QLineEdit()
        self.output_shapefile_button = QPushButton(self.tr('Browse'))
        self.output_shapefile_button.clicked.connect(self.browse_output_shapefile)
        h_layout = QHBoxLayout()
        h_layout.addWidget(self.output_shapefile_label)
        h_layout.addWidget(self.output_shapefile_edit)
        h_layout.addWidget(self.output_shapefile_button)
        self.layout.addLayout(h_layout)
        self.set_temporary_output_path()

    def setup_initial_point_field(self):
        self.initial_point_label = QLabel(self.tr('Initial Coordinate:'))
        self.initial_point_edit = QLineEdit()
        self.set_canvas_center_as_initial_point()
        self.initial_point_button = QPushButton(self.tr('Select on Canvas'))
        self.initial_point_button.clicked.connect(self.select_initial_point)
        h_layout = QHBoxLayout()
        h_layout.addWidget(self.initial_point_label)
        h_layout.addWidget(self.initial_point_edit)
        h_layout.addWidget(self.initial_point_button)
        self.layout.addLayout(h_layout)

    def setup_distance_azimuth_table(self):
        self.distance_azimuth_label = QLabel(self.tr('List of Vertices, Azimuths/Bearings, Distances, and Adjacencies:'))
        info_icon_path = os.path.join(os.path.dirname(__file__), 'icon_info.png')
        self.info_button = QPushButton()
        self.info_button.setIcon(QIcon(info_icon_path))
        self.info_button.clicked.connect(self.show_info)
        label_layout = QHBoxLayout()
        label_layout.addWidget(self.distance_azimuth_label)
        label_layout.addWidget(self.info_button)
        label_layout.addStretch()
        self.layout.addLayout(label_layout)
        self.table = QTableWidget(10, 4)
        self.table.setSortingEnabled(False)
        self.table.setHorizontalHeaderLabels([
            self.tr('Vertex'),
            self.tr('Angle'),
            self.tr('Distance (m)'),
            self.tr('Adjacency')
        ])
        self.table.horizontalHeader().setStretchLastSection(True)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.layout.addWidget(self.table)

    def setup_buttons(self):
        row_management_layout = QHBoxLayout()
        self.remove_row_button = QPushButton(self.tr('-'))
        self.remove_row_button.clicked.connect(self.remove_selected_rows)
        self.add_row_button = QPushButton(self.tr('+'))
        self.add_row_button.clicked.connect(self.add_row)
        row_management_layout.addWidget(self.remove_row_button)
        row_management_layout.addWidget(self.add_row_button)
        self.move_up_button = QPushButton(self.tr('↑'))
        self.move_up_button.clicked.connect(self.move_rows_up)
        self.move_down_button = QPushButton(self.tr('↓'))
        self.move_down_button.clicked.connect(self.move_rows_down)
        row_management_layout.addWidget(self.move_up_button)
        row_management_layout.addWidget(self.move_down_button)
        io_buttons_layout = QHBoxLayout()
        self.import_button = QPushButton(self.tr('Import from .txt'))
        self.import_button.clicked.connect(self.import_from_txt)
        self.export_button = QPushButton(self.tr('Export to .txt'))
        self.export_button.clicked.connect(self.export_to_txt)
        self.import_polygon_button = QPushButton(self.tr('Import from Line/Polygon'))
        self.import_polygon_button.clicked.connect(self.import_from_line_or_polygon)
        io_buttons_layout.addWidget(self.import_button)
        io_buttons_layout.addWidget(self.export_button)
        io_buttons_layout.addWidget(self.import_polygon_button)
        self.import_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.export_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.import_polygon_button.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.layout.addLayout(row_management_layout)
        self.layout.addLayout(io_buttons_layout)

    def setup_process_button(self):
        process_layout = QHBoxLayout()
        self.process_point_button = QPushButton(self.tr('Process as Point'))
        self.process_line_button = QPushButton(self.tr('Process as Line'))
        self.process_polygon_button = QPushButton(self.tr('Process as Polygon'))
        self.process_point_button.clicked.connect(self.process_as_point)
        self.process_line_button.clicked.connect(self.process_as_line)
        self.process_polygon_button.clicked.connect(self.process_as_polygon)
        process_layout.addWidget(self.process_point_button)
        process_layout.addWidget(self.process_line_button)
        process_layout.addWidget(self.process_polygon_button)
        self.layout.addLayout(process_layout)

    def set_temporary_output_path(self):
        self.output_shapefile_edit.setText(self.tr('Temporary Layer'))

    def set_canvas_center_as_initial_point(self):
        canvas = iface.mapCanvas()
        extent = canvas.extent()
        center = extent.center()
        self.initial_point_edit.setText(f"{center.x()},{center.y()}")

    def browse_output_shapefile(self):
        filename, _ = QFileDialog.getSaveFileName(
            self,
            self.tr('Select output file'),
            '',
            self.tr('GeoPackages (*.gpkg);;Shapefiles (*.shp)')
        )
        if filename:
            self.output_shapefile_edit.setText(filename)

    def select_initial_point(self):
        selected_layers = iface.layerTreeView().selectedLayers()
        if len(selected_layers) == 1 and isinstance(selected_layers[0], QgsVectorLayer):
            self.show_message(self.tr("Snapping is enabled for the selected vector layer."))
        else:
            self.show_message(
                self.tr("You can enable snapping if you select exactly one vector layer before clicking 'Select on Canvas'.")
            )
        self.mapTool = SnappingPointMapTool(iface.mapCanvas())
        self.mapTool.canvasClicked.connect(self.set_initial_point)
        iface.mapCanvas().setMapTool(self.mapTool)
        self.configure_snapping()

    def configure_snapping(self):
        selected_layers = iface.layerTreeView().selectedLayers()
        if not selected_layers or len(selected_layers) != 1:
            self.show_message(self.tr("Select only one vector layer for snapping."))
            return
        layer = selected_layers[0]
        if not isinstance(layer, QgsVectorLayer):
            self.show_message(self.tr("Select a vector layer for snapping."))
            return
        project = QgsProject.instance()
        snapping_config = QgsSnappingConfig()
        snapping_config.setEnabled(True)
        snapping_config.setMode(QgsSnappingConfig.AdvancedConfiguration)
        individual_settings = QgsSnappingConfig.IndividualLayerSettings(
            True,
            QgsSnappingConfig.VertexFlag | QgsSnappingConfig.SegmentFlag,
            10,
            QgsTolerance.Pixels
        )
        snapping_config.setIndividualLayerSettings(layer, individual_settings)
        project.setSnappingConfig(snapping_config)
        iface.mapCanvas().snappingUtils().setConfig(snapping_config)

    def set_initial_point(self, point):
        self.initial_point_edit.setText(f"{point.x()},{point.y()}")
        iface.mapCanvas().unsetMapTool(self.mapTool)
        self.mapTool.canvasClicked.disconnect(self.set_initial_point)
        self.mapTool = None

    def add_row(self):
        selected_rows = self.table.selectionModel().selectedRows()
        if selected_rows:
            index = selected_rows[0].row() + 1
        else:
            index = self.table.rowCount()
        self.table.insertRow(index)

    def remove_selected_rows(self):
        indices = self.table.selectionModel().selectedRows()
        for index in sorted(indices, reverse=True):
            self.table.removeRow(index.row())

    def move_rows_up(self):
        selected_rows = sorted(self.table.selectionModel().selectedRows(), key=lambda x: x.row())
        if not selected_rows or selected_rows[0].row() == 0:
            return
        new_positions = []
        for index in selected_rows:
            row = index.row()
            if row > 0:
                self.table.insertRow(row - 1)
                for col in range(self.table.columnCount()):
                    item = self.table.takeItem(row + 1, col)
                    self.table.setItem(row - 1, col, item)
                self.table.removeRow(row + 1)
                new_positions.append(row - 1)
            else:
                new_positions.append(row)
        self.table.selectionModel().clearSelection()
        for pos in new_positions:
            idx = self.table.model().index(pos, 0)
            self.table.selectionModel().select(idx, QItemSelectionModel.Select | QItemSelectionModel.Rows)

    def move_rows_down(self):
        selected_rows = sorted(self.table.selectionModel().selectedRows(), key=lambda x: x.row(), reverse=True)
        if not selected_rows or selected_rows[0].row() == self.table.rowCount() - 1:
            return
        new_positions = []
        for index in selected_rows:
            row = index.row()
            if row < self.table.rowCount() - 1:
                self.table.insertRow(row + 2)
                for col in range(self.table.columnCount()):
                    item = self.table.takeItem(row, col)
                    self.table.setItem(row + 2, col, item)
                self.table.removeRow(row)
                new_positions.append(row + 1)
            else:
                new_positions.append(row)
        self.table.selectionModel().clearSelection()
        for pos in new_positions:
            idx = self.table.model().index(pos, 0)
            self.table.selectionModel().select(idx, QItemSelectionModel.Select | QItemSelectionModel.Rows)

    def import_from_txt(self):
        filename, _ = QFileDialog.getOpenFileName(
            self, self.tr('Import from TXT'), '', self.tr('Text Files (*.txt)')
        )
        if filename:
            self.table.setRowCount(0)
            with open(filename, 'r', encoding='utf-8') as file:
                lines = file.readlines()
                if lines:
                    first_line_parts = lines[0].strip().split(';')
                    if first_line_parts[0] == "Initial Point":
                        self.initial_point_edit.setText(first_line_parts[1])
                        lines = lines[1:]
                for line in lines:
                    parts = line.strip().split(';')
                    parts += [''] * (4 - len(parts))
                    vertex, azimuth, distance, adjacency = parts
                    row_position = self.table.rowCount()
                    self.table.insertRow(row_position)
                    self.table.setItem(row_position, 0, QTableWidgetItem(vertex))
                    self.table.setItem(row_position, 1, QTableWidgetItem(azimuth))
                    self.table.setItem(row_position, 2, QTableWidgetItem(distance))
                    self.table.setItem(row_position, 3, QTableWidgetItem(adjacency))

    def export_from_txt_normalization(self, raw_azimuth):
        if any(letter in raw_azimuth.upper() for letter in ['N', 'S', 'E', 'W']):
            return self.export_format_rumo(raw_azimuth)
        else:
            return self.export_format_azimuth(raw_azimuth)

    def export_format_azimuth(self, azimuth_text):
        sep = ',' if ',' in azimuth_text else '.'
        try:
            value = float(azimuth_text.replace(',', '.'))
            if sep == '.':
                decimals = len(azimuth_text.split('.')[1]) if '.' in azimuth_text else 0
            else:
                decimals = len(azimuth_text.split(',')[1]) if ',' in azimuth_text else 0
            if decimals == 0:
                decimals = 2
            degrees = int(value)
            minutes_full = (value - degrees) * 60
            minutes = int(minutes_full)
            seconds = (minutes_full - minutes) * 60
            seconds_str = f"{seconds:0.{decimals}f}"
            if '.' in seconds_str or ',' in seconds_str:
                if '.' in seconds_str:
                    int_part, frac_part = seconds_str.split('.')
                else:
                    int_part, frac_part = seconds_str.split(',')
                seconds_str = f"{int(int_part):02d}{sep}{frac_part}"
            else:
                seconds_str = f"{int(seconds):02d}"
            return f"{degrees:02d}-{minutes:02d}-{seconds_str}"
        except ValueError:
            parts = azimuth_text.split('-')
            if len(parts) == 2:
                try:
                    degrees = int(parts[0])
                    minutes = int(parts[1])
                except:
                    return azimuth_text
                seconds = 0.0
                decimals = 2
                seconds_str = f"{seconds:0.{decimals}f}"
                if '.' in seconds_str or ',' in seconds_str:
                    if '.' in seconds_str:
                        int_part, frac_part = seconds_str.split('.')
                    else:
                        int_part, frac_part = seconds_str.split(',')
                    seconds_str = f"{int(int_part):02d}{sep}{frac_part}"
                else:
                    seconds_str = f"{int(seconds):02d}"
                return f"{degrees:02d}-{minutes:02d}-{seconds_str}"
            elif len(parts) == 3:
                try:
                    degrees = int(parts[0])
                    minutes = int(parts[1])
                    seconds = float(parts[2].replace(',', '.'))
                except:
                    return azimuth_text
                if sep == '.':
                    decimals = len(parts[2].split('.')[1]) if '.' in parts[2] else 2
                else:
                    decimals = len(parts[2].split(',')[1]) if ',' in parts[2] else 2
                seconds_str = f"{seconds:0.{decimals}f}"
                if '.' in seconds_str or ',' in seconds_str:
                    if '.' in seconds_str:
                        int_part, frac_part = seconds_str.split('.')
                    else:
                        int_part, frac_part = seconds_str.split(',')
                    seconds_str = f"{int(int_part):02d}{sep}{frac_part}"
                else:
                    seconds_str = f"{int(seconds):02d}"
                return f"{degrees:02d}-{minutes:02d}-{seconds_str}"
            else:
                return azimuth_text

    def export_format_rumo(self, rumo_text):
        allowed_directions = {"NE", "NW", "SE", "SW"}
        sep = ',' if ',' in rumo_text else '.'
        parts = rumo_text.split('-')
        parts = [p.strip() for p in parts if p.strip()]
        n = len(parts)
        if n == 2:
            try:
                deg = int(parts[0])
            except:
                raise ValueError(self.tr("Invalid degrees."))
            direction = parts[1].upper()
            if direction not in allowed_directions:
                raise ValueError(self.tr("Invalid direction."))
            minutes = 0
            seconds = 0.0
            raw_seconds = "0"
        elif n == 3:
            try:
                deg = int(parts[0])
                minutes = int(parts[1])
            except:
                raise ValueError(self.tr("Invalid degrees or minutes."))
            direction = parts[2].upper()
            if direction not in allowed_directions:
                raise ValueError(self.tr("Invalid direction."))
            seconds = 0.0
            raw_seconds = "0"
        elif n == 4:
            try:
                deg = int(parts[0])
                minutes = int(parts[1])
            except:
                raise ValueError(self.tr("Invalid degrees or minutes."))
            raw_seconds = parts[2]
            try:
                seconds = float(raw_seconds.replace(',', '.'))
            except:
                raise ValueError(self.tr("Invalid seconds."))
            direction = parts[3].upper()
            if direction not in allowed_directions:
                raise ValueError(self.tr("Invalid direction."))
        else:
            raise ValueError(self.tr("Invalid number of parts for bearing."))
        if sep == ',':
            decimals = len(raw_seconds.split(',')[1]) if ',' in raw_seconds else 0
        else:
            decimals = len(raw_seconds.split('.')[1]) if '.' in raw_seconds else 0
        if decimals == 0:
            decimals = 2
        seconds_str = f"{seconds:0.{decimals}f}"
        if '.' in seconds_str or ',' in seconds_str:
            if '.' in seconds_str:
                int_part, frac_part = seconds_str.split('.')
            else:
                int_part, frac_part = seconds_str.split(',')
            seconds_str = f"{int(int_part):02d}{sep}{frac_part}"
        else:
            seconds_str = f"{int(seconds):02d}"
        return f"{deg:02d}-{minutes:02d}-{seconds_str}-{direction}"

    def export_format_distance(self, distance_text):
        return distance_text

    def export_to_txt(self):
        filename, _ = QFileDialog.getSaveFileName(
            self, self.tr('Export to TXT'), '', self.tr('Text Files (*.txt)')
        )
        if filename:
            with open(filename, 'w', encoding='utf-8') as file:
                initial_point = self.initial_point_edit.text()
                file.write(f"{self.tr('Initial Point')};{initial_point}\n")
                for row in range(self.table.rowCount()):
                    vertex_item = self.table.item(row, 0)
                    azimuth_item = self.table.item(row, 1)
                    distance_item = self.table.item(row, 2)
                    adjacency_item = self.table.item(row, 3)
                    vertex = vertex_item.text() if vertex_item else ''
                    raw_azimuth = azimuth_item.text() if azimuth_item else ''
                    distance_raw = distance_item.text() if distance_item else ''
                    adjacency = adjacency_item.text() if adjacency_item else ''
                    if raw_azimuth or distance_raw or adjacency or vertex:
                        azimuth_export = raw_azimuth if raw_azimuth else ''
                        distance_export = self.export_format_distance(distance_raw) if distance_raw else ''
                        file.write(f"{vertex};{azimuth_export};{distance_export};{adjacency}\n")

    def import_from_line_or_polygon(self):
        options_dialog = ImportOptionsDialog(self)
        if options_dialog.exec_() == QDialog.Accepted:
            options = options_dialog.get_options()
            mode = options['angle_type']
            decimal_separator_option = options['decimal_separator']
            angle_precision = options['angle_precision']
            distance_precision = options['distance_precision']
            selected_layers = iface.layerTreeView().selectedLayers()
            if not selected_layers:
                self.show_message(self.tr('No layer selected.'))
                return
            layer = selected_layers[0]
            selected_features = layer.selectedFeatures()
            if not selected_features:
                self.show_message(self.tr('No feature selected.'))
                return
            selected_features = sorted(selected_features, key=lambda f: f.id())
            self.table.setRowCount(0)
            project_crs = QgsProject.instance().crs()
            layer_crs = layer.crs()
            transform = QgsCoordinateTransform(layer_crs, project_crs, QgsProject.instance())
            first_vertex_set = False
            for feature in selected_features:
                geometry = feature.geometry()
                if geometry.type() in (QgsWkbTypes.PolygonGeometry, QgsWkbTypes.CurvePolygon):
                    if geometry.isMultipart():
                        polygons = geometry.asMultiPolygon() if geometry.type() == QgsWkbTypes.PolygonGeometry else geometry.asMultiSurface()
                    else:
                        polygons = [geometry.asPolygon()] if geometry.type() == QgsWkbTypes.PolygonGeometry else [geometry.asCurvePolygon()]
                    for polygon in polygons:
                        exterior_ring = polygon[0]
                        if not first_vertex_set and exterior_ring:
                            first_vertex = transform.transform(exterior_ring[0])
                            self.initial_point_edit.setText(f"{first_vertex.x()},{first_vertex.y()}")
                            first_vertex_set = True
                        for i in range(len(exterior_ring) - 1):
                            vertex1 = transform.transform(exterior_ring[i])
                            vertex2 = transform.transform(exterior_ring[i + 1])
                            dx = vertex2.x() - vertex1.x()
                            dy = vertex2.y() - vertex1.y()
                            angle = math.atan2(dx, dy)
                            azimuth_decimal = math.degrees(angle) if angle >= 0 else math.degrees(angle) + 360
                            if mode == QCoreApplication.translate('ImportOptionsDialog', 'Azimuth'):
                                azimuth_formatted = self.convert_decimal_to_dms(azimuth_decimal, precision=angle_precision)
                            else:
                                azimuth_formatted = self.convert_decimal_to_rumo(azimuth_decimal, precision=angle_precision)
                            if decimal_separator_option == "Comma":
                                azimuth_formatted = azimuth_formatted.replace('.', ',')
                            distance = self.calculate_distance(vertex1, vertex2)
                            distance_str = f"{distance:.{distance_precision}f}"
                            if decimal_separator_option == "Comma":
                                distance_str = distance_str.replace('.', ',')
                            row_position = self.table.rowCount()
                            self.table.insertRow(row_position)
                            self.table.setItem(row_position, 0, QTableWidgetItem(''))
                            self.table.setItem(row_position, 1, QTableWidgetItem(azimuth_formatted))
                            self.table.setItem(row_position, 2, QTableWidgetItem(distance_str))
                            self.table.setItem(row_position, 3, QTableWidgetItem(''))
                elif geometry.type() == QgsWkbTypes.LineGeometry:
                    if geometry.isMultipart():
                        lines = geometry.asMultiPolyline()
                    else:
                        lines = [geometry.asPolyline()]
                    for line in lines:
                        if len(line) < 2:
                            continue
                        transformed_line = [transform.transform(vertex) for vertex in line]
                        if not first_vertex_set:
                            self.initial_point_edit.setText(f"{transformed_line[0].x()},{transformed_line[0].y()}")
                            first_vertex_set = True
                        for i in range(len(transformed_line) - 1):
                            vertex1 = transformed_line[i]
                            vertex2 = transformed_line[i + 1]
                            dx = vertex2.x() - vertex1.x()
                            dy = vertex2.y() - vertex1.y()
                            angle = math.atan2(dx, dy)
                            azimuth_decimal = math.degrees(angle) if angle >= 0 else math.degrees(angle) + 360
                            if mode == QCoreApplication.translate('ImportOptionsDialog', 'Azimuth'):
                                azimuth_formatted = self.convert_decimal_to_dms(azimuth_decimal, precision=angle_precision)
                            else:
                                azimuth_formatted = self.convert_decimal_to_rumo(azimuth_decimal, precision=angle_precision)
                            if decimal_separator_option == "Comma":
                                azimuth_formatted = azimuth_formatted.replace('.', ',')
                            distance = self.calculate_distance(vertex1, vertex2)
                            distance_str = f"{distance:.{distance_precision}f}"
                            if decimal_separator_option == "Comma":
                                distance_str = distance_str.replace('.', ',')
                            row_position = self.table.rowCount()
                            self.table.insertRow(row_position)
                            self.table.setItem(row_position, 0, QTableWidgetItem(''))
                            self.table.setItem(row_position, 1, QTableWidgetItem(azimuth_formatted))
                            self.table.setItem(row_position, 2, QTableWidgetItem(distance_str))
                            self.table.setItem(row_position, 3, QTableWidgetItem(''))
        else:
            return

    def calculate_distance(self, point1, point2):
        return math.sqrt((point2.x() - point1.x())**2 + (point2.y() - point1.y())**2)

    def convert_decimal_to_dms(self, decimal_degrees, precision=3):
        degrees = int(decimal_degrees)
        minutes_full = (decimal_degrees - degrees) * 60
        minutes = int(minutes_full)
        seconds = round((minutes_full - minutes) * 60, precision)
        if seconds >= 60.0:
            seconds = 0.0
            minutes += 1
        if minutes >= 60:
            minutes = 0
            degrees += 1
        if precision == 0:
            return f"{degrees:02d}-{minutes:02d}-{int(seconds):02d}"
        else:
            return f"{degrees:02d}-{minutes:02d}-{seconds:0.{precision}f}"

    def convert_dms_to_decimal(self, dms):
        dms = dms.replace(',', '.')
        parts = dms.split('-')
        d = float(parts[0])
        m = float(parts[1]) if len(parts) > 1 else 0.0
        s = float(parts[2]) if len(parts) > 2 else 0.0
        return d + (m / 60.0) + (s / 3600.0)

    def convert_rumo_to_decimal(self, rumo):
        rumo = rumo.replace(',', '.').upper()
        parts = rumo.split('-')
        if len(parts) == 2:
            d = float(parts[0])
            m = 0.0
            s = 0.0
            direction = parts[1]
        elif len(parts) == 3:
            d = float(parts[0])
            m = float(parts[1])
            s = 0.0
            direction = parts[2]
        elif len(parts) == 4:
            d = float(parts[0])
            m = float(parts[1])
            s = float(parts[2])
            direction = parts[3]
        else:
            raise ValueError(self.tr("Invalid bearing format."))
        decimal_degrees = d + (m / 60.0) + (s / 3600.0)
        if 'S' in direction:
            decimal_degrees = 180 - decimal_degrees
        if 'W' in direction:
            decimal_degrees = 360 - decimal_degrees
        return decimal_degrees

    def convert_decimal_to_rumo(self, decimal_degrees, precision=3):
        if decimal_degrees < 90:
            direction = 'NE'
            angle = decimal_degrees
        elif decimal_degrees < 180:
            direction = 'SE'
            angle = 180 - decimal_degrees
        elif decimal_degrees < 270:
            direction = 'SW'
            angle = decimal_degrees - 180
        else:
            direction = 'NW'
            angle = 360 - decimal_degrees
        degrees = int(angle)
        minutes_full = (angle - degrees) * 60
        minutes = int(minutes_full)
        seconds = round((minutes_full - minutes) * 60, precision)
        if seconds >= 60.0:
            seconds = 0.0
            minutes += 1
        if minutes >= 60:
            minutes = 0
            degrees += 1
        if precision == 0:
            return f"{degrees:02d}-{minutes:02d}-{int(seconds):02d}-{direction}"
        else:
            return f"{degrees:02d}-{minutes:02d}-{seconds:0.{precision}f}-{direction}"

    def show_info(self):
        info_text = (
            self.tr("In this plugin, the Angle field accepts values in the following format: ")
            + self.tr("AZIMUTH: such as 80-00-00 or similar (without decimals unless specified). Examples: ")
            + self.tr("80 -> 80-00-00 | 80-00 -> 80-00-00 | 80-00-00.00 -> 80-00-00.00. ")
            + self.tr("BEARING: such as 80-00-00-NE (with directions NE, NW, SE, SW). Examples: ")
            + self.tr("80-NE -> 80-00-00-NE | 80-00-NE -> 80-00-00-NE | 80-00-00.00-NE -> 80-00-00.00-NE | ")
            + self.tr("80-45-NE -> 80-45-00-NE | 80-45-38.00-NE -> 80-45-38.00-NE. ")
            + self.tr("Directions: can be uppercase or lowercase; extra text is not accepted. ")
            + self.tr("Decimals: both Angle and Distance accept commas or dots as separators. ")
            + self.tr("Note: Vertex and Adjacency fields are optional and can be left blank.")
        )
        QMessageBox.information(self, self.tr('Information'), info_text)

    def parse_angle(self, angle_str):
        original_angle = angle_str
        if ',' in angle_str:
            sep = ','
        elif '.' in angle_str:
            sep = '.'
        else:
            sep = None
        parts = angle_str.split('-')
        if len(parts) >= 2 and parts[-1].upper() in ['NE', 'NW', 'SE', 'SW']:
            direction = parts[-1].upper()
            numeric_parts = parts[:-1]
            if len(numeric_parts) == 1:
                degrees = numeric_parts[0]
                minutes = '00'
                seconds = '00'
                if sep:
                    seconds += sep + '000'
            elif len(numeric_parts) == 2:
                degrees = numeric_parts[0]
                minutes = numeric_parts[1]
                seconds = '00'
                if sep:
                    seconds += sep + '000'
            elif len(numeric_parts) == 3:
                degrees = numeric_parts[0]
                minutes = numeric_parts[1]
                seconds = numeric_parts[2]
            else:
                raise ValueError(self.tr("Invalid rumo format"))
            try:
                int(degrees)
                int(minutes)
                if seconds:
                    if sep:
                        float(seconds.replace(sep, '.'))
                    else:
                        float(seconds)
            except ValueError:
                raise ValueError(self.tr("Invalid numeric values in angle"))
            return 'rumo', degrees, minutes, seconds, direction, sep
        else:
            if len(parts) == 1:
                degrees = parts[0]
                minutes = '00'
                seconds = '00'
                if sep:
                    seconds += sep + '000'
            elif len(parts) == 2:
                degrees = parts[0]
                minutes = parts[1]
                seconds = '00'
                if sep:
                    seconds += sep + '000'
            elif len(parts) == 3:
                degrees = parts[0]
                minutes = parts[1]
                seconds = parts[2]
            else:
                raise ValueError(self.tr("Invalid azimute format"))
            try:
                int(degrees)
                int(minutes)
                if seconds:
                    if sep:
                        float(seconds.replace(sep, '.'))
                    else:
                        float(seconds)
            except ValueError:
                raise ValueError(self.tr("Invalid numeric values in angle"))
            return 'azimute', degrees, minutes, seconds, sep

    def show_message(self, message):
        QMessageBox.information(self, self.tr('Information'), message)

    def _parse_and_validate_data(self):
        initial_point_text = self.initial_point_edit.text()
        if not initial_point_text:
            self.show_message(self.tr('Initial coordinate is required.'))
            return None, None, None
        try:
            x, y = map(float, initial_point_text.split(','))
            initial_point = QgsPointXY(x, y)
        except ValueError:
            self.show_message(self.tr('Invalid format for initial coordinate.'))
            return None, None, None
        distances_azimuths = []
        max_precision = 0
        for row in range(self.table.rowCount()):
            vertex_item = self.table.item(row, 0)
            azimuth_item = self.table.item(row, 1)
            distance_item = self.table.item(row, 2)
            adjacency_item = self.table.item(row, 3)
            if not azimuth_item or not distance_item or not azimuth_item.text() or not distance_item.text():
                continue
            try:
                azimuth_dms = azimuth_item.text().replace(',', '.')
                if any(x in azimuth_dms.upper() for x in ['N', 'S', 'E', 'W']):
                    azimuth = self.convert_rumo_to_decimal(azimuth_dms)
                else:
                    azimuth = self.convert_dms_to_decimal(azimuth_dms)
                distance_text = distance_item.text().replace(',', '.')
                distance = float(distance_text)
                if '.' in distance_text:
                    precision = len(distance_text.split('.')[1])
                    if precision > max_precision:
                        max_precision = precision
                vertex = vertex_item.text() if vertex_item else ''
                adjacency = adjacency_item.text() if adjacency_item else ''
                distances_azimuths.append((vertex, azimuth, distance, adjacency, azimuth_item.text()))
            except (ValueError, IndexError):
                err_template = self.tr("Invalid data in row %1.")
                err_text = err_template.replace("%1", str(row + 1))
                self.show_message(err_text)
                return None, None, None
        if not distances_azimuths:
            self.show_message(self.tr('No valid azimuth and distance provided.'))
            return None, None, None
        return initial_point, distances_azimuths, max_precision

    def process_as_point(self):
        initial_point, distances_azimuths, max_precision = self._parse_and_validate_data()
        if initial_point is None:
            return
        points = self.calculate_points(initial_point, distances_azimuths)
        self.create_point_shapefile(self.output_shapefile_edit.text(), points, distances_azimuths, max_precision)

    def process_as_line(self):
        initial_point, distances_azimuths, max_precision = self._parse_and_validate_data()
        if initial_point is None:
            return
        points = self.calculate_points(initial_point, distances_azimuths)
        self.create_line_shapefile(self.output_shapefile_edit.text(), points, distances_azimuths, max_precision)

    def process_as_polygon(self):
        initial_point, distances_azimuths, max_precision = self._parse_and_validate_data()
        if initial_point is None:
            return
        points = self.calculate_points(initial_point, distances_azimuths)
        if len(points) < 3:
            self.show_message(self.tr('At least 2 segments are required to create a polygon.'))
            return
        reply = QMessageBox.question(self,
                                       self.tr('Confirm Last Vertex'),
                                       self.tr('Do you want to use the last vertex to close the polygon?'),
                                       QMessageBox.Yes | QMessageBox.No,
                                       QMessageBox.Yes)
        if reply == QMessageBox.Yes:
            points_for_polygon = points
        else:
            points_for_polygon = points[:-1]
        if len(points_for_polygon) < 3:
            self.show_message(self.tr('At least 2 segments are required to create a polygon.'))
            return
        self.create_polygon_shapefile(self.output_shapefile_edit.text(), points_for_polygon, max_precision)

    def calculate_points(self, initial_point, distance_azimuths):
        points = [initial_point]
        for vertex, azimuth, distance, adjacency, azimuth_dms in distance_azimuths:
            last_point = points[-1]
            azimuth_rad = math.radians(azimuth)
            dx = distance * math.sin(azimuth_rad)
            dy = distance * math.cos(azimuth_rad)
            new_point = QgsPointXY(last_point.x() + dx, last_point.y() + dy)
            points.append(new_point)
        return points

    def create_line_shapefile(self, shapefile_path, points, distances_azimuths, max_precision):
        fields = QgsFields()
        fields.append(QgsField('ID', QVariant.Int))
        fields.append(QgsField(self.tr('Vertex'), QVariant.String))
        fields.append(QgsField(self.tr('Angle'), QVariant.String))
        fields.append(QgsField(self.tr('Distance'), QVariant.Double, 'double', 20, max_precision))
        fields.append(QgsField(self.tr('Adjacency'), QVariant.String))
        crs = QgsProject.instance().crs()
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            if shapefile_path.lower().endswith('.shp'):
                driver_name = 'ESRI Shapefile'
            elif shapefile_path.lower().endswith('.gpkg'):
                driver_name = 'GPKG'
            else:
                driver_name = 'ESRI Shapefile'
            writer = QgsVectorFileWriter(
                shapefile_path, 'UTF-8', fields,
                QgsWkbTypes.LineString, crs, driver_name
            )
            if writer.hasError() != QgsVectorFileWriter.NoError:
                self.show_message(self.tr(f'Error creating file: {writer.errorMessage()}'))
                return
            del writer
            layer = QgsVectorLayer(shapefile_path, os.path.basename(shapefile_path), 'ogr')
        else:
            layer = QgsVectorLayer(
                f'LineString?crs={crs.authid()}',
                self.tr('Output Lines'),
                'memory'
            )
            pr = layer.dataProvider()
            pr.addAttributes(fields)
            layer.updateFields()
        pr = layer.dataProvider()
        for i in range(len(points) - 1):
            feature = QgsFeature()
            feature.setGeometry(QgsGeometry.fromPolylineXY([points[i], points[i + 1]]))
            vertex, azimuth, distance, adjacency, azimuth_dms = distances_azimuths[i]
            try:
                parsed = self.parse_angle(azimuth_dms)
                if parsed[0] == 'rumo':
                    angle_type, degrees, minutes, seconds, direction, sep = parsed
                    if sep and sep in seconds:
                        sec_parts = seconds.split(sep)
                        sec_int = sec_parts[0].zfill(2)
                        sec_dec = sec_parts[1] if len(sec_parts) > 1 else '000'
                        sec_formatted = f"{sec_int}{sep}{sec_dec}"
                    else:
                        sec_formatted = seconds.zfill(2)
                    angle_formatted = f"{degrees.zfill(2)}°{minutes.zfill(2)}'{sec_formatted}\"{direction}"
                else:
                    angle_type, degrees, minutes, seconds, sep = parsed
                    if sep and sep in seconds:
                        sec_parts = seconds.split(sep)
                        sec_int = sec_parts[0].zfill(2)
                        sec_dec = sec_parts[1] if len(sec_parts) > 1 else '000'
                        sec_formatted = f"{sec_int}{sep}{sec_dec}"
                    else:
                        sec_formatted = seconds.zfill(2)
                    angle_formatted = f"{degrees.zfill(2)}°{minutes.zfill(2)}'{sec_formatted}\""
            except ValueError:
                err_template = self.tr("Invalid data in row %1.")
                err_text = err_template.replace("%1", str(i + 1))
                self.show_message(err_text)
                return
            distance_formatted = round(distance, max_precision)
            attributes = [i + 1, vertex, angle_formatted, distance_formatted, adjacency]
            feature.setAttributes(attributes)
            pr.addFeature(feature)
        layer.updateExtents()
        QgsProject.instance().addMapLayer(layer)
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            self.show_message(self.tr(f'File created at {shapefile_path}'))
        else:
            self.show_message(self.tr('Temporary layer created.'))

    def create_point_shapefile(self, shapefile_path, points, distances_azimuths, max_precision):
        fields = QgsFields()
        fields.append(QgsField('ID', QVariant.Int))
        fields.append(QgsField(self.tr('Vertex'), QVariant.String))
        fields.append(QgsField(self.tr('Angle'), QVariant.String))
        fields.append(QgsField(self.tr('Distance'), QVariant.Double, 'double', 20, max_precision))
        fields.append(QgsField(self.tr('Adjacency'), QVariant.String))
        crs = QgsProject.instance().crs()
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            if shapefile_path.lower().endswith('.shp'):
                driver_name = 'ESRI Shapefile'
            elif shapefile_path.lower().endswith('.gpkg'):
                driver_name = 'GPKG'
            else:
                driver_name = 'ESRI Shapefile'
            writer = QgsVectorFileWriter(
                shapefile_path, 'UTF-8', fields,
                QgsWkbTypes.Point, crs, driver_name
            )
            if writer.hasError() != QgsVectorFileWriter.NoError:
                self.show_message(self.tr(f'Error creating file: {writer.errorMessage()}'))
                return
            del writer
            layer = QgsVectorLayer(shapefile_path, os.path.basename(shapefile_path), 'ogr')
        else:
            layer = QgsVectorLayer(
                f'Point?crs={crs.authid()}',
                self.tr('Output Points'),
                'memory'
            )
            pr = layer.dataProvider()
            pr.addAttributes(fields)
            layer.updateFields()
        pr = layer.dataProvider()
        for i in range(len(points) - 1):
            feature = QgsFeature()
            feature.setGeometry(QgsGeometry.fromPointXY(points[i]))
            vertex, azimuth, distance, adjacency, azimuth_dms = distances_azimuths[i]
            try:
                parsed = self.parse_angle(azimuth_dms)
                if parsed[0] == 'rumo':
                    angle_type, degrees, minutes, seconds, direction, sep = parsed
                    if sep and sep in seconds:
                        sec_parts = seconds.split(sep)
                        sec_int = sec_parts[0].zfill(2)
                        sec_dec = sec_parts[1] if len(sec_parts) > 1 else '000'
                        sec_formatted = f"{sec_int}{sep}{sec_dec}"
                    else:
                        sec_formatted = seconds.zfill(2)
                    angle_formatted = f"{degrees.zfill(2)}°{minutes.zfill(2)}'{sec_formatted}\"{direction}"
                else:
                    angle_type, degrees, minutes, seconds, sep = parsed
                    if sep and sep in seconds:
                        sec_parts = seconds.split(sep)
                        sec_int = sec_parts[0].zfill(2)
                        sec_dec = sec_parts[1] if len(sec_parts) > 1 else '000'
                        sec_formatted = f"{sec_int}{sep}{sec_dec}"
                    else:
                        sec_formatted = seconds.zfill(2)
                    angle_formatted = f"{degrees.zfill(2)}°{minutes.zfill(2)}'{sec_formatted}\""
            except ValueError:
                err_template = self.tr("Invalid data in row %1.")
                err_text = err_template.replace("%1", str(i + 1))
                self.show_message(err_text)
                return
            distance_formatted = round(distance, max_precision)
            attributes = [i + 1, vertex, angle_formatted, distance_formatted, adjacency]
            feature.setAttributes(attributes)
            pr.addFeature(feature)
        layer.updateExtents()
        QgsProject.instance().addMapLayer(layer)
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            self.show_message(self.tr(f'File created at {shapefile_path}'))
        else:
            self.show_message(self.tr('Temporary layer created.'))

    def create_polygon_shapefile(self, shapefile_path, points, max_precision):
        fields = QgsFields()
        fields.append(QgsField('ID', QVariant.Int))
        crs = QgsProject.instance().crs()
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            if shapefile_path.lower().endswith('.shp'):
                driver_name = 'ESRI Shapefile'
            elif shapefile_path.lower().endswith('.gpkg'):
                driver_name = 'GPKG'
            else:
                driver_name = 'ESRI Shapefile'
            writer = QgsVectorFileWriter(
                shapefile_path, 'UTF-8', fields,
                QgsWkbTypes.Polygon, crs, driver_name
            )
            if writer.hasError() != QgsVectorFileWriter.NoError:
                self.show_message(self.tr(f'Error creating file: {writer.errorMessage()}'))
                return
            del writer
            layer = QgsVectorLayer(shapefile_path, os.path.basename(shapefile_path), 'ogr')
        else:
            layer = QgsVectorLayer(
                f'Polygon?crs={crs.authid()}',
                self.tr('Output Polygon'),
                'memory'
            )
            pr = layer.dataProvider()
            pr.addAttributes(fields)
            layer.updateFields()
        pr = layer.dataProvider()
        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromPolygonXY([points]))
        feature.setAttributes([1])
        pr.addFeature(feature)
        layer.updateExtents()
        QgsProject.instance().addMapLayer(layer)
        if shapefile_path and shapefile_path != self.tr('Temporary Layer'):
            self.show_message(self.tr(f'File created at {shapefile_path}'))
        else:
            self.show_message(self.tr('Temporary layer created.'))

class AzimuthToolPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.dialog = None
        self.toolbar = None
        self.plugin_dir = os.path.dirname(__file__)
        self.translator = QTranslator()
        settings = QSettings()
        locale = settings.value("locale/userLocale", QLocale.system().name()).split('_')[0]
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'azimuth_tool_{locale}.qm')
        if os.path.exists(locale_path):
            self.translator.load(f'azimuth_tool_{locale}.qm', os.path.join(self.plugin_dir, 'i18n'))
            QgsApplication.instance().installTranslator(self.translator)
        else:
            fallback_locale_path = os.path.join(self.plugin_dir, 'i18n', 'azimuth_tool_en.qm')
            if os.path.exists(fallback_locale_path):
                self.translator.load('azimuth_tool_en.qm', os.path.join(self.plugin_dir, 'i18n'))
                QgsApplication.instance().installTranslator(self.translator)

    def tr(self, message):
        return QCoreApplication.translate('AzimuthToolDialog', message)

    def initGui(self):
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.png')
        self.toolbar = QToolBar(self.tr('Azimuth Tool'))
        self.iface.addToolBar(self.toolbar)
        self.add_action(icon_path, self.tr('Azimuth Tool'), self.run, parent=self.toolbar)

    def add_action(self, icon_path, text, callback, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        parent.addAction(action)
        self.action = action

    def unload(self):
        if self.toolbar:
            self.iface.mainWindow().removeToolBar(self.toolbar)
        self.iface.removePluginMenu(self.tr('&Azimuth Tool'), self.action)

    def run(self):
        if not self.dialog:
            self.dialog = AzimuthToolDialog()
        self.dialog.show()
