# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AutoLayoutTool
                                 A QGIS plugin
 Create a layout with the current map canvas extent
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
 Icon from https://www.freepik.com/free-icon/layout_14181101.htm
 Usefull Links:

                              -------------------
        begin                : 2021-11-19
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Sylvain Théry
        email                : sylvain.thery@cnrs.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon, QColor, QKeySequence
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QShortcut
from qgis.core import QgsProject, QgsPrintLayout, QgsLayoutItemMap, QgsLayoutItemLegend, QgsLayoutPoint, \
    QgsLayoutItemScaleBar, QgsUnitTypes, QgsLayoutItemPicture, QgsLayoutSize, QgsApplication, QgsLayoutItemPage

import os.path
# Initialize Qt resources from file resources.py
from .resources import *


class AutoLayoutTool:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # debug state for pycharm (pro version only :-() python debug server
        self.debug = False
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'AutoLayoutTool_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)
        try:
            locale.setlocale(
                locale.LC_ALL,
                QSettings().value('locale/userLocale')
            )
        except:
            pass

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&AutoLayoutTool')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        if self.debug:
            try:
                import pydevd_pycharm
                pydevd_pycharm.settrace('localhost', port=53100, stdoutToServer=True, stderrToServer=True)
            except:
                pass

    # noinspection PyMethodMayBeStatic

    def tr(self, message):
        """
        Get the translation for a string using Qt translation API.
        We implement this ourselves since we do not inherit QObject.
        :param message: String for translation.
        :type message: str, QString
        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('AutoLayoutTool', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = ':/plugins/create_layout/layout.png'
        self.my_action = self.add_action(
            icon_path,
            text=self.tr(u'Create a new layout based on current extent '),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True
        self.my_action = QAction("Test Plugin", self.iface.mainWindow())
        self.iface.registerMainWindowAction(self.my_action, "Ctrl+I")  # action triggered by Ctrl+I

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&AutoLayoutTool'),
                action)
            self.iface.removeToolBarIcon(action)
        self.iface.unregisterMainWindowAction(self.my_action)

    def create_layout(self, layout_name):
        """
        Layout Management: creates new layout and delete previous
        :param layout_name:
        :return
        :rtype:QgsLayout, QgsProject.instance().layoutManager()
        """
        project = QgsProject.instance()
        manager = project.layoutManager()
        layouts_list = manager.printLayouts()
        for layout in layouts_list:
            if layout.name() == layout_name:
                reply = QMessageBox.question(None, self.tr(u'Delete layout...'),
                                             self.tr(
                                                 u"There's already a layout named '%s'\nDo you want to delete it?")
                                             % layout_name,
                                             QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
                if reply == QMessageBox.No:
                    return
                else:
                    manager.removeLayout(layout)
                    print(self.tr(u"Previous layout named '%s' removed... ") % layout_name)
        layout = QgsPrintLayout(project)
        layout.initializeDefaults()
        layout.setName(layout_name)
        return layout, manager

    def compute_layout_orientation(self, e, layout):
        """
        Determine and set best layout orientation
        :param e: iface.mapCanvas().extent()
        :param layout: QgsLayout
        :rtype: bool, float, float, float, float, float
        """
        print('Creating layout')
        map_width = e.xMaximum() - e.xMinimum()
        map_height = e.yMaximum() - e.yMinimum()

        page_size = QgsApplication.pageSizeRegistry().find(layout.pageCollection().page(0).pageSize())  # eg. 'A4' str
        landscape = False
        if map_width <= map_height:
            # portrait
            if self.debug: print("Portrait")
            layout.pageCollection().page(0).setPageSize(page_size, QgsLayoutItemPage.Orientation.Portrait)
        else:
            # landscape
            if self.debug: print("Landscape")
            layout.pageCollection().page(0).setPageSize(page_size, QgsLayoutItemPage.Orientation.Landscape)
            landscape = True
        # Calculate scale ratio between layout size and map size
        layout_width = layout.pageCollection().page(0).pageSize().width()
        layout_height = layout.pageCollection().page(0).pageSize().height()
        if landscape:
            scale_ratio = (layout_width / map_width)
            if map_height * scale_ratio > layout_height:
                scale_ratio = map_height / layout_height
        else:
            scale_ratio = layout_height / map_height
            if map_width * scale_ratio > layout_width:
                scale_ratio = map_height / layout_height
        return landscape, layout_height, layout_width, map_height, map_width, scale_ratio

    def calculate_map_scale(self, landscape, layout, layout_height, layout_width, map_height, map_width, scale_ratio):
        """
        Calculate map scale
        :param landscape: boolean
        :param layout: QgsLayout
        :param layout_height: float
        :param layout_width: float
        :param map_height: float
        :param map_width: float
        :param scale_ratio: float
        :rtype: float, float, QgsLayoutItemMap
        """
        print(self.tr(u'Calculating scale'))
        my_map = QgsLayoutItemMap(layout)
        if self.debug: print("mapw: %r / maph: %r" % (map_width, map_height))
        if self.debug: print("scale ratio: %r" % scale_ratio)
        previous_height = map_height
        previous_width = map_width
        if landscape:
            if self.debug: print("Landscape")
            map_width = layout_width
            map_height = round(map_height * scale_ratio, 3)  # makes qgis bug if not rounded 3
            if self.debug: print("ori_mapw: %r / ori_maph: %r" % (map_width, map_height))
            # workaround don't now why in special case it has to be changed !:#
            if map_height > layout_height:
                map_height = layout_height
                map_width = round(previous_width / scale_ratio, 3)
                if self.debug: print("corr_mapw: %r / corr_maph: %r" % (map_width, map_height))
        else:
            if self.debug: print("Portrait")
            map_width = round(map_width * scale_ratio, 3)
            map_height = layout_height
            if self.debug: print("ori_mapw: %r / ori_maph: %r" % (map_width, map_height))
            # workaround: don't now why in special case it has to be changed ! (extent is nearly a square 4 instance)
            if map_width > map_height:
                map_width = layout_width
                map_height = round(previous_height / scale_ratio, 3)
                if self.debug: print("corr_mapw: %r / corr_maph: %r" % (map_width, map_height))

        if self.debug: print("final_mapw: %r / final_maph: %r" % (map_width, map_height))
        return map_height, map_width, my_map

    def add_map(self, e, layout, layout_height, layout_width, map_height, map_width, margin, my_map):
        """
        Add map to the layout
        :param e: iface.mapCanvas().extent()
        :param layout: QgsLayout
        :param layout_height: float
        :param layout_width: float
        :param map_height: float
        :param map_width: float
        :param margin: float
        :param my_map: QgsLayoutItemMap
        :rtype: float, float, float, float
        """
        print('Adding map')
        map_width = map_width - margin
        map_height = map_height - margin
        my_map.setRect(0, 0, map_width, map_height)
        my_map.setExtent(e)
        layout.addLayoutItem(my_map)
        my_map.refresh()
        map_real_width = my_map.rect().size().width()
        map_real_height = my_map.rect().size().height()
        x_offset = (layout_width - map_real_width) / 2
        y_offset = (layout_height - map_real_height) / 2
        if self.debug: print("x_offset: %r / y_offset: %r" % (x_offset, y_offset))
        if self.debug: print("real_mapw: %r / real_maph: %r" % (map_real_width, map_real_height))
        my_map.setBackgroundColor(QColor(255, 255, 255, 0))
        my_map.setFrameEnabled(True)
        my_map.attemptMove(QgsLayoutPoint(x_offset, y_offset, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(my_map)
        return map_real_height, map_real_width, x_offset, y_offset

    def add_legend(self, layout, x_offset, y_offset):
        """
        Add legend to the layout
        :param layout: QgsLayout
        :param x_offset: float
        :param y_offset: float
        :return: None
        """
        print(self.tr(u"Adding legend"))
        layers_to_add = [l for l in QgsProject().instance().layerTreeRoot().children() if l.isVisible()]
        legend = QgsLayoutItemLegend(layout)
        legend.setTitle(self.tr(u'Legend'))
        legend.setAutoUpdateModel(False)
        group = legend.model().rootGroup()
        group.clear()
        for l in layers_to_add:
            if l.nodeType() == 0:
                subgroup = group.addGroup(l.name())
                checked = l.checkedLayers()
                for c in checked:
                    subgroup.addLayer(c)
            elif l.nodeType() == 1:
                group.addLayer(l.layer())
        layout.addItem(legend)
        legend.adjustBoxSize()
        legend.setFrameEnabled(True)
        legend.attemptMove(QgsLayoutPoint(x_offset, y_offset, QgsUnitTypes.LayoutMillimeters))
        legend.refresh()

    def add_scalebar(self, layout, map_real_height, map_real_width, my_map, x_offset, y_offset):
        """
        Add scalebar to the layout
        :param layout: QgsLayout
        :param map_real_height: float
        :param map_real_width: float
        :param my_map: QgsLayoutItemMap
        :param x_offset: float
        :param y_offset: float
        :return: None
        """
        print(self.tr(u'Adding scale bar'))
        scalebar = QgsLayoutItemScaleBar(layout)
        scalebar.setStyle('Single Box')
        scalebar.setLinkedMap(my_map)
        scalebar.applyDefaultSize()
        scalebar.applyDefaultSettings()
        scalebar.setUnits(QgsUnitTypes.DistanceKilometers)
        scalebar.setUnitsPerSegment(scalebar.unitsPerSegment() / 1000)
        scalebar.setUnitLabel('km')
        scalebar.update()
        layout.addLayoutItem(scalebar)
        scalebar.attemptMove(QgsLayoutPoint(map_real_width + x_offset - scalebar.rect().size().width() - 5,
                                            map_real_height + y_offset - scalebar.rect().size().height() - 5,
                                            QgsUnitTypes.LayoutMillimeters))

    def add_north_arrow(self, layout, manager, map_real_height, x_offset, y_offset):
        """
        Adds north arrow to the layout
        :param layout: QgsLayout
        :param manager: QgsProject.instance().layoutManager()
        :param map_real_height: float
        :param x_offset: float
        :param y_offset: float
        :return: None
        """
        print(self.tr(u"Adding north arrow"))
        north = QgsLayoutItemPicture(layout)
        north.setPicturePath(os.path.dirname(__file__) + "/north-arrow.svg")
        layout.addLayoutItem(north)
        north.attemptResize(QgsLayoutSize(8, 13, QgsUnitTypes.LayoutMillimeters))
        north.attemptMove(QgsLayoutPoint(3 + x_offset, map_real_height + y_offset - 15, QgsUnitTypes.LayoutMillimeters))

    def run(self):
        """
        Creates a layout with a map of the current interface extent, with legend, scalebar and north arrow
        :return: None
        """
        if self.first_start == True:
            self.first_start = False
            shortcut = QShortcut(QKeySequence(Qt.ControlModifier + Qt.ShiftModifier + Qt.Key_1),
                                 self.iface.mainWindow())
            shortcut.setContext(Qt.ApplicationShortcut)
            shortcut.activated.connect(self.run)


        margin = 10
        layout_name = self.tr('Automatic layout')
        print('--------------------------------')
        print(self.tr(u'AutoLayoutTool starts'))
        print('--------------------------------')

        # Create layout
        layout, manager = self.create_layout(layout_name)

        # Determine and set best layout orientation
        landscape, layout_height, layout_width, map_height, map_width, scale_ratio = self.compute_layout_orientation(
                                                                             self.iface.mapCanvas().extent(), layout)

        # Calculate scale
        map_height, map_width, my_map = self.calculate_map_scale(landscape, layout, layout_height, layout_width,
                                                                 map_height, map_width, scale_ratio)

        # Add map
        map_real_height, map_real_width, x_offset, y_offset = self.add_map(self.iface.mapCanvas().extent(), layout,
                                                                           layout_height, layout_width, map_height,
                                                                           map_width, margin, my_map)

        # Add legend
        self.add_legend(layout, x_offset, y_offset)

        # Add scale bar
        self.add_scalebar(layout, map_real_height, map_real_width, my_map, x_offset, y_offset)

        # Add north arrow
        self.add_north_arrow(layout, manager, map_real_height, x_offset, y_offset)

        # Finally add layout to the project via its manager
        manager.addLayout(layout)
        self.iface.openLayoutDesigner(layout)
        print('--------------------------------')
        print("♪♪ This is the end, my friend ♪♪")
        print('--------------------------------')
