from qgis.core import (
    QgsProject, QgsVectorLayer, QgsRasterLayer, QgsSymbol, QgsSingleSymbolRenderer,
    QgsFillSymbol, QgsLineSymbol, QgsMarkerSymbol, QgsPalLayerSettings,
    QgsTextFormat, QgsVectorLayerSimpleLabeling, QgsCoordinateReferenceSystem,
    QgsLayerTreeLayer, QgsLayerTreeGroup, QgsSimpleLineSymbolLayer
)
from qgis.gui import QgsProjectionSelectionDialog
from qgis.PyQt.QtWidgets import (QDialog, QVBoxLayout, QPushButton, QAction, 
                                   QCheckBox, QDialogButtonBox, QGroupBox, QHBoxLayout,
                                   QToolButton, QMenu, QLabel, QDoubleSpinBox, QMessageBox)
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtCore import QTimer, QSettings, Qt
from qgis.PyQt.QtWidgets import QListWidget, QListWidgetItem
from qgis.utils import iface
import os
import logging

# Set up logging
logging.basicConfig()
logger = logging.getLogger('AutoLayer')
logger.setLevel(logging.INFO)


# ==========================================
# CRS HANDLING COMPONENT
# ==========================================

class CRSSelectionDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Select Project CRS")
        self.setMinimumWidth(320)
        self.layout = QVBoxLayout()
        
        # Load preferred CRS from settings
        settings = QSettings()
        crs_string = settings.value('AutoLayer/preferred_crs_list', '')
        
        if crs_string:
            # User has preferred CRS
            crs_codes = [code.strip() for code in crs_string.split(',') if code.strip()]
            for epsg_code in crs_codes:
                crs = QgsCoordinateReferenceSystem(epsg_code)
                if crs.isValid():
                    btn = QPushButton(f"{epsg_code}\n{crs.description()}")
                    btn.setStyleSheet("text-align: center; padding: 8px;")
                    btn.clicked.connect(lambda _, e=epsg_code: self.select_crs(e))
                    self.layout.addWidget(btn)
        else:
            # No preferred CRS - show placeholder
            placeholder = QLabel(
                "Preferred CRS list will appear here\n\n"
                "Choose preferred CRS from\nAutoLayer Setting"
            )
            placeholder.setStyleSheet(
                "border: 1px solid #ccc; padding: 20px; "
                "text-align: center; color: #666;"
            )
            placeholder.setAlignment(Qt.AlignCenter)
            self.layout.addWidget(placeholder)
        
        # Always show "Choose Other..." button
        other_btn = QPushButton("Choose Other...")
        other_btn.setStyleSheet("text-align: center; padding: 8px; font-weight: bold;")
        other_btn.clicked.connect(self.open_native_crs_selector)
        self.layout.addWidget(other_btn)
        
        self.setLayout(self.layout)
        self.selected_crs = None
    
    def select_crs(self, crs):
        self.selected_crs = crs
        self.accept()
        
    def open_native_crs_selector(self):
        dlg = QgsProjectionSelectionDialog(self)
        if dlg.exec_():
            crs = dlg.crs()
            if crs.isValid():
                self.selected_crs = crs.authid()
                self.accept()


# ==========================================
# PREFERRED CRS MANAGER DIALOG
# ==========================================

class PreferredCRSManagerDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Manage Preferred CRS")
        self.resize(500, 400)
        
        layout = QVBoxLayout()
        
        # Info label
        info_label = QLabel("Manage your preferred CRS list (Maximum 10):")
        layout.addWidget(info_label)
        
        # List widget to show CRS
        self.crs_list_widget = QListWidget()
        layout.addWidget(self.crs_list_widget)
        
        # Buttons layout
        button_layout = QHBoxLayout()
        
        self.add_button = QPushButton("Add CRS")
        self.add_button.clicked.connect(self.add_crs)
        button_layout.addWidget(self.add_button)
        
        self.remove_button = QPushButton("Remove CRS")
        self.remove_button.clicked.connect(self.remove_crs)
        button_layout.addWidget(self.remove_button)
        
        button_layout.addStretch()
        layout.addLayout(button_layout)
        
        # OK/Cancel buttons
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
        
        # Load existing preferred CRS
        self.load_preferred_crs()
        self.update_button_states()
    
    def load_preferred_crs(self):
        """Load preferred CRS from settings"""
        settings = QSettings()
        crs_string = settings.value('AutoLayer/preferred_crs_list', '')
        
        if crs_string:
            crs_codes = [code.strip() for code in crs_string.split(',') if code.strip()]
            for epsg_code in crs_codes:
                crs = QgsCoordinateReferenceSystem(epsg_code)
                if crs.isValid():
                    display_text = f"{epsg_code} - {crs.description()}"
                    item = QListWidgetItem(display_text)
                    item.setData(Qt.UserRole, epsg_code)  # Store EPSG code
                    self.crs_list_widget.addItem(item)
    
    def add_crs(self):
        """Add a new CRS to the list"""
        # Check limit
        if self.crs_list_widget.count() >= 10:
            QMessageBox.warning(
                self,
                "Maximum Limit Reached",
                "You can add a maximum of 10 preferred CRS."
            )
            return
        
        # Open QGIS CRS selector
        dlg = QgsProjectionSelectionDialog(self)
        if dlg.exec_():
            crs = dlg.crs()
            if crs.isValid():
                epsg_code = crs.authid()
                
                # Check for duplicates
                for i in range(self.crs_list_widget.count()):
                    if self.crs_list_widget.item(i).data(Qt.UserRole) == epsg_code:
                        QMessageBox.warning(
                            self,
                            "Duplicate CRS",
                            "This CRS already exists in the list."
                        )
                        return
                
                # Add to list
                display_text = f"{epsg_code} - {crs.description()}"
                item = QListWidgetItem(display_text)
                item.setData(Qt.UserRole, epsg_code)
                self.crs_list_widget.addItem(item)
                self.update_button_states()
    
    def remove_crs(self):
        """Remove selected CRS from the list"""
        current_item = self.crs_list_widget.currentItem()
        if current_item:
            row = self.crs_list_widget.row(current_item)
            self.crs_list_widget.takeItem(row)
            self.update_button_states()
    
    def update_button_states(self):
        """Update button enabled/disabled states"""
        self.remove_button.setEnabled(self.crs_list_widget.count() > 0)
        self.add_button.setEnabled(self.crs_list_widget.count() < 10)
    
    def get_preferred_crs_list(self):
        """Get list of EPSG codes"""
        crs_codes = []
        for i in range(self.crs_list_widget.count()):
            item = self.crs_list_widget.item(i)
            crs_codes.append(item.data(Qt.UserRole))
        return crs_codes
    
    def save_preferred_crs(self):
        """Save preferred CRS list to settings"""
        crs_codes = self.get_preferred_crs_list()
        settings = QSettings()
        settings.setValue('AutoLayer/preferred_crs_list', ','.join(crs_codes))
    
    def accept(self):
        """Save and close"""
        self.save_preferred_crs()
        super().accept()


# ==========================================
# SETTINGS DIALOG COMPONENT
# ==========================================

class SettingsDialog(QDialog):
    def __init__(self, settings_dict, parent=None):
        super().__init__(parent)
        self.setWindowTitle("AutoLayer Settings")
        self.setMinimumWidth(400)
        
        self.settings_dict = settings_dict
        self.checkboxes = {}
        
        main_layout = QVBoxLayout()
        
        # CRS Group
        crs_group = QGroupBox("CRS Settings")
        crs_layout = QVBoxLayout()
        
        # CRS dialog checkbox with button
        crs_dialog_layout = QHBoxLayout()
        self.checkboxes['crs_dialog'] = QCheckBox("CRS selection dialog on startup")
        crs_dialog_layout.addWidget(self.checkboxes['crs_dialog'])
        
        preferred_crs_button = QPushButton("Choose Preferred CRS")
        preferred_crs_button.clicked.connect(self.open_preferred_crs_manager)
        crs_dialog_layout.addWidget(preferred_crs_button)
        crs_dialog_layout.addStretch()
        
        crs_layout.addLayout(crs_dialog_layout)
        
        self.checkboxes['auto_crs_vector'] = QCheckBox("Auto-assign CRS for vector layers")
        self.checkboxes['auto_crs_raster'] = QCheckBox("Auto-assign CRS for raster layers")
        crs_layout.addWidget(self.checkboxes['auto_crs_vector'])
        crs_layout.addWidget(self.checkboxes['auto_crs_raster'])
        crs_group.setLayout(crs_layout)
        main_layout.addWidget(crs_group)
        
        # Symbology Group
        symbology_group = QGroupBox("Symbology Settings")
        symbology_layout = QVBoxLayout()
        self.checkboxes['auto_symbology_point'] = QCheckBox("Auto symbology for point layers")
        self.checkboxes['auto_symbology_polygon'] = QCheckBox("Auto symbology for polygon layers")
        self.checkboxes['auto_symbology_line'] = QCheckBox("Auto symbology for line layers")
        self.checkboxes['auto_symbology_custom'] = QCheckBox("Auto symbology for EO, Tile, Grid and Cutline")
        symbology_layout.addWidget(self.checkboxes['auto_symbology_point'])
        symbology_layout.addWidget(self.checkboxes['auto_symbology_polygon'])
        symbology_layout.addWidget(self.checkboxes['auto_symbology_line'])
        symbology_layout.addWidget(self.checkboxes['auto_symbology_custom'])
        symbology_group.setLayout(symbology_layout)
        main_layout.addWidget(symbology_group)
        
        # AutoSave Group
        autosave_group = QGroupBox("AutoSave Settings")
        autosave_layout = QVBoxLayout()
        
        # AutoSave Project
        project_save_layout = QHBoxLayout()
        self.checkboxes['autosave_project_enable'] = QCheckBox("AutoSave project")
        project_save_layout.addWidget(self.checkboxes['autosave_project_enable'])
        project_save_layout.addWidget(QLabel("Interval:"))
        self.project_interval_spin = QDoubleSpinBox()
        self.project_interval_spin.setMinimum(0.1)
        self.project_interval_spin.setMaximum(1440)  # 24 hours max
        self.project_interval_spin.setSingleStep(0.5)
        self.project_interval_spin.setValue(settings_dict.get('autosave_project_interval', 5.0))
        self.project_interval_spin.setSuffix(" (m)")
        project_save_layout.addWidget(self.project_interval_spin)
        self.checkboxes['autosave_project_prompt'] = QCheckBox("Prompt")
        project_save_layout.addWidget(self.checkboxes['autosave_project_prompt'])
        project_save_layout.addStretch()
        autosave_layout.addLayout(project_save_layout)
        
        # AutoSave Layer Edit
        layer_save_layout = QHBoxLayout()
        self.checkboxes['autosave_layer_enable'] = QCheckBox("AutoSave layer edit")
        layer_save_layout.addWidget(self.checkboxes['autosave_layer_enable'])
        layer_save_layout.addWidget(QLabel("Interval:"))
        self.layer_interval_spin = QDoubleSpinBox()
        self.layer_interval_spin.setMinimum(0.1)
        self.layer_interval_spin.setMaximum(1440)
        self.layer_interval_spin.setSingleStep(0.5)
        self.layer_interval_spin.setValue(settings_dict.get('autosave_layer_interval', 5.0))
        self.layer_interval_spin.setSuffix(" (m)")
        layer_save_layout.addWidget(self.layer_interval_spin)
        self.checkboxes['autosave_layer_prompt'] = QCheckBox("Prompt")
        layer_save_layout.addWidget(self.checkboxes['autosave_layer_prompt'])
        layer_save_layout.addStretch()
        autosave_layout.addLayout(layer_save_layout)
        
        autosave_group.setLayout(autosave_layout)
        main_layout.addWidget(autosave_group)
        
        # Other Settings Group
        other_group = QGroupBox("Other Settings")
        other_layout = QVBoxLayout()
        self.checkboxes['auto_selection'] = QCheckBox("Auto enable 'Selection' mode")
        self.checkboxes['auto_collapse_raster'] = QCheckBox("Auto-collapse raster layers")
        self.checkboxes['auto_enable_editing'] = QCheckBox("Auto Enable Toggle Editing")
        self.checkboxes['auto_show_feature_count'] = QCheckBox("Auto show feature count")
        other_layout.addWidget(self.checkboxes['auto_selection'])
        other_layout.addWidget(self.checkboxes['auto_collapse_raster'])
        other_layout.addWidget(self.checkboxes['auto_enable_editing'])
        other_layout.addWidget(self.checkboxes['auto_show_feature_count'])
        other_group.setLayout(other_layout)
        main_layout.addWidget(other_group)
        
        # Load current settings into checkboxes
        for key, checkbox in self.checkboxes.items():
            checkbox.setChecked(self.settings_dict.get(key, True))
        
        # Buttons
        button_box = QDialogButtonBox(
            QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Apply
        )
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        button_box.button(QDialogButtonBox.Apply).clicked.connect(self.apply_settings)
        main_layout.addWidget(button_box)
        
        self.setLayout(main_layout)
    
    def apply_settings(self):
        for key, checkbox in self.checkboxes.items():
            self.settings_dict[key] = checkbox.isChecked()
        # Save interval values
        self.settings_dict['autosave_project_interval'] = self.project_interval_spin.value()
        self.settings_dict['autosave_layer_interval'] = self.layer_interval_spin.value()
    
    def accept(self):
        self.apply_settings()
        super().accept()

    def open_preferred_crs_manager(self):
        """Open the Preferred CRS Manager dialog"""
        dlg = PreferredCRSManagerDialog(self)
        dlg.exec_()


# ==========================================
# MAIN PLUGIN CLASS
# ==========================================

class AutoLayer:
    def __init__(self, iface):
        self.iface = iface
        self.project = QgsProject.instance()
        self.plugin_dir = os.path.dirname(__file__)
        self.is_enabled = True
        
        # Initialize settings
        self.settings = QSettings()
        self.feature_settings = {
            'crs_dialog': self.settings.value('AutoLayer/crs_dialog', True, type=bool),
            'auto_crs_vector': self.settings.value('AutoLayer/auto_crs_vector', True, type=bool),
            'auto_crs_raster': self.settings.value('AutoLayer/auto_crs_raster', True, type=bool),
            'auto_selection': self.settings.value('AutoLayer/auto_selection', True, type=bool),
            'auto_symbology_point': self.settings.value('AutoLayer/auto_symbology_point', True, type=bool),
            'auto_symbology_polygon': self.settings.value('AutoLayer/auto_symbology_polygon', True, type=bool),
            'auto_symbology_line': self.settings.value('AutoLayer/auto_symbology_line', True, type=bool),
            'auto_symbology_custom': self.settings.value('AutoLayer/auto_symbology_custom', True, type=bool),
            'auto_collapse_raster': self.settings.value('AutoLayer/auto_collapse_raster', True, type=bool),
            'auto_labeling': self.settings.value('AutoLayer/auto_labeling', True, type=bool),
            'autosave_project_enable': self.settings.value('AutoLayer/autosave_project_enable', True, type=bool),
            'autosave_project_interval': self.settings.value('AutoLayer/autosave_project_interval', 5.0, type=float),
            'autosave_project_prompt': self.settings.value('AutoLayer/autosave_project_prompt', True, type=bool),
            'autosave_layer_enable': self.settings.value('AutoLayer/autosave_layer_enable', True, type=bool),
            'autosave_layer_interval': self.settings.value('AutoLayer/autosave_layer_interval', 5.0, type=float),
            'autosave_layer_prompt': self.settings.value('AutoLayer/autosave_layer_prompt', True, type=bool),
            'auto_enable_editing': self.settings.value('AutoLayer/auto_enable_editing', True, type=bool),
            'auto_show_feature_count': self.settings.value('AutoLayer/auto_show_feature_count', True, type=bool),
        }
        
        # New 16-color palette
        self.color_palette = [
            '#e41a1c', '#3579b1', '#00e4f6', '#0000ff', 
            '#ff00ff', '#ff69b4', '#5e17eb', '#ffa500',
            '#00fa9a', '#e10052', '#9bbce7', '#c8ff6d',
            '#22c89e', '#ffd93d', '#008e9b', '#ff9671'
        ]
        self.current_color_index = 0
        
        # Color indices for each custom layer type
        self.custom_layer_color_indices = {
            'eo': 3,
            'tile': 3,
            'grid': 3,
            'cutline': 3
        }
        
        # Custom symbology rules
        self.custom_rules = {
            'eo': {'type': 'point', 'color': '#fdbf6f', 'keep_default_symbol': True},
            'tile': {
                'type': 'polygon',
                'fill': 'no',
                'outline_width': '1.0',
                'outline_color': '#3579b1',
                'label': {'field': 'TileID', 'size': 11, 'color': '#ffc843'}
            },
            'grid': {
                'type': 'polygon',
                'fill': 'no',
                'outline_width': '1.0',
                'outline_color': '#74cee3',
                'label': {'field': 'id', 'size': 11, 'color': '#ffc843'}
            },
            'cutline': {
                'type': 'line_or_polygon',
                'width': '0.5',
                'color': '#4daf4a'
            }
        }
        
        # AutoSave tracking
        self.project_autosave_timer = QTimer()
        self.project_autosave_timer.timeout.connect(self.autosave_project)
        self.layer_autosave_timers = {}  # {layer_id: QTimer}
        self.layer_edit_connections = {}  # {layer_id: connection}

    # ==========================================
    # GUI INITIALIZATION COMPONENT
    # ==========================================

    def initGui(self):
        self.toolbar = self.iface.addToolBar("AutoLayer")
        self.toolbar.setObjectName("AutoLayerToolbar")
        
        # Main toggle action
        icon_path = os.path.join(self.plugin_dir, 'autolayer.png')
        self.action = QAction(
            QIcon(icon_path),
            "AutoLayer",
            self.iface.mainWindow()
        )
        
        self.action.setCheckable(True)
        self.action.setChecked(self.is_enabled)
        self.action.triggered.connect(self.toggle_plugin)
        self.action.setWhatsThis("Click to enable/disable automatic layer processing")
        
        # Create tool button with dropdown
        self.tool_button = QToolButton()
        self.tool_button.setDefaultAction(self.action)
        self.tool_button.setPopupMode(QToolButton.MenuButtonPopup)
        
        # Create menu for dropdown
        self.settings_menu = QMenu()
        settings_action = self.settings_menu.addAction("Settings...")
        settings_action.triggered.connect(self.open_settings_dialog)
        
        self.tool_button.setMenu(self.settings_menu)
        self.toolbar.addWidget(self.tool_button)
        
        # Connect signals
        self.iface.initializationCompleted.connect(self.on_init_complete)
        self.project.layersAdded.connect(self.process_new_layers)
        
        # AutoSave signal connections
        self.project.layersAdded.connect(self.on_layers_added_for_autosave)
        self.project.layersRemoved.connect(self.on_layers_removed_for_autosave)
        self.project.projectSaved.connect(self.on_project_saved)
        self.project.dirtySet.connect(self.on_project_modified)
        self.project.layersAdded.connect(self.setup_layer_edit_tracking)
        self.project.layersAdded.connect(self.enable_editing_for_new_layers)
        self.project.layersAdded.connect(self.show_feature_count_for_new_layers)

    def toggle_plugin(self):
        self.is_enabled = self.action.isChecked()
        
        if self.is_enabled:
            self.iface.messageBar().pushMessage(
                "AutoLayer", "Plugin ENABLED - Will process new layers", 
                level=0, duration=3
            )
        else:
            self.iface.messageBar().pushMessage(
                "AutoLayer", "Plugin DISABLED - Will ignore new layers", 
                level=1, duration=3
            )

    def open_settings_dialog(self):
        dlg = SettingsDialog(self.feature_settings, parent=self.iface.mainWindow())
        if dlg.exec_():
            # Save settings to QSettings
            for key, value in self.feature_settings.items():
                self.settings.setValue(f'AutoLayer/{key}', value)
            
            # Stop project autosave timer if disabled
            if not self.feature_settings.get('autosave_project_enable', True):
                self.project_autosave_timer.stop()
                logger.info("Project autosave disabled")
            
            # Stop all layer autosave timers if disabled
            if not self.feature_settings.get('autosave_layer_enable', True):
                for timer in self.layer_autosave_timers.values():
                    timer.stop()
                logger.info("Layer autosave disabled")

    # ==========================================
    # LAYER DETECTION COMPONENT
    # ==========================================

    def has_plugin_styling(self, layer):
        """
        Check if the layer already has styling that indicates it was processed by this plugin
        """
        if not isinstance(layer, QgsVectorLayer):
            return False
            
        # Check if layer has custom property set by our plugin
        if layer.customProperty("autolayer_processed"):
            return True
            
        # For custom rule layers, check if they have the expected styling
        layer_name_lower = layer.name().lower()
        
        for rule_name, rule_data in self.custom_rules.items():
            if rule_name in layer_name_lower:
                # Check if this layer type has the expected renderer
                renderer = layer.renderer()
                if renderer and isinstance(renderer, QgsSingleSymbolRenderer):
                    symbol = renderer.symbol()
                    if symbol:
                        # For line layers and cutline polygons
                        if rule_data['type'] in ['line', 'line_or_polygon'] and layer.geometryType() in [1, 2]:
                            if symbol.symbolLayerCount() > 0:
                                symbol_layer = symbol.symbolLayer(0)
                                if (hasattr(symbol_layer, 'width') and 
                                    abs(symbol_layer.width() - float(rule_data.get('width', rule_data.get('outline_width', '1.0')))) < 0.1):
                                    return True
                break
                
        return False

    # ==========================================
    # INITIALIZATION HANDLING COMPONENT
    # ==========================================

    def on_init_complete(self):
        if self.feature_settings['crs_dialog']:
            if not self.project.fileName() and not self.project.mapLayers():
                dlg = CRSSelectionDialog(parent=self.iface.mainWindow())
                if dlg.exec_():
                    if dlg.selected_crs:
                        crs = QgsCoordinateReferenceSystem(dlg.selected_crs)
                        if crs.isValid():
                            self.project.setCrs(crs)
                            logger.info(f"Project CRS set to: {crs.authid()}")
        
        if self.feature_settings['auto_selection']:
            self.iface.actionSelect().trigger()

    # ==========================================
    # LAYER PROCESSING COMPONENT
    # ==========================================

    def process_new_layers(self, layers):
        if not self.is_enabled:
            return
        
        project_crs = self.project.crs()
        has_target_raster = False
        processed_count = 0
        
        for layer in layers:
            try:
                if not layer or not isinstance(layer, (QgsVectorLayer, QgsRasterLayer)):
                    continue
                
                # Skip layers that already have plugin styling
                if self.has_plugin_styling(layer):
                    logger.info(f"Skipping already processed layer: {layer.name()}")
                    continue
                
                logger.info(f"Processing NEW layer: {layer.name()} (ID: {layer.id()})")
                processed_count += 1
                
                # CRS assignment based on layer type
                layer_crs = layer.crs()
                should_assign_crs = False
                
                if isinstance(layer, QgsVectorLayer) and self.feature_settings['auto_crs_vector']:
                    should_assign_crs = True
                elif isinstance(layer, QgsRasterLayer) and self.feature_settings['auto_crs_raster']:
                    should_assign_crs = True
                
                # Only assign CRS if layer has no CRS or invalid CRS
                if should_assign_crs:
                    if not layer_crs.isValid() or layer_crs.authid() == '' or layer_crs.authid() == 'USER:100000':
                        layer.setCrs(project_crs, False)
                        logger.info(f"Assigned project CRS: {project_crs.authid()}")
                
                if isinstance(layer, QgsVectorLayer):
                    self.apply_symbology(layer)
                    # Mark as processed by our plugin
                    layer.setCustomProperty("autolayer_processed", "true")
                elif isinstance(layer, QgsRasterLayer):
                    if self.feature_settings['auto_collapse_raster']:
                        source_lower = layer.source().lower()
                        if any(source_lower.endswith(ext) for ext in ['.pix', '.tif', '.tiff']):
                            has_target_raster = True
                            logger.info(f"Found target raster: {layer.name()} ({layer.source()})")
                    
            except Exception as e:
                logger.error(f"Error processing layer {layer.name()}: {str(e)}")
        
        logger.info(f"Processed {processed_count} new layers")
        
        if has_target_raster and self.feature_settings['auto_collapse_raster']:
            QTimer.singleShot(1000, self.process_raster_layers)

    def enable_editing_for_new_layers(self, layers):
        """Enable edit mode for newly added vector layers"""
        if not self.is_enabled:
            return
        
        if not self.feature_settings.get('auto_enable_editing', True):
            return
        
        for layer in layers:
            try:
                if not isinstance(layer, QgsVectorLayer):
                    continue
                
                if not layer.isEditable():
                    layer.startEditing()
                    logger.info(f"Edit mode enabled for layer: {layer.name()}")
                    
            except Exception as e:
                logger.error(f"Error enabling edit mode for {layer.name()}: {str(e)}")

    def show_feature_count_for_new_layers(self, layers):
        """Enable show feature count for newly added vector layers"""
        if not self.is_enabled:
            return
        
        if not self.feature_settings.get('auto_show_feature_count', True):
            return
        
        # Use a small delay to ensure layer tree is updated
        QTimer.singleShot(50, lambda: self._apply_feature_count(layers))
    
    def _apply_feature_count(self, layers):
        """Internal method to apply feature count"""
        root = self.project.layerTreeRoot()
        
        for layer in layers:
            try:
                if not isinstance(layer, QgsVectorLayer):
                    continue
                
                # Find the layer tree node for this layer
                layer_tree_node = root.findLayer(layer.id())
                
                if layer_tree_node:
                    # Set the property to show feature count
                    layer_tree_node.setCustomProperty("showFeatureCount", True)
                    logger.info(f"Feature count enabled for layer: {layer.name()}")
                else:
                    logger.warning(f"Layer tree node not found for: {layer.name()}")
                    
            except Exception as e:
                logger.error(f"Error enabling feature count for {layer.name()}: {str(e)}")

    # ==========================================
    # RASTER PROCESSING COMPONENT
    # ==========================================

    def process_raster_layers(self):
        try:
            root = QgsProject.instance().layerTreeRoot()
            target_extensions = ['.pix', '.tif', '.tiff']
            
            def process_group(group):
                for child in group.children():
                    if isinstance(child, QgsLayerTreeGroup):
                        process_group(child)
                    elif isinstance(child, QgsLayerTreeLayer):
                        layer = child.layer()
                        if isinstance(layer, QgsRasterLayer):
                            source_lower = layer.source().lower()
                            if any(source_lower.endswith(ext) for ext in target_extensions):
                                child.setExpanded(False)
            
            process_group(root)
            
        except Exception as e:
            logger.error(f"Error processing raster layers: {str(e)}")

    # ==========================================
    # COLOR MANAGEMENT COMPONENT
    # ==========================================

    def get_next_custom_color(self, layer_type):
        if layer_type not in self.custom_layer_color_indices:
            return None
            
        if self.custom_layer_color_indices[layer_type] == 3:
            self.custom_layer_color_indices[layer_type] += 1
            # Return the appropriate color based on rule type
            if layer_type == 'cutline':
                return self.custom_rules[layer_type]['color']
            elif layer_type == 'eo':
                return self.custom_rules[layer_type]['color']
            else:
                return self.custom_rules[layer_type].get('outline_color', self.custom_rules[layer_type].get('color'))
        
        color_index = self.custom_layer_color_indices[layer_type]
        color = self.color_palette[color_index % len(self.color_palette)]
        
        self.custom_layer_color_indices[layer_type] = (color_index + 1)
        if self.custom_layer_color_indices[layer_type] >= len(self.color_palette):
            self.custom_layer_color_indices[layer_type] = 3
            
        return color

    # ==========================================
    # SYMBOLOGY APPLICATION COMPONENT
    # ==========================================

    def apply_symbology(self, layer):
        layer_name_lower = layer.name().lower()
        rule = None
        layer_type = None
        
        # Check if this is a custom layer (EO, Tile, Grid, Cutline)
        is_custom_layer = False
        for rule_name in self.custom_rules.keys():
            if rule_name in layer_name_lower:
                is_custom_layer = True
                layer_type = rule_name
                if self.feature_settings['auto_symbology_custom']:
                    rule = self.custom_rules[rule_name]
                break
        
        # If it's a custom layer and custom symbology is enabled, apply custom rule
        if rule:
            self.apply_custom_rule(layer, rule, layer_type)
        # If it's a custom layer but custom symbology is disabled, skip styling
        elif is_custom_layer and not self.feature_settings['auto_symbology_custom']:
            logger.info(f"Skipping custom symbology for {layer.name()} (custom symbology disabled)")
            return
        # Otherwise apply geometry-based symbology if enabled
        else:
            geom_type = layer.geometryType()
            should_apply = False
            
            if geom_type == 0 and self.feature_settings['auto_symbology_point']:
                should_apply = True
            elif geom_type == 1 and self.feature_settings['auto_symbology_line']:
                should_apply = True
            elif geom_type == 2 and self.feature_settings['auto_symbology_polygon']:
                should_apply = True
            
            if should_apply:
                self.apply_sequential_symbology(layer)
            else:
                logger.info(f"Skipping symbology for {layer.name()} (geometry type disabled)")
                return
        
        layer.triggerRepaint()

    def apply_custom_rule(self, layer, rule, layer_type):
        try:
            geom_type = layer.geometryType()
            color = self.get_next_custom_color(layer_type)
            
            # Handle cutline rule for both line and polygon geometry types
            if layer_type == 'cutline' and geom_type in [1, 2]:
                if geom_type == 1:  # Line
                    symbol = QgsLineSymbol.createSimple({'width': rule['width']})
                    symbol.setColor(QColor(color))
                else:  # Polygon
                    symbol = QgsFillSymbol()
                    symbol.deleteSymbolLayer(0)
                    line_layer = QgsSimpleLineSymbolLayer(QColor(color), float(rule['width']))
                    symbol.appendSymbolLayer(line_layer)
                layer.setRenderer(QgsSingleSymbolRenderer(symbol))
                
            # Handle EO point rule
            elif rule['type'] == 'point' and geom_type == 0:
                symbol = QgsMarkerSymbol.createSimple({
                    'name': 'circle',
                    'size': rule.get('size', '4.0')
                }) if not rule.get('keep_default_symbol', False) else QgsMarkerSymbol()
                symbol.setColor(QColor(color))
                layer.setRenderer(QgsSingleSymbolRenderer(symbol))
                
            # Handle tile and grid polygon rules
            elif rule['type'] == 'polygon' and geom_type == 2:
                symbol = QgsFillSymbol()
                symbol.deleteSymbolLayer(0)
                line_layer = QgsSimpleLineSymbolLayer(QColor(color), float(rule['outline_width']))
                symbol.appendSymbolLayer(line_layer)
                layer.setRenderer(QgsSingleSymbolRenderer(symbol))
                
                # Apply labels if specified and labeling is enabled
                if 'label' in rule and self.feature_settings['auto_labeling']:
                    label_settings = QgsPalLayerSettings()
                    label_settings.fieldName = rule['label']['field']
                    text_format = QgsTextFormat()
                    text_format.setSize(rule['label']['size'])
                    text_format.setColor(QColor(rule['label']['color']))
                    label_settings.setFormat(text_format)
                    layer.setLabeling(QgsVectorLayerSimpleLabeling(label_settings))
                    layer.setLabelsEnabled(True)
                
        except Exception as e:
            logger.error(f"Error applying custom rule: {str(e)}")

    def apply_sequential_symbology(self, layer):
        try:
            color = QColor(self.color_palette[self.current_color_index])
            self.current_color_index = (self.current_color_index + 1) % len(self.color_palette)
            
            geom_type = layer.geometryType()
            
            if geom_type == 0:
                symbol = QgsMarkerSymbol.createSimple({'name': 'diamond', 'size': '4.4'})
            elif geom_type == 1:
                symbol = QgsLineSymbol.createSimple({'width': '0.6'})
            else:
                symbol = QgsFillSymbol()
                symbol.deleteSymbolLayer(0)
                line_layer = QgsSimpleLineSymbolLayer(color, 1.0)
                symbol.appendSymbolLayer(line_layer)
            
            if geom_type != 2:
                symbol.setColor(color)
            layer.setRenderer(QgsSingleSymbolRenderer(symbol))
            
        except Exception as e:
            logger.error(f"Error applying sequential symbology: {str(e)}")

    # ==========================================
    # AUTOSAVE COMPONENT
    # ==========================================
    
    def on_layers_added_for_autosave(self, layers):
        """Trigger project autosave timer when layers are added"""
        self.on_project_modified()
    
    def on_layers_removed_for_autosave(self, layer_ids):
        """Trigger project autosave timer when layers are removed"""
        self.on_project_modified()
        
        # Clean up timers for removed layers
        for layer_id in layer_ids:
            if layer_id in self.layer_autosave_timers:
                self.layer_autosave_timers[layer_id].stop()
                del self.layer_autosave_timers[layer_id]
            if layer_id in self.layer_edit_connections:
                del self.layer_edit_connections[layer_id]
    
    def on_project_modified(self, *args):
        """Trigger project autosave timer when project is modified"""
        if not self.feature_settings.get('autosave_project_enable', True):
            return
        if not self.project.fileName():
            return
        self.restart_project_autosave_timer()
    
    def on_project_saved(self):
        """Stop autosave timer when project is saved manually"""
        self.project_autosave_timer.stop()
        logger.info("Project saved - autosave timer stopped")

    def restart_project_autosave_timer(self):
        """Restart the project autosave timer"""
        self.project_autosave_timer.stop()
        interval_minutes = self.feature_settings.get('autosave_project_interval', 5.0)
        interval_ms = int(interval_minutes * 60 * 1000)
        self.project_autosave_timer.start(interval_ms)
        logger.info(f"Project autosave timer restarted: {interval_minutes} minutes")
    
    def autosave_project(self):
        """Auto-save the project"""
        try:
            # Double-check settings before saving
            if not self.feature_settings.get('autosave_project_enable', True):
                self.project_autosave_timer.stop()
                return
                
            if not self.project.fileName():
                logger.info("Project not saved yet, skipping autosave")
                self.project_autosave_timer.stop()
                return
            
            if self.feature_settings.get('autosave_project_prompt', True):
                reply = QMessageBox.question(
                    self.iface.mainWindow(),
                    "AutoSave Project",
                    "Do you want to save the project now?",
                    QMessageBox.Yes | QMessageBox.No
                )
                if reply == QMessageBox.No:
                    logger.info("User cancelled project autosave")
                    self.project_autosave_timer.stop()
                    return
            
            self.iface.actionSaveProject().trigger()
            self.iface.messageBar().pushMessage(
                "AutoLayer",
                "Project auto-saved successfully",
                level=0,
                duration=2
            )
            logger.info("Project auto-saved successfully")
            # Stop timer after saving (will restart on next layer add/remove)
            self.project_autosave_timer.stop()
            
        except Exception as e:
            logger.error(f"Error auto-saving project: {str(e)}")
            self.iface.messageBar().pushMessage(
                "AutoLayer",
                f"Error auto-saving project: {str(e)}",
                level=2,
                duration=3
            )
            self.project_autosave_timer.stop()
    
    def setup_layer_edit_tracking(self, layers):
        """Setup tracking for layer edits"""
        if not self.feature_settings.get('autosave_layer_enable', True):
            return
        
        for layer in layers:
            if isinstance(layer, QgsVectorLayer):
                layer_id = layer.id()
                
                # Connect to all edit signals
                if layer_id not in self.layer_edit_connections:
                    try:
                        # Geometry changes
                        layer.geometryChanged.connect(
                            lambda fid, geom, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        # Attribute value changes
                        layer.attributeValueChanged.connect(
                            lambda fid, idx, val, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        # Feature added
                        layer.featureAdded.connect(
                            lambda fid, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        # Feature deleted
                        layer.featureDeleted.connect(
                            lambda fid, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        # Field added
                        layer.attributeAdded.connect(
                            lambda idx, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        # Field deleted
                        layer.attributeDeleted.connect(
                            lambda idx, lid=layer_id, lyr=layer: self.on_layer_edited(lid, lyr)
                        )
                        
                        self.layer_edit_connections[layer_id] = True
                        logger.info(f"Edit tracking setup for layer: {layer.name()}")
                    except Exception as e:
                        logger.error(f"Error setting up edit tracking for {layer.name()}: {str(e)}")
    
    def on_layer_edited(self, layer_id, layer):
        """Called when a layer is edited"""
        if not self.feature_settings.get('autosave_layer_enable', True):
            return
        
        if not layer.isEditable():
            return
        
        # Stop existing timer for this layer if any
        if layer_id in self.layer_autosave_timers:
            self.layer_autosave_timers[layer_id].stop()
        else:
            # Create new timer for this layer
            timer = QTimer()
            timer.timeout.connect(lambda lid=layer_id, lyr=layer: self.autosave_layer(lid, lyr))
            self.layer_autosave_timers[layer_id] = timer
        
        # Start/restart timer
        interval_minutes = self.feature_settings.get('autosave_layer_interval', 5.0)
        interval_ms = int(interval_minutes * 60 * 1000)
        self.layer_autosave_timers[layer_id].start(interval_ms)
        logger.info(f"Layer autosave timer started for {layer.name()}: {interval_minutes} minutes")
    
    def autosave_layer(self, layer_id, layer):
        """Auto-save a specific layer"""
        try:
            if not layer.isEditable():
                logger.info(f"Layer {layer.name()} is not in edit mode, skipping autosave")
                return
            
            if self.feature_settings.get('autosave_layer_prompt', True):
                reply = QMessageBox.question(
                    self.iface.mainWindow(),
                    "AutoSave Layer",
                    f"Do you want to save edits for layer '{layer.name()}'?",
                    QMessageBox.Yes | QMessageBox.No
                )
                if reply == QMessageBox.No:
                    logger.info(f"User cancelled layer autosave for {layer.name()}")
                    if layer_id in self.layer_autosave_timers:
                        self.layer_autosave_timers[layer_id].stop()
                    return
            
            layer.commitChanges()
            
            # Re-enable editing mode immediately after saving
            layer.startEditing()
            
            self.iface.messageBar().pushMessage(
                "AutoLayer",
                f"Layer {layer.name()} auto-saved successfully",
                level=0,
                duration=2
            )
            logger.info(f"Layer {layer.name()} auto-saved successfully")
            
            # Stop timer after successful save (will restart on next edit)
            if layer_id in self.layer_autosave_timers:
                self.layer_autosave_timers[layer_id].stop()
                
        except Exception as e:
            logger.error(f"Error auto-saving layer {layer.name()}: {str(e)}")
            self.iface.messageBar().pushMessage(
                "AutoLayer",
                f"Error auto-saving layer {layer.name()}: {str(e)}",
                level=2,
                duration=3
            )

    # ==========================================
    # CLEANUP COMPONENT
    # ==========================================

    def unload(self):
        self.iface.initializationCompleted.disconnect(self.on_init_complete)
        self.project.layersAdded.disconnect(self.process_new_layers)
        
        # Disconnect AutoSave signals
        try:
            self.project.layersAdded.disconnect(self.on_layers_added_for_autosave)
            self.project.layersRemoved.disconnect(self.on_layers_removed_for_autosave)
            self.project.projectSaved.disconnect(self.on_project_saved)
            self.project.dirtySet.disconnect(self.on_project_modified)
            self.project.layersAdded.disconnect(self.setup_layer_edit_tracking)
            self.project.layersAdded.disconnect(self.enable_editing_for_new_layers)
            self.project.layersAdded.disconnect(self.show_feature_count_for_new_layers)
        except:
            pass
        
        # Stop all timers
        self.project_autosave_timer.stop()
        for timer in self.layer_autosave_timers.values():
            timer.stop()
        
        del self.toolbar