# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AreaAlongVector
                                 A QGIS plugin
 Create an area along a vector depending on a field value.
 Right and left sides can be distinguished.
 
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-08-07
        copyright            : (C) 2018 by Alexis DUPONT-ROC
        email                : alexis.dupont-roc@internet.lu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexis DUPONT-ROC'
__date__ = '2018-08-07'
__copyright__ = '(C) 2018 by Alexis DUPONT-ROC'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os.path
from PyQt5.QtCore import QCoreApplication, QVariant, QUrl
from PyQt5.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFeatureSink,
                       QgsFeature,
                       QgsGeometry,
                       QgsWkbTypes,
                       QgsFields,
                       QgsField,
                       QgsPointXY)



class AreaAlongVectorAlgorithm(QgsProcessingAlgorithm):
    """
    For each vector of the vector INPUT layer, two rectangles are created:
    one on the right side for the forth value, the other on the left for the return value.
    Rectangles widths are proportional to values read from the selected fields (for example VOL and R_VOL).
    The resulting rectangles are stored in an OUTPUT layer.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT' 
    INPUT = 'INPUT'
    FORTHFIELD = 'FORTHFIELD'
    BACKFIELD = 'BACKFIELD'
    MINLIMIT = 'MINLIMIT'
    MAXLIMIT = 'MAXLIMIT'
    MINWIDTH = 'MINWIDTH'
    MAXWIDTH = 'MAXWIDTH'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
 
        # We add the input vector features source. It can have only VectorLine geometry type.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input layer (MultiLine Vector Layer)'),
                [QgsProcessing.TypeVectorLine]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.FORTHFIELD,
                self.tr('Field to use for forth values as Int (right)'),
                None,
                self.INPUT,
                QgsProcessingParameterField.Numeric,
                allowMultiple=False,
                optional=False
            )
        )
        
        self.addParameter(
            QgsProcessingParameterField(
                self.BACKFIELD,
                self.tr('Field to use for back values as Int (left)'),
                None,
                self.INPUT,
                QgsProcessingParameterField.Numeric,
                allowMultiple=False,
                optional=False
            )
        )
                
        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINLIMIT,
                self.tr('Minimum value to be drawn. All values under are set to this limit.'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINWIDTH,
                self.tr('Rectangle''s width for minimum.'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterNumber(
                self.MAXLIMIT,
                self.tr('Maximum value to be drawn. All values over are set to this limit.'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=10000.0,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MAXWIDTH,
                self.tr('Rectangle''s width for maximum.'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=100.0,
                optional=True
            )
        )     
          
        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        
        # New fields in destination OUTPUT file
        outputfields = QgsFields()
        outputfields.append(QgsField("WAY", QVariant.Int))
        outputfields.append(QgsField("VAL", QVariant.Int))
        #fields.append(QgsField( "desc", QVariant.String ))
        
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, outputfields, QgsWkbTypes.Polygon, source.sourceCrs())

        # Retrieve other parameters        
        
        FORTHFIELD = self.parameterAsString(parameters, self.FORTHFIELD, context)
        BACKFIELD = self.parameterAsString(parameters, self.BACKFIELD, context)
        
        MINLIMIT = self.parameterAsDouble(parameters, self.MINLIMIT, context)
        MAXLIMIT = self.parameterAsDouble(parameters, self.MAXLIMIT, context)
        MINWIDTH = self.parameterAsDouble(parameters, self.MINWIDTH, context)
        MAXWIDTH = self.parameterAsDouble(parameters, self.MAXWIDTH, context)       
                
        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
  
# =================== Algorithm ======================================
  
            # fetch QgsGeometry from feature
            geom = feature.geometry()

            # Check geometry information of the feature
                 
            if geom.wkbType() == QgsWkbTypes.MultiLineString:
                # Process for every PolyLine
                multipolyline = geom.asMultiPolyline()
                for x in multipolyline:
                    #feedback.pushInfo("nbr items : {0}, {1}".format(len(multipolyline), len(x)))
                    for i in range(0, len(x) - 1):
                                         
                        # Read field values as int
                        fieldValueForth = feature[FORTHFIELD]
                        fieldValueBack = feature[BACKFIELD]
                        
                        #feedback.pushInfo("{0} Values F - B : {1} - {2}".format(i, fieldValueForth, fieldValueBack))
                        
                        # Transform NULL values to 0
                        if not fieldValueForth:
                            fieldValueForth = 0
                        
                        if not fieldValueBack:
                            fieldValueBack = 0

                        # Distance calculation
                        distR = self.calculateDistance(fieldValueForth, MINLIMIT, MINWIDTH, MAXLIMIT, MAXWIDTH)
                        distL = self.calculateDistance(fieldValueBack, MINLIMIT, MINWIDTH, MAXLIMIT, MAXWIDTH)

                        # add right rectangle
                        fields = source.fields()
                        featR = QgsFeature(fields)
                        featR.setGeometry(QgsGeometry.fromPolygonXY([self.calculateRightPolygon(x[i], x[i + 1], distR)]))
                        featR[0] = 1
                        featR[1] = fieldValueForth
                        sink.addFeature(featR, QgsFeatureSink.FastInsert)

                        # add left rectangle
                        featL = QgsFeature(fields)
                        featL.setGeometry(QgsGeometry.fromPolygonXY([self.calculateLeftPolygon(x[i], x[i + 1], distL)]))
                        featL[0] = 2
                        featL[1] = fieldValueBack
                        sink.addFeature(featL, QgsFeatureSink.FastInsert)
            else:
                feedback.pushInfo("Bad Layer geometry type : {}".format(geom.wkbType()))
            
# =========================================================     
            
            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm.
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Create areas along vectors'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Algorithms for vector layers'

    def icon(self):
        file=os.path.realpath(__file__)
        file = os.path.join(os.path.dirname(file),'icon.png')
        if not os.path.exists(file):
            return ''
        return QIcon(file)

    def helpUrl(self):
        file=os.path.realpath(__file__)
        file = os.path.join(os.path.dirname(file),'index.html')
        if not os.path.exists(file):
            return ''
        return QUrl.fromLocalFile(file).toString(QUrl.FullyEncoded)

    def shortHelpString(self):
        file=os.path.realpath(__file__)
        file = os.path.join(os.path.dirname(file),'AreaAlongVectorShortHelp.help')
        if not os.path.exists(file):
            return ''
        with open(file) as helpf:
            help=helpf.read()
        return help    
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return AreaAlongVectorAlgorithm()


# ==================== Calculation Functions =====================================   
    
    def calculateDistance(self, fieldValue, minV, minW, maxV, maxW):
        """
        Returns the field value converted into a distance regarding fixed limits
        """
        maxRectangleWidth = maxW
        minRectangleWidth = minW
        # over maxFieldLimit, draw a rectangle width of maxRectangleWidth
        maxFieldLimit = maxV
        # under minFieldLimit, draw a rectangle width of minRectangleWidth
        minFieldLimit = minV
        if fieldValue <= minFieldLimit:
            return minRectangleWidth
        elif fieldValue >= maxFieldLimit:
            return maxRectangleWidth
        else:
            return fieldValue * maxRectangleWidth / maxFieldLimit


    def calculateRightPolygon(self, p1, p2, dist):
        """
        Returns the points of the right side polygon
        """
        if dist == 0:
            points = [p1, p2, p2, p1]
            return points

        dn = ((p1.x() - p2.x()) ** 2 + (p1.y() - p2.y()) ** 2) ** 0.5
        if dn == 0:
            points = [p1, p2, p2, p1]
            return points

        # P3 = P2 to the right
        x3 = p2.x() + dist * (p2.y() - p1.y()) / dn
        y3 = p2.y() - dist * (p2.x() - p1.x()) / dn
        p3 = QgsPointXY(x3, y3)

        # P4 = P1 to the right
        x4 = p1.x() + dist * (p2.y() - p1.y()) / dn
        y4 = p1.y() - dist * (p2.x() - p1.x()) / dn
        p4 = QgsPointXY(x4, y4)

        points = [p1, p2, p3, p4]
        return points


    def calculateLeftPolygon(self, p1, p2, dist):
        """
        Returns the points of the left side polygon
        """
        if dist == 0:
            points = [p1, p2, p2, p1]
            return points

        dn = ((p1.x() - p2.x()) ** 2 + (p1.y() - p2.y()) ** 2) ** 0.5
        if dn == 0:
            points = [p1, p2, p2, p1]
            return points

        # P3 = P2 to the left
        x3 = p2.x() - dist * (p2.y() - p1.y()) / dn
        y3 = p2.y() + dist * (p2.x() - p1.x()) / dn
        p3 = QgsPointXY(x3, y3)

        # P4 = P1 to the left
        x4 = p1.x() - dist * (p2.y() - p1.y()) / dn
        y4 = p1.y() + dist * (p2.x() - p1.x()) / dn
        p4 = QgsPointXY(x4, y4)

        points = [p2, p1, p4, p3]
        return points