# -*- coding: utf-8 -*-
import os

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QFormLayout, QLineEdit, QCheckBox,
    QDialogButtonBox, QLabel, QHBoxLayout, QPushButton,
    QFileDialog, QSizePolicy
)
from qgis.PyQt.QtCore import QCoreApplication

def tr(self, message):
    return QCoreApplication.translate('AlterrixDefaultSaveLocation', message)

# Keys are not used in the dialog anymore, but left here for clarity/compatibility
ORG_KEY = 'plugins/AlterrixDefaultSaveLocation'
SET_KEY_ENABLED = ORG_KEY + '/enabled'
SET_KEY_PARENT = ORG_KEY + '/parentPath'
SET_KEY_FOLDER = ORG_KEY + '/folderName'


class SettingsDialog(QDialog):
    """
        The plugin reads the attributes after exec_():
        dlg.parent_path
        dlg.folder_name
        dlg.enabled
    """

    def __init__(self, parent=None, parent_path='', folder_name='', enabled=True):
        super().__init__(parent)
        self.setWindowTitle(self.tr('Default Save Location – Settings'))

        # Store initial values
        self.parent_path = parent_path or os.path.expanduser('~')
        # Do NOT fall back to '100 GIS-Daten'; keep what we received
        self.folder_name = folder_name
        self.enabled = enabled

        # --- UI: parent path label (read-only)
        # If no folder_name is set, the actual save path is the parent itself.
        # We therefore show the parent path in the label.
        self.lbl_parent = QLabel(self.parent_path)
        self.lbl_parent.setToolTip(self.tr('The path of the folder your save folder is located in.'))
        self.lbl_parent.setTextInteractionFlags(Qt.TextSelectableByMouse)

        # --- Folder entry
        self.ed_folder = QLineEdit(self.folder_name)
        self.ed_folder.setToolTip(self.tr(
            'Leave empty to save directly in the folder shown above (Path). '
            'If you enter a name, a subfolder will be used or created.'
        ))
        self.ed_folder.setMinimumWidth(180)
        self.ed_folder.setMaximumWidth(260)
        self.ed_folder.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)

        # --- Buttons: Browse / Reset
        self.btn_browse_folder = QPushButton(self.tr('Browse…'))
        self.btn_browse_folder.setMaximumWidth(100)

        self.btn_reset = QPushButton(self.tr('Reset'))
        self.btn_reset.setToolTip(self.tr(
            'Reset to project "100 GIS-Daten", "100 GIS Daten", "Geodata" or "Geodaten" - if present. '
            'Otherwise creates "100 GIS-Daten".'
        ))
        self.btn_reset.setMaximumWidth(80)

        # --- Activate checkbox
        self.chk_enabled = QCheckBox(self.tr('Activate'))
        self.chk_enabled.setToolTip(self.tr('Enable or disable the plugin here.'))
        self.chk_enabled.setChecked(self.enabled)

        # --- Layout
        lay = QVBoxLayout(self)

        # Path row
        path_row = QHBoxLayout()
        path_row.addWidget(QLabel(self.tr('Path:')))
        path_row.addWidget(self.lbl_parent, 1)
        lay.addLayout(path_row)

        # Folder + buttons
        form = QFormLayout()
        row = QHBoxLayout()
        row.addWidget(self.ed_folder)
        row.addWidget(self.btn_browse_folder)
        row.addWidget(self.btn_reset)
        form.addRow(QLabel(self.tr('Folder name:')), row)
        lay.addLayout(form)
        lay.addSpacing(16)

        # Buttons + checkbox
        btns = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, parent=self)

        bottom = QHBoxLayout()
        bottom.addStretch(1)
        bottom.addWidget(self.chk_enabled)
        bottom.addSpacing(12)
        bottom.addWidget(btns)
        lay.addLayout(bottom)

        # Signals
        self.btn_browse_folder.clicked.connect(self._on_browse_folder)
        self.btn_reset.clicked.connect(self._on_reset)
        btns.accepted.connect(self.accept)
        btns.rejected.connect(self.reject)

        # Optional tab order
        self.setTabOrder(self.ed_folder, self.btn_browse_folder)
        self.setTabOrder(self.btn_browse_folder, self.btn_reset)
        self.setTabOrder(self.btn_reset, self.chk_enabled)

    # ------------------------------------------------------------------

    def _on_browse_folder(self):
        """Select an actual folder and split into parent + subfolder name."""
        start = os.path.join(self.parent_path, self.folder_name)
        folder = QFileDialog.getExistingDirectory(self, 'Select save folder', start)
        if folder:
            folder = os.path.normpath(folder)
            self.parent_path, self.folder_name = os.path.dirname(folder), os.path.basename(folder)
            self.lbl_parent.setText(self.parent_path)
            self.ed_folder.setText(self.folder_name)

    # ------------------------------------------------------------------

    def _on_reset(self):
        """Use project-aware defaults; if no project path, use user-home fallback."""
        from .main import project_dir, detect_project_default_folder, get_fallback_unsaved_dir, DEFAULT_HYPHEN
        base = project_dir()
        detected, name = detect_project_default_folder(base)
        if not detected:  # no project path yet -> fallback under home
            detected = get_fallback_unsaved_dir()
            name = ''  # we treat it as the full parent folder directly
            self.parent_path = detected
            self.folder_name = ''
        else:
            self.parent_path = os.path.dirname(detected)
            self.folder_name = name

        self.lbl_parent.setText(self.parent_path)
        self.ed_folder.setText(self.folder_name)

    # ------------------------------------------------------------------

    def accept(self):
        """Expose the user's final selections to the plugin."""
        raw = (self.ed_folder.text() or '').strip()

        # IMPORTANT:
        # If the user leaves the field truly empty, treat it as "use parent folder" (empty subfolder).
        # If the user typed something, use that something.
        #
        # Corner case: If folder_name was previously empty, we showed the parent's basename
        # just as a *visual hint*. If the user didn’t touch the field, it might still contain
        # that basename text. We must NOT save that as a real subfolder unless the user changed it.
        #
        # Strategy:
        # - If original folder_name was empty AND raw equals os.path.basename(parent_path),
        #   we interpret that as "unchanged hint" => save empty.
        # - Otherwise, save whatever is in raw.
        if not (getattr(self, 'folder_name', '') or '').strip():
            # It was empty before opening the dialog
            parent_basename = os.path.basename(self.parent_path)
            if raw == parent_basename:
                # User didn’t change the hint -> keep empty (use parent folder)
                name = ''
            else:
                name = raw
        else:
            # Previously had a value; honor what's typed now
            name = raw

        # Update state ready for the plugin
        self.folder_name = name
        self.enabled = self.chk_enabled.isChecked()

        super().accept()