# -*- coding: utf-8 -*-
import os
import traceback

from qgis.PyQt.QtCore import QObject, QEvent, QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtWidgets import (
    QAction, QApplication, QFileDialog, QLineEdit, QMessageBox, QWidget, QMenu
)
from qgis.PyQt.QtGui import QIcon

from qgis.core import QgsProject, QgsSettings

from qgis.gui import QgsFileWidget   # guaranteed in QGIS ≥ 3.40

from pathlib import Path


# ---- Settings keys ----
ORG_KEY = 'plugins/AlterrixDefaultSaveLocation'
SET_KEY_ENABLED = ORG_KEY + '/enabled'
SET_KEY_PARENT = ORG_KEY + '/parentPath'
SET_KEY_FOLDER = ORG_KEY + '/folderName'

APP_FILTER_KEY = '_dsl_dialog_filter_installed'
APP_LINEFILTER_KEY = '_dsl_line_filter_installed'

DEFAULT_HYPHEN = '100 GIS-Daten'
DEFAULT_SPACE = '100 GIS Daten'
DEFAULT_FR = 'Geodata'
DEFAULT_ALT = 'Geodaten'

MENU_TEXT_SRC = 'Default Save Location...'
FALLBACK_UNSAVED_SUBFOLDER_SRC = "GIS data from unsaved QGIS projects"
FALLBACK_UNSAVED_SUBFOLDER = None


def get_fallback_unsaved_dir() -> str:
    home = Path.home()  # robust across Windows/macOS/Linux
    sub = FALLBACK_UNSAVED_SUBFOLDER or FALLBACK_UNSAVED_SUBFOLDER_SRC
    return str(home.joinpath(sub))




# ---- Utilities ----

def _critical(parent, title, text):
    try:
        QMessageBox.critical(parent, title, text)
    except Exception:
        pass


def project_dir():
    p = QgsProject.instance()

    # Prefer API function absolutePath() if available
    try:
        path = (p.absolutePath() or '').strip()
    except Exception:
        path = ''

    # Fallback: derive from fileName()
    if not path:
        fn = p.fileName()
        if fn:
            path = os.path.dirname(fn)

    # VERY IMPORTANT: if nothing found -> return None
    return path or None


def detect_project_default_folder(base_dir):
    """
    Returns (full_path, chosen_name).
    Prefers <project>/100 GIS-Daten, then <project>/100 GIS Daten,
    else creates 100 GIS-Daten.

    Returns (None, None) if base_dir is invalid.
    """
    if not base_dir:
        return None, None

    candidates = [
        (DEFAULT_HYPHEN, DEFAULT_HYPHEN),
        (DEFAULT_SPACE,  DEFAULT_SPACE),
        (DEFAULT_FR,     DEFAULT_FR),     
        (DEFAULT_ALT,    DEFAULT_ALT),     
    ]

    for dirname, label in candidates:
        path = os.path.join(base_dir, dirname)
        if os.path.isdir(path):            # only check, do not create
            return path, label

    # Nothing exists → **do not create**; just point to preferred default
    default_path = os.path.join(base_dir, DEFAULT_HYPHEN)
    return default_path, DEFAULT_HYPHEN



def read_parent_and_folder(s):
    parent = (s.value(SET_KEY_PARENT, '', str) or '').strip()
    folder = (s.value(SET_KEY_FOLDER, '', str) or '').strip()

    if not parent:
        parent = project_dir()

    # If folder is empty → return parent folder
    return parent, folder


def update_qgis_last_dirs(path):
    if not path:
        return
    qs = QgsSettings()
    for key in (
        '/UI/lastFileNameWidgetDir',
        '/UI/lastRasterFileDir',
        '/UI/lastVectorFileFilterDir',
        '/Processing/LastInputPath',
        '/Processing/LastOutputPath',
    ):
        qs.setValue(key, path)

def active_full_path(s):
    
    # Respect the Activate checkbox
    enabled = QgsSettings().value(SET_KEY_ENABLED, True, bool)
    parent, folder = read_parent_and_folder(s)

    
    # If disabled, just compute a path and return it — no directory creation,
    # no "last dirs" sync, no side effects.
    if not enabled:
        if not parent:
            # unsaved project fallback (but do not create)
            return os.path.normpath(get_fallback_unsaved_dir())
        return os.path.normpath(os.path.join(parent, folder)) if folder else parent


    # If there is NO project path yet (brand-new unsaved project),
    # use the cross-platform fallback under the user's home.
    if not parent:
        full = os.path.normpath(get_fallback_unsaved_dir())
        try:
            os.makedirs(full, exist_ok=True)
        except Exception:
            pass
        update_qgis_last_dirs(full)
        return full

    # Normal behavior once a project path exists:
    if folder:
        full = os.path.normpath(os.path.join(parent, folder))
    else:
        full = parent

    try:
        os.makedirs(full, exist_ok=True)
    except Exception:
        pass

    update_qgis_last_dirs(full)
    return full


# ---- Line edit normalization ----

class LineEditKeyFilter(QObject):
    """Prefix bare filenames with active folder; create parents for absolute paths."""

    def __init__(self, settings):
        super().__init__()
        self.s = settings

    def _normalize(self, le: QLineEdit):
        enabled = QgsSettings().value(SET_KEY_ENABLED, True, bool)
        if not enabled:
            return


        txt = (le.text() or '').strip()
        if not txt:
            return

        # Absolute path or path-like → ensure directory exists
        if os.path.isabs(txt) or any(sep in txt for sep in ('/', os.sep)):
            try:
                d = os.path.dirname(txt)
                if d:
                    os.makedirs(d, exist_ok=True)
                    update_qgis_last_dirs(d)
            except Exception:
                pass
            return

        # Bare filename → prefix with active folder
        base = active_full_path(self.s)
        le.setText(os.path.normpath(os.path.join(base, txt)))

    def eventFilter(self, obj, ev):
        try:
            if isinstance(obj, QLineEdit):
                if ev.type() == QEvent.KeyPress:
                    from qgis.PyQt.QtCore import Qt
                    if ev.key() in (Qt.Key_Return, Qt.Key_Enter):
                        self._normalize(obj)
        except Exception:
            pass
        return False


# ---- File widget hook ----

class FileWidgetHook(QObject):
    """Installs a normalizer into every QgsFileWidget's internal QLineEdit."""

    def __init__(self, settings):
        super().__init__()
        self.s = settings
        self.le_filter = LineEditKeyFilter(settings)

    def _attach(self, fw):
        try:
            le = fw.lineEdit()
            if le and not le.property('_dsl_attached'):
                le.installEventFilter(self.le_filter)
                try:
                    le.editingFinished.connect(lambda le=le: self.le_filter._normalize(le))
                except Exception:
                    pass
                le.setProperty('_dsl_attached', True)
        except Exception:
            pass

    def eventFilter(self, obj, ev):
        try:
            if ev.type() == QEvent.Show and isinstance(obj, QWidget):
                for fw in obj.findChildren(QgsFileWidget):
                    self._attach(fw)
        except Exception:
            pass
        return False


# ---- Browse dialog filter ----

class BrowseDialogFilter(QObject):
    """Set the starting directory for save dialogs."""

    def __init__(self, settings):
        super().__init__()
        self.s = settings

    def eventFilter(self, obj, ev):
        try:
            if ev.type() == QEvent.Show and isinstance(obj, QFileDialog):
                if obj.acceptMode() == QFileDialog.AcceptSave:
                    if QgsSettings().value(SET_KEY_ENABLED, True, bool):
                        p = active_full_path(self.s)
                        obj.setDirectory(p)
        except Exception:
            pass
        return False


# ---- Safe, i18n‑robust Settings menu insertion ----

def add_action_to_settings_menu(main_window, action):
    try:
        menu = main_window.findChild(QMenu, 'mSettingsMenu')
        if menu:
            menu.addAction(action)
            return menu
    except Exception:
        pass
    return None


def remove_action_from_settings_menu(menu, action):
    try:
        if menu and action:
            menu.removeAction(action)
    except Exception:
        pass


# ---- Plugin class ----

class AlterrixDefaultSaveLocationPlugin:

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.s = QgsSettings()

        self.action = None
        self.settings_menu_ref = None

        self.fw_hook = None
        self.browse_filter = None

        self._defer_connected = False

    # --------------------------

    def _load_project_folder(self):
        """Return folder stored in the project (or empty if none)."""
        proj = QgsProject.instance()
        folder = proj.readEntry("AlterrixDefaultSaveLocation", "folderName", "")[0]
        return folder.strip()

    def _save_project_folder(self, folder):
        """Store manually chosen folder inside the project file."""
        proj = QgsProject.instance()
        proj.writeEntry("AlterrixDefaultSaveLocation", "folderName", folder)

    def _initialize_from_current_project(self):
        parent = project_dir()
        if not parent:
            return False

        full, folder = detect_project_default_folder(parent)
        if not full:
            return False

        self.s.setValue(SET_KEY_PARENT, parent)
        self.s.setValue(SET_KEY_FOLDER, folder)
        self.s.setValue("plugins/AlterrixDefaultSaveLocation/firstRunDone", True)
        active_full_path(self.s)
        return True

    def _on_project_available(self, *args):
        if self._initialize_from_current_project():
            try: self.iface.projectRead.disconnect(self._on_project_available)
            except Exception: pass
            try: self.iface.newProjectCreated.disconnect(self._on_project_available)
            except Exception: pass
            try: QgsProject.instance().projectSaved.disconnect(self._on_project_available)
            except Exception: pass
            self._defer_connected = False

    def _clear_folder_on_project_change(self):
        """
        Clear only the stored subfolder name so a new project starts in auto-detect mode.
        Users who explicitly choose a folder in the dialog will set this again.
        """
        try:
            self.s.remove(SET_KEY_FOLDER)
        except Exception:
            pass

    def _refresh_active_base(self):
        parent = project_dir()

        if parent:
            # 1. Check project-specific folder
            project_folder = self._load_project_folder()

            if project_folder:
                # User manually set a folder for this project
                self.s.setValue(SET_KEY_PARENT, parent)
                self.s.setValue(SET_KEY_FOLDER, project_folder)

            else:
                # 2. No folder saved in project → auto-detect
                full, name = detect_project_default_folder(parent)
                self.s.setValue(SET_KEY_PARENT, parent)
                self.s.setValue(SET_KEY_FOLDER, name)

        else:
            # Unsaved project → fallback logic
            self.s.setValue(SET_KEY_PARENT, '')

        active_full_path(self.s)


    def initGui(self):
        # Detect user language, e.g. "de", "en"
        locale = QSettings().value("locale/userLocale")[0:2]

        # Build the expected filename
        qm_file = f"AlterrixDefaultSaveLocation_{locale}.qm"

        # Build full path
        locale_path = os.path.join(self.plugin_dir, "i18n", qm_file)

        # Load translation if exists
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Translate texts
        menu_text = QCoreApplication.translate('AlterrixDefaultSaveLocation', MENU_TEXT_SRC)
        fallback_unsaved = QCoreApplication.translate('AlterrixDefaultSaveLocation', FALLBACK_UNSAVED_SUBFOLDER_SRC)

        # set global so that get_fallback_unsaved_dir() uses localized name 
        global FALLBACK_UNSAVED_SUBFOLDER
        FALLBACK_UNSAVED_SUBFOLDER = fallback_unsaved

        s = QgsSettings()
        is_first_run = s.value("plugins/AlterrixDefaultSaveLocation/firstRunDone", False, bool) == False

        if is_first_run:
            if not self._initialize_from_current_project():
                try:
                    self.iface.projectRead.connect(self._on_project_available)
                    self.iface.newProjectCreated.connect(self._on_project_available)
                    QgsProject.instance().projectSaved.connect(self._on_project_available)
                    self._defer_connected = True
                except Exception:
                    pass


        try:
            # --- Icon
            icon_path = os.path.join(os.path.dirname(__file__), "icons", "folder.svg")
            icon = QIcon(icon_path)

            self.action = QAction(icon, menu_text, self.iface.mainWindow())
            try:
                self.action.setIconVisibleInMenu(True)
            except Exception:
                pass

            self.action.triggered.connect(self.open_settings)

            # Insert into Settings menu
            self.settings_menu_ref = add_action_to_settings_menu(
                self.iface.mainWindow(), self.action
            )

            # Install filters once per application
            app = QApplication.instance()

            if not getattr(app, APP_LINEFILTER_KEY, False):
                self.fw_hook = FileWidgetHook(self.s)
                app.installEventFilter(self.fw_hook)
                setattr(app, APP_LINEFILTER_KEY, True)

            if not getattr(app, APP_FILTER_KEY, False):
                self.browse_filter = BrowseDialogFilter(self.s)
                app.installEventFilter(self.browse_filter)
                setattr(app, APP_FILTER_KEY, True)

            # Ensure we re-evaluate the project state (saved vs. unsaved) and sync last dirs
            self._refresh_active_base()

            # Always react when the project context changes
            try:
                self.iface.projectRead.connect(self._refresh_active_base)
            except Exception:
                pass

            try:
                self.iface.newProjectCreated.connect(self._refresh_active_base)
            except Exception:
                pass

            try:
                QgsProject.instance().projectSaved.connect(self._refresh_active_base)
            except Exception:
                pass

        except Exception:
            title = QCoreApplication.translate('AlterrixDefaultSaveLocation', MENU_TEXT_SRC)
            _critical(self.iface.mainWindow(), title + " – initGui() error", traceback.format_exc())

    # --------------------------

    def unload(self):
        try:
            remove_action_from_settings_menu(self.settings_menu_ref, self.action)
        except Exception:
            pass

        try:
            app = QApplication.instance()

            if self.fw_hook:
                app.removeEventFilter(self.fw_hook)
            if self.browse_filter:
                app.removeEventFilter(self.browse_filter)

            if hasattr(app, APP_LINEFILTER_KEY):
                delattr(app, APP_LINEFILTER_KEY)
            if hasattr(app, APP_FILTER_KEY):
                delattr(app, APP_FILTER_KEY)

        except Exception:
            pass

        # Remove settings when plugin is disabled or uninstalled
        # s = QgsSettings()
        # s.remove("plugins/AlterrixDefaultSaveLocation")

        try:
            self.iface.projectRead.disconnect(self._refresh_active_base)
        except Exception:
            pass
        try:
            self.iface.newProjectCreated.disconnect(self._refresh_active_base)
        except Exception:
            pass
        try:
            QgsProject.instance().projectSaved.disconnect(self._refresh_active_base)
        except Exception:
            pass

        # Also disconnect deferred "first-run" hooks if they were set
        try:
            if getattr(self, "_defer_connected", False):
                try: self.iface.projectRead.disconnect(self._on_project_available)
                except Exception: pass
                try: self.iface.newProjectCreated.disconnect(self._on_project_available)
                except Exception: pass
                try: QgsProject.instance().projectSaved.disconnect(self._on_project_available)
                except Exception: pass
                self._defer_connected = False
        except Exception:
            pass

        self.action = None
        self.fw_hook = None
        self.browse_filter = None
        self.settings_menu_ref = None

    # --------------------------

    def open_settings(self):
        try:
            from .settings_dialog import SettingsDialog

            parent, folder = read_parent_and_folder(self.s)

            enabled = self.s.value(SET_KEY_ENABLED, True, bool)

            dlg = SettingsDialog(
                self.iface.mainWindow(),
                parent_path=parent,
                folder_name=folder,
                enabled=enabled
            )
            
            if dlg.exec_():
                # Persist
                self.s.setValue(SET_KEY_PARENT, dlg.parent_path)
                # persist in project file (per-project setting)
                self._save_project_folder(dlg.folder_name)

                # also persist in global settings (for active_full_path)
                self.s.setValue(SET_KEY_FOLDER, dlg.folder_name)
                self.s.setValue(SET_KEY_ENABLED, bool(getattr(dlg, 'enabled', True)))

                # Apply and sync
                active_full_path(self.s)

        except Exception:
            _critical(self.iface.mainWindow(), MENU_TEXT + " – Settings error",
                      traceback.format_exc())