# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AGLgis Dialog
                                 A QGIS plugin
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-24
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Caio Arantes
        email                : caiosimplicioarante@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

AGLgis is a graphical interface designed to simplify the use of the 
Sentinel-1 SAR Backscatter Analysis Ready Data Preparation in Google Earth Engine.
This plugin allows users to configure and run Sentinel-1 SAR data processing 
without the need to write code manually.
"""

# =============================================================================
# STANDARD LIBRARY IMPORTS
# =============================================================================
import os
import sys
import re
import json
import array
import tempfile
import datetime
import platform
import subprocess
import traceback
import zipfile
import webbrowser
import io
from functools import partial
from datetime import timedelta

# =============================================================================
# THIRD-PARTY IMPORTS
# =============================================================================
import requests
import pandas as pd
import numpy as np
import geopandas as gpd
import plotly.express as px
import plotly.graph_objects as go
import urllib.request

# Earth Engine imports
from ee_s1_ard import S1ARDImageCollection
import ee

# Additional dependencies
import qgis
# Additional scientific computing imports
import processing
from dateutil.relativedelta import relativedelta
from shapely.geometry import shape
from scipy.signal import savgol_filter
from osgeo import gdal

# =============================================================================
# QGIS IMPORTS
# =============================================================================

# QGIS Core imports
from qgis.core import (
    # Basic QGIS classes
    QgsMessageLog,
    Qgis,
    QgsWkbTypes,
    QgsProject,
    QgsApplication,
    
    # Vector handling
    QgsVectorLayer,
    QgsVectorFileWriter,
    QgsFeatureRequest,
    QgsFeature,
    QgsGeometry,
    QgsField,
    
    # Raster handling
    QgsRasterLayer,
    QgsRasterShader,
    QgsColorRampShader,
    QgsSingleBandPseudoColorRenderer,
    QgsMultiBandColorRenderer,
    QgsContrastEnhancement,
    QgsRasterBandStats,
    
    # Style and visualization
    QgsStyle,
    QgsColorRamp,
    QgsLayerTreeLayer,
    
    # Coordinate systems and geometry
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsRectangle,
    
    # Processing and feedback
    QgsProcessingFeedback,
    
    # Map layers
    QgsMapLayer,
)

# Qt GUI imports
from qgis.PyQt.QtGui import (
    QFont, 
    QColor
)

from qgis.PyQt.QtCore import (
    QDate, 
    Qt, 
    QVariant, 
    QSettings, 
    QTimer, 
    QEvent
)

from qgis.PyQt.QtWidgets import (
    # Application and main window
    QApplication,
    QMainWindow,
    QWidget,
    QDialog,
    
    # Layout managers
    QVBoxLayout,
    QHBoxLayout,
    QGridLayout,
    
    # Input widgets
    QCheckBox,
    QDateEdit,
    QPushButton,
    QToolButton,
    
    # Display widgets
    QLabel,
    QTextBrowser,
    QScrollArea,
    
    # Dialog components
    QMessageBox,
    QFileDialog,
    QDialogButtonBox,
    
    # Size policies
    QSizePolicy,
)

# QGIS PyQt imports
from qgis.PyQt import uic, QtWidgets

# QGIS GUI tools
from qgis.gui import (
    QgsMapToolEmitPoint,
    QgsRubberBand,
    QgsMapToolCapture,
    QgsMapToolExtent,
    QgsMapToolPan,
)

# QGIS utilities
from qgis.utils import iface

# =============================================================================
# LOCAL MODULE IMPORTS
# =============================================================================
from .modules import (
    map_tools,
    save_utils,
    authentication,
)

# =============================================================================
# UI CONFIGURATION
# =============================================================================

# Load the UI file for the main dialog
ui_file = os.path.join("ui", "aglgis_dialog_base.ui")
FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), ui_file))


# =============================================================================
# MAIN DIALOG CLASS
# =============================================================================

class AGLgisDialog(QDialog, FORM_CLASS):
    """
    Main dialog class for AGLgis plugin.
    
    This class provides a graphical interface for Sentinel-1 SAR Backscatter 
    Analysis Ready Data Preparation using Google Earth Engine. It simplifies
    the process of configuring and running Sentinel-1 SAR data processing
    without requiring manual coding.
    
    Attributes:
        iface: QGIS interface object
        focus_timer: Timer for managing window focus
        Various data processing attributes (aoi, df, collection, etc.)
    """
    
    def __init__(self, parent=None, iface=None):
        """
        Initialize the AGLgis dialog.
        
        Args:
            parent: Parent widget (default: None)
            iface: QGIS interface object (default: None)
        """
        super(AGLgisDialog, self).__init__(parent)
        self.setupUi(self)
        self.iface = iface
        
        # Configure window properties
        self.setWindowFlags(
            Qt.WindowType.Window |
            Qt.WindowType.WindowCloseButtonHint |
            Qt.WindowType.WindowMinimizeButtonHint |
            Qt.WindowType.WindowMaximizeButtonHint
        )
        self.setModal(False)
        
        # Initialize focus management timer
        self.focus_timer = QTimer()
        self.focus_timer.timeout.connect(self.check_focus)

        # Load authentication settings
        authentication.loadProjectId(self)

        # Initialize instance variables
        self.inicialize_variables()

        # Setup UI components and signal connections
        self.setup_ui()
        self.connect_signals()

        # Set default values and initial state
        self.last_clicked(3)  # Default to 3 months
        self.load_intro()
        self.tabWidget.setCurrentIndex(0)
        
        # Initialize window size
        QTimer.singleShot(0, lambda: self.resizeEvent("small"))

    def load_intro(self):
        """Load the introduction HTML content into the text browser."""
        intro_path = os.path.join(os.path.dirname(__file__), "ui", "intro.html")
        with open(intro_path, "r", encoding="utf-8") as f:
            html_content = f.read()
        self.QTextBrowser.setHtml(html_content)

    # =========================================================================
    # WINDOW EVENT HANDLERS
    # =========================================================================
    
    def showEvent(self, event):
        """
        Handle dialog show event.
        
        Starts the focus timer and ensures proper window sizing.
        
        Args:
            event: The show event object
        """
        super().showEvent(event)
        
        # Ensure window size is locked when first shown
        if not hasattr(self, '_size_locked'):
            self.resizeEvent("small")
            self._size_locked = True
            
        # Start focus management timer
        if hasattr(self, 'focus_timer'):
            self.focus_timer.start(100)

    def hideEvent(self, event):
        """
        Handle dialog hide event.
        
        Stops the focus timer to prevent unnecessary processing.
        
        Args:
            event: The hide event object
        """
        super().hideEvent(event)
        if hasattr(self, 'focus_timer'):
            try:
                self.focus_timer.stop()
            except RuntimeError:
                # Timer might already be stopped or deleted
                pass

    def check_focus(self):
        """
        Check if the plugin window should be brought to front.
        
        This method implements a less aggressive focus management strategy
        that only raises the window when appropriate, avoiding interference
        with internal QGIS events.
        """
        if not self.isVisible() or self.isMinimized():
            return
            
        # Only act when clicking on map canvas specifically
        active_window = QApplication.activeWindow()
        if (active_window == self.iface.mainWindow() and 
            not QApplication.activeModalWidget() and
            not self.isActiveWindow()):
            
            # Only raise window, don't activate to avoid event interference
            self.raise_()

    def closeEvent(self, event):
        """
        Handle dialog close event.
        
        Instead of actually closing the dialog, this method hides it to
        preserve the current state and settings. This allows the user
        to reopen the dialog with all their previous work intact.
        
        Args:
            event: The close event object
        """
        # Stop the focus timer to prevent memory leaks
        if hasattr(self, 'focus_timer'):
            try:
                self.focus_timer.stop()
            except RuntimeError:
                # Timer might already be stopped
                pass
        
        # Hide the dialog instead of closing to preserve state
        self.hide()
        event.ignore()

    # =========================================================================
    # INITIALIZATION METHODS
    # =========================================================================

    def inicialize_variables(self):
        """
        Initialize all instance variables to their default values.
        
        This method sets up all the necessary attributes for data processing,
        UI state management, and Earth Engine operations.
        """
        # Plotting and visualization
        self.plot1 = None
        
        # Authentication and settings
        self.autentication = False
        self.folder_set = False
        
        # Date range variables
        self.inicio = None
        self.final = None
        self.nuvem = None
        
        # Vector and geometry data
        self.vector_path = None
        self.aoi = None  # Area of Interest (Earth Engine geometry)
        self.aoi_checked = False
        self.selected_aoi_layer_path = None
        
        # Data processing variables
        self.df = None  # Main dataframe for time series
        self.df_aux = None  # Auxiliary dataframe for filtered data
        self.recorte_datas = None  # Date filtering
        self.selected_dates = []
        self.collection = None  # Earth Engine image collection
        
        # File system
        self.output_folder = None

    def setup_ui(self):
        """
        Perform initial UI component configuration.
        
        This method configures UI elements with appropriate settings,
        permissions, and default values.
        """
        # Configure text browser properties
        self.QTextBrowser.setReadOnly(True)
        self.QTextBrowser.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        
        # Configure project ID input field
        self.project_QgsPasswordLineEdit.setEchoMode(QtWidgets.QLineEdit.EchoMode.Normal)
        
        # Populate year combo box with available years (Sentinel-1 data starts from 2017)
        current_year = datetime.datetime.now().year
        available_years = [str(year) for year in range(2017, current_year + 1)]
        self.combo_year.addItems(available_years)

    def connect_signals(self):
        """
        Connect UI signals to their respective handler methods.
        
        This method establishes all the signal-slot connections for the dialog,
        organizing them by functional areas for better maintainability.
        """
        # Authentication signals
        self.autenticacao.clicked.connect(lambda: authentication.auth(self))
        self.desautenticacao.clicked.connect(lambda: authentication.auth_clear(self))
        self.project_QgsPasswordLineEdit.textChanged.connect(
            lambda new_text: authentication.autoSaveProjectId(self, new_text)
        )
        
        # Vector layer management signals
        self.update_vector.clicked.connect(self.update_vector_clicked)
        self.update_vector_2.clicked.connect(self.update_vector_clicked)
        self.vector_layer_combobox.currentIndexChanged.connect(self.get_selected_layer_path)
        self.vector_layer_combobox_2.currentIndexChanged.connect(self.combobox_2_update)
        
        # Tab navigation signals
        self.tabWidget.currentChanged.connect(self.on_tab_changed)
        
        # Image loading signals
        self.load_1index.clicked.connect(self.load_image)
        self.load_1index_preview.clicked.connect(lambda: self.load_image(True))
        self.load_batch.clicked.connect(self.load_batch_clicked)
        
        # Navigation button signals
        self.QPushButton_next.clicked.connect(self.next_clicked)
        self.QPushButton_next_2.clicked.connect(self.next_clicked)
        self.QPushButton_next_4.clicked.connect(self.next_clicked)
        self.QPushButton_back.clicked.connect(self.back_clicked)
        self.QPushButton_back_2.clicked.connect(self.back_clicked)
        self.QPushButton_back_4.clicked.connect(self.back_clicked)
        self.QPushButton_back_7.clicked.connect(self.back_clicked)
        self.QPushButton_skip.clicked.connect(lambda: self.tabWidget.setCurrentIndex(4))
        
        # Data processing signals
        self.loadtimeseries.clicked.connect(self.loadtimeseries_clicked)
        self.datasrecorte.clicked.connect(self.datasrecorte_clicked)
        self.salvar.clicked.connect(self.salvar_clicked)
        self.QTextBrowser.anchorClicked.connect(self.open_link)
        
        # UI interaction signals
        self.navegador.clicked.connect(self.open_browser)
        self.drawing.stateChanged.connect(self.drawing_clicked)
        self.mQgsFileWidget.fileChanged.connect(self.on_file_changed)
        self.hybrid.clicked.connect(map_tools.hybrid_function)

        # Date range selection signals
        self.radioButton_all.clicked.connect(self.all_clicked)
        self.radioButton_3months.clicked.connect(lambda: self.last_clicked(3))
        self.radioButton_6months.clicked.connect(lambda: self.last_clicked(6))
        self.radioButton_12months.clicked.connect(lambda: self.last_clicked(12))
        self.radioButton_3years.clicked.connect(lambda: self.last_clicked(3 * 12))
        self.radioButton_5years.clicked.connect(lambda: self.last_clicked(5 * 12))
        self.combo_year.currentIndexChanged.connect(self.selected_year_clicked)
        
        # Date input synchronization signals
        self.incioedit.dateChanged.connect(self.reload_update)
        self.finaledit.dateChanged.connect(self.reload_update)


    def load_batch_clicked (self):

        length = self.dataunica.count()
        print(f"Number of dates in the collection: {length}")
        for i in range(length):
            self.dataunica.setCurrentIndex(i)
            print(f"Loading image for date: {self.dataunica.currentText()}")
            self.load_image(preview=False)  # Load each image in download mode


    # =========================================================================
    # IMAGE LOADING AND PROCESSING
    # =========================================================================


    def load_image(self, preview=False):
        """
        Load a Sentinel-1 image based on the selected date.
        
        This method downloads and loads a single Sentinel-1 image from the
        processed collection for the selected date, allowing users to visualize
        individual images from their time series.
        
        Args:
            preview (bool): If True, saves to temporary directory. 
                          If False, saves to user-specified output folder.
        """
        print("Loading image...")
        
        # Get selected date and prepare date range
        date = self.dataunica.currentText()
        next_date = (datetime.datetime.strptime(date, "%Y-%m-%d") + 
                    datetime.timedelta(days=1)).strftime("%Y-%m-%d")
        
        # Extract single image from collection
        selected_image = (self.collection
                         .filterDate(date, next_date)
                         .first()
                         .select(['VV', 'VH', 'VVVH_ratio'])
                         .clip(self.aoi))

        mask = ee.Image(1).clip(self.aoi).mask()

        # 2. Apply this mask to the final composite image.
        # This operation sets pixels outside the mask to NoData (transparent).
        final_image_masked = selected_image.updateMask(mask)

        # Prepare download URL for the clipped image
        url = final_image_masked.getDownloadURL({
            "scale": 10,
            "region": self.aoi.geometry().bounds().getInfo(),
            "format": "GeoTIFF",
            "crs": "EPSG:4326",
        })
        
        # Generate unique filename
        base_output_file = f"Sentinel1_{date}.tiff"
        output_file = self.get_unique_filename(base_output_file, temporary=preview)

        # Download the image
        response = requests.get(url)
        if response.status_code == 200:
            with open(output_file, "wb") as f:
                f.write(response.content)
            print(f"Sentinel1 image downloaded as {output_file}")
        else:
            print(f"Failed to download image. HTTP Status: {response.status_code}")
            return

        # Generate unique layer name
        layer_name = f"Sentinel1 {date}"
        base_name = layer_name
        counter = 1
        while QgsProject.instance().mapLayersByName(layer_name):
            layer_name = f"{base_name}_{counter}"
            counter += 1
        print(f"Layer name adjusted to '{layer_name}' to ensure uniqueness.")

        # Load the downloaded image as a raster layer
        raster_layer = QgsRasterLayer(output_file, layer_name, "gdal")
        if not raster_layer.isValid():
            print(f"Failed to load raster layer from {output_file}")
            return
        
        # Configure raster layer properties
        raster_layer.setCrs(QgsCoordinateReferenceSystem("EPSG:4326"))
        QgsProject.instance().addMapLayer(raster_layer)
        print(f"Raster layer '{layer_name}' added to the project.")

    # =========================================================================
    # UI UPDATE AND SYNCHRONIZATION METHODS
    # =========================================================================

    def combobox_2_update(self):
        """Synchronize the second vector layer combobox with the first one."""
        print("combobox_2_update called")
        self.vector_layer_combobox.setCurrentIndex(
            self.vector_layer_combobox_2.currentIndex()
        )

    def reload_update(self):
        """Synchronize date inputs between different tabs."""
        self.finaledit_2.setDate(self.finaledit.date())
        self.incioedit_2.setDate(self.incioedit.date())

    def open_link(self, url):
        """
        Open a clicked link in the default web browser.
        
        Args:
            url: The URL object to open
        """
        print(f"Opening URL: {url.toString()}")
        webbrowser.open(url.toString())

    # =========================================================================
    # DATE RANGE SELECTION METHODS
    # =========================================================================

    def last_clicked(self, months):
        """
        Set date range to the last N months from today.
        
        Args:
            months (int): Number of months to go back from today
        """
        today = datetime.datetime.today().strftime("%Y-%m-%d")
        start_date = (datetime.datetime.today() - 
                     relativedelta(months=months)).strftime("%Y-%m-%d")
        
        self.finaledit.setDate(QDate.fromString(today, "yyyy-MM-dd"))
        self.incioedit.setDate(QDate.fromString(start_date, "yyyy-MM-dd"))

    def all_clicked(self):
        """Set date range to cover all available Sentinel data (since 2014)."""
        today = datetime.datetime.today().strftime("%Y-%m-%d")
        since = "2014-04-13"  # Sentinel-1A launch date
        self.finaledit.setDate(QDate.fromString(today, "yyyy-MM-dd"))
        self.incioedit.setDate(QDate.fromString(since, "yyyy-MM-dd"))

    def selected_year_clicked(self):
        """
        Set the date range to the selected year from the combo box.
        
        This method updates both start and end dates to cover the entire
        selected year (January 1st to December 31st).
        """
        year = self.combo_year.currentText()
        start_date = f"{year}-01-01"
        end_date = f"{year}-12-31"
        self.incioedit.setDate(QDate.fromString(start_date, "yyyy-MM-dd"))
        self.finaledit.setDate(QDate.fromString(end_date, "yyyy-MM-dd"))

    # =========================================================================
    # VECTOR DRAWING AND GEOMETRY METHODS
    # =========================================================================

    def drawing_clicked(self):
        """
        Handle the drawing checkbox state change.
        
        When the checkbox is checked, activates the vector builder tool.
        When unchecked, deactivates the drawing tool and returns to pan mode.
        """
        print("Drawing clicked")
        if self.drawing.isChecked():
            self.vector_builder()
        else:
            # Deactivate the extent tool and return to pan mode
            iface.mapCanvas().setMapTool(QgsMapToolPan(iface.mapCanvas()))

    def vector_builder(self):
        """
        Initialize the vector building process for drawing AOI.
        
        This method checks if an output folder is selected and then
        activates the extent drawing tool for creating vector layers.
        """
        if self.output_folder is None:
            self.pop_warning("Please select an output folder first.")
            return

        # Activate the extent drawing tool
        self.extent_tool = QgsMapToolExtent(iface.mapCanvas())
        self.extent_tool.extentChanged.connect(self.process_extent)
        iface.mapCanvas().setMapTool(self.extent_tool)
        print("Extent tool activated.")

    def process_extent(self, extent: QgsRectangle):
        """
        Process the drawn extent and save it as a vector layer.
        
        This method takes a drawn rectangle extent, transforms it to EPSG:4326,
        creates a vector layer, saves it to disk, and loads it into the project.
        
        Args:
            extent (QgsRectangle): The drawn extent rectangle
        """
        # Transform the extent to EPSG:4326 for Earth Engine compatibility
        canvas_crs = iface.mapCanvas().mapSettings().destinationCrs()
        target_crs = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(canvas_crs, target_crs, QgsProject.instance())
        extent_4326 = transform.transformBoundingBox(extent)

        # Create a temporary vector layer in memory
        layer = QgsVectorLayer("Polygon?crs=EPSG:4326", "drawn_extent", "memory")
        pr = layer.dataProvider()

        # Add ID field to the layer
        pr.addAttributes([QgsField("id", QVariant.Int)])
        layer.updateFields()

        # Create a feature with the transformed extent geometry
        feature = QgsFeature()
        geometry = QgsGeometry.fromRect(extent_4326)
        feature.setGeometry(geometry)
        feature.setAttributes([1])
        pr.addFeature(feature)

        # Update layer extents
        layer.updateExtents()

        # Generate unique filename and directory structure
        shp_path, shp_name = self.get_subdirectory_filename("drawn_extent")
        print(f"Shapefile path: {shp_path}")
        print(f"Shapefile name: {shp_name}")

        # Configure save options for ESRI Shapefile
        save_options = QgsVectorFileWriter.SaveVectorOptions()
        save_options.driverName = "ESRI Shapefile"
        save_options.fileEncoding = "UTF-8"

        # Write the layer to disk
        QgsVectorFileWriter.writeAsVectorFormat(layer, shp_path, save_options)

        # Load the saved shapefile back into the project
        loaded_layer = QgsVectorLayer(shp_path, shp_name, "ogr")
        if loaded_layer.isValid():
            # Verify and handle CRS if needed
            if loaded_layer.crs().authid() != "EPSG:4326":
                print("Warning: Layer CRS is not EPSG:4326, reprojecting...")
                # Reproject to ensure EPSG:4326
                params = {
                    'INPUT': loaded_layer,
                    'TARGET_CRS': target_crs,
                    'OUTPUT': 'memory:'
                }
                reprojected = processing.run("native:reprojectlayer", params)['OUTPUT']
                loaded_layer = reprojected

            # Add the layer to the project
            QgsProject.instance().addMapLayer(loaded_layer)
            print(f"Layer added successfully with CRS: {loaded_layer.crs().authid()}")
            
            # Update UI components and process the new layer
            self.load_vector_layers()
            self.get_selected_layer_path()
            self.load_vector_function()
            self.find_area()

    # =========================================================================
    # DATA EXPORT AND SAVING METHODS
    # =========================================================================

    def salvar_clicked(self):
        """
        Handle the save button click event.
        
        This method exports the processed time series data to a CSV file
        using the configured output folder and file naming conventions.
        """
        df = self.df_aux
        name = "Sentinel1_time_series.csv"
        save_utils.save(df, name, self)

    # =========================================================================
    # DATE SELECTION DIALOG METHODS
    # =========================================================================

    def datasrecorte_clicked(self):
        """
        Open a dialog for selecting specific dates for the time series.
        
        This method creates a hierarchical date selection dialog organized
        by years and months, allowing users to filter which dates to include
        in their time series analysis.
        """
        dialog = QDialog(self)
        dialog.setWindowTitle("Date Selection for Time Series")
        dialog.setGeometry(100, 100, 400, 500)

        layout = QVBoxLayout(dialog)

        # Create scrollable area for date checkboxes
        scroll_area = QScrollArea(dialog)
        scroll_area.setWidgetResizable(True)
        scroll_content = QWidget()
        scroll_layout = QVBoxLayout(scroll_content)

        # Initialize checkbox storage containers
        self.checkboxes = []
        self.group_checkboxes = {}  # Month group checkboxes
        self.year_checkboxes = {}   # Year group checkboxes
        self.group_widgets = {}     # Month group content widgets

        # Prepare date data for grouping
        self.df["dates"] = pd.to_datetime(self.df["dates"])
        grouped = self.df.groupby([self.df["dates"].dt.year, self.df["dates"].dt.month])

        # Process each year
        years = self.df["dates"].dt.year.unique()
        for year in sorted(years):
            # Create a year-level widget / Cria um widget de nível de ano
            year_widget = QWidget(dialog)
            year_layout = QVBoxLayout(year_widget)

            # Year-level checkbox (above all content for the year) / Checkbox
            # de nível de ano (acima de todo o conteúdo para o ano)
            year_checkbox = QCheckBox(f"Select All in {year}", dialog)
            year_checkbox.setChecked(
                True
                if self.recorte_datas is None
                else all(
                    str(date.date()) in self.recorte_datas
                    for date in self.df[self.df["dates"].dt.year == year]["dates"]
                )
            )
            year_checkbox.stateChanged.connect(
                lambda state, yr=year: self.toggle_year_checkboxes(yr, state)
            )
            scroll_layout.addWidget(year_checkbox)
            self.year_checkboxes[year] = year_checkbox

            # Indented content for the year / Conteúdo indentado para o ano
            year_content_widget = QWidget(dialog)
            year_content_layout = QVBoxLayout(year_content_widget)
            year_content_layout.setContentsMargins(
                20, 0, 0, 0
            )  # Add indentation for year content
            scroll_layout.addWidget(year_content_widget)

            # Add months under each year / Adiciona meses sob cada ano
            for (group_year, month), group in grouped:
                if group_year != year:
                    continue

                # Create a month-level widget / Cria um widget de nível de mês
                group_label = f"{group_year}-{month:02d}"
                month_widget = QWidget(dialog)
                month_layout = QVBoxLayout(month_widget)

                # Month toggle button / Botão de alternância do mês
                month_toggle_button = QToolButton(dialog)
                month_toggle_button.setText(f"▶ {group_label}")
                month_toggle_button.setCheckable(True)
                month_toggle_button.setChecked(True)
                month_toggle_button.setStyleSheet("text-align: left;")
                month_toggle_button.toggled.connect(
                    lambda checked, grp=month_widget, btn=month_toggle_button, lbl=group_label: self.toggle_group_visibility(
                        grp, btn, lbl
                    )
                )

                # Month-level checkbox / Checkbox de nível de mês
                group_checkbox = QCheckBox(f"Select All in {group_label}", dialog)
                group_checkbox.setChecked(
                    True
                    if self.recorte_datas is None
                    else all(
                        str(date.date()) in self.recorte_datas for date in group["dates"]
                    )
                )
                group_checkbox.stateChanged.connect(
                    lambda state, grp=group_label: self.toggle_group_checkboxes(
                        grp, state
                    )
                )
                month_layout.addWidget(group_checkbox)
                self.group_checkboxes[group_label] = group_checkbox

                # Add individual checkboxes with further indentation / Adiciona
                # checkboxes individuais com mais indentação
                for date in group["dates"]:
                    date_str = str(date.date())
                    checkbox = QCheckBox(date_str, dialog)
                    checkbox.setChecked(
                        True
                        if self.recorte_datas is None
                        else date_str in self.recorte_datas
                    )
                    month_layout.addWidget(checkbox)
                    checkbox.setContentsMargins(
                        20, 0, 0, 0
                    )  # Further indentation for dates
                    self.checkboxes.append((checkbox, group_label, group_year))

                # Add month layout to the year content layout / Adiciona o layout
                # do mês ao layout do conteúdo do ano
                year_content_layout.addWidget(month_toggle_button)
                year_content_layout.addWidget(month_widget)
                self.group_widgets[group_label] = month_widget

        scroll_content.setLayout(scroll_layout)
        scroll_area.setWidget(scroll_content)
        layout.addWidget(scroll_area)

        # Buttons / Botões
        # Use QDialogButtonBox.StandardButton for Qt6 compatibility
        button_box = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, dialog
        )
        button_layout = QHBoxLayout()
        apply_button = QPushButton("Apply", dialog)
        select_button = QPushButton("Select All", dialog)
        deselect_button = QPushButton("Deselect All", dialog)
        button_layout.addWidget(apply_button)
        button_layout.addWidget(select_button)
        button_layout.addWidget(deselect_button)
        layout.addLayout(button_layout)
        layout.addWidget(button_box)

        # Signal Connections / Conexões de Sinal
        button_box.accepted.connect(dialog.accept)
        button_box.rejected.connect(dialog.reject)
        apply_button.clicked.connect(
            self.apply_changes
        )  # Apply without closing / Aplica sem fechar
        select_button.clicked.connect(self.select_all_checkboxes)
        deselect_button.clicked.connect(self.deselect_all_checkboxes)

        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.apply_changes()  # Ensure changes are applied before closing
        else:
            print("Time series dialog canceled. No changes made.")

    def apply_changes(self):
        """
        Apply changes to the selected dates for time series analysis.

        This method updates the selected dates based on the checked checkboxes
        in the date selection dialog and refreshes the time series plot accordingly.
        It does not close the dialog, allowing users to see the immediate effects
        of their selections.
        """
        self.selected_dates = [
            cb.text() for cb, _, _ in self.checkboxes if cb.isChecked()
        ]
        self.recorte_datas = self.selected_dates
        print(f"Selected dates for time series (applied): {self.recorte_datas}")
        
        # Update the auxiliary dataframe and refresh the plot
        self.df_ajust()
        self.plot_timeseries()

    def df_ajust(self):
        """
        Adjust the main DataFrame based on the selected dates.
        
        This method filters the main dataframe to include only the dates
        selected by the user in the date selection dialog. The filtered
        data is stored in df_aux for further processing and visualization.
        """
        df = self.df.copy()
        if self.recorte_datas:
            # Filter dataframe to include only selected dates
            df = df[df["dates"].astype(str).isin([str(d) for d in self.recorte_datas])]
            self.df_aux = df.copy()
        else:
            # If no date filtering, use all data
            self.df_aux = df.copy()

    def toggle_group_visibility(self, group_widget, toggle_button, group_label):
        """
        Toggle the visibility of a month group widget.

        This method shows or hides a month group's content and updates
        the toggle button's appearance accordingly.

        Args:
            group_widget (QWidget): The widget representing the month group
            toggle_button (QToolButton): The button used to toggle visibility
            group_label (str): The label of the group (YYYY-MM format)
        """
        group_widget.setVisible(toggle_button.isChecked())
        toggle_button.setText(
            f"▶ {group_label}" if not toggle_button.isChecked() 
            else f"▼ {group_label}"
        )

    def toggle_group_checkboxes(self, group_label, state):
        """
        Toggle all checkboxes within a specific month group.

        Args:
            group_label (str): The label of the month group (YYYY-MM format)
            state (int): The checkbox state (Qt.Checked or Qt.Unchecked)
        """
        for checkbox, group, _ in self.checkboxes:
            if group == group_label:
                checkbox.setChecked(state == Qt.Checked)

    def toggle_year_checkboxes(self, year, state):
        """
        Toggle all checkboxes within a specific year group.

        This method affects all individual date checkboxes and month group
        checkboxes for the specified year.

        Args:
            year (int): The year to toggle
            state (int): The checkbox state (Qt.Checked or Qt.Unchecked)
        """
        # Toggle individual date checkboxes for the specified year
        for checkbox, _, group_year in self.checkboxes:
            if group_year == year:
                checkbox.setChecked(state == Qt.Checked)
        
        # Toggle month group checkboxes for the specified year
        for group_label, group_checkbox in self.group_checkboxes.items():
            if group_label.startswith(str(year)):
                group_checkbox.setChecked(state == Qt.Checked)

    def select_all_checkboxes(self):
        """Select all checkboxes in the date selection dialog."""
        for checkbox, _, _ in self.checkboxes:
            checkbox.setChecked(True)
        for group_checkbox in self.group_checkboxes.values():
            group_checkbox.setChecked(True)
        for year_checkbox in self.year_checkboxes.values():
            year_checkbox.setChecked(True)

    def deselect_all_checkboxes(self):
        """Deselect all checkboxes in the date selection dialog."""
        for checkbox, _, _ in self.checkboxes:
            checkbox.setChecked(False)
        for group_checkbox in self.group_checkboxes.values():
            group_checkbox.setChecked(False)
        for year_checkbox in self.year_checkboxes.values():
            year_checkbox.setChecked(False)

    # =========================================================================
    # WINDOW POSITIONING AND RESIZING METHODS
    # =========================================================================

    def centralizar(self):
        """
        Center the window on the current screen.

        This method calculates the geometry of the window frame and moves it to
        the center of the available screen space on the current screen without
        changing which screen the dialog is displayed on.
        """
        # Get the current geometry of the window frame
        qtRectangle = self.frameGeometry()
        
        # Determine the center point of the available screen space
        # Uses QApplication.primaryScreen() as replacement for deprecated QDesktopWidget
        screen = QApplication.screenAt(self.frameGeometry().center())
        if screen is None:
            screen = QApplication.primaryScreen()
        centerPoint = screen.availableGeometry().center()
        
        # Move the center of the window frame to the center point of the screen
        qtRectangle.moveCenter(centerPoint)
        # Move the window to the new position
        self.move(qtRectangle.topLeft())

    def resizeEvent(self, size):
        """
        Handle window resize events with predefined size configurations.
        
        This method manages two window size modes: "small" for basic operations
        and "big" for data visualization and analysis. It locks the window
        size to prevent manual resizing.
        
        Args:
            size (str): Either "small" or "big" to set the window size
        """
        # Remove any existing size constraints
        self.setMinimumSize(0, 0)
        self.setMaximumSize(16777215, 16777215)

        if size == "small":
            # Small window for basic operations (tabs 0-4)
            self.resize(765, 371)
            self.setFixedSize(self.width(), self.height())
        elif size == "big":
            # Large window for data visualization (tabs 5+)
            self.resize(1145, 620)
            self.setFixedSize(self.width(), self.height())

    # =========================================================================
    # TAB NAVIGATION AND CONTROL METHODS
    # =========================================================================

    def on_tab_changed(self, index):
        """
        Handle tab change events with comprehensive validation and setup.
        
        This method manages tab transitions, enforces step-by-step workflow,
        handles authentication checks, and performs tab-specific initialization.
        
        Args:
            index (int): The index of the newly selected tab
        """
        # Prevent recursion during programmatic tab changes
        if getattr(self, '_programmatic_tab_change', False):
            return

        def _safe_set_current_index(target_index):
            """Safely change tab index without triggering recursive events."""
            if not hasattr(self, 'tabWidget'):
                return False

            if self.tabWidget.currentIndex() != target_index:
                self._programmatic_tab_change = True
                try:
                    self.tabWidget.setCurrentIndex(target_index)
                finally:
                    self._programmatic_tab_change = False
                return True
            return False

        # Authentication check - must be authenticated to access tabs beyond 0
        if not getattr(self, 'autentication', False) and index != 0:
            QMessageBox.warning(self, "Access Denied", "Please authenticate first!")
            _safe_set_current_index(0)
            return

        # Handle window resizing based on tab
        if index < 5:
            self.resizeEvent("small")
        else:
            self.resizeEvent("big")
            # Center window only once for large tabs
            if not getattr(self, '_centralized', False):
                self.centralizar()
                self._centralized = True

        # Tab-specific initialization
        if index == 1 and not getattr(self, 'path_suggestion_loaded', False):
            # Tab 1: Load output folder suggestion
            try:
                self.load_last_output_folder()
            except Exception as e:
                print(f"Error loading last output folder: {e}. Using suggestion.")
                self.load_path_sugestion()
            finally:
                self.path_suggestion_loaded = True

        elif index == 2 and not getattr(self, 'vector_layers_loaded', False):
            # Tab 2: Load vector layers
            try:
                self.load_vector_layers()
                self.get_selected_layer_path()
                self.vector_layers_loaded = True
            except Exception as e:
                print(f"Error loading vector layers: {e}")
                # Keep vector_layers_loaded as False so it retries

        # Progression checks - enforce step-by-step workflow
        if index > 1:
            next_button_4 = getattr(self, 'QPushButton_next_4', None)
            if next_button_4 and not next_button_4.isEnabled():
                QMessageBox.warning(self, "Proceed Step-by-Step", "Please complete Step 2 before proceeding.")
                _safe_set_current_index(1)
                return

        if index > 2:
            next_button = getattr(self, 'QPushButton_next', None)
            if next_button and not next_button.isEnabled():
                QMessageBox.warning(self, "Proceed Step-by-Step", "Please complete Step 3 before proceeding.")
                _safe_set_current_index(2)
                self.resizeEvent("small")
                return

        # Disable coordinate capture when not on tab 2
        if index != 2:
            drawing_checkbox = getattr(self, 'drawing', None)
            if drawing_checkbox:
                try:
                    drawing_checkbox.setChecked(False)
                except Exception as e:
                    print(f"Error unchecking coordinate capture checkbox: {e}")

    # =========================================================================
    # NAVIGATION BUTTON METHODS
    # =========================================================================

    def next_clicked(self):
        """Navigate to the next tab in the tab widget."""
        current_index = self.tabWidget.currentIndex()
        next_index = (current_index + 1) % self.tabWidget.count()
        self.tabWidget.setCurrentIndex(next_index)

    def back_clicked(self):
        """Navigate to the previous tab in the tab widget."""
        current_index = self.tabWidget.currentIndex()
        previous_index = (current_index - 1) % self.tabWidget.count()
        self.tabWidget.setCurrentIndex(previous_index)

    # =========================================================================
    # FILE AND FOLDER MANAGEMENT METHODS
    # =========================================================================

    def load_path_sugestion(self):
        """
        Load output folder suggestion based on the user's operating system.
        
        Sets the default output folder to the user's Downloads directory
        across different operating systems (Windows, Linux, macOS).
        """
        system = platform.system()
        if system == "Windows":
            self.output_folder = os.path.join(os.environ["USERPROFILE"], "Downloads")
        elif system == "Linux":
            self.output_folder = os.path.join(os.environ["HOME"], "Downloads")
        elif system == "Darwin":  # macOS
            self.output_folder = os.path.join(os.environ["HOME"], "Downloads")
        else:
            # Fallback for unknown systems
            self.output_folder = os.path.expanduser("~/Downloads")

        # Set the suggested directory in the file widget
        self.mQgsFileWidget.setFilePath(self.output_folder)

    def pop_warning(self, message):
        """
        Display a warning dialog with the specified message.
        
        Args:
            message (str): The warning message to display
        """
        QApplication.restoreOverrideCursor()
        msg = QMessageBox(self)
        msg.setWindowTitle("Warning!")
        msg.setIcon(QMessageBox.Icon.Warning)
        msg.setText(message)
        msg.setStandardButtons(QMessageBox.StandardButton.Ok)
        msg.button(QMessageBox.StandardButton.Ok).setText("OK")
        msg.setStyleSheet("font-size: 10pt;")
        msg.exec()

    # =========================================================================
    # VECTOR LAYER MANAGEMENT METHODS
    # =========================================================================

    def update_vector_clicked(self):
        """
        Update vector layers and reload related functions.
        
        This method refreshes the vector layer list, updates the selected
        layer path, and reloads vector-related functionality.
        """
        self.load_vector_layers()
        self.get_selected_layer_path()
        self.load_vector_function()

    def load_vector_layers(self):
        """
        Load and populate the combo box with available vector layers from the current QGIS project.
        
        This method retrieves all vector layers from the current QGIS project,
        filters them by geometry type, and populates the layer selection combo box.
        """
        # Get all layers from the current QGIS project
        layers = list(QgsProject.instance().mapLayers().values())

        # Filter for polygon and multipolygon vector layers only
        vector_layers = [
            layer
            for layer in layers
            if layer.type() == QgsMapLayer.VectorLayer
            and layer.geometryType() == QgsWkbTypes.PolygonGeometry
        ]

        # Get current layer names for comparison
        current_layer_names = set(
            self.vector_layer_combobox.itemText(i)
            for i in range(self.vector_layer_combobox.count())
        )

        # Clear combo boxes and layer ID dictionary
        self.vector_layer_combobox.clear()
        self.vector_layer_ids = {}

        # Populate combo box and track new layers
        new_layer_name = None
        for layer in vector_layers:
            layer_name = layer.name()
            self.vector_layer_combobox.addItem(layer_name)
            self.vector_layer_ids[layer_name] = layer.id()

            # Check if this is a newly added layer
            if layer_name not in current_layer_names:
                new_layer_name = layer_name

        # Synchronize the second combo box
        self.vector_layer_combobox_2.clear()
        for i in range(self.vector_layer_combobox.count()):
            self.vector_layer_combobox_2.addItem(
                self.vector_layer_combobox.itemText(i)
            )
        self.vector_layer_combobox_2.setCurrentIndex(
            self.vector_layer_combobox.currentIndex()
        )

        # If a new layer was found, select it automatically
        if new_layer_name:
            index = self.vector_layer_combobox.findText(new_layer_name)
            if index >= 0:
                self.vector_layer_combobox.setCurrentIndex(index)

        # Handle case when no vector layers are available
        if self.vector_layer_combobox.count() == 0:
            self.aoi = None
            self.tabWidget.setCurrentIndex(2)
            self.pop_warning(
                "No vector layers found in the project. Please add a polygon or multipolygon layer to continue."
            )

    def get_selected_layer_path(self):
        """
        Retrieve the path of the currently selected layer and trigger processing.
        
        This method gets the selected layer from the combo box, validates it,
        and initiates further processing including zooming to the layer and
        calculating its area.
        
        Returns:
            str or None: The layer path if successful, None if no layer selected
        """
        # Get the currently selected layer name from the combo box
        layer_name = self.vector_layer_combobox.currentText().strip()
        
        if not layer_name:
            print("No layer selected.")
            self.aoi_area.setText("Total Area:")
            self.QPushButton_next.setEnabled(False)
            self.QPushButton_skip.setEnabled(False)
            return None
            
        print(f"Processing layer: '{layer_name}'")
        self.zoom_to_layer(layer_name)

        # Get the corresponding layer ID
        layer_id = self.vector_layer_ids.get(layer_name)
        print(f"Layer ID: {layer_id}")

        if layer_id is None:
            print(f"Error: Layer ID not found for '{layer_name}'")
            print(f"Available layers: {list(self.vector_layer_ids.keys())}")
            return None

        # Get the layer using its ID
        layer = QgsProject.instance().mapLayer(layer_id)
        if layer:
            print(f"Layer found: {layer.name()}, ID: {layer_id}")
            # Extract the data source path from the layer
            self.selected_aoi_layer_path = (
                layer.dataProvider().dataSourceUri().split("|")[0]
            )
            print(f"Selected layer path: {self.selected_aoi_layer_path}")

            # Process the layer and update UI
            self.aoi = self.load_vector_function()
            area = self.find_area()
            self.QPushButton_next.setEnabled(True)
            self.QPushButton_skip.setEnabled(True)
            self.loadtimeseries.setEnabled(True)

            return self.selected_aoi_layer_path
        else:
            print(f"Layer '{layer_name}' with ID '{layer_id}' not found in project")
            return None

    def zoom_to_layer(self, layer_name, margin_ratio=0.3):
        """
        Zoom the map canvas to the extent of the specified layer with optional margin.
        
        Args:
            layer_name (str): Name of the layer to zoom to
            margin_ratio (float): Fraction of the extent to add as margin (default 0.3 or 30%)
        """
        project = QgsProject.instance()
        layers = project.mapLayersByName(layer_name)

        if not layers:
            print(f"Layer '{layer_name}' not found.")
            return

        layer = layers[0]  # Use the first matching layer
        iface = qgis.utils.iface
        canvas = iface.mapCanvas()

        # Ensure the canvas CRS matches the layer CRS
        canvas.setDestinationCrs(layer.crs())

        # Get the layer's extent and add margin
        layer_extent = layer.extent()
        x_margin = layer_extent.width() * margin_ratio
        y_margin = layer_extent.height() * margin_ratio

        expanded_extent = QgsRectangle(
            layer_extent.xMinimum() - x_margin,
            layer_extent.yMinimum() - y_margin,
            layer_extent.xMaximum() + x_margin,
            layer_extent.yMaximum() + y_margin,
        )

        # Set the expanded extent and refresh canvas
        canvas.setExtent(expanded_extent)
        canvas.refresh()

        print(f"Zoomed to layer with margin: {expanded_extent.toString()}")

    # =========================================================================
    # FILE MANAGEMENT UTILITY METHODS
    # =========================================================================

    def get_subdirectory_filename(self, base_name, temporary=False):
        """
        Create a unique layer name with incrementing number and matching subdirectory.
        
        Args:
            base_name (str): Base name for the layer and directory
            temporary (bool): If True, use temporary directory instead of output folder
            
        Returns:
            str: Path to the unique filename
        """
        if temporary:
            base_output_folder = tempfile.gettempdir()
        else:
            base_output_folder = self.output_folder
        
        # Find a unique layer name with incrementing number
        counter = 1
        layer_name = f"{base_name}{counter}"
        subdirectory_path = os.path.join(base_output_folder, layer_name)
        
        # Keep incrementing until we find an unused name
        while os.path.exists(subdirectory_path):
            counter += 1
            layer_name = f"{base_name}{counter}"
            subdirectory_path = os.path.join(base_output_folder, layer_name)
        
        # Create the directory if it doesn't exist
        while os.path.exists(subdirectory_path):
            counter += 1
            layer_name = f"{base_name}{counter}"
            subdirectory_path = os.path.join(base_output_folder, layer_name)
        
        os.makedirs(subdirectory_path)
        
        # Create the filename with the same base name
        file_name = f"{layer_name}.shp"
        file_path = os.path.join(subdirectory_path, file_name)
        
        print(f"Unique layer name: {layer_name}")
        print(f"Unique filename: {file_path}")
        
        return file_path, layer_name

    def get_unique_filename(self, base_file_name, temporary=False):
        """
        Generate a unique filename by appending a counter if the file already exists.
        
        Args:
            base_file_name (str): The base filename to make unique
            temporary (bool): If True, use temporary directory instead of output folder
            
        Returns:
            str: Path to a unique filename
        """
        name, extension = os.path.splitext(base_file_name)
        
        if temporary:
            output_folder = tempfile.gettempdir()
            output_file = os.path.join(output_folder, base_file_name)
        else:
            output_folder = self.output_folder
            output_file = os.path.join(output_folder, base_file_name)
            
        counter = 1
        while os.path.exists(output_file):
            output_file = os.path.join(
                output_folder, f"{name}_{counter}{extension}"
            )
            counter += 1

        print(f"Unique filename: {output_file}")
        return output_file

    def on_file_changed(self, file_path):
        """
        Handle file selection changes in the file widget.
        
        This method is called when the user selects a new file/folder path
        and updates the output folder accordingly.
        
        Args:
            file_path (str): The selected file path
        """
        if self.mQgsFileWidget.filePath():
            print(f"File selected: {file_path}")
            self.output_folder = file_path
            self.folder_set = True
            self.QPushButton_next_4.setEnabled(True)
            
            # Save the selected file path to QGIS settings for persistence
            QSettings().setValue("aglgis_plugin/last_output_folder", file_path)
            print(f"Last output folder saved: {file_path}")
        else:
            print("No file selected.")
            self.folder_set = False
            self.QPushButton_next_4.setEnabled(False)

    def load_last_output_folder(self):
        """
        Load the last selected output folder from QGIS settings.
        
        This method restores the previously used output folder to provide
        continuity between plugin sessions.
        """
        last_folder = QSettings().value("aglgis_plugin/last_output_folder", "")
        if last_folder:
            self.mQgsFileWidget.setFilePath(last_folder)
            self.output_folder = last_folder
            self.folder_set = True
            self.QPushButton_next_4.setEnabled(True)
            print(f"Last output folder loaded: {last_folder}")
        else:
            print("No previously selected output folder found.")

    # =========================================================================
    # EARTH ENGINE INTEGRATION METHODS
    # =========================================================================

    def load_vector_function(self, shapefile_path=None):
        print("Loading vector function...")


        shapefile_path = self.selected_aoi_layer_path
        print(f"Shapefile path: {shapefile_path}")

        if shapefile_path is None:
            return None

        gpd_aoi = None # Use a distinct name for the geopandas DataFrame

        if shapefile_path.endswith(".zip"):
            with zipfile.ZipFile(shapefile_path, "r") as zip_ref:
                shapefile_within_zip = None
                for file in zip_ref.namelist():
                    if file.lower().endswith(".shp"): # Use .lower() for case-insensitivity
                        shapefile_within_zip = file
                        break
                if not shapefile_within_zip:
                    return None

                # Read shapefile directly from the zip archive.
                gpd_aoi = gpd.read_file(f"zip://{shapefile_path}/{shapefile_within_zip}")
        else:
            gpd_aoi = gpd.read_file(shapefile_path)

        # Reproject the GeoDataFrame to EPSG:4326 to ensure correct
        # coordinates for Earth Engine.
        gpd_aoi = gpd_aoi.to_crs(epsg=4326)

        if gpd_aoi.empty:
            raise ValueError(f"The shapefile at {shapefile_path} does not contain any geometries.")

        # Dissolve multiple features into a single geometry if necessary.
        if len(gpd_aoi) > 1:
            gpd_aoi = gpd_aoi.dissolve()

        # Extract the first geometry.
        # It's safer to ensure there's at least one geometry before iloc[0]
        if gpd_aoi.empty: # Re-check after dissolve if it became empty (unlikely but robust)
            raise ValueError(f"The shapefile at {shapefile_path} became empty after dissolve.")

        geometry = gpd_aoi.geometry.iloc[0]

        # Convert the geometry to GeoJSON format.
        geojson = geometry.__geo_interface__

        # Remove any third dimension from the coordinates.
        # Note: GeoJSON can support 3D, but Earth Engine often expects 2D.
        # This part of the logic is robust.
        if geojson["type"] == "Polygon":
            geojson["coordinates"] = [
                list(map(lambda coord: coord[:2], ring)) for ring in geojson["coordinates"]
            ]
        elif geojson["type"] == "MultiPolygon":
            geojson["coordinates"] = [
                [list(map(lambda coord: coord[:2], ring)) for ring in polygon]
                for polygon in geojson["coordinates"]
            ]

        # Create an Earth Engine geometry object.
        ee_geometry = ee.Geometry(geojson)
        feature = ee.Feature(ee_geometry)
        ee_feature_collection = ee.FeatureCollection([feature])

        print("AOI defined successfully.")
        
        # Update UI elements
        self.QPushButton_next.setEnabled(True)
        self.QPushButton_skip.setEnabled(True)
        self.aoi_set = True
        self.vector_layer_combobox_2.setCurrentIndex(
            self.vector_layer_combobox.currentIndex()
        )
        self.aoi_checked = True

        return ee_feature_collection


    def find_centroid(self):
        """
        Calculate and return the centroid coordinates and area of the AOI.
        
        Returns:
            float: Area of the AOI in square kilometers
        """
        centroid = self.aoi.geometry().centroid()
        coordinates = centroid.getInfo().get("coordinates")
        self.lat = coordinates[1]
        self.lon = coordinates[0]
        print(f"Centroid: {round(self.lat, 4)}, {round(self.lon, 4)}")
        
        # Calculate area in square kilometers
        area = self.aoi.geometry().area().getInfo() / 1e6
        print(f"Area: {area:.2f} km²")
        return area

    def find_area(self):
        """
        Calculate and display the area of the AOI in multiple units.
        
        This method calculates the area in square kilometers and hectares,
        updates the UI label, and returns the area value.
        
        Returns:
            float: Area of the AOI in square kilometers
        """
        try:
            # Convert from square meters to square kilometers
            area_km2 = self.aoi.geometry().area().getInfo() / 1e6
            # Convert from square kilometers to hectares
            area_ha = area_km2 * 100
            
            print(f"Area: {area_km2:.2f} km² ({area_ha:.2f} ha)")
            self.aoi_area.setText(
                f"Total Area: {area_km2:.2f} km² ({area_ha:.2f} hectares)"
            )
            return area_km2
        except Exception as e:
            print(f"Error in find_area: {e}")
            self.aoi_area.setText("Total Area:")
            return 0

    def aoi_checked_function(self):
        """
        Check if both AOI and folder are set to enable progression buttons.
        
        This method validates that both the area of interest is checked and
        the output folder is set before enabling the next/skip buttons.
        """
        if self.aoi_checked and self.folder_set:
            self.QPushButton_next.setEnabled(True)
            self.QPushButton_skip.setEnabled(True)
        else:
            self.QPushButton_next.setEnabled(False)
            self.QPushButton_skip.setEnabled(False)

    def resetting(self):
        """
        Reset processing variables to their initial state.
        
        This method clears previously processed data and reloads the AOI
        to prepare for a new analysis run.
        """
        self.recorte_datas = None
        self.aoi = self.load_vector_function()
        self.inicio = self.incioedit.date().toString("yyyy-MM-dd")
        self.final = self.finaledit.date().toString("yyyy-MM-dd")
        self.df = None
        self.collection = None
        self.fig = None
        self.df_aux = None

    # =========================================================================
    # TIME SERIES PROCESSING METHODS
    # =========================================================================

    def loadtimeseries_clicked(self):
        """
        Handle the load time series button click event.
        
        This method initiates the complete time series processing workflow,
        including Earth Engine data processing, date loading, and visualization.
        """
        try:
            QApplication.setOverrideCursor(Qt.CursorShape.WaitCursor)
            self.resetting()
            self.collection = self.ee_process()
            self.load_dates()
            self.plot_timeseries()
            self.next_clicked()
        except Exception as e:
            print(f"Error during timeseries loading: {e}")
            self.pop_warning(f"Error loading time series: {str(e)}")
        finally:
            QApplication.restoreOverrideCursor()

    def ee_process(self):

        #self.cheching()
        """
        Process Earth Engine data to create a time series collection.
        
        This method configures and executes the Sentinel-1 ARD processing
        based on user-selected parameters and returns the processed collection.
        
        Returns:
            ee.ImageCollection: Processed Sentinel-1 image collection with time series data
        """
        # Determine orbit direction from user selection
        orbit = self.QComboBox_orbit.currentText() != "DESCENDING"

        # Configure the S1ARD processor with user parameters

        processor = S1ARDImageCollection(
            geometry=self.aoi,
            start_date=self.inicio,
            stop_date=self.final,
            polarization=self.QComboBox_polarization.currentText(),
            apply_border_noise_correction=self.QCheckBox_apply_border_noise_correction.isChecked(),
            apply_terrain_flattening=self.QCheckBox_apply_terrain_flattening.isChecked(),
            apply_speckle_filtering=self.QCheckBox_apply_speckle_filtering.isChecked(),
            output_format=self.QComboBox_output.currentText(),
            ascending=orbit  # False for descending orbit, True for ascending orbit
        )
        # print("AOI:", self.aoi.getInfo())
        # print("start date:", self.inicio)
        # print("end date:", self.final)
        # print("Polarization:", self.QComboBox_polarization.currentText())
        # print("Orbit ascending:", orbit)
        # print("Border noise correction:", self.QCheckBox_apply_border_noise_correction.isChecked())
        # print("Terrain flattening:", self.QCheckBox_apply_terrain_flattening.isChecked())
        # print("Speckle filtering:", self.QCheckBox_apply_speckle_filtering.isChecked())
        # print("Output format:", self.QComboBox_output.currentText())

        collection = processor.get_collection().sort('system:time_start')

        size = collection.size().getInfo()
        print(f"Number of images in collection: {size}")

        if size == 0:
            self.pop_warning("No images found for the selected criteria.")
            return None

        # Add VV/VH ratio band to each image
        def add_vvvh_ratio_band(image):
            ratio = image.select("VV").divide(image.select("VH")).rename("VVVH_ratio")
            return image.addBands(ratio)

        # Map the function to add the VV/VH ratio band
        try:
            collection = collection.map(add_vvvh_ratio_band)
        except Exception as e:
            print(f"Error adding VV/VH ratio band: {e}")
            self.pop_warning(f"Error processing images: {str(e)}")
            return None

        # Extract the VVVH_ratio band and calculate mean values for each image
        def get_vvvh_ratio_mean(image):
            stats = image.select("VVVH_ratio").reduceRegion(
                reducer=ee.Reducer.mean(),
                geometry=self.aoi,
                scale=10,
                maxPixels=1e9
            )
            # Get image date
            date = image.date().format('YYYY-MM-dd')
            return ee.Feature(None, {
                'date': date,
                'VVVH_ratio_mean': stats.get('VVVH_ratio')
            })

        # Map the function over the collection and get the results
        vvvh_ratio_ts = collection.map(get_vvvh_ratio_mean).getInfo()

        # Extract data for DataFrame creation
        data = [
            {'dates': f['properties']['date'], 'AOI_average': f['properties']['VVVH_ratio_mean']}
            for f in vvvh_ratio_ts['features']
        ]

        # Create and store DataFrames
        df = pd.DataFrame(data)
        #print(df)
        self.df = df.copy()
        self.df_aux = df.copy()

        return collection

    def df_adjust(self):
        """
        Adjust the main DataFrame based on selected dates.
        
        This method filters the DataFrame to include only the dates
        that have been selected by the user for analysis.
        """
        df = self.df.copy()
        if self.recorte_datas:
            # Filter DataFrame to include only selected dates
            df = df[df["dates"].astype(str).isin([str(d) for d in self.recorte_datas])]
            self.df_aux = df.copy()
        else:
            self.df_aux = df.copy()

    # =========================================================================
    # VISUALIZATION AND PLOTTING METHODS
    # =========================================================================

    def plot_timeseries(self):
        """
        Plot the time series data using Plotly.
        
        This method creates an interactive time series plot with customized
        configuration and styling for the VV/VH ratio data.
        """
        # Configure Plotly display options
        self.config = {
            "displaylogo": False,
            "modeBarButtonsToRemove": [
                "toImage",
                "sendDataToCloud",
                "zoom2d",
                "pan2d",
                "select2d",
                "lasso2d",
                "zoomIn2d",
                "zoomOut2d",
                "autoScale2d",
                "resetScale2d",
                "hoverClosestCartesian",
                "hoverCompareCartesian",
                "zoom3d",
                "pan3d",
                "orbitRotation",
                "tableRotation",
                "resetCameraLastSave",
                "resetCameraDefault3d",
                "hoverClosest3d",
                "zoomInGeo",
                "zoomOutGeo",
                "resetGeo",
                "hoverClosestGeo",
                "hoverClosestGl2d",
                "hoverClosestPie",
                "toggleHover",
                "toggleSpikelines",
                "resetViews",
            ],
        }

        # Create the time series plot
        self.fig = px.line(
            self.df_aux, 
            x='dates', 
            y='AOI_average', 
            markers=True, 
            title='VV/VH Time Series'
        )
        self.fig.update_layout(
            xaxis_title='Date', 
            yaxis_title='VV/VH Ratio'
        )

        # Render the plot in the web view
        self.QWebView.setHtml(
            self.fig.to_html(include_plotlyjs="cdn", config=self.config)
        )

    def open_browser(self):
        """
        Open the time series plot in the default web browser.
        
        This method exports the current plot to an HTML file and opens it
        in the system's default web browser for better viewing.
        """
        if self.fig:
            temp_file = tempfile.NamedTemporaryFile(
                mode='w', suffix='.html', delete=False
            )
            temp_file.write(
                self.fig.to_html(include_plotlyjs="cdn", config=self.config)
            )
            temp_file.close()
            
            # Open in default browser
            webbrowser.open('file://' + temp_file.name)
        else:
            self.pop_warning("No plot available to open in browser.")

    def load_dates(self):
        """
        Load and populate the date selection interface.
        
        This method extracts unique dates from the processed DataFrame
        and populates the date selection widgets for user interaction.
        """
        if self.df is not None and not self.df.empty:
            # Get unique dates and sort them
            dates = self.df['dates'].unique().astype(str).tolist()
            self.available_dates = sorted(dates)
            
            # Populate the date selection combo box
            self.dataunica.clear()
            self.dataunica.addItems(dates)
            # Set to the most recent date by default
            self.dataunica.setCurrentIndex(self.dataunica.count() - 1)
            
            print(f"Available dates loaded: {len(self.available_dates)} dates")
        else:
            print("No data available for date loading")
            self.available_dates = []
